/*
 * Copyright (C) 2014 Advanced Driver Information Technology GmbH
 * Written by Norbert Uetrecht <nuetrecht@de.adit-jv.com) and
 *          Christoph Gellner <cgellner@de.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 */

#ifndef _CAAM_KM_H
#define _CAAM_KM_H

#include <linux/kernel.h>
#include <linux/caam_key.h>

#define CAAM_KM_DEV_NAME "keyman"

/* IOCTL */
/*
 * Install keystore key ioctl
 * Add a new keystore key to the keystore
 *
 * The following key_flags are fixed
 *      CAAM_KEY_FLAG_TYPE_KEYSTORE
 *
 * The ioctl supports black and red keys.
 * Black keys - CAAM_KEY_FLAG_STOR_BLACK are limited to
 * symmetric keys (CAAM_KEY_FLAG_SECRET_... can be BYTES_PLAIN, BYTES_PARITY).
 * Red keys - CAAM_KEY_FLAG_STOR_RED are limited to
 * asymmetric keys (CAAM_KEY_FLAG_SECRET_... can be RSA_BER_PUB, RSA_BER_PRI).
 *
 * In case CAAM_KEY_FLAG_SUBTYPE_PLAIN is part of spec.key_flags
 * the key secret is specified by user-space.
 * key_secret and key_secret_len must be != NULL/0 in this case.
 * key_secret_len has to match the length as specified by key_bits.
 * CAAM_KEY_FLAG_SECRET_... can be BYTES_PLAIN, BYTES_PARITY,
 *                                 RSA_BER_PUB or RSA_BER_PRI.
 * CAAM_KEY_FLAG_AUTH_... can be UNKNOWN, LOW, MEDIUM.
 * CAAM_KEY_FLAG_CONFIDENT_... can be UNKNOWN, LOW, MEDIUM.
 *
 * In case CAAM_KEY_FLAG_SUBTYPE_RANDOM is part of spec.key_flags
 * the key secret is internally generated using DRNG.
 * key_secret and key_secret_len must be == NULL/0 in this case.
 * CAAM_KEY_FLAG_SECRET_... can be BYTES_PLAIN or BYTES_PARITY.
 * CAAM_KEY_FLAG_AUTH_... can be UNKNOWN, LOW, MEDIUM.
 * CAAM_KEY_FLAG_CONFIDENT_... can be UNKNOWN, LOW, MEDIUM, HIGH
 *
 * Other CAAM_KEY_FLAG_SUBTYPE_... are currently not supported
 *
 * If CAAM_KEY_FLAG_MODIFIER_NON is part of the spec.key_flags
 * spec.modifier == NULL and spec.modifier_bytes == 0
 * If CAAM_KEY_FLAG_MODIFIER_PUBLIC or CAAM_KEY_FLAG_MODIFIER_PRIVATE
 * is part of spec.key_flags spec.modifier != NULL and spec.modifier_bytes != 0
 * Note: CAAM_KEY_FLAG_MODIFIER_PRIVATE is not supported for
 * CAAM_KEY_FLAG_SECRET_RSA_BER_PUB, CAAM_KEY_FLAG_SECRET_RSA_BER_PRI.
 *
 * The userspace can set any value for bits of CAAM_KEY_FLAG_USER_MASK
 *
 * All other bits of spec.key_flags need to be unset.
 *
 * The length of the key is specified using spec.key_bits.
 * spec.uid and spec.gid are used to specify the owner and the group of
 * the key.
 * spec.perms is used to specify the permissions.
 *
 * kid, kid_min, kid_max are used to specify the desired key ID.
 * In order to get a specific key ID all three values need to be set
 * to the desired value.
 * If this is not the case the driver will search for the first
 * available key ID.
 *  - first in the range [kid, kid_max]
 *  - second in the range [kid_min, kid-1]
 * kid_min needs to be less or equal to key_max.
 * The found key ID is written to kid in case the ioctl succeeds.
 *
 * In case key_encap != NULL the driver will create a blob of the key
 * which is encrypted and signed by a key derived from OTPMK using a
 * hash including all information of spec as a modifier.
 * The available length of key_encap is specified by key_encap_bytes.
 * After the ioctl has completed without error key_encap_bytes is
 * set to the amount of bytes used for storing the blob.
 *
 * additional_key_data and additional_key_data_bytes are reserved for
 * future use and will be set to NULL/0 at the moment.
 *
 * Using install_flags it is possible to specify if the key
 * can be used immediately (CAAM_KM_INSTALL_ACTIVATE) or only
 * after an additional call to
 * CAAM_KM_ACTIVATE_KEYSTORE_KEY (CAAM_KM_INSTALL_DONT_ACTIVATE).
 * This is important to ensure the blob is stored securely before
 * any application is able to use the key for any operation.
 *
 * For CAAM_KEY_FLAG_SECRET_RSA_BER_PUB, CAAM_KEY_FLAG_SECRET_RSA_BER_PRI
 * the encoding of the key_secret can be specified
 * using CAAM_KM_INSTALL_ENCODING.
 * In case CAAM_KM_INSTALL_NO_ENC   DER/BER encoding is assumed.
 * In case CAAM_KM_INSTALL_PEM      PEM encoding is assumed.
 * For other CAAM_KEY_FLAG_SECRET_... CAAM_KM_INSTALL_NO_ENC
 * needs to be used.
 *
 * Corresponding data structure : struct caam_km_install_keystore_key
 *
 * Error codes:
 * -EINVAL      : Invalid install_flags.
 *                Invalid combination of spec.key_flags,
 *                spec.key_bits, spec.perms,
 *                spec.modifier, spec.modifier_bytes,
 *                key_secret or key_secret_bytes.
 *                key_encap_bytes is to small to store the key blob.
 * -EINTR       : interrupted while waiting for other install, load to complete
 * -ENOSPC      : The maximum amount of keystore keys is already reached
 * -EXISTS      : All key IDs in the requested range are already used
 * -ENOMEM      : No memory to allocate temporary buffers in kernel
 * -EBUSY       : CAAM queue is full
 * -EIO	        : CAAM internal error
 * -EFAULT      : Accessing ioctl data structure failed
 * -EPERM	: Requested permission exceeds allowed permission
 * -EBADMSG     : The keysecret is not a valid RSA public or private key
 * -EMSGSIZE    : The size of RSA public or private key does not match
 *		  to the number of bits specified by key_bits.
 */
#define CAAM_KM_INSTALL_KEYSTORE_KEY    _IOWR('m', 0,\
					struct caam_km_install_keystore_key)

/*
 * Load keystore key ioctl
 * Load a key blob generated with CAAM_KM_INSTALL_KEYSTORE_KEY
 *
 * This IOCTL is used to restore keystore keys generated
 * using CAAM_KM_INSTALL_KEYSTORE_KEY in a previous power cycle
 *
 * In order to do so CAAM_KM_LOAD_WITH_EXT_SPEC needs to be set in
 * load_flags and kid as well as all elements of spec, key_encap
 * and key_encap_bytes need to be identical to the values used
 * during CAAM_KM_INSTALL_KEYSTORE_KEY.
 *
 * In order to allow loading of key blobs which have been
 * generated by the previous version of this driver
 * CAAM_KM_LOAD_NO_EXT_SPEC can be set in load_flags.
 * In this case only spec.uid, spec.gid as well as kid,
 * key_encap and key_encap_bytes need to be set.
 * The other values are determined internally.
 *
 * These values are used in combination with the OTPMK to
 * decrypt and verify that the key blob or its spec has
 * not been altered.
 *
 * In case load_flags sets CAAM_KM_LOAD_DONT_ACTIVATE
 * an additional call to CAAM_KM_ACTIVATE_KEYSTORE_KEY is
 * required before any application can use the keystore key.
 * In case load_flags sets CAAM_KM_LOAD_ACTIVATE the
 * keystore key can be used immediately.
 *
 * Corresponding data structure : struct caam_km_load_keystore_key
 *
 * Error codes:
 * -EINVAL      : Invalid load_flags.
 *                Invalid combination of spec.key_flags,
 *                spec.key_bits, spec.perms,
 *                spec.modifier or spec.modifier_bytes.
 *                Invalid length of key_encap_bytes.
 * -EINTR       : interrupted while waiting for other install, load to complete
 * -ENOSPC      : The maximum amount of keystore keys is already reached
 * -EXISTS      : key ID is already used
 * -ENOMEM      : No memory to allocate temporary buffers in kernel
 * -EBUSY       : CAAM queue is full
 * -EIO	        : CAAM internal error
 * -EFAULT      : Accessing ioctl data structure failed
 * -EKEYREVOKED : Authentication of key blob failed
 */
#define CAAM_KM_LOAD_KEYSTORE_KEY       _IOW('m', 1,\
					struct caam_km_load_keystore_key)

/*
 * Remove keystore key ioctl
 * Remove a key from the keystore
 *
 * Corresponding data structure : struct caam_km_remove_keystore_key
 *
 * Error codes:
 * -EBUSY       : Key can't be removed -- still used
 * -EACCESS     : UID is not allowed to remove key
 * -ENOKEY      : Key referenced by kid not in keystore
 * -EFAULT      : Accessing ioctl data structure failed
 */
#define CAAM_KM_REL_KEYSTORE_KEY        _IOW('m', 2,\
					struct caam_km_remove_keystore_key)

/*
 * Activate keystore key ioctl
 * Activate a keystore key which has not been activated by
 * CAAM_KM_INSTALL_KEYSTORE_KEY or CAAM_KM_LOAD_KEYSTORE_KEY *
 *
 * Corresponding data structure : struct caam_km_activate_keystore_key
 *
 * Error codes:
 * -ENOKEY      : Key referenced by kid not in keystore
 * -EFAULT      : Accessing ioctl data structure failed
 */
#define CAAM_KM_ACTIVATE_KEYSTORE_KEY   _IOW('m', 3,\
					struct caam_km_activate_keystore_key)

/*
 * Unlock keystore ioctl
 * Until this ioctl is called no keystore key can be used.
 * This is needed to ensure all previous keys are restored before
 * any application can use any key.
 *
 * Corresponding data structure : NONE
 *
 * Error codes:
 * NONE
 */
#define CAAM_KM_UNLOCK_KEYSTORE         _IO('m', 4)

/* this IOCTL must not be changed */
#define CAAM_KM_GET_KERNEL_VERSION_OLD _IOR('l', 0, struct sm_version *)
#define CAAM_KM_GET_KERNEL_VERSION _IOR('l', 0, struct caam_km_version)

/* Defines */
/* Minimal size of key to be used */
#define CAAM_KM_KEY_SIZE_MIN		8

/*
 * define max / min data size for
 * caam_km_install_keystore_key - key_secret
 *
 * 4096 bytes will be sufficient to even handle 4096bit RSA keys
 */
#define CAAM_KM_KEYSECRET_LEN_MAX (4096)
#define CAAM_KM_KEYSECRET_LEN_MIN (CAAM_KM_KEY_SIZE_MIN)

/*
 * A blob consist out of 32bit random key for encrypting the blob key
 * This key is protected by OPTMK
 * 16bit authentication data
 * The blob key itself (variable length)
 */
#define CAAM_KM_BLOB_ADD (32 + 16) /* random key + auth data */

/*
 * define max / min data size for
 * caam_km_install_keystore_key - key_encap and
 * caam_km_load_keystore_key - key_encap
 *
 * 4096 bytes will be sufficient to even handle 4096bit RSA keys
 */
#define CAAM_KM_ENCAP_KEY_SIZE_MAX (4096)
#define CAAM_KM_ENCAP_KEY_SIZE_MIN (CAAM_KM_BLOB_ADD + CAAM_KM_KEY_SIZE_MIN)

/* flags for install_flags of caam_km_install_keystore_key */
#define CAAM_KM_INSTALL_ACTIVATE        0x01
#define CAAM_KM_INSTALL_DONT_ACTIVATE	0x00
#define CAAM_KM_INSTALL_ACTIVATE_MASK	0x01

#define CAAM_KM_INSTALL_ENCODING        0xF0
#define CAAM_KM_INSTALL_NO_ENC		0x00
#define CAAM_KM_INSTALL_PEM		0x10

#define CAAM_KM_INSTALL_ALL_FLAG_MASKS	(CAAM_KM_INSTALL_ACTIVATE_MASK |\
					 CAAM_KM_INSTALL_ENCODING)

/* flags for load_flags of caam_km_load_keystore_key */
#define CAAM_KM_LOAD_WITH_EXT_SPEC	0x02
/*
 * CAAM_KM_LOAD_NO_EXT_SPEC is needed to allow loading key blobs
 * which have been generated by a previous version of caam_km
 */
#define CAAM_KM_LOAD_NO_EXT_SPEC	0x00
#define CAAM_KM_LOAD_EXT_SPEC_MASK      0x02

#define CAAM_KM_LOAD_ACTIVATE           0x01
#define CAAM_KM_LOAD_DONT_ACTIVATE	0x00
#define CAAM_KM_LOAD_ACTIVATE_MASK	0x01

#define CAAM_KM_LOAD_ALL_FLAG_MASKS	(CAAM_KM_LOAD_EXT_SPEC_MASK |\
					 CAAM_KM_LOAD_ACTIVATE_MASK)


/* IOCTL data structures */

/* structure of CAAM_KM_INSTALL_KEY IOCTL */
struct caam_km_install_keystore_key {
	uint32_t  install_flags; /* in */

	uint32_t kid; /* inout */
	uint32_t kid_min; /* in */
	uint32_t kid_max; /* in */

	struct caam_key_spec spec; /* in */

	const uint8_t *key_secret; /* in */
	uint32_t key_secret_bytes; /* in */

	uint8_t *key_encap; /* out */
	uint32_t key_encap_bytes; /* inout */

	uint8_t *additional_key_data; /* out */
	uint32_t additional_key_data_bytes; /* inout */
};

/* structure of CAAM_KM_LOAD_KEYSTORE_KEY IOCTL */
struct caam_km_load_keystore_key {
	uint32_t load_flags; /* in */

	uint32_t kid; /* in */

	struct caam_key_spec spec; /* in */

	const uint8_t *key_encap; /* in */
	uint32_t key_encap_bytes; /* in */
};

/* structure of CAAM_KM_ACTIVATE_KEYSTORE_KEY IOCTL */
struct caam_km_activate_keystore_key {
	uint32_t  kid; /* in */
};

/* structure of CAAM_KM_REL_KEY IOCTL */
struct caam_km_remove_keystore_key {
	uint32_t  kid; /* in */
	uint32_t  uid; /* in - UID trying to remove key */
	uint32_t  gid; /* in - GID trying to remove key */
};



/*
 * The below structure allows the user-space to decide if it is compatible
 * to the current kernel version.
 * In order to do so the user-space needs to know if new IOCTLs
 * have been added, IOCTLs have been removed or data structures have
 * been changed. Last but not least it needs to know if there are any
 * user-space relevant changes regarding the behavior of IOCTLs
 * (e.g. functionality of IOCTL has been implemented, extended)
 *
 * Whenever a completely new set of IOCTLs is provided the version number
 * needs to be incremented. But as the already existing IOCTLs are not
 * touched the ioctl_comp_version can stay the same.
 * subversion is set to 0 in this case
 *
 * In case an already existing IOCTL is changed (with regards to the
 * input parameters or removed) the version gets incremented and
 * ioctl_comp_version is set to the same value.
 * subversion is set to 0 in this case
 *
 * For minor changes. E.g. functionality of IOCTL implemented, extended
 * only subversion is incremented
 */

#define CAAM_KM_CURRENT_VERSION			3
#define CAAM_KM_CURRENT_SUBVERSION		3
#define CAAM_KM_CURRENT_IOCTL_COMP_VERSION	3

struct caam_km_version {
	uint32_t version;
	uint32_t subversion;
	uint32_t ioctl_comp_version;
};


#endif /* _CAAM_KM_H */
