/**
* \file: AditInputSource.h
*
* \version: $Id:$
*
* \release: $Name:$
*
* Input Source Endpoint interfaces
*
* \component: Baidu CarLife
*
* \author: P. Acar / ADIT/ESM / pacar@de.adit-jv.com
*
* \copyright (c) 2017 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#ifndef BDCL_ADITINPUTSOURCE_H
#define BDCL_ADITINPUTSOURCE_H

#include <string>
#include "bdclErrorCodes.h"

namespace adit { namespace bdcl {

class CoreCallbackDealer;

/** @interface  IAditInputSourceCallbacks
 *  @brief      Callbacks to notify the MC application. This interface must be implemented by the MC application using
 *              ADIT Input Source implementation.
 */
class IAditInputSourceCallbacks
{
public:
    virtual ~IAditInputSourceCallbacks() { }

    /** @brief Callback initiated by ADIT to inform MC application about an error.
     */
    virtual void onError(bdclErrorCodes inErrorCode) { (void) inErrorCode; };
};

/** @class      AditInputSource
 *  @brief      Abstract input source interface to be used by the MC. MC has to implement a class of a concrete
 *              implementation of AditInputSource, but use AditInputSource interfaces for easier portability.
 */
class AditInputSource
{
public:
    // shared ptr, no document
    // after the teardown is called, it has to be ensured that these callbacks won't be triggered anymore.
    // So MC is free to destroy them, but MC has to guarantee that the IAditInputSourceCallbacks object is alive before
    // they call the teardown.

    /** @brief AditInputSource constructor
     *  @param inCallbacks: object implementing the callback interface. MC has to ensure that IAditInputSourceCallbacks
     *         object is alive until they call teardown.
     *  @param inCallbackDealer: dispatcher responsible for callbacks transmitted by Baidu core.
     */
    AditInputSource(IAditInputSourceCallbacks* inCallbacks, CoreCallbackDealer* inCallbackDealer)
    {
        (void)inCallbacks;
        (void)inCallbackDealer;
    }

    virtual ~AditInputSource() { };

    /** @brief Configuration interface for each parameter
     *  @param inKey: name of the configuration parameter
     *  @param inValue: the actual value for configuration
     */
    virtual void setConfigItem(std::string inKey, std::string inValue) = 0;

    /** @brief Method to initialize input endpoint
     *  @return true on success
     *          false on failure
     */
    virtual bool initialize() = 0;
    /** @brief Method to shutdown input endpoint. After teardown call, MC is free to destroy
     *         IAditInputSourceCallbacks object
     */
    virtual void teardown() = 0;
};

} } /* namespace adit { namespace bdcl { */

#endif /* BDCL_ADITINPUTSOURCE_H */
