/**
* \file: AditAudioSink.h
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: Android Auto
*
* \author: I. Hayashi / ADITJ/SW / ihayashi@jp.adit-jv.com
* 
* \copyright (c) 2015 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#ifndef AAUTO_ADITAUDIOSINK_H
#define AAUTO_ADITAUDIOSINK_H

#include <aauto/AudioSink.h>
#include <aauto/MessageRouter.h>
#include "aauto/AautoErrorCodes.h"

namespace adit { namespace aauto {

/**
 * @brief callback function to receive notification from MD.
 * All callbacks have empty default implementations.
 * If necessary, need to be implemented.
 * */
 
class IAditAudioSinkCallbacks
{
public:
    virtual ~IAditAudioSinkCallbacks() {}

    /** @brief Called when playback is going to start.
    * The audio environment must be fully prepared to accept audio frames once this method returns. 
    * @param sessionId  The sessionId is a unique id different from each playback.
    * */
    virtual void playbackStartCallback(int32_t sessionId) {};

    /** @brief Called when playback of this audio session has ended. 
    * This is invoked after outputting all data from GalReceiver to audio environment.
    * Therefore, audio data are not left in AditAudioSink.
    * @param sessionId  The sessionId when playback stops. This is the same as the sessionId of playbacStarCallback.
    * */
    virtual void playbackStopCallback(int32_t sessionId) {};

    /** @brief Called when an error happens in the underlying layer.
    * The callback is introduced to inform the upper layer about error cases,
    * for which another notification, such as return values, is not possible.
    * @param inErrorCode  Error code
    * */
    virtual void notifyErrorCallback(aautoErrorCodes inErrorCode) {(void) inErrorCode;}
};

/** @brief AditAudioSink provides a generic base class to audio sink implementations provided by ADIT.
 *  It hides some AudioSink methods and replaces them by setConfigItem.
 */
class AditAudioSink : public AudioSink
{
public:
    /** @brief Constructor of AditAudioSink.
     * @param inSessionId      inSessionId is used in GalReceiver as identification data on endpoint session.
     * @param inMessageRouter  inMessageRouter is message router obtained from GalReceiver.
     * */
    AditAudioSink(uint8_t inSessionId, MessageRouter* inMessageRouter)
        : AudioSink(inSessionId, inMessageRouter) {}

    /** @brief Set configuration for platform and AudioSink-related items. Must be called before init().
     * @param  inKey        The key string for configuration (e.g. "audio-sink-sampling-rate")
     *  - [AudioSink Configuration]
     *  - audio-sink-max-unacked-frames  [optional, default=4096] : The maximum number of frames that will be kept outstanding.
     *                                                              Once this limit is hit, 
     *                                                              the other end will no longer send frames.
     *  - audio-sink-stream-type [mandatory] :  Stream type. Can be one of AUDIO_STREAM_GUIDANCE, AUDIO_STREAM_SYSTEM_AUDIO,
     *                                          AUDIO_STREAM_MEDIA, AUDIO_STREAM_TELEPHONY.
     *                                          [AUDIO_STREAM_TELEPHONY is not supported in current GalReciver]
     *  - audio-sink-stream-codec [mandatory] : Codec type. Can be one of MEDIA_CODEC_AUDIO_PCM, MEDIA_CODEC_AUDIO_AAC_LC,
     *                                          MEDIA_CODEC_AUDIO_AAC_LC_ADTS
     *                                          [current implementatin (AIL environment) is only supported MEDIA_CODEC_AUDIO_PCM]
     *  - audio-sink-sampling-rate [optional, default=48000] :  The sampling rate. choose 16000 or 48000.
     *  - audio-sink-bits-per-sample [optional, default=16] :   Number of bits per sample. current GalReciver is fixed 16.
     *  - audio-sink-channels [optional, default=2] :           Number of channels. choose 1 or 2.
     *    (Setting combination refer to "Integrating Audio" chapter of "AA_HUIGv1_0_1.pdf" document.)
     *  - enable-verbose-logging [optional, default=0] : Enable flag in order to get detail log data. 
     *                                                   0:disable. 1:enable.
     *  - disable-real-time-priority-audio [optional, default=0] : Configuration to manually disable real-time priority
     *                                                             in audio threads. 0:enable. 1:disable.
     *  - audio-threads-real-time-priority [optional, default=61] : Real-time thread priority. set value is thread priority.
     * @param  inValue      The actual value for configuration (e.g. "48000")
     * */
    virtual void setConfigItem(string inKey, string inValue) = 0;

    /** @brief Register callback interface to receive AudioSink-related notifications.
     *  See IAditAudioSinkCallbacks for details.
     *
     * @param   inCallbacks object implementing the callback interface
     * */
    virtual void registerCallbacks(IAditAudioSinkCallbacks* inCallbacks) = 0;

    /** @brief Initialize the audio sink. All configuration items must be set.
     *
     * @return true if initialization succeeds.
     *         false if configuration items without default value is not set or initialization
     *          fails.
     * */
    virtual bool init() = 0;

    /**
     * @brief  Shut down the audio sink, stop running threads and release allocated resources */
    virtual void shutdown() = 0;

protected:
    /** @brief Method removed from public interface. Use setConfigItem instead. */
    uint32_t addSupportedConfiguration(uint32_t samplingRate, uint32_t numberOfBits, uint32_t numberOfChannels);
    /** @brief Method removed from public interface. Use setConfigItem instead. */
    void setMaxUnackedFrames(int maxUnackedFrames);
    /** @brief Method removed from public interface. Use setConfigItem instead. */
    void setAudioType(int audioType);
    /** @brief Method removed from public interface. Use setConfigItem instead. */
    void setCodecType(int codecType);
    /** @brief Method removed from public interface. */
    bool resetConfigurationPreference(uint32_t *configIndices, int size);
};

} } /* namespace adit { namespace aauto { */

#endif /* AAUTO_ADITAUDIOSINK_H */
