/*
 * (c) 2015 Advanced Driver Information Technology GmbH
 *          Christoph Gellner <cgellner@de.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 */

#ifndef _CAAM_KEY_H
#define _CAAM_KEY_H

#include <linux/kernel.h>

/*
 * definition of key length encoding
 * limited to 32 values due to type of key_len_bmsk
 */
#define CAAM_KEY_LEN_64BIT_IDX	0
#define CAAM_KEY_LEN_128BIT_IDX	1
#define CAAM_KEY_LEN_192BIT_IDX	2
#define CAAM_KEY_LEN_256BIT_IDX	3
#define CAAM_KEY_LEN_56BIT_IDX	4
#define CAAM_KEY_LEN_112BIT_IDX	5
#define CAAM_KEY_LEN_168BIT_IDX	6
#define CAAM_KEY_LEN_1024BIT_IDX	7
#define CAAM_KEY_LEN_2048BIT_IDX	8
#define CAAM_KEY_LEN_4096BIT_IDX	9

/* values as used in descriptors */
#define CAAM_KEY_LEN_64BIT_BMSK	\
	(1 << CAAM_KEY_LEN_64BIT_IDX)
#define CAAM_KEY_LEN_128BIT_BMSK	\
	(1 << CAAM_KEY_LEN_128BIT_IDX)
#define CAAM_KEY_LEN_192BIT_BMSK	\
	(1 << CAAM_KEY_LEN_192BIT_IDX)
#define CAAM_KEY_LEN_256BIT_BMSK	\
	(1 << CAAM_KEY_LEN_256BIT_IDX)
#define CAAM_KEY_LEN_56BIT_BMSK	\
	(1 << CAAM_KEY_LEN_56BIT_IDX)
#define CAAM_KEY_LEN_112BIT_BMSK \
	(1 << CAAM_KEY_LEN_112BIT_IDX)
#define CAAM_KEY_LEN_168BIT_BMSK \
	(1 << CAAM_KEY_LEN_168BIT_IDX)
#define CAAM_KEY_LEN_1024BIT_BMSK \
	(1 << CAAM_KEY_LEN_1024BIT_IDX)
#define CAAM_KEY_LEN_2048BIT_BMSK \
	(1 << CAAM_KEY_LEN_2048BIT_IDX)
#define CAAM_KEY_LEN_4096BIT_BMSK \
	(1 << CAAM_KEY_LEN_4096BIT_IDX)

/*
 * definition of key secret type encoding
 * limited to 31 real values due to type of key_secret_bmsk
 * IDX 0 is regarded as invalid
 */
#define CAAM_KEY_SECRET_BYTES_PLAIN_IDX	  1
#define CAAM_KEY_SECRET_BYTES_PARITY_IDX  2
#define CAAM_KEY_SECRET_RSA_BER_PUB_IDX  3
#define CAAM_KEY_SECRET_RSA_BER_PRI_IDX  4

#define CAAM_KEY_SECRET_IDX_TO_BMSK(idx) \
	(1 << (idx))

#define CAAM_KEY_SECRET_BYTES_PLAIN_BMSK \
	CAAM_KEY_SECRET_IDX_TO_BMSK(CAAM_KEY_SECRET_BYTES_PLAIN_IDX)
#define CAAM_KEY_SECRET_BYTES_PARITY_BMSK \
	CAAM_KEY_SECRET_IDX_TO_BMSK(CAAM_KEY_SECRET_BYTES_PARITY_IDX)
#define CAAM_KEY_SECRET_RSA_BER_PUB_BMSK \
	CAAM_KEY_SECRET_IDX_TO_BMSK(CAAM_KEY_SECRET_RSA_BER_PUB_IDX)
#define CAAM_KEY_SECRET_RSA_BER_PRI_BMSK \
	CAAM_KEY_SECRET_IDX_TO_BMSK(CAAM_KEY_SECRET_RSA_BER_PRI_IDX)

#define CAAM_KEY_SECRET_ALL_BMSK (CAAM_KEY_SECRET_BYTES_PLAIN_BMSK |\
				  CAAM_KEY_SECRET_BYTES_PARITY_BMSK |\
				  CAAM_KEY_SECRET_RSA_BER_PUB_BMSK |\
				  CAAM_KEY_SECRET_RSA_BER_PRI_BMSK)

#define CAAM_KEY_SECRET_RSA_BER_BMSKS (CAAM_KEY_SECRET_RSA_BER_PUB_BMSK |\
				       CAAM_KEY_SECRET_RSA_BER_PRI_BMSK)

/* key length description */
struct caam_key_len_desc {
	uint32_t bits;
	uint32_t bytes_inc_padding; /* bytes including padding */
	uint32_t key_len_bmsk; /* corresponding key_len bmsk */
	uint32_t key_secret_bmsk; /* supported key secret types */
};

static const struct caam_key_len_desc caam_key_len_descs[] = {
	/* CAAM_KEY_LEN_64BIT_IDX */
	{64, 8, CAAM_KEY_LEN_64BIT_BMSK, CAAM_KEY_SECRET_BYTES_PLAIN_BMSK},
	/* CAAM_KEY_LEN_128BIT_IDX */
	{128, 16, CAAM_KEY_LEN_128BIT_BMSK, CAAM_KEY_SECRET_BYTES_PLAIN_BMSK},
	/* CAAM_KEY_LEN_192BIT_IDX */
	{192, 24, CAAM_KEY_LEN_192BIT_BMSK, CAAM_KEY_SECRET_BYTES_PLAIN_BMSK},
	/* CAAM_KEY_LEN_256BIT_IDX */
	{256, 32, CAAM_KEY_LEN_256BIT_BMSK, CAAM_KEY_SECRET_BYTES_PLAIN_BMSK},
	/* CAAM_KEY_LEN_56BIT_IDX */
	{56, 8, CAAM_KEY_LEN_56BIT_BMSK, CAAM_KEY_SECRET_BYTES_PARITY_BMSK},
	/* CAAM_KEY_LEN_112BIT_IDX */
	{112, 16, CAAM_KEY_LEN_112BIT_BMSK, CAAM_KEY_SECRET_BYTES_PARITY_BMSK},
	/* CAAM_KEY_LEN_168BIT_IDX */
	{168, 24, CAAM_KEY_LEN_168BIT_BMSK, CAAM_KEY_SECRET_BYTES_PARITY_BMSK},
	/* CAAM_KEY_LEN_1024BIT_IDX */
	{1024, 128, CAAM_KEY_LEN_1024BIT_BMSK, CAAM_KEY_SECRET_RSA_BER_BMSKS},
	/* CAAM_KEY_LEN_2048BIT_IDX */
	{2048, 256, CAAM_KEY_LEN_2048BIT_BMSK, CAAM_KEY_SECRET_RSA_BER_BMSKS},
	/* CAAM_KEY_LEN_4096BIT_IDX */
	{4096, 512, CAAM_KEY_LEN_4096BIT_BMSK, CAAM_KEY_SECRET_RSA_BER_BMSKS},
};

/* storage flags */
#define CAAM_KEY_FLAG_STOR_SHIFT        0
#define CAAM_KEY_FLAG_STOR_MASK         0x0007
#define CAAM_KEY_FLAG_STOR_INVALID      (0 << CAAM_KEY_FLAG_STOR_SHIFT)
/* key is stored as black blob / black key (always encrypted) */
#define CAAM_KEY_FLAG_STOR_BLACK        (1 << CAAM_KEY_FLAG_STOR_SHIFT)
/* key is stored as red blob / red key (red keys are unencrypted in memory) */
#define CAAM_KEY_FLAG_STOR_RED          (2 << CAAM_KEY_FLAG_STOR_SHIFT)

/* key type flags */
#define CAAM_KEY_FLAG_TYPE_SHIFT        3
#define CAAM_KEY_FLAG_TYPE_MASK	        0x0038
#define CAAM_KEY_FLAG_TYPE_INVALID      (0 << CAAM_KEY_FLAG_TYPE_SHIFT)
#define CAAM_KEY_FLAG_TYPE_KEYSTORE     (1 << CAAM_KEY_FLAG_TYPE_SHIFT)
#define CAAM_KEY_FLAG_TYPE_BUILTIN      (2 << CAAM_KEY_FLAG_TYPE_SHIFT)

/*
 * key subtype flags
 * not all combinations are valid
 */
#define CAAM_KEY_FLAG_SUBTYPE_SHIFT     6
#define CAAM_KEY_FLAG_SUBTYPE_MASK      0x03C0
#define CAAM_KEY_FLAG_SUBTYPE_INVALID   (0 << CAAM_KEY_FLAG_SUBTYPE_SHIFT)
/*
 * keysecret has been created from plain keysecret provided by userspace
 * valid for CAAM_KEY_FLAG_TYPE_KEYSTORE
 */
#define CAAM_KEY_FLAG_SUBTYPE_PLAIN     (1 << CAAM_KEY_FLAG_SUBTYPE_SHIFT)
/*
 * keysecret has been created from internally generated random number
 * valid for CAAM_KEY_FLAG_TYPE_KEYSTORE
 */
#define CAAM_KEY_FLAG_SUBTYPE_RANDOM    (2 << CAAM_KEY_FLAG_SUBTYPE_SHIFT)
/*
 * keysecret has been imported from an encrypted data buffer
 * valid for CAAM_KEY_FLAG_TYPE_KEYSTORE
 */
#define CAAM_KEY_FLAG_SUBTYPE_IMPORTED	(3 << CAAM_KEY_FLAG_SUBTYPE_SHIFT)
/*
 * keysecret has been derived from the OTPMK
 * valid for CAAM_KEY_FLAG_TYPE_BUILTIN
 */
#define CAAM_KEY_FLAG_SUBTYPE_DERIVED	(4 << CAAM_KEY_FLAG_SUBTYPE_SHIFT)

/*
 * keysecret type flags
 * format of the keysecret - e.g. plain stream of bytes
 */
#define CAAM_KEY_FLAG_SECRET_SHIFT      10
#define CAAM_KEY_FLAG_SECRET_MASK       0x7C00
/* keysecret is marked invalid */
#define CAAM_KEY_FLAG_SECRET_INVALID \
	(0 << CAAM_KEY_FLAG_SECRET_SHIFT)

#define CAAM_KEY_SECRET_IDX_TO_FLAG(idx) \
	((idx) << CAAM_KEY_FLAG_SECRET_SHIFT)
#define CAAM_KEY_SECRET_FLAG_TO_IDX(flag) \
	(((flag) & CAAM_KEY_FLAG_SECRET_MASK) >> CAAM_KEY_FLAG_SECRET_SHIFT)

/* plain stream of bytes */
#define CAAM_KEY_FLAG_SECRET_BYTES_PLAIN \
	CAAM_KEY_SECRET_IDX_TO_FLAG(CAAM_KEY_SECRET_BYTES_PLAIN_IDX)
/* same as above but with odd parity for each byte */
#define CAAM_KEY_FLAG_SECRET_BYTES_PARITY \
	CAAM_KEY_SECRET_IDX_TO_FLAG(CAAM_KEY_SECRET_BYTES_PARITY_IDX)
/* RSA private key - BER data */
#define CAAM_KEY_FLAG_SECRET_RSA_BER_PRI \
	CAAM_KEY_SECRET_IDX_TO_FLAG(CAAM_KEY_SECRET_RSA_BER_PRI_IDX)
/* RSA private key - BER data */
#define CAAM_KEY_FLAG_SECRET_RSA_BER_PUB \
	CAAM_KEY_SECRET_IDX_TO_FLAG(CAAM_KEY_SECRET_RSA_BER_PUB_IDX)

/*
 * authenticity of keysecret
 */
#define CAAM_KEY_FLAG_AUTH_SHIFT        18
#define CAAM_KEY_FLAG_AUTH_MASK         0xC0000
#define CAAM_KEY_FLAG_AUTH_UNKNOWN \
	(0 << CAAM_KEY_FLAG_AUTH_SHIFT)
#define CAAM_KEY_FLAG_AUTH_LOW \
	(1 << CAAM_KEY_FLAG_AUTH_SHIFT)
#define CAAM_KEY_FLAG_AUTH_MEDIUM \
	(2 << CAAM_KEY_FLAG_AUTH_SHIFT)
#define CAAM_KEY_FLAG_AUTH_HIGH \
	(3 << CAAM_KEY_FLAG_AUTH_SHIFT)

/*
 * confidentiality of keysecret
 */
#define CAAM_KEY_FLAG_CONFIDENT_SHIFT    22
#define CAAM_KEY_FLAG_CONFIDENT_MASK     0xC00000
#define CAAM_KEY_FLAG_CONFIDENT_UNKNOWN \
	(0 << CAAM_KEY_FLAG_CONFIDENT_SHIFT)
#define CAAM_KEY_FLAG_CONFIDENT_LOW \
	(1 << CAAM_KEY_FLAG_CONFIDENT_SHIFT)
#define CAAM_KEY_FLAG_CONFIDENT_MEDIUM \
	(2 << CAAM_KEY_FLAG_CONFIDENT_SHIFT)
#define CAAM_KEY_FLAG_CONFIDENT_HIGH \
	(3 << CAAM_KEY_FLAG_CONFIDENT_SHIFT)

/*
 * modifier spec flags
 * specify if the modifier is used and if it contains secret data
 */
#define CAAM_KEY_FLAG_MODIFIER_SHIFT      24
#define CAAM_KEY_FLAG_MODIFIER_MASK       0x3000000
#define CAAM_KEY_FLAG_MODIFIER_NON \
	(0 << CAAM_KEY_FLAG_MODIFIER_SHIFT)
#define CAAM_KEY_FLAG_MODIFIER_PUBLIC \
	(1 << CAAM_KEY_FLAG_MODIFIER_SHIFT)
#define CAAM_KEY_FLAG_MODIFIER_PRIVATE \
	(2 << CAAM_KEY_FLAG_MODIFIER_SHIFT)


/*
 * the upper 4bits of the flag are available to be defined by userspace
 * These flags will be included into the modifier when computing
 * blobs for storage or builtin keys
 */
#define CAAM_KEY_FLAG_USER_SHIFT      28
#define CAAM_KEY_FLAG_USER_MASK       0xF0000000

/* data permissions */
#define CAAM_PERM_DATA_ENCRYPT_IDX	0
#define CAAM_PERM_DATA_DECRYPT_IDX	1
#define CAAM_PERM_DATA_SIGN_IDX		2
#define CAAM_PERM_DATA_VERIFY_IDX	3
#define CAAM_PERM_DATA_WRAP_IDX		4
#define CAAM_PERM_DATA_UNWRAP_IDX	5

/* allow to wrap a key-secret */
#define CAAM_PERM_KEY_WRAP_IDX		8
/*
 * allow to unwrap and use the data as key
 * in contrast to CAAM_PERM_DATA_UNWRAP_IDX
 * the content will be kept in kernel-space
 * and can't be written to user-space
 */
#define CAAM_PERM_KEY_UNWRAP_IDX	9
/* remove a key from keystore */
#define CAAM_PERM_KEY_RELEASE_IDX	10

#define CAAM_PERM_DATA_ENCRYPT_BMSK \
	(1 << CAAM_PERM_DATA_ENCRYPT_IDX)
#define CAAM_PERM_DATA_DECRYPT_BMSK \
	(1 << CAAM_PERM_DATA_DECRYPT_IDX)
#define CAAM_PERM_DATA_SIGN_BMSK \
	(1 << CAAM_PERM_DATA_SIGN_IDX)
#define CAAM_PERM_DATA_VERIFY_BMSK \
	(1 << CAAM_PERM_DATA_VERIFY_IDX)
#define CAAM_PERM_DATA_WRAP_BMSK \
	(1 << CAAM_PERM_DATA_WRAP_IDX)
#define CAAM_PERM_DATA_UNWRAP_BMSK \
	(1 << CAAM_PERM_DATA_UNWRAP_IDX)
#define CAAM_PERM_KEY_WRAP_BMSK \
	(1 << CAAM_PERM_KEY_WRAP_IDX)
#define CAAM_PERM_KEY_UNWRAP_BMSK \
	(1 << CAAM_PERM_KEY_UNWRAP_IDX)
#define CAAM_PERM_KEY_RELEASE_BMSK \
	(1 << CAAM_PERM_KEY_RELEASE_IDX)

#define CAAM_PERM_ALL_DATA_OPS (CAAM_PERM_DATA_ENCRYPT_BMSK |\
		CAAM_PERM_DATA_DECRYPT_BMSK |\
		CAAM_PERM_DATA_SIGN_BMSK |\
		CAAM_PERM_DATA_VERIFY_BMSK |\
		CAAM_PERM_DATA_WRAP_BMSK |\
		CAAM_PERM_DATA_UNWRAP_BMSK)
#define CAAM_PERM_ALL_KEY_OPS (CAAM_PERM_KEY_RELEASE_BMSK |\
		CAAM_PERM_KEY_WRAP_BMSK |\
		CAAM_PERM_KEY_UNWRAP_BMSK)

#define CAAM_PERM_NON 0

/* default permissions */
#define CAAM_PERM_DEFAULT_OTHERS (CAAM_PERM_NON)
#define CAAM_PERM_DEFAULT_GID (CAAM_PERM_ALL_DATA_OPS)
#define CAAM_PERM_DEFAULT_UID (CAAM_PERM_ALL_DATA_OPS | CAAM_PERM_ALL_KEY_OPS)

#define CAAM_INHERITANCE_MASK_DEFAULT_OTHERS (CAAM_PERM_NON)
#define CAAM_INHERITANCE_MASK_DEFAULT_GID (CAAM_PERM_ALL_DATA_OPS |\
					   CAAM_PERM_ALL_KEY_OPS)
#define CAAM_INHERITANCE_MASK_DEFAULT_UID (CAAM_PERM_ALL_DATA_OPS |\
					   CAAM_PERM_ALL_KEY_OPS)

/* max permissions */
#define CAAM_PERM_MAX_OTHERS (CAAM_PERM_ALL_DATA_OPS |\
			      CAAM_PERM_KEY_UNWRAP_BMSK)
#define CAAM_PERM_MAX_GID    (CAAM_PERM_ALL_DATA_OPS |\
			      CAAM_PERM_KEY_UNWRAP_BMSK)
#define CAAM_PERM_MAX_UID    (CAAM_PERM_ALL_DATA_OPS |\
			      CAAM_PERM_ALL_KEY_OPS)
#define CAAM_INHERITANCE_MASK_MAX (CAAM_PERM_ALL_DATA_OPS |\
				   CAAM_PERM_ALL_KEY_OPS)

struct caam_key_perms {
	/* permissions relevant when using this key */
	uint32_t perms_uid;
	uint32_t perms_gid;
	uint32_t perms_others;

	/*
	 * upper border of permissions which can
	 * be set when importing a key using
	 * this key for protection of the imported key
	 */
	uint32_t inheritance_mask_uid;
	uint32_t inheritance_mask_gid;
	uint32_t inheritance_mask_others;
};

/* max length of modifier */
#define CAAM_KEY_MOD_LEN_MAX 8192

/* used to specify key */
struct caam_key_spec {
	uint32_t uid; /* owner of key */
	uint32_t gid; /* group of key */

	/* extended key spec */
	uint32_t key_flags; /* CAAM_KEY_FLAG_... */
	uint32_t key_bits; /* length of key in bits */
	struct caam_key_perms perms; /* permissions */
	const uint8_t *modifier; /* modifier */
	/* 0 <= length of modifier <= CAAM_KEY_MOD_LEN_MAX */
	uint32_t modifier_bytes;
};


#define caam_key_perms_set_default(P)\
({\
	(P)->perms_uid = CAAM_PERM_DEFAULT_UID;\
	(P)->perms_gid = CAAM_PERM_DEFAULT_GID;\
	(P)->perms_others = CAAM_PERM_DEFAULT_OTHERS;\
	(P)->inheritance_mask_uid = CAAM_INHERITANCE_MASK_DEFAULT_UID;\
	(P)->inheritance_mask_gid = CAAM_INHERITANCE_MASK_DEFAULT_GID;\
	(P)->inheritance_mask_others = CAAM_INHERITANCE_MASK_DEFAULT_OTHERS;\
})

/*
 * permissions to be used when loading a key blob
 * generated by a previous version
 */
#define caam_key_perms_compatibility_perms(P)\
		caam_key_perms_set_default(P)

/* used to mark uid or group as invalid */
#define CAAM_KEY_INVALID_UID 0xFFFFFFFF
#define CAAM_KEY_INVALID_GID 0xFFFFFFFF

#define CAAM_KEY_INVALID_KID 0

#define CAAM_KEY_INFO_INVALID_KID (CAAM_KEY_INVALID_KID)

#define CAAM_KEY_INFO_KEY_ACTIVATED     1
#define CAAM_KEY_INFO_KEY_DEACTIVATED	0

/*
 * key info struct to be used by
 * caam_ee ioctl returning info of the currently loaded key
 * caam_km ioctl returning info of an specific keystore key
 */
struct caam_key_info {
	uint32_t key_flags;
	uint32_t key_bits;
	uint32_t key_bytes;
	uint32_t key_id;
	uint32_t key_activated;
	uint32_t uid;
	uint32_t gid;
	struct caam_key_perms perms;
	/*
	 * If modifier == NULL while calling no modifier is
	 * provided by kernel.
	 * If modifier != NULL but the length is not sufficient
	 * modifier is set to NULL and no modifier is provided
	 * In all cases modifier_bytes is set to the
	 * modifier length available in kernel
	 */
	uint8_t *modifier;
	uint32_t modifier_bytes;
	/*
	 * Additional key specific data - reserved for future use
	 *
	 * If add_data == NULL while calling no add_data is
	 * provided by kernel.
	 * If add_data != NULL but the length is not sufficient
	 * add_data is set to NULL and no add_data is provided
	 * In all cases add_data_bytes is set to the
	 * add_data length available in kernel
	 */
	uint8_t *add_data;
	uint32_t add_data_bytes;
};

/*
 * Info struct passed along with an encapsulated key
 * Furthermore it will used as a modifier to assure
 * that it can't be altered.
 *
 * uid will be used to ensure that only the user
 * who triggered the encapsulation of the key
 * can install the encapsulated key
 *
 * Inheritance mask will ensure that permissions
 * given during install can't exceed the limit
 * defined during encapsulation.
 *
 * Authenticity and confidentiality values will
 * ensure that the values given during install can't
 * exceed the limit defined during encapsulation.
 */
struct caam_key_encap_info {
	uint32_t encap_uid;
	uint32_t encap_inherit_mask;
	uint32_t encap_auth; /* authenticity */
	uint32_t encap_conf; /* confidentiality */
};

#define CAAM_KEY_ENCAP_INFO_LEN (sizeof(struct caam_key_encap_info))

#endif /* _CAAM_KEY_H */
