/**
* \file: AditAudioSink.h
*
* \version: $Id:$
*
* \release: $Name:$
*
* Audio Sink Endpoint using AIL
*
* \component: Baidu CarLife
*
* \author: P. Acar / ADIT/SW2 / pacar@de.adit-jv.com
*
* \copyright (c) 2017 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#ifndef BDCL_ADITAUDIOSINK_H
#define BDCL_ADITAUDIOSINK_H

#include <string>
#include "bdclErrorCodes.h"

namespace adit { namespace bdcl {

class CoreCallbackDealer;

/** @interface  IAditAudioSinkCallbacks
 *  @brief      Callbacks to notify the MC application. Most of the callbacks are initiated by Baidu stack and match
 *              their signature. This interface must be implemented by the MC application using ADIT Audio Sink
 *              implementation.
 */
class IAditAudioSinkCallbacks
{
public:
    virtual ~IAditAudioSinkCallbacks() { }

    /** @brief Callback initiated by Baidu core to initialize and start audio playback. The input parameters must be
     *         used to configure the audio device with each starting stream.
     */
    virtual void onPlaybackInit(unsigned int inSampleRate, unsigned int inChannelConfig, unsigned int inSampleFormat)
    {
        (void)inSampleRate;
        (void)inChannelConfig;
        (void)inSampleFormat;
    };

    /** @brief Callback initiated by Baidu core to stop audio playback.
     */
    virtual void onPlaybackStop() { };

    /** @brief Callback initiated by Baidu core to resume audio playback. Resume callback is only relevant for media
     *         channel.
     */
    virtual void onPlaybackResume() { };

    /** @brief Callback initiated by Baidu core to pause audio playback. Pause callback is only relevant for media
     *         channel.
     */
    virtual void onPlaybackPause() { };

    /** @brief Callback initiated by Baidu core to interrupt audio playback. This callback is only relevant for voice
     *         recognition text-to-speech channel
     */
    virtual void onPlaybackInterrupt() { };

    /** @brief Callback initiated by ADIT to inform MC application about an error.
     */
    virtual void onError(bdclErrorCodes inErrorCode) { (void) inErrorCode; };
};

/** @class      AditAudioSink
 *  @brief      Abstract audio sink interface to be used by the MC. MC has to implement a class of a concrete
 *              implementation of AditAudioSink, but use AditAudioSink interfaces for easier portability.
 */
class AditAudioSink
{
public:
    // shared ptr, no document
    // after the teardown is called, it has to be ensured that these callbacks won't be triggered anymore.
    // So MC is free to destroy them, but MC has to guarantee that the IAditAudioSinkCallbacks object is alive before
    // they call the teardown.

    /** @brief AditAudioSink constructor
     *  @param inCallbacks: object implementing the callback interface. MC has to ensure that IAditAudioSinkCallbacks
     *         object is alive until they call teardown.
     */
    AditAudioSink(IAditAudioSinkCallbacks* inCallbacks, CoreCallbackDealer* inCallbackDealer)
    {
        (void)inCallbacks;
        (void)inCallbackDealer;
    }

    virtual ~AditAudioSink() { };

    // todo write the config params as comment: mediaType,
    //      std::string deviceName, int bufferSizeMin, int bufferSizeMax, int threadPriority, int initTimeOut
    //      unsigned int inSampleRate, unsigned int inChannelConfig, unsigned int inSampleFormat
    /** @brief Configuration interface for each parameter
     *  @param inKey: name of the configuration parameter
     *  @param inValue: the actual value for configuration
     */
    virtual void setConfigItem(std::string inKey, std::string inValue) = 0;

    /** @brief Method to initialize audio sink endpoint
     *         TODO: decide whether initialize() / teardown() should be accessible by MC. The alternative would be MC
     *         only accessing the abdcl interface for initializing or ending whole session, which will initialize / end
     *         audio session
     *  @return true on success
     *          false on failure
     */
    virtual bool initialize() = 0;
    /** @brief Method to shutdown audio sink endpoint. After teardown call, MC is free to destroy
     *         IAditAudioSinkCallbacks object
     */
    virtual void teardown() = 0;

    /** @brief Method to start audio playback. MC has to ensure that the configuration is done before calling starting
     *         audio playback.
     *  @return true on success
     *          false on failure
     */
    virtual bool playbackStart() = 0;
    /** @brief Method to stop audio playback.
     *  @param inFlushBuffer: flag to clean prebuffered audio when stopping audio playback
     *  @return true on success
     *          false on failure
     */
    virtual bool playbackStop(bool inFlushBuffer) = 0;

};

} } /* namespace adit { namespace bdcl { */

#endif /* BDCL_ADITAUDIOSINK_H */
