/* -*-c++-*-
mapengine_oss

Copyright (C) 1998-2006 Robert Osfield
Copyright (C) 2015 Robert Bosch Car Multimedia GmbH

This library is open source and may be redistributed and/or modified under
the terms of the OpenSceneGraph Public License (OSGPL) version 0.0 or
(at your option) any later version.  The full license is in LICENSE file
included with this distribution, and on the openscenegraph.org website.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
OpenSceneGraph Public License for more details.

History:
Copied from https://github.com/openscenegraph/osg/blob/OpenSceneGraph-3.2.0/src/osgDB/DatabasePager.cpp
List of changes:
1. Support several different kinds of requests instead of just 'file' and 'http' requests.
2. Checking if a specific request-type is supported by the DBPager thread looking for requests to execute.
3. Support deletion of objects in a specific type of DBPager thread, instead of only using one particular thread.
4. Support different cache limits in cache-management.
5. Disable incremental-compile of GL objects completely.
*/
#include <assert.h>
#include <exception>
#include <osg/ProxyNode>
#include <osg/ValueObject>
#include "oss/dbpager/DatabasePager.h"
#include "oss/dbpager/DatabasePagerBase.h"
#include <osgDB/Registry>
#include <osgDB/FileNameUtils>

using namespace mapengine::oss;

const char* const EXCLUSIVE_REQUEST_QUEUE = "EReqQ";
const char* const SHARED_REQUEST_QUEUE = "SReqQ";

void me_DatabasePager::ThreadConfig::determineTotalThreadsForAsyncSingleRequest(unsigned int numFMChannels)
{
   _totalNumberOfThreads[DatabasePagerConfig::ASYNCHRONOUS_SINGLE_REQUEST]
      = numFMChannels
      - (_numberOfExclusiveThreads[mapengine::oss::DatabasePagerConfig::HIGH_PRIO_ASYNCHRONOUS_SINGLE_REQUEST]
         + _totalNumberOfThreads[mapengine::oss::DatabasePagerConfig::HLIF_SINGLE_REQUEST]);
}

me_DatabasePager::me_DatabaseRequest::me_DatabaseRequest( mapengine::oss::DatabasePagerConfig::ThreadWorkloadType workloadType,
                                                          const std::string& fileName, const osg::FrameStamp* frameStamp,
                                                          float prio ) : _workloadType(workloadType)
{
   _valid = true;
   _fileName = fileName;
   _timestampFirstRequest = frameStamp ? frameStamp->getReferenceTime() : 0.0;
   _frameNumberFirstRequest = frameStamp ? frameStamp->getFrameNumber() : 0;
   _priorityFirstRequest = prio;
}

unsigned int me_DatabasePager::me_DatabaseRequest::getFlag( const std::vector<mapengine::oss::DatabasePagerConfig::ThreadWorkloadType>& workloads )
{
   unsigned int workloadsFlag = 0;
   for (unsigned int i = 0; i < workloads.size(); ++i)
   {
      workloadsFlag |= 1 << workloads[i];
   }
   return workloadsFlag;
}

void me_DatabasePager::me_DatabaseRequest::getRequestsTypeString(const std::vector<mapengine::oss::DatabasePagerConfig::ThreadWorkloadType>& workloads, std::string& requestsTypeString)
{
   std::stringstream ss;
   ss << "DBPgr-";
   for (unsigned int i = 0; i < workloads.size(); ++i)
   {
      ss << workloads[i] << "_";
   }
   requestsTypeString = ss.str();
}

bool me_DatabasePager::me_DatabaseRequest::isObsolete(unsigned int frameNumber, unsigned int frameNumberOfLastRequestReset) const
{
   bool obsolete = false;
   if (isPerFrameRequest())
   {
      obsolete = !isRequestCurrent(frameNumber);
   }

   // Check OBSOLESCENCE for all kinds of requests
   if (!obsolete && _loadOptions.valid())
   {
      int reqOptions = 0;
      if (_loadOptions->getUserValue(DATABASE_REQUEST_OPTIONS_STRING, reqOptions))
      {
         obsolete = (reqOptions & me_DatabasePager::E_CHECK_FOR_OBSOLESCENCE) && (frameNumber < frameNumberOfLastRequestReset);
         if (frameNumber < frameNumberOfLastRequestReset)
         {
            reqOptions |= me_DatabasePager::E_COULD_BE_OBSOLETE;
         }
         _loadOptions->setUserValue(DATABASE_REQUEST_OPTIONS_STRING, reqOptions); // store it for the ReaderWriter
      }
   }
   return obsolete;
}

me_DatabasePager::me_DatabaseThread::me_DatabaseThread(DatabasePager* pager, Mode mode, const std::string& name, const std::vector<mapengine::oss::DatabasePagerConfig::ThreadWorkloadType>& supportedWorkloads)
: osgDB::DatabasePager::DatabaseThread(pager, mode, name)
, _supportedWorkloadsFlag(0)
{
   me_DatabasePager::me_DatabaseRequest::getRequestsTypeString(supportedWorkloads, _name);
   _supportedWorkloadsFlag = me_DatabasePager::me_DatabaseRequest::getFlag(supportedWorkloads);
}

int me_DatabasePager::me_DatabaseThread::cancel()
{
   int result = 0;

   if( isRunning() )
   {
      setDone(true);
      osg::ref_ptr<me_DatabasePager::ReadQueue> requestQueue = static_cast<me_DatabasePager*>(_pager)->getRequestQueueByFlag(_supportedWorkloadsFlag); //lint !e1774
      if (requestQueue.valid())
      {
         requestQueue->release();
      }
      
      // then wait for the the thread to stop running.
      while(isRunning())
      {
         // commenting out debug info as it was cashing crash on exit, presumable
         // due to OSG_NOTICE or std::cout destructing earlier than this destructor.
         // OSG_INFO<<"Waiting for DatabasePager::DatabaseThread to cancel"<<std::endl;
         OpenThreads::Thread::YieldCurrentThread();
      }

      // _startThreadCalled = false;
   }

   //OSG_NOTICE<<"DatabasePager::DatabaseThread stopped running"<<std::endl;
   return result;
}

void me_DatabasePager::me_DatabaseThread::run()
{
   OSG_INFO<<_name<<": CustomDatabasePager::CustomDatabaseThread::run"<<std::endl;
   
   
   bool firstTime = true;

   // thread is not processing a request yet
   _threadActiveBlock.release();
   
   osg::ref_ptr<me_DatabasePager::me_ReadQueue> read_queue;
   osg::ref_ptr<DatabasePager::ReadQueue> out_queue;
   
   me_DatabasePager* pager = (me_DatabasePager*)(_pager); //lint !e1774
   read_queue = pager->getRequestQueueByFlag(_supportedWorkloadsFlag);

#if defined(LINUX)
#if defined(__APPLE__)
   int threadNameError = pthread_setname_np(_name.c_str());
#else
   int threadNameError = pthread_setname_np(pthread_self(), _name.c_str());
#endif 
   if (0 != threadNameError)
   {
      OSG_FATAL <<"Unable to set thread name " << _name << " error: "<< threadNameError << std::endl;
   }
#endif

   do
   {
      _active = false;

      read_queue->block();

      if (_done)
      {
         break;
      }

      // thread is now active processing a request
      _threadActiveBlock.reset();
      _active = true;

      if (!pager->_databasePagerThreadPaused)
      {
         //@af: removed this trace as it accesses _childrenToDeleteList without locking (AIMAPENG-2077)
         //OSG_INFO<<_name<<": myPager->size()= "<<read_queue->size()<<" to delete = "<<read_queue->_childrenToDeleteList.size()<<std::endl;

         //
         // delete any children if required.
         //
         if (pager->_deleteRemovedSubgraphsInDatabaseThread) // && !(read_queue->_childrenToDeleteList.empty()))
         {
            if (pager->getDeletionCallback().valid())
            {
               pager->getDeletionCallback()->deletion(read_queue);
            }
            else
            {
               osgDB::DatabasePager::ObjectList deleteList;
               pager->deletion(read_queue, deleteList);
            }
         }

         //
         // load any subgraphs that are required.
         //
         osg::ref_ptr<DatabaseRequest> databaseRequest;
         pager->takeFirstRequest(read_queue, databaseRequest, this);

         bool readFromFileCache = false;

         osg::ref_ptr<osgDB::FileCache> fileCache = osgDB::Registry::instance()->getFileCache();
         osg::ref_ptr<osgDB::FileLocationCallback> fileLocationCallback = osgDB::Registry::instance()->getFileLocationCallback();
         osg::ref_ptr<osgDB::Options> dr_loadOptions;
         std::string fileName;
         if (databaseRequest.valid())
         {
            {
               OpenThreads::ScopedLock<OpenThreads::Mutex> drLock(pager->_dr_mutex);
               dr_loadOptions = databaseRequest->_loadOptions;
               fileName = databaseRequest->_fileName;
            }
            OSG_INFO << "[DBP]: run valid " << fileName.c_str();
            if (dr_loadOptions.valid())
            {
               if (dr_loadOptions->getFileCache()) fileCache = dr_loadOptions->getFileCache();
               if (dr_loadOptions->getFileLocationCallback()) fileLocationCallback = dr_loadOptions->getFileLocationCallback();

               dr_loadOptions = dr_loadOptions->cloneOptions();
            }
            else
            {
               dr_loadOptions = new osgDB::Options;
            }

            dr_loadOptions->setTerrain(databaseRequest->_terrain);

            // disable the FileCache if the fileLocationCallback tells us that it isn't required for this request.
            if (fileLocationCallback.valid() && !fileLocationCallback->useFileCache()) fileCache = 0;

            bool requestIsObsolete = false;
            {
               OpenThreads::ScopedLock<OpenThreads::Mutex> drLock(pager->_dr_mutex);
               requestIsObsolete = pager->isRequestObsolete(databaseRequest.get());
            }
            if (!requestIsObsolete)
            {
               if (fileCache.valid() && fileCache->isFileAppropriateForFileCache(fileName))
               {
                  if (fileCache->existsInCache(fileName))
                  {
                     readFromFileCache = true;
                  }
               }
            }
            else
            {
               databaseRequest = 0;
            }
         }

         if (databaseRequest.valid())
         {
            // load the data, note safe to write to the databaseRequest since once
            // it is created this thread is the only one to write to the _loadedModel pointer.
            //OSG_NOTICE<<"In DatabasePager thread readNodeFile("<<databaseRequest->_fileName<<")"<<std::endl;
            //osg::Timer_t before = osg::Timer::instance()->tick();

            // prevent termination of DBPager thread by any plug-in not handling exceptions
            try
            {
               // assume that readNode is thread safe...
               osgDB::ReaderWriter::ReadResult rr = readFromFileCache ?
               fileCache->readNode(fileName, dr_loadOptions.get(), false) :
               osgDB::Registry::instance()->readNode(fileName, dr_loadOptions.get(), false);

               osg::ref_ptr<osg::Node> loadedModel;
               if (rr.validNode()) loadedModel = dynamic_cast<osg::Node*>(rr.getNode());
               if (rr.error()) OSG_WARN<<"Error in reading file "<<fileName<<" : "<<rr.message() << std::endl;
               if (rr.notEnoughMemory()) OSG_INFO<<"Not enough memory to load file "<<fileName << std::endl;

               if (loadedModel.valid() &&
                   fileCache.valid() &&
                   fileCache->isFileAppropriateForFileCache(fileName) &&
                   !readFromFileCache)
               {
                  fileCache->writeNode(*(loadedModel), fileName, dr_loadOptions.get());
               }

               {
                  OpenThreads::ScopedLock<OpenThreads::Mutex> drLock(pager->_dr_mutex);
                  if (pager->isRequestObsolete(databaseRequest.get()))
                  {
                     OSG_INFO<<_name<<": Warning DatabaseRquest no longer required."<<std::endl;
                     loadedModel = 0;
                  }
               }

               //OSG_NOTICE<<"     node read in "<<osg::Timer::instance()->delta_m(before,osg::Timer::instance()->tick())<<" ms"<<std::endl;

               if (loadedModel.valid())
               {
                  loadedModel->getBound();

#ifdef INCREMENTAL_COMPILE_BY_DB_PAGER
                  // find all the compileable rendering objects
                  DatabasePager::FindCompileableGLObjectsVisitor stateToCompile(pager);
                  loadedModel->accept(stateToCompile);

                  bool loadedObjectsNeedToBeCompiled = pager->_doPreCompile &&
                  pager->_incrementalCompileOperation.valid() &&
                  pager->_incrementalCompileOperation->requiresCompile(stateToCompile);

                  // move the databaseRequest from the front of the fileRequest to the end of
                  // dataToCompile or dataToMerge lists.
                  osg::ref_ptr<osgUtil::IncrementalCompileOperation::CompileSet> compileSet = 0;
                  if (loadedObjectsNeedToBeCompiled)
                  {
                     // OSG_NOTICE<<"Using IncrementalCompileOperation"<<std::endl;

                     compileSet = new osgUtil::IncrementalCompileOperation::CompileSet(loadedModel.get());
                     compileSet->buildCompileMap(pager->_incrementalCompileOperation->getContextSet(), stateToCompile);
                     compileSet->_compileCompletedCallback = new DatabasePagerCompileCompletedCallback(pager, databaseRequest.get());
                     pager->_incrementalCompileOperation->add(compileSet.get(), false);
                  }
#else //INCREMENTAL_COMPILE_BY_DB_PAGER
                  bool loadedObjectsNeedToBeCompiled = false;
                  osg::ref_ptr<osgUtil::IncrementalCompileOperation::CompileSet> compileSet = 0;
#endif //INCREMENTAL_COMPILE_BY_DB_PAGER
                  {
                     OpenThreads::ScopedLock<OpenThreads::Mutex> drLock(pager->_dr_mutex);
                     databaseRequest->_loadedModel = loadedModel;
                     databaseRequest->_compileSet = compileSet;
                  }
                  // Dereference the databaseRequest while the queue is
                  // locked. This prevents the request from being
                  // deleted at an unpredictable time within
                  // addLoadedDataToSceneGraph.
                  if (loadedObjectsNeedToBeCompiled)
                  {
                     OpenThreads::ScopedLock<OpenThreads::Mutex> listLock(
                                                                          pager->_dataToCompileList->_requestMutex);
                     OpenThreads::ScopedLock<OpenThreads::Mutex> drLock(pager->_dr_mutex);
                     // We have to ensure that the serviced request has not been canceled so we check for obsolescence again. We might get duplicates of the same data without this check.
                     if (pager->isRequestObsolete(databaseRequest.get()))
                     {
                        OSG_INFO << _name << ": Warning DatabaseRquest no longer required." << std::endl;
                        loadedModel = 0;
                     }
                     else
                     {
                        pager->_dataToCompileList->addNoLock(databaseRequest.get());
                     }
                     databaseRequest = 0;
                  }
                  else
                  {
                     OpenThreads::ScopedLock<OpenThreads::Mutex> listLock(
                                                                          pager->_dataToMergeList->_requestMutex);
                     OpenThreads::ScopedLock<OpenThreads::Mutex> drLock(pager->_dr_mutex);
                     // We have to ensure that the serviced request has not been canceled so we check for obsolescence again. We might get duplicates of the same data without this check.
                     if (pager->isRequestObsolete(databaseRequest.get()))
                     {
                        OSG_INFO << _name << ": Warning DatabaseRquest no longer required." << std::endl;
                        loadedModel = 0;
                     }
                     else
                     {
                        pager->_dataToMergeList->addNoLock(databaseRequest.get());
                     }
                     databaseRequest = 0;
                  }
               }
               pager->_dataToCompileList->pruneOldRequestsAndCheckIfEmpty();
            }
            catch (std::exception& ex)
            {
               OSG_ALWAYS << _name << ": Unhandled exception occurred while reading " << fileName << " what: " << ex.what() << std::endl;
#ifdef _DEBUG
               assert(false); // force termination in debug version to reveal possible bugs
#endif
            }
            catch (...)
            {
               OSG_ALWAYS << _name << ": Unhandled exception occurred while reading " << fileName << std::endl;
#ifdef _DEBUG
               assert(false); // force termination in debug version to reveal possible bugs
#endif
            }
         }
         else
         {
            // release block before yielding the thread
            _threadActiveBlock.release();

            OpenThreads::Thread::YieldCurrentThread();
         }
      }

      // thread is now done processing a request
      _threadActiveBlock.release();

      // go to sleep till our the next time our thread gets scheduled.

      if (firstTime)
      {
         // do a yield to get round a peculiar thread hang when testCancel() is called
         // in certain circumstances - of which there is no particular pattern.
         YieldCurrentThread();
         firstTime = false;
      }

   } while (!testCancel() && !_done);
}

me_DatabasePager::me_DatabasePager()
   : _frameNumberOfLastRequestReset(0)
   , _targetDesiredMaximumNumberOfPagedLOD(200)
   , _supportedWorkloads(0)
{
   _startThreadCalled = true;
   createRequestQueues();
   memset(_numberOfExclusiveThreadsCreated, 0, sizeof(_numberOfExclusiveThreadsCreated));
   memset(_totalNumberOfThreadsCreated, 0, sizeof(_totalNumberOfThreadsCreated));
  
   setUpThreads(
      osg::DisplaySettings::instance()->getNumOfDatabaseThreadsHint(),
      osg::DisplaySettings::instance()->getNumOfHttpDatabaseThreadsHint());

   const char* str = getenv("OSG_DATABASE_PAGER_PRIORITY");
   if (str)
   {
      if (strcmp(str,"DEFAULT")==0)
      {
         setSchedulePriority(OpenThreads::Thread::THREAD_PRIORITY_DEFAULT);
      }
      else if (strcmp(str,"MIN")==0)
      {
         setSchedulePriority(OpenThreads::Thread::THREAD_PRIORITY_MIN);
      }
      else if (strcmp(str,"LOW")==0)
      {
         setSchedulePriority(OpenThreads::Thread::THREAD_PRIORITY_LOW);
      }
      else if (strcmp(str,"NOMINAL")==0)
      {
         setSchedulePriority(OpenThreads::Thread::THREAD_PRIORITY_NOMINAL);
      }
      else if (strcmp(str,"HIGH")==0)
      {
         setSchedulePriority(OpenThreads::Thread::THREAD_PRIORITY_HIGH);
      }
      else if (strcmp(str,"MAX")==0)
      {
         setSchedulePriority(OpenThreads::Thread::THREAD_PRIORITY_MAX);
      }
   }
}


me_DatabasePager::~me_DatabasePager()
{
   std::map< unsigned int, osg::ref_ptr<me_ReadQueue> >::iterator itr;
   for (itr = _requestQueues.begin(); itr != _requestQueues.end(); ++itr)
   {
      itr->second = 0;
   }
}


unsigned int me_DatabasePager::addDatabaseThread( DatabaseThread::Mode mode, const std::string& name )
{
   OSG_INFO<<"CustomDatabasePager::addDatabaseThread() "<<name<<std::endl;
   
   unsigned int pos = static_cast<unsigned int>(_databaseThreads.size());
   DatabaseThread* thread = NULL;

   std::vector<mapengine::oss::DatabasePagerConfig::ThreadWorkloadType> supportedWorkloads;
   getWorkloadsToBeSupported(supportedWorkloads);

   thread = new me_DatabaseThread(this, mode, name, supportedWorkloads);

   if (thread)
   {
      _databaseThreads.push_back(thread);
      _supportedWorkloads |= static_cast<me_DatabaseThread*>(thread)->getSupportedWorkloadsFlag();

      if (_startThreadCalled)
      {
         OSG_INFO<<"DatabasePager::startThread()"<<std::endl;
         thread->startThread();
      }
   }
  
   return pos; //lint !e429
}

void me_DatabasePager::requestNodeFile(const std::string& fileName, osg::NodePath& nodePath,
                                             float priority, const osg::FrameStamp* framestamp,
                                             osg::ref_ptr<osg::Referenced>& databaseRequestRef,
                                             const osg::Referenced* options)
{
   osgDB::Options* loadOptions = dynamic_cast<osgDB::Options*>(const_cast<osg::Referenced*>(options));
   if (!loadOptions)
   {
      loadOptions = osgDB::Registry::instance()->getOptions();
      
      // OSG_NOTICE<<"Using options from Registry "<<std::endl;
   }
   else
   {
      // OSG_NOTICE<<"options from requestNodeFile "<<std::endl;
   }
   
   
   if (!_acceptNewRequests) return;
   
   
   if (nodePath.empty())
   {
      OSG_NOTICE<<"Warning: DatabasePager::requestNodeFile(..) passed empty NodePath, so nowhere to attach new subgraph to."<<std::endl;
      return;
   }
   
   osg::Group* group = nodePath.back()->asGroup();
   if (!group)
   {
      OSG_NOTICE<<"Warning: DatabasePager::requestNodeFile(..) passed NodePath without group as last node in path, so nowhere to attach new subgraph to."<<std::endl;
      return;
   }
   
   osg::Node* terrain = 0;
   for(osg::NodePath::reverse_iterator itr = nodePath.rbegin();
       itr != nodePath.rend();
       ++itr)
   {
      if ((*itr)->asTerrain()) terrain = *itr;
   }
   
   double timestamp = framestamp?framestamp->getReferenceTime():0.0;
   unsigned int frameNumber = framestamp?framestamp->getFrameNumber():static_cast<unsigned int>(_frameNumber);
   
   // #define WITH_REQUESTNODEFILE_TIMING
#ifdef WITH_REQUESTNODEFILE_TIMING
   osg::Timer_t start_tick = osg::Timer::instance()->tick();
   static int previousFrame = -1;
   static double totalTime = 0.0;
   
   if (previousFrame!=frameNumber)
   {
      OSG_NOTICE<<"requestNodeFiles for "<<previousFrame<<" time = "<<totalTime<<std::endl;
      
      previousFrame = frameNumber;
      totalTime = 0.0;
   }
#endif
   
   // search to see if filename already exist in the file loaded list.
   bool foundEntry = false;
   
   bool httpReq = false;
   if (loadOptions && loadOptions->getFileLocationCallback())
   {
      httpReq = loadOptions->getFileLocationCallback()->fileLocation(fileName, loadOptions) == osgDB::FileLocationCallback::REMOTE_FILE ? true : httpReq;
   }
   
   //ME_TR_TYPES_INFO((__LINE__, __FILE__, "[DBP]: '%s'", fileName.c_str()));
   osg::ref_ptr<ReadQueue> requestQueue;
   me_DatabaseRequest* dbReq = dynamic_cast<me_DatabaseRequest*>(databaseRequestRef.get());
   if (dbReq)
   {
      requestQueue = getRequestQueueByFlag(dbReq->getWorkloadFlag());
   }
   else
   {
      requestQueue = getRequestQueueByFlag(1 << mapengine::oss::DatabasePagerConfig::SYNCHRONOUS_PER_FRAME_REQUEST); //the default (pagedLOD)
   }

   if (databaseRequestRef.valid())
   {
      DatabaseRequest* databaseRequest = dynamic_cast<DatabaseRequest*>(databaseRequestRef.get());
      bool requeue = false;
      if (databaseRequest)
      {
         //ME_TR_TYPES_INFO((__LINE__, __FILE__, "[DBP]: rereq '%s'", fileName.c_str()));
         OpenThreads::ScopedLock<OpenThreads::Mutex> drLock(_dr_mutex);
         if (!(databaseRequest->valid()))
         {
            OSG_INFO<<"DatabaseRequest has been previously invalidated whilst still attached to scene graph."<<std::endl;
            databaseRequest = 0;
         }
         else
         {
            //ME_TR_TYPES_INFO((__LINE__, __FILE__, "[DBP]: rereq valid '%s'", fileName.c_str()));
            OSG_INFO<<"DatabasePager::requestNodeFile("<<fileName<<") updating already assigned."<<std::endl;
            
            
            databaseRequest->_valid = true;
            databaseRequest->_frameNumberLastRequest = frameNumber;
            databaseRequest->_timestampLastRequest = timestamp;
            databaseRequest->_priorityLastRequest = priority;
            if (databaseRequest->_loadOptions.valid()) 
            {
               databaseRequest->_loadOptions->setUserValue(DATABASE_REQUEST_FRAMENUMBER_STRING, frameNumber);
            }
            ++(databaseRequest->_numOfRequests);
            
            foundEntry = true;
            
            if (databaseRequestRef->referenceCount()==1)
            {
               OSG_INFO<<"DatabasePager::requestNodeFile("<<fileName<<") orphaned, resubmitting."<<std::endl;
               
               databaseRequest->_frameNumberLastRequest = frameNumber;
               databaseRequest->_timestampLastRequest = timestamp;
               databaseRequest->_priorityLastRequest = priority;
               databaseRequest->_group = group;
               databaseRequest->_terrain = terrain;
               databaseRequest->_loadOptions = loadOptions;
               if (databaseRequest->_loadOptions.valid())
               {
                  databaseRequest->_loadOptions->setUserValue(DATABASE_REQUEST_FRAMENUMBER_STRING, frameNumber);
               }
               requeue = true;
            }
            
         }
      }
      if (requeue)
      {
         requestQueue->add((DatabaseRequest*)(databaseRequestRef.get()));//lint !e1774
      }
   }
   
   if (!foundEntry)
   {
      OSG_INFO<<"In DatabasePager::requestNodeFile("<<fileName<<")"<<std::endl;
      OpenThreads::ScopedLock<OpenThreads::Mutex> lock(requestQueue->_requestMutex);
      
      if (!databaseRequestRef.valid() || databaseRequestRef->referenceCount()==1)
      {
         mapengine::oss::DatabasePagerConfig::ThreadWorkloadType workloadType = mapengine::oss::DatabasePagerConfig::SYNCHRONOUS_PER_FRAME_REQUEST;
         me_DatabaseRequest* meDbRequest = static_cast<me_DatabaseRequest*>(databaseRequestRef.get());
         if (meDbRequest)
         {
            workloadType = meDbRequest->_workloadType;
         }

         osg::ref_ptr<DatabaseRequest> databaseRequest = new me_DatabaseRequest(workloadType, fileName, framestamp, priority);

         //ME_TR_TYPES_INFO((__LINE__, __FILE__, "[DBP]: new req '%s'", fileName.c_str()));
         databaseRequestRef = databaseRequest.get();
         
         databaseRequest->_valid = true;
         databaseRequest->_fileName = fileName;
         databaseRequest->_frameNumberFirstRequest = frameNumber;
         databaseRequest->_timestampFirstRequest = timestamp;
         databaseRequest->_priorityFirstRequest = priority;
         databaseRequest->_frameNumberLastRequest = frameNumber;
         databaseRequest->_timestampLastRequest = timestamp;
         databaseRequest->_priorityLastRequest = priority;
         databaseRequest->_group = group;
         databaseRequest->_terrain = terrain;
         databaseRequest->_loadOptions = loadOptions;

         requestQueue->addNoLock(databaseRequest);
      }
   }
   
   if (!_startThreadCalled)
   {
      OpenThreads::ScopedLock<OpenThreads::Mutex> lock(_run_mutex);
      
      if (!_startThreadCalled) //lint !e774 // multithreading synchronization
      {
         _startThreadCalled = true;
         _done = false;
         OSG_INFO<<"DatabasePager::startThread()"<<std::endl;
         
         if (_databaseThreads.empty())
         {
            setUpThreads(
                         osg::DisplaySettings::instance()->getNumOfDatabaseThreadsHint(),
                         osg::DisplaySettings::instance()->getNumOfHttpDatabaseThreadsHint());
         }
         
         for(DatabaseThreadList::const_iterator dt_itr = _databaseThreads.begin();
             dt_itr != _databaseThreads.end();
             ++dt_itr)
         {
            (*dt_itr)->startThread();
         }
      }
   }
   
#ifdef WITH_REQUESTNODEFILE_TIMING
   totalTime += osg::Timer::instance()->delta_m(start_tick, osg::Timer::instance()->tick());
#endif
}

void me_DatabasePager::takeFirstRequest( osg::ref_ptr<me_DatabasePager::me_ReadQueue>& requestQueue, 
                                         osg::ref_ptr<DatabaseRequest>& databaseRequest, const DatabaseThread* dbPagerThread ) const
{
   const me_DatabaseThread* sharingThread = dynamic_cast<const me_DatabaseThread*>(dbPagerThread);
   if (sharingThread)
   {
      if (requestQueue.valid())
      {
         OpenThreads::ScopedLock<OpenThreads::Mutex> lock(requestQueue->_requestMutex);
         if (!(requestQueue->_requestList.empty()))
         {
            DatabasePager::SortFileRequestFunctor highPriority;
            me_DatabasePager::SortFileRequestFunctorPerFrame highPriorityPerFrame;

            me_DatabasePager* pager = (me_DatabasePager*)(requestQueue->_pager); //lint !e1774
            RequestQueue::RequestList::iterator selected_itr = requestQueue->_requestList.end();

            unsigned int frameNumber = pager->_frameNumber;

            for(RequestQueue::RequestList::iterator citr = requestQueue->_requestList.begin(); citr != requestQueue->_requestList.end(); )
            {
               OpenThreads::ScopedLock<OpenThreads::Mutex> drLock(pager->_dr_mutex);
               //             if (selected_itr != requestQueue->_requestList.end())
               //             {
               //                ME_TR_BMD_INFO(( __LINE__, __FILE__, "comparing req %.9f %.9f %u with %.9f %.9f %u and picked %u",
               //                   (*citr)->_priorityLastRequest, (*citr)->_timestampLastRequest, (*citr)->_frameNumberLastRequest,
               //                   (*selected_itr)->_priorityLastRequest, (*selected_itr)->_timestampLastRequest, (*selected_itr)->_frameNumberLastRequest,
               //                   highPriority(*citr, *selected_itr) ? 1 : 2));
               //             }
               mapengine::oss::DatabasePagerConfig::ThreadWorkloadType workloadType = ((me_DatabaseRequest*)((*citr).get()))->_workloadType; //lint !e1774
               if (((me_DatabaseRequest*)((*citr).get()))->isPerFrameRequest())
               {
                  if ((*citr)->isRequestCurrent(frameNumber))
                  {
                     if (isWorkloadCurrentlyAllowed(workloadType, sharingThread->getSupportedWorkloadsFlag()))
                     {
                        if (selected_itr==requestQueue->_requestList.end() || highPriorityPerFrame(*citr, *selected_itr))
                        {
                           selected_itr = citr;
                        }
                     }

                     ++citr;
                  }
                  else
                  {
                     requestQueue->invalidate(citr->get());

                     OSG_INFO<<"DatabasePager::RequestQueue::takeFirst(): Pruning "<<(*citr)<<std::endl;
                     citr = requestQueue->_requestList.erase(citr);
                  }
               }
               else
               {
                  if (isWorkloadCurrentlyAllowed(workloadType, sharingThread->getSupportedWorkloadsFlag()) && (selected_itr==requestQueue->_requestList.end() || highPriority(*citr, *selected_itr)))
                  {
                     selected_itr = citr;
                  }
                  ++citr;
               }
            }

            requestQueue->_frameNumberLastPruned = frameNumber;

            if (selected_itr != requestQueue->_requestList.end())
            {
               databaseRequest = *selected_itr;
               requestQueue->_requestList.erase(selected_itr);
               OSG_INFO<<" DatabasePager::RequestQueue::takeFirst() Found DatabaseRequest size()="<<requestQueue->_requestList.size()<<std::endl;
            }
            else
            {
               OSG_INFO<<" DatabasePager::RequestQueue::takeFirst() No suitable DatabaseRequest found size()="<<requestQueue->_requestList.size()<<std::endl;
            }

            requestQueue->updateBlock();
         }
      }
   }
   else
   {
      requestQueue->takeFirst(databaseRequest);
   }
}


void me_DatabasePager::addToChildrenToDeleteList( osgDB::DatabasePager::ObjectList& objectsToDeleteList, mapengine::oss::DatabasePagerConfig::ThreadWorkloadType workloadType ) const
{
   if (!objectsToDeleteList.empty())
   {
      if (_deleteRemovedSubgraphsInDatabaseThread)
      {
         unsigned int workloadFlag = 1 << workloadType;
         bool dataPassedToDatabaseThread = false;
         if (workloadFlag & _supportedWorkloads)
         {
            osg::ref_ptr<DatabasePager::ReadQueue> read_queue = getRequestQueueByFlag(workloadFlag);
            if (read_queue.valid())
            {
               OpenThreads::ScopedLock<OpenThreads::Mutex> lock(read_queue->_requestMutex);
               // splice transfers the entire list in constant time.
               read_queue->_childrenToDeleteList.splice(read_queue->_childrenToDeleteList.end(),
                  objectsToDeleteList);
               dataPassedToDatabaseThread = true;
               read_queue->updateBlock();
            }
         }
         if (!dataPassedToDatabaseThread)
         {
            assert(0);
            OSG_FATAL << "No thread exists to delete " << workloadType << " type of data. So deleting it right here" << std::endl;
         }
      }
   }
   objectsToDeleteList.clear(); // This will not do anything if the data was passed to the DatabaseThread for deletion.
}


osg::ref_ptr<me_DatabasePager::me_ReadQueue> me_DatabasePager::getRequestQueueByFlag( unsigned int workloadsFlag ) const
{
   osg::ref_ptr<me_DatabasePager::me_ReadQueue> read_queue;
   std::map< unsigned int, osg::ref_ptr<me_DatabasePager::me_ReadQueue> >::const_iterator itr = _requestQueues.find(workloadsFlag);
   if (itr == _requestQueues.end())
   {
      //search for shared queue
      for (itr = _requestQueues.begin(); itr != _requestQueues.end(); ++itr)
      {
         if (itr->first & workloadsFlag)
         {
            read_queue = itr->second;
            break;
         }
      }
   }
   else
   {
      read_queue = itr->second;
   }
   return read_queue;
}

void me_DatabasePager::getWorkloadsToBeSupported( std::vector<DatabasePagerConfig::ThreadWorkloadType>& supportedWorkloads )
{
   osg::ref_ptr<osgDB::Options> theOptions = osgDB::Registry::instance()->getOptions();
   if (theOptions.valid())
   {
      ThreadConfig* threadConfig = (ThreadConfig*)(theOptions->getPluginData(DB_PAGER_THREAD_CONFIG));

      if (threadConfig)
      {
         unsigned int i = 0;
         bool exclusiveThread = false;
         for (i = 0; i < DatabasePagerConfig::WORKLOAD_TYPE_COUNT; ++i)
         {
            if (_numberOfExclusiveThreadsCreated[i] < threadConfig->_numberOfExclusiveThreads[i])
            {
               exclusiveThread = true;
               break;
            }
         }
         if (exclusiveThread && i < DatabasePagerConfig::WORKLOAD_TYPE_COUNT)
         {
            //we need an exclusive thread
            supportedWorkloads.push_back(DatabasePagerConfig::ThreadWorkloadType(i));
            ++_numberOfExclusiveThreadsCreated[i];
            ++_totalNumberOfThreadsCreated[i];
         }
         else
         {
            //now create shared threads
            DatabasePagerConfig::ThreadGroup currThreadGroup = DatabasePagerConfig::THREAD_GROUP_INVALID;
            for (i = 0; i < DatabasePagerConfig::WORKLOAD_TYPE_COUNT; ++i)
            {
               if ((_totalNumberOfThreadsCreated[i] < threadConfig->_totalNumberOfThreads[i]) &&
                   ((currThreadGroup == DatabasePagerConfig::THREAD_GROUP_INVALID) || (currThreadGroup == threadConfig->_threadGroupOfWorkload[i])))
               {
                  currThreadGroup = threadConfig->_threadGroupOfWorkload[i];
                  supportedWorkloads.push_back(DatabasePagerConfig::ThreadWorkloadType(i));
                  ++_totalNumberOfThreadsCreated[i];
               }
            }
         }
      }
   }
}

void me_DatabasePager::requestData( const std::string& fileName, osg::NodePath& nodePath, float priority,
                                    const osg::FrameStamp* framestamp, osg::ref_ptr<osg::Referenced>& databaseRequest, 
                                    DatabasePagerConfig::ThreadWorkloadType workloadType, const osg::Referenced* options )
{
   if( !databaseRequest.valid() )
   {
      databaseRequest = new me_DatabasePager::me_DatabaseRequest(workloadType, fileName, framestamp, priority);
   }
   requestNodeFile(fileName, nodePath, priority, framestamp, databaseRequest, options);
}

void me_DatabasePager::createRequestQueues()
{
   osg::ref_ptr<osgDB::Options> theOptions = osgDB::Registry::instance()->getOptions();
   std::vector<DatabasePagerConfig::ThreadWorkloadType> exclusives;
   std::vector<DatabasePagerConfig::ThreadWorkloadType> shared[DatabasePagerConfig::THREAD_GROUP_COUNT];
   if (theOptions.valid())
   {
      ThreadConfig* threadConfig = (ThreadConfig*)(theOptions->getPluginData(DB_PAGER_THREAD_CONFIG));
      if (threadConfig)
      {
         for (unsigned int i = 0; i < DatabasePagerConfig::WORKLOAD_TYPE_COUNT; ++i)
         {
            if ((threadConfig->_totalNumberOfThreads[i] == threadConfig->_numberOfExclusiveThreads[i]) && (threadConfig->_totalNumberOfThreads[i] > 0))
            {
               exclusives.push_back(DatabasePagerConfig::ThreadWorkloadType(i));
            }
            else
            {
               shared[threadConfig->_threadGroupOfWorkload[i]].push_back(DatabasePagerConfig::ThreadWorkloadType(i));
            }
         }
      }
   }
   for (unsigned int i = 0; i < exclusives.size(); ++i)
   {
      _requestQueues[1 << exclusives[i]] = new me_DatabasePager::me_ReadQueue(this, EXCLUSIVE_REQUEST_QUEUE);
   }

   //now for shared request Queues
   for (unsigned int i = 0; i < DatabasePagerConfig::THREAD_GROUP_COUNT; ++i)
   {
      if (!shared[i].empty())
      {
         _requestQueues[me_DatabaseRequest::getFlag(shared[i])] = new me_DatabasePager::me_ReadQueue(this, SHARED_REQUEST_QUEUE);
      }
   }
}

int me_DatabasePager::cancel()
{
   int result = 0;

   for(DatabaseThreadList::const_iterator dt_itr = _databaseThreads.begin();
      dt_itr != _databaseThreads.end();
      ++dt_itr)
   {
      (*dt_itr)->setDone(true);
   }

   // release the queue blocks in case they are holding up thread cancellation.
   _fileRequestQueue->release();
   _httpRequestQueue->release();

   for (std::map< unsigned int, osg::ref_ptr<me_DatabasePager::me_ReadQueue> >::const_iterator itr = _requestQueues.begin(); itr != _requestQueues.end(); ++itr)
   {
      itr->second->release();
   }

   for(DatabaseThreadList::const_iterator dt_itr = _databaseThreads.begin();
      dt_itr != _databaseThreads.end();
      ++dt_itr)
   {
      (*dt_itr)->cancel();
   }

   _done = true;
   _startThreadCalled = false;

   return result;
}

void mapengine::oss::me_DatabasePager::removeExpiredSubgraphs( const osg::FrameStamp &frameStamp )
{

   static double s_total_iter_stage_a = 0.0;
   static double s_total_time_stage_a = 0.0;
   static double s_total_max_stage_a = 0.0;

   static double s_total_iter_stage_b = 0.0;
   static double s_total_time_stage_b = 0.0;
   static double s_total_max_stage_b = 0.0;

   static double s_total_iter_stage_c = 0.0;
   static double s_total_time_stage_c = 0.0;
   static double s_total_max_stage_c = 0.0;

   if (frameStamp.getFrameNumber()==0)
   {
      // No need to remove anything on first frame.
      return;
   }


   osg::Timer_t startTick = osg::Timer::instance()->tick();

   // numPagedLODs >= actual number of PagedLODs. There can be
   // invalid observer pointers in _activePagedLODList.
   unsigned int numPagedLODs = _activePagedLODList->size();

   osg::Timer_t end_a_Tick = osg::Timer::instance()->tick();
   double time_a = osg::Timer::instance()->delta_m(startTick,end_a_Tick);

   s_total_iter_stage_a += 1.0;
   s_total_time_stage_a += time_a;
   if (s_total_max_stage_a<time_a) s_total_max_stage_a = time_a;


   if (numPagedLODs <= _targetMaximumNumberOfPageLOD)
   {
      // nothing to do
      return;
   }

   int numToPrune = numPagedLODs - _targetMaximumNumberOfPageLOD;

   ObjectList childrenRemoved;

   double expiryTime = frameStamp.getReferenceTime() - 0.1;
   unsigned int expiryFrame = frameStamp.getFrameNumber() - 1;

   // First traverse inactive PagedLODs, as their children will
   // certainly have expired. Then traverse active nodes if we still
   // need to prune.
   //OSG_NOTICE<<"numToPrune "<<numToPrune;
   if (numToPrune>0 && _activePagedLODList.valid())
      _activePagedLODList->removeExpiredChildren(
      numToPrune, expiryTime, expiryFrame, childrenRemoved, false);
   numToPrune = _activePagedLODList->size() - _targetMaximumNumberOfPageLOD;
   if (numToPrune>0 && _activePagedLODList.valid())
      _activePagedLODList->removeExpiredChildren(
      numToPrune, expiryTime, expiryFrame, childrenRemoved, true);

   osg::Timer_t end_b_Tick = osg::Timer::instance()->tick();
   double time_b = osg::Timer::instance()->delta_m(end_a_Tick,end_b_Tick);

   s_total_iter_stage_b += 1.0;
   s_total_time_stage_b += time_b;
   if (s_total_max_stage_b<time_b) s_total_max_stage_b = time_b;

   //OSG_NOTICE<<" childrenRemoved.size()="<<childrenRemoved.size()<<std::endl;

   if (!childrenRemoved.empty())
   {
      // pass the objects across to the database pager delete list
      if (_deleteRemovedSubgraphsInDatabaseThread)
      {
         osg::ref_ptr<osgDB::DatabasePager::ReadQueue> requestQueue = getRequestQueueByFlag(1 << mapengine::oss::DatabasePagerConfig::SYNCHRONOUS_PER_FRAME_REQUEST);
         OpenThreads::ScopedLock<OpenThreads::Mutex> lock(requestQueue->_requestMutex);
         // splice transfers the entire list in constant time.
         requestQueue->_childrenToDeleteList.splice(
            requestQueue->_childrenToDeleteList.end(),
            childrenRemoved);
         requestQueue->updateBlock();
      }
      else
         childrenRemoved.clear();
   }

   osg::Timer_t end_c_Tick = osg::Timer::instance()->tick();
   double time_c = osg::Timer::instance()->delta_m(end_b_Tick,end_c_Tick);

   s_total_iter_stage_c += 1.0;
   s_total_time_stage_c += time_c;
   if (s_total_max_stage_c<time_c) s_total_max_stage_c = time_c;

   OSG_INFO<<"active="<<_activePagedLODList->size()<<" overall = "<<osg::Timer::instance()->delta_m(startTick,end_c_Tick)<<
      " A="<<time_a<<" avg="<<s_total_time_stage_a/s_total_iter_stage_a<<" max = "<<s_total_max_stage_a<<
      " B="<<time_b<<" avg="<<s_total_time_stage_b/s_total_iter_stage_b<<" max = "<<s_total_max_stage_b<<
      " C="<<time_c<<" avg="<<s_total_time_stage_c/s_total_iter_stage_c<<" max = "<<s_total_max_stage_c<<std::endl;
}

unsigned int me_DatabasePager::getFrameNumberOfLastRequestReset()
{
   OpenThreads::ScopedLock<OpenThreads::Mutex> drLock(_dr_mutex);
   return _frameNumberOfLastRequestReset;
}

void me_DatabasePager::clearRequests(DatabaseRequestList& requestList, DatabasePagerConfig::ThreadWorkloadType workloadType,
   const std::vector<std::string>& extensions /*= std::vector<std::string>()*/, me_RequestIdentifierFunctor* requestIdentifier /*= NULL*/)
{
   unsigned int workloadFlag = 1 << workloadType;
   if (workloadFlag & _supportedWorkloads)
   {
      osg::ref_ptr<me_DatabasePager::me_ReadQueue> requestQueue = getRequestQueueByFlag(workloadFlag);
      if (requestQueue.valid())
      {
         requestQueue->clearRequests(requestList, workloadType, extensions, requestIdentifier);
      }
   }
}

void me_DatabasePager::setFrameNumberOfLastRequestReset(unsigned int val)
{
   OpenThreads::ScopedLock<OpenThreads::Mutex> drLock(_dr_mutex);
   _frameNumberOfLastRequestReset = val;
}

void me_DatabasePager::setDatabasePagerThreadPause(bool pause)
{
   bool wasPaused = getDatabasePagerThreadPause(); // back up the value since the base class function would reset it
   DatabasePager::setDatabasePagerThreadPause(pause);
   if (wasPaused != pause)
   {
      for (std::map< unsigned int, osg::ref_ptr<me_DatabasePager::me_ReadQueue> >::const_iterator itr = _requestQueues.begin(); itr != _requestQueues.end(); ++itr)
      {
         OpenThreads::ScopedLock<OpenThreads::Mutex> lock(itr->second->_requestMutex);
         itr->second->updateBlock();
      }
   }
}

void me_DatabasePager::pauseDatabasePagerThreadsAndWaitForCompletion()
{
   // pause the waking up of pager threads on new requests
   setDatabasePagerThreadPause(true);

   // block until all already running requests in pager threads have been processed
   for (DatabaseThreadList::const_iterator dt_itr = _databaseThreads.begin(); dt_itr != _databaseThreads.end(); ++dt_itr)
   {
      me_DatabasePager::me_DatabaseThread* dbThread = static_cast<me_DatabasePager::me_DatabaseThread*>((*dt_itr).get());
      dbThread->block();
   }
}

void me_DatabasePager::clearAllReadQueue()
{
   std::map< unsigned int, osg::ref_ptr<me_ReadQueue> >::iterator itr;
   for (itr = _requestQueues.begin(); itr != _requestQueues.end(); ++itr)
   {
      if (itr->second.valid())
      {
         itr->second->clearAllData();
      }
   }
}

unsigned int me_DatabasePager::getFileRequestListSize() const
{
   unsigned int reqQueueSize = 0;
   for (std::map< unsigned int, osg::ref_ptr<me_DatabasePager::me_ReadQueue> >::const_iterator itr = _requestQueues.begin(); itr != _requestQueues.end(); ++itr)
   {
      reqQueueSize += itr->second->size();
   }
   return reqQueueSize;
}

unsigned int me_DatabasePager::getNumOfRequests(DatabasePagerConfig::ThreadWorkloadType workloadType) const
{
   osg::ref_ptr<me_DatabasePager::me_ReadQueue> reqQueue = getRequestQueueByFlag(1 << workloadType);
   unsigned int reqQueueSize = reqQueue.valid() ? reqQueue->size() : 0;
   return reqQueueSize;
}

bool me_DatabasePager::isRequestObsolete( const DatabaseRequest* databaseRequest ) const
{
   const me_DatabaseRequest* dbRequest = static_cast<const me_DatabaseRequest*>(databaseRequest); //lint !e1774
   if (dbRequest)
   {
      return dbRequest->isObsolete(_frameNumber, _frameNumberOfLastRequestReset);
   }
   return false;
}

void me_DatabasePager::deletion(osg::ref_ptr<mapengine::oss::me_DatabasePager::me_ReadQueue> readQueue, osgDB::DatabasePager::ObjectList& deleteList) const
{
   // Don't hold lock during destruction of deleteList
   OpenThreads::ScopedLock<OpenThreads::Mutex> lock(readQueue->_requestMutex);
   if (!readQueue->_childrenToDeleteList.empty())
   {
      deleteList.swap(readQueue->_childrenToDeleteList);
      readQueue->updateBlock();
   }
}

void me_DatabasePager::updateSceneGraph(const osg::FrameStamp& frameStamp)
{

#define UPDATE_TIMING 0
#if UPDATE_TIMING
   osg::ElapsedTime timer;
   double timeFor_removeExpiredSubgraphs, timeFor_addLoadedDataToSceneGraph;
#endif

   {
      removeExpiredSubgraphs(frameStamp);

#if UPDATE_TIMING
      timeFor_removeExpiredSubgraphs = timer.elapsedTime_m();
#endif

      customAddLoadedDataToSceneGraph(frameStamp);

#if UPDATE_TIMING
      timeFor_addLoadedDataToSceneGraph = timer.elapsedTime_m() - timeFor_removeExpiredSubgraphs;
#endif

   }

#if UPDATE_TIMING
   double elapsedTime = timer.elapsedTime_m();
   if (elapsedTime > 0.4)
   {
      OSG_NOTICE << "DatabasePager::updateSceneGraph() total time = " << elapsedTime << "ms" << std::endl;
      OSG_NOTICE << "   timeFor_removeExpiredSubgraphs    = " << timeFor_removeExpiredSubgraphs << "ms" << std::endl;
      OSG_NOTICE << "   timeFor_addLoadedDataToSceneGraph = " << timeFor_addLoadedDataToSceneGraph << "ms" << std::endl;
      OSG_NOTICE << "   _activePagedLODList.size()        = " << _activePagedLODList->size() << std::endl;
      OSG_NOTICE << "   _inactivePagedLODList.size()      = " << _inactivePagedLODList->size() << std::endl;
      OSG_NOTICE << "   total                             = " << _activePagedLODList->size() + _inactivePagedLODList->size() << std::endl;
   }
#endif
}

void me_DatabasePager::customAddLoadedDataToSceneGraph(const osg::FrameStamp &frameStamp)
{
   double timeStamp = frameStamp.getReferenceTime();
   unsigned int frameNumber = frameStamp.getFrameNumber();

   osg::Timer_t before = osg::Timer::instance()->tick(), mid, last;

   RequestQueue::RequestList localFileLoadedList;

   // get the data from the _dataToMergeList, leaving it empty via a std::vector<>.swap.
   _dataToMergeList->swap(localFileLoadedList);

   // Remove requests which are obsolete 
   for (RequestQueue::RequestList::iterator itr = localFileLoadedList.begin();
      itr != localFileLoadedList.end();
      )
   {
      DatabaseRequest* databaseRequest = itr->get();
      if (isRequestObsolete(databaseRequest))
      {
         itr = localFileLoadedList.erase(itr);
      }
      else
      {
         ++itr;
      }
   }

   mid = osg::Timer::instance()->tick();

   // add the loaded data into the scene graph.
   for (RequestQueue::RequestList::iterator itr = localFileLoadedList.begin();
      itr != localFileLoadedList.end();
      ++itr)
   {
      DatabaseRequest* databaseRequest = itr->get();

      // No need to take _dr_mutex. The pager threads are done with
      // the request; the cull traversal -- which might redispatch
      // the request -- can't run at the sametime as this update traversal.
      osg::ref_ptr<osg::Group> group;
      if (!databaseRequest->_groupExpired && databaseRequest->_group.lock(group))
      {
         if (osgDB::Registry::instance()->getSharedStateManager())
            osgDB::Registry::instance()->getSharedStateManager()->share(databaseRequest->_loadedModel.get());

         osg::PagedLOD* plod = dynamic_cast<osg::PagedLOD*>(group.get());
         if (plod)
         {
            plod->setTimeStamp(plod->getNumChildren(), timeStamp);
            plod->setFrameNumber(plod->getNumChildren(), frameNumber);
            plod->getDatabaseRequest(plod->getNumChildren()) = 0;
         }
         else
         {
            osg::ProxyNode* proxyNode = dynamic_cast<osg::ProxyNode*>(group.get());
            if (proxyNode)
            {
               proxyNode->getDatabaseRequest(proxyNode->getNumChildren()) = 0;
            }
         }

         group->addChild(databaseRequest->_loadedModel.get());

         // Check if parent plod was already registered if not start visitor from parent
         if (plod &&
            !_activePagedLODList->containsPagedLOD(plod))
         {
            registerPagedLODs(plod, frameNumber);
         }
         else
         {
            registerPagedLODs(databaseRequest->_loadedModel.get(), frameNumber);
         }

         // OSG_NOTICE<<"merged subgraph"<<databaseRequest->_fileName<<" after "<<databaseRequest->_numOfRequests<<" requests and time="<<(timeStamp-databaseRequest->_timestampFirstRequest)*1000.0<<std::endl;

         double timeToMerge = timeStamp - databaseRequest->_timestampFirstRequest;

         if (timeToMerge < _minimumTimeToMergeTile) _minimumTimeToMergeTile = timeToMerge;
         if (timeToMerge > _maximumTimeToMergeTile) _maximumTimeToMergeTile = timeToMerge;

         _totalTimeToMergeTiles += timeToMerge;
         ++_numTilesMerges;
      }
      else
      {
         OSG_INFO << "DatabasePager::addLoadedDataToSceneGraph() node in parental chain deleted, discarding subgaph." << std::endl;
      }

      // reset the loadedModel pointer
      databaseRequest->_loadedModel = 0;

      // OSG_NOTICE<<"curr = "<<timeToMerge<<" min "<<getMinimumTimeToMergeTile()*1000.0<<" max = "<<getMaximumTimeToMergeTile()*1000.0<<" average = "<<getAverageTimToMergeTiles()*1000.0<<std::endl;
   }

   last = osg::Timer::instance()->tick();

   if (!localFileLoadedList.empty())
   {
      OSG_INFO << "Done DatabasePager::addLoadedDataToSceneGraph" <<
         osg::Timer::instance()->delta_m(before, mid) << "ms,\t" <<
         osg::Timer::instance()->delta_m(mid, last) << "ms" <<
         "  objects" << localFileLoadedList.size() << std::endl << std::endl;
   }
}

void me_DatabasePager::me_ReadQueue::clearRequests(DatabaseRequestList& requestList, DatabasePagerConfig::ThreadWorkloadType workloadType,
   const std::vector<std::string>& extensions /*= std::vector<std::string>()*/, me_RequestIdentifierFunctor* requestIdentifier /*= NULL*/)
{
   bool hasExtensions = !extensions.empty();
   bool sharedRequestQueue = (_name == SHARED_REQUEST_QUEUE);

   OpenThreads::ScopedLock<OpenThreads::Mutex> lock(_requestMutex);

   for (RequestQueue::RequestList::iterator itr = _requestList.begin(); itr != _requestList.end();)
   {
      bool sameWorkloadType = true;
      if (sharedRequestQueue)
      {
         sameWorkloadType = ((static_cast<me_DatabaseRequest*>(itr->get()))->_workloadType == workloadType);
      }

      bool sameExtension = true;
      if (hasExtensions && sameWorkloadType)
      {
         sameExtension = (std::find(extensions.begin(), extensions.end(), osgDB::getLowerCaseFileExtension((*itr)->_fileName)) != extensions.end());
      }

      bool deleteRequest = false;
      if (sameWorkloadType && sameExtension)
      {
         deleteRequest = true;
         if (NULL != requestIdentifier)
         {
            deleteRequest = ((*requestIdentifier)(static_cast<me_DatabaseRequest*>(itr->get())));
         }
      }

      if (deleteRequest)
      {
         requestList.push_back(static_cast<me_DatabaseRequest*>(itr->get()));
         itr = _requestList.erase(itr);
      }
      else
      {
         ++itr;
      }
   }
}

void me_DatabasePager::me_ReadQueue::clearAllData()
{
   release();
   clear();

   {
      OpenThreads::ScopedLock<OpenThreads::Mutex> lock(_requestMutex);
      _childrenToDeleteList.clear();
   }
}
