#include <sstream>
#include <time.h>
#include <osg/MatrixTransform>
#include <osg/ShapeDrawable>
#include <osg/PointSprite>
#include <osg/BlendFunc>
#include <osg/FrontFace>
#include <osg/CullFace>
#include <osg/Program>
#include <osg/Camera>
#include <osg/Point>
#include <osg/Shape>
#include <osg/Depth>
#include <osg/Quat>
#include <osg/Geometry>
#include <osg/Geode>
#include <osg/StateSet>
#include <osg/Billboard>
#include <osg/Texture2D>
#include <osgUtil/CullVisitor>
#include <osgDB/ReadFile>
#include "config/MapConfig.h"
#include "config/LightingScriptSystem.h"
#include "common/utils/GeoMath.h"
#include "common/utils/GeoRect.h"
#include "common/utils/OSGUtils.h"
#include "common/utils/StateSet.h"
#include "common/types/global/UniformUtils.h"
#include "common/types/global/MapContextualInfo.h"
#include "common/types/global/MapGlobalInfo.h"
#include "common/types/base/ViewBase.h"
#include "common/types/global/CameraParameters.h"
#include "common/types/interface/IWeatherControlSystem.h"
#include "common/types/interface/IWeatherEffectsEngine.h"
#include "SkyNodeWithCloud.h"
#include "oss/sky/IcoSphereCreator.h"

using namespace mapengine::oss;
using namespace mapengine::terrain;
using namespace mapengine::config;
using namespace mapengine::types;
using namespace mapengine::utils;

// A utility functions which are only used here
namespace mapengine {
namespace terrain {
   // linear function f(x) created from p1 p2
   float linearFunction(const osg::Vec2& p1, const osg::Vec2& p2, float x) {
      return (p2[1] - p1[1]) / (p2[0] - p1[0])*(x - p1[0]) + p1[1];
   }
   // Linear function with clamp
   float linearFunctionWithClamp(const osg::Vec2& p1, const osg::Vec2& p2, float x) {
      float min_ = osg::minimum(p1[0], p2[0]);
      float max_ = osg::maximum(p1[0], p2[0]);
      x = osg::minimum(max_, osg::maximum(min_, x));
      return linearFunction(p1, p2, x);
   }
}
}

namespace mapengine {
namespace terrain {
   static char s_atmosphereVertexExtShader[] = {
      // Cloud code
      "varying vec3 atmos_v3WorldPos; \n"
      "void mainExt() {\n"
      "  atmos_v3WorldPos = gl_Vertex.xyz;\n"
      "}"

   };
   static char s_atmosphereFragmentExtShader[] = {

      // cloud's declaration
      "varying vec3 atmos_v3WorldPos; \n"
      "uniform sampler2D atmos_cloudNoiseTexture; \n"
      "uniform mat4 osg_ViewMatrixInverse;     // camera position \n"
      "uniform mat4 atmos_mat4CameraFrame; \n"
      "uniform vec3 sunAmbient; // (1.7955, 1.7871, 1.89);\n"
      "uniform vec2 perturbTextureOffset;// = vec2(0.5, 0.5)*0.0005;\n"
      "uniform vec2 cloudTextureOffset;// = vec2(0.5, 0.5)*0.0005;\n"
      "uniform float cloudiness;\n"
      "uniform vec3 cloudExtincColor;// = vec3(0.5, 0.5, 0.5);\n"
      "uniform vec3 sunDir;// = vec3(0.5, 0.5, 0.5);\n"
      "uniform float skycolorInterp;// = 1.0;\n"
      "uniform vec3 artistSkyColor;// = vec3(0.0);\n"
      "uniform float parabolicMapCurveness;// = 0.02;\n"
      "uniform float parabolicMapScale; // 9.0\n"
      "uniform vec4 horizonClamp;// = vec4(0.06, 0.1, 0.9, 1.0);\n"
      "uniform float cloudBlendFactor;// = 1.0;\n"

      // cloud
      "\n"
      "// Parabolic mapping (winner in our mapping test)\n"
      "vec2 ParabolicMap( vec3 view, float c) {\n"
      "// c should always be smaller than 1, smaller sky more looks like planar\n"
      "//const float c = 0.05;\n"
      "float one_minus_ysqr = 1.0 - view.z*view.z;\n"
      "float k = sqrt(1.0 - one_minus_ysqr + 4.0*one_minus_ysqr*c) - view.z;\n"
      "k = k *0.5 / one_minus_ysqr;\n"
      "return vec2(view.x, view.y)*k;\n"
      "}\n"
      "\n"
      "// Similar to smoothstep\n"
      "float linearStep(float edge0, float edge1, float x) {\n"
      "    return clamp((x - edge0) / (edge1 - edge0), 0.0, 1.0);\n"
      "}\n"
      "\n"
      "// Input: ray is the vector under camera frame, skyColor is the color of original sky\n"
      "vec4 computeCloudSky(vec3 ray, vec3 skyColor)\n"
      "{\n"
      "    const float PERTURB_FACTOR = 0.12;\n"
      "    const float EXTINCT_MAX_FACTOR = 16.0;\n"
      "\n"
      "    // Compute sun/moon ambient color or get it from input\n"
      "    // cloud mapping, it turns out parabolic is the best, cylinder is good but suffers from line artifacts and cannot handle the top distort\n"
      "    //   spherical has distortion at the horizon; plane mapping is too boring and show strong pattern. If you're interested, try them out and\n"
      "    //   tweak the parameters in mapping function\n"
      "    vec2 texCoord = ParabolicMap(ray,  parabolicMapCurveness);\n"
      "\n"
      "    // sample perturb (cloud texture's blue channel), since we just need perturbance (magnitude) along windDir\n"
      "    float cloudscale = parabolicMapScale;\n"
      "    vec4 perturb = texture2D(atmos_cloudNoiseTexture, texCoord*cloudscale + perturbTextureOffset);\n"
      "\n"
      "    // use perturb to affect cloud noise\n"
      "    vec4 cloud = texture2D(atmos_cloudNoiseTexture, vec2(perturb.b*PERTURB_FACTOR) + texCoord*cloudscale + cloudTextureOffset);\n"
      "\n"
      "    // Below line is to make the cloud look like rolling instead of drifting rigidly, which seems to be more realistic and partially looks interesting\n"
      "    // Can be commented out if the effect is not desired\n"
      "    cloud.r = (1.0 + 0.5*(perturb.r-0.5))*cloud.r;\n"
      "    \n"
      "#define CLOUD_NOISE_CLAMP\n"
      "#ifdef CLOUD_NOISE_CLAMP \n"
      "    // it's hard to tweak this, a suggestion would be: don't change these magic number, unless you know what you're doing\n"
      "    //    to understand more, please refer to http://freespace.virgin.net/hugo.elias/models/m_clouds.htm\n"
      "    float cloudcoverage =  1.0 - cloudiness;\n"
      "    float cloudfluffiness = 0.9;\n"
      "    cloud.r = 2.0*clamp(1.0 - pow(cloudfluffiness, max(0.0, cloud.r - cloudcoverage)*5.0)*1.0, 0.0, 1.0);\n"
      "    cloud.r = clamp(cloud.r, 0.0, 1.0);\n"
      "#endif\n"
      "\n"
      "    // horizon cloud merge, this line control how cloud will smooth out approaching horizon, just leave the magic here, basically a linear decrement\n"
      "    cloud.r = cloud.r*linearStep(horizonClamp.x, horizonClamp.y, ray.z)\n"
      "    *(1.0 - linearStep(horizonClamp.z, horizonClamp.w, ray.z))\n"
      "    *cloudBlendFactor;\n"
      "\n"
      "    // How sky color get extinct transmitting in cloud, approximation!\n"
      "    // I guess extinct is also related to sun dir and cloud thickness\n"
      "    // draw out the function will help a little bit\n"
      "    vec3 extinct = (cloudExtincColor*(EXTINCT_MAX_FACTOR*max(1.0, dot(ray, sunDir)+1.0)*0.5 + 1.0) )*cloudiness ;\n"
      "\n"
      "    // scale the skycolor, e.g. when raining or sandstorm, skycolor will change to dark, or another color\n"
      "    vec3 comp = mix(skyColor, artistSkyColor, (1.0 - skycolorInterp)*cloudBlendFactor); // original term\n"
      "\n"
      "    // blend sky color with cloud, not exp extinction, just interpolation, it means sky color left after block of cloud\n"
      "    vec3 transSkyColor = comp * (1.0-clamp(cloud.r*8.0,0.0,1.0));\n"
      "\n"
      "    // cloud color due to outscattering of cloud react to sun, in blue sky day, it should be white, however, in sunset or sunrise,\n"
      "    //   this color would change to red or yellow, this color is related to sun's ambient color.\n"
      "    vec3 cloudColor =  sunAmbient *exp(-cloud.r*extinct);\n"
      "    // Final color = extinct  + cloudColor, alpha channel is to block the sun's rendering\n"
      "    return vec4(transSkyColor + (cloudColor) *clamp(cloud.r*8.0, 0.0, 1.0), cloud.r*8.0);\n"
      "}\n"

      "void mainExt(vec3 color) {\n"
      // Cloud code
      "    vec3 viewPoint = osg_ViewMatrixInverse[3].xyz; \n"
      "    vec3 ray = normalize(atmos_v3WorldPos - viewPoint);\n"
      "    ray = (atmos_mat4CameraFrame * vec4(ray, 0.0)).xyz;\n"
      "    gl_FragColor = computeCloudSky(ray, color.rgb*atmos_fWeather);\n"
      "}\n"
      "\n"

   };
}

}

me_SkyNodeWithCloud::me_SkyNodeWithCloud(osg::Node* parent, float minStarMagnitude)
: me_SkyNode(parent, minStarMagnitude)
, _lastStampTime(0.0)
{
   setName("SkyNodeWithCloud");
}

void me_SkyNodeWithCloud::traverse(osg::NodeVisitor& nv)
{
   if (nv.getVisitorType() == osg::NodeVisitor::CULL_VISITOR)
   {
      cullTraversal(nv);
   }
   else
   {
      osg::Group::traverse(nv);
   }
}

void me_SkyNodeWithCloud::cullTraversal(osg::NodeVisitor& nv)
{
   osgUtil::CullVisitor* cv = me_fast_downcast<osgUtil::CullVisitor*>(&nv);

   // If there's a custom projection matrix clamper installed, remove it temporarily.
   // We don't want it mucking with our sky elements.
   osg::ref_ptr<osg::CullSettings::ClampProjectionMatrixCallback> cb = cv->getClampProjectionMatrixCallback();
   cv->setClampProjectionMatrixCallback(0L);

   // restore a custom clamper.
   if (cb.valid()) cv->setClampProjectionMatrixCallback(cb.get());

   // If our node are cull we definitely will do some update here
   // Get camera and calculate the frame
   me_ViewBase* view = me_fast_downcast<me_ViewBase*>(mapengine::utils::getView(cv));
   ViewID viewID = view->getViewID();
   me_ICamera* iCamera = me_MapGlobalInfo::getCameraInterface(viewID);
   osg::Matrixf mat4;

   if (iCamera)
   {
      // Get camera frame matrix that convert local -> world
      computeCoordinateFramef(static_cast<float>(iCamera->getCameraParameters().getViewPosition()._latitude),
         static_cast<float>(iCamera->getCameraParameters().getViewPosition()._longitude),
         mat4);
   }
   else
   {
      ME_TR_FATAL(TR_CLASS_ME_TERRAIN, ("Camera for view %d is NULL!", viewID));
   }

   // get weather control system from weather effects engine, if it is available
   me_IWeatherEffectsEngine* weatherEffectsEngine = me_MapContextualInfo::getWeatherEffectsEngineInterfacePtr();
   me_IWeatherControlSystem* weatherControlSystem = NULL;
   if (NULL != weatherEffectsEngine)
   {
      weatherControlSystem = weatherEffectsEngine->getWeatherControlSystem(viewID);
   }

   // Make sure it's orthogonal
   mat4 = osg::Matrix::orthoNormal(mat4);

   // Inverse, world -> local: For orthogonal matrix transpose == inverse
   // But funny I didn't find any transpose for mat4x4 in our engine
   osg::Matrixf transposedMat(
      mat4(0, 0), mat4(1, 0), mat4(2, 0), mat4(3, 0),
      mat4(0, 1), mat4(1, 1), mat4(2, 1), mat4(3, 1),
      mat4(0, 2), mat4(1, 2), mat4(2, 2), mat4(3, 2),
      mat4(0, 3), mat4(1, 3), mat4(2, 3), mat4(3, 3)
      );

   _cameraFrame->set(transposedMat);
   _cameraFrameMat4 = transposedMat;

   // Get camera position
   osg::Vec3 eyePoint = cv->getEyePoint();

   // Cloud parameters
   float longitude, latitude, height;
   cartesianToWgs84f(eyePoint, latitude, longitude, height);

   // Get local time in [0, 24)
   int year, month, date;
   double utcHours, localHours = 0.0;
   getDateTime(year, month, date, utcHours);
   localHours = mapengine::config::me_LightingScriptSystem::getLocalHours(longitude, utcHours);

   // Get sun elevation
   double sunElevation = mapengine::config::me_LightingScriptSystem::getSunElevation(latitude, longitude, _lightPos);

   // Sample based on sunElevation
   // SunAmbient
   osg::Vec3f sunAmbient = mapengine::config::me_LightingScriptSystem::getSunAmbient(sunElevation);
   if ((NULL != weatherEffectsEngine) && (NULL != weatherControlSystem) && (NULL != iCamera))
   {
      weatherControlSystem->moderateSunAmbient(weatherEffectsEngine->getTime(), iCamera->getCameraParameters().getViewPosition(), sunAmbient);
   }
   _sunAmbient->set(sunAmbient);

   // CloudExtinctColor
   osg::Vec3f extinctColor = mapengine::config::me_LightingScriptSystem::getCloudExtinctColor(sunElevation);
   if ((NULL != weatherEffectsEngine) && (NULL != weatherControlSystem) && (NULL != iCamera))
   {
      weatherControlSystem->moderateCloudExtinctColor(weatherEffectsEngine->getTime(), iCamera->getCameraParameters().getViewPosition(), extinctColor);
   }
   _cloudExtincColor->set(extinctColor);

   // Sample based on local hours
   float cloudiness = mapengine::config::me_LightingScriptSystem::getCloudiness(static_cast<float>(localHours));
   if ((NULL != weatherEffectsEngine) && (NULL != weatherControlSystem) && (NULL != iCamera))
   {
      cloudiness = weatherControlSystem->getCloudiness(weatherEffectsEngine->getTime(), iCamera->getCameraParameters().getViewPosition());
   }
   _cloudiness->set(cloudiness);

   // Get artist color
   osg::Vec4f artistColor = mapengine::config::me_LightingScriptSystem::getArtistColor(static_cast<float>(localHours));
   if ((NULL != weatherEffectsEngine) && (NULL != weatherControlSystem) && (NULL != iCamera))
   {
      weatherControlSystem->getArtistColor(weatherEffectsEngine->getTime(), iCamera->getCameraParameters().getViewPosition(), sunElevation, artistColor);
   }
   _artistSkyColor->set(osg::Vec3f(artistColor.x(), artistColor.y(), artistColor.z()));
   _skycolorInterp->set(1.0f - artistColor.w());

   // Get cloud texture offset 
   osg::Vec2 earthOffset = osg::Vec2(longitude, latitude)*mapengine::config::me_LightingScriptSystem::getRenderConstants().cloudTextureOffsetScalor;

   // Get windDirection and its perturb, we should stop using this hard coded
   osg::Vec2 windDir;
   osg::Vec2 perturbDir;
   if ((NULL != weatherEffectsEngine) && (NULL != weatherControlSystem) && (NULL != iCamera))
   {
      weatherControlSystem->getWindVector(weatherEffectsEngine->getTime(), iCamera->getCameraParameters().getViewPosition(), windDir);
      perturbDir = osg::Vec2(1.0, 1.0);
      perturbDir = perturbDir / perturbDir.length()*3.0;
   }
   else
   {
      windDir = osg::Vec2(5.5f, 3.5f);
      perturbDir = osg::Vec2(1.0f, -windDir.x() / windDir.y()); // vertical direction, actually you can select any direction
      perturbDir = perturbDir / perturbDir.length() * windDir.length();
   }

   // Disable cloud animation if animations are disabled from the registry
   double currentStamp = nv.getFrameStamp()->getReferenceTime();
   float deltaTime = (0 == getConfigValue<TestingGetter>().disableAnimations) ? static_cast<float>((currentStamp - _lastStampTime)*0.001) : 0.0f; //MS Seconds
   _lastStampTime = currentStamp;
   _windOffset = _windOffset + windDir*deltaTime;
   // TODO: in case windOffset get too large, we lose precision, we need to find the scale boundary and reset it

   _perturbTextureOffset->set(earthOffset + perturbDir*deltaTime);
   _cloudTextureOffset->set(earthOffset + _windOffset);

   calculateRenderConstants(height);

   _cullContainer->accept(nv);
}

void me_SkyNodeWithCloud::calculateRenderConstants(float& height)
{
   // Calculate 2 horizontal clamp (in our requirement we need to clamp the cloud into horizon when zoom out at 
   //    certain level) and blend factor to make cloud finally disappear when zoom out a lot...
   const SkyCloudRenderConstantsConfig& renderConstants = mapengine::config::me_LightingScriptSystem::getRenderConstants();
   float curvness = linearFunctionWithClamp(osg::Vec2(renderConstants.curvenessLinearParameters.x1, renderConstants.curvenessLinearParameters.y1),
      osg::Vec2(renderConstants.curvenessLinearParameters.x2, renderConstants.curvenessLinearParameters.y2),
      height);
   _parabolicMapCurveness->set(curvness);
   // How cloud texture scale when zoom in/out
   float scale = linearFunctionWithClamp(osg::Vec2(renderConstants.cloudScaleLinearParameters.x1, renderConstants.cloudScaleLinearParameters.y1),
      osg::Vec2(renderConstants.cloudScaleLinearParameters.x2, renderConstants.cloudScaleLinearParameters.y2),
      height);
   _parabolicMapScale->set(scale);
   // How cloud clamp when zoom out
   float upperAngle = linearFunctionWithClamp(osg::Vec2(renderConstants.cloudUpperClampLinearParameters.x1, renderConstants.cloudUpperClampLinearParameters.y1 - renderConstants.upperSkylineRange),
      osg::Vec2(renderConstants.cloudUpperClampLinearParameters.x2, renderConstants.cloudUpperClampLinearParameters.y2),
      height);

   float lowerAngle = linearFunctionWithClamp(osg::Vec2(renderConstants.cloudLowerClampLinearParameters.x1, renderConstants.cloudLowerClampLinearParameters.y1),
      osg::Vec2(renderConstants.cloudLowerClampLinearParameters.x2, renderConstants.cloudLowerClampLinearParameters.y2),
      height);

   static const float pi_2 = static_cast<float>(osg::PI_2);
   _horizonClamp->set(osg::Vec4(sinf(pi_2*(lowerAngle - renderConstants.lowerSkylineRange)), sinf(lowerAngle*pi_2), sinf(upperAngle*pi_2), sinf(pi_2*(upperAngle + renderConstants.upperSkylineRange))));
   // How cloud are blend when zoom in/out
   float blendFactor = linearFunctionWithClamp(osg::Vec2(renderConstants.blendFactorLinearParameters.x1, renderConstants.blendFactorLinearParameters.y1),
      osg::Vec2(renderConstants.blendFactorLinearParameters.x2, renderConstants.blendFactorLinearParameters.y2),
      height);
   _cloudBlendFactor->set(blendFactor);

   // Avoid complex clouds fragment shader calculations at higher scales by switching the Atmosphere shader program based on visibility of the Clouds.
   osg::ref_ptr<osg::StateSet> atmosStateset = getAtmosphereStateset();
   if (atmosStateset.valid())
   {
      if (blendFactor > 0.0)
      {
         atmosStateset->setAttributeAndModes(_atomsphereWithCloudsShaderProgram, osg::StateAttribute::ON);
      }
      else
      {
         atmosStateset->setAttributeAndModes(_basicAtmosphereShaderProgram, osg::StateAttribute::ON);
      }
   }
}

void me_SkyNodeWithCloud::loadCloudNoise()
{
   // TODO: check valid etc.
   osg::ref_ptr<osg::Image> cloudNoiseImg;
   cloudNoiseImg = osgDB::readImageFile(mapengine::config::getConfigValue<mapengine::config::CloudNoiseTextureGetter>().getRequestFilename());

   if (cloudNoiseImg)
   {
      _cloudNoiseTexture = new osg::Texture2D(cloudNoiseImg.get());
      _cloudNoiseTexture->setFilter(osg::Texture::MIN_FILTER, osg::Texture::LINEAR_MIPMAP_NEAREST);
      _cloudNoiseTexture->setFilter(osg::Texture::MAG_FILTER, osg::Texture::LINEAR);
      _cloudNoiseTexture->setWrap(osg::Texture2D::WRAP_S, osg::Texture2D::REPEAT);
      _cloudNoiseTexture->setWrap(osg::Texture2D::WRAP_T, osg::Texture2D::REPEAT);
   }
}

osg::Geode* me_SkyNodeWithCloud::makeAtmosphere(const osg::EllipsoidModel* em)
{
   // Load images
   loadCloudNoise();

   // make old atmosphere
   osg::Geode* geode = me_SkyNode::makeAtmosphere(em);

   // Fix stuff of old
   osg::StateSet* set = geode->getOrCreateStateSet();
   set->setAttributeAndModes(new osg::BlendFunc(GL_ONE, GL_ONE_MINUS_SRC_ALPHA), osg::StateAttribute::ON);

   _basicAtmosphereShaderProgram = me_fast_downcast<osg::Program*>(set->getAttribute(osg::StateAttribute::PROGRAM, 0));
   _atomsphereWithCloudsShaderProgram = generateAtmosphereShaderProgram("AtmosphereWithCloudsShader", s_atmosphereVertexExtShader, s_atmosphereFragmentExtShader);
   MAP_ASSERT(_basicAtmosphereShaderProgram.valid() && _atomsphereWithCloudsShaderProgram.valid());

   // Set cloud texture for shader use @ channel 0
   set->setTextureAttribute(0, _cloudNoiseTexture, osg::StateAttribute::OFF | osg::StateAttribute::PROTECTED);

   // Uniform for cloud
   set->getOrCreateUniform("atmos_cloudNoiseTexture", osg::Uniform::SAMPLER_2D)->set(0);
   // Runtime changing uniform
   _cameraFrame = new osg::Uniform("atmos_mat4CameraFrame", osg::Matrixf::identity());
   set->addUniform(_cameraFrame);
#define uniformInitLessTyping(x, y) _##x = new osg::Uniform(#x, y);\
    set->addUniform(_##x)
   uniformInitLessTyping(sunAmbient, osg::Vec3(1.7955f, 1.7871f, 1.89f));
   uniformInitLessTyping(perturbTextureOffset, osg::Vec2(0.0f, 0.0f));
   uniformInitLessTyping(cloudTextureOffset, osg::Vec2(0.0f, 0.0f));
   uniformInitLessTyping(cloudiness, 0.5f);
   uniformInitLessTyping(cloudExtincColor, osg::Vec3(0.5f, 0.5f, 0.5f));
   uniformInitLessTyping(sunDir, osg::Vec3(0.5f, 0.5f, 0.5f));
   uniformInitLessTyping(skycolorInterp, 1.0f);
   uniformInitLessTyping(artistSkyColor, osg::Vec3(0.0f, 0.0f, 0.0f));
   uniformInitLessTyping(parabolicMapCurveness, 0.02f);
   uniformInitLessTyping(parabolicMapScale, 1.0f);
   uniformInitLessTyping(horizonClamp, osg::Vec4(0.0f, 0.06f, 0.9f, 1.0f));
   uniformInitLessTyping(cloudBlendFactor, 1.0f);
#undef uniformInitLessTyping
   return geode;
}

osg::ref_ptr<osg::StateSet> me_SkyNodeWithCloud::getAtmosphereStateset() const
{
   osg::ref_ptr<osg::StateSet> ss;
   osg::ref_ptr<osg::Camera> cam = me_fast_downcast<osg::Camera*>(_atmosphere.get());
   if (cam.valid())
   {
      osg::ref_ptr<osg::Geode> atmosphereGeode = cam->getChild(0)->asGeode();
      if (atmosphereGeode.valid())
      {
         ss = atmosphereGeode->getStateSet();
      }
   }

   MAP_ASSERT(ss.valid());
   return ss;
}
