/** @file
 *
 * ADIT Pixmap API header.
 *
 * @copyright:	(c) 2013 ADIT Corporation
 * @author	Vincent Stehle <vincent.stehle@freescale.com>
 */

#ifndef __APX_H_INCLUDED__
#define __APX_H_INCLUDED__

#include <wayland-client.h>

/** @struct apx
 *
 * API context.
 *
 * This, to make this API thread-safe.
 */
struct apx {
    void *viv;
};

/**
 * \brief Enumeration for supported pixelformats
 **/
typedef enum e_apxPixelFormat
{
    APX_PIXELFORMAT_R_8 = 0,           /*!< Pixelformat value, to describe a 8 bit luminance surface */
    APX_PIXELFORMAT_GR_88 = 1,         /*!< Pixelformat value, to describe a GR88 */
    APX_PIXELFORMAT_ARGB_8888 = 2,     /*!< Pixelformat value, to describe a 24 bit rgb surface with 8 bit alpha */
    APX_PIXELFORMAT_ABGR_8888 = 3,     /*!< Pixelformat value, to describe a 24 bit bgr surface with 8 bit alpha */
    APX_PIXELFORMAT_XRGB_8888 = 4,     /*!< Pixelformat value, to describe a 24 bit rgb surface with 8 bit ignored */
    APX_PIXELFORMAT_XBGR_8888 = 5,     /*!< Pixelformat value, to describe a 24 bit bgr surface with 8 bit ignored */
    APX_PIXELFORMAT_RGBA_8888 = 6,     /*!< Pixelformat value, to describe a 24 bit rgb surface with 8 bit alpha */
    APX_PIXELFORMAT_BGRA_8888 = 7,     /*!< Pixelformat value, to describe a 24 bit bgr surface with 8 bit alpha */
    APX_PIXELFORMAT_RGBX_8888 = 8,     /*!< Pixelformat value, to describe a 24 bit rgb surface with 8 bit ignored */
    APX_PIXELFORMAT_BGRX_8888 = 9,     /*!< Pixelformat value, to describe a 24 bit bgr surface with 8 bit ignored */
    APX_PIXELFORMAT_RGB_888 = 10,      /*!< Pixelformat value, to describe a 24 bit rgb surface */
    APX_PIXELFORMAT_RGB_565 = 11,      /*!< Pixelformat value, to describe a 16 bit rgb surface */
    APX_PIXELFORMAT_RGBA_5551 = 12,    /*!< Pixelformat value, to describe a 16 bit rgb surface, with binary mask */
    APX_PIXELFORMAT_RGBA_6661 = 13,    /*!< Pixelformat value, to describe a 18 bit rgb surface, with binars mask */
    APX_PIXELFORMAT_RGBA_4444 = 14,    /*!< Pixelformat value, to describe a 12 bit rgb surface, with 4 bit alpha */
    APX_PIXEL_FORMAT_UNKNOWN = 15,     /*!< Pixelformat not known */
    APX_PIXELFORMAT_RGBX_4444 = 16,    /*!< Pixelformat value, to describe a 12 bit rgb surface, with 4 bit ignored */
    APX_PIXELFORMAT_YUV420 = 17,       /*!< Pixelformat value, to describe a 4:2:0 planar YUV surface */
    APX_PIXELFORMAT_UYVY = 18,         /*!< Pixelformat value, to describe a 4:2:2 packed YUV surface */
    APX_PIXELFORMAT_YUY2 = 19,         /*!< Pixelformat value, to describe a 4:2:2 packed YUV surface */
    APX_PIXELFORMAT_YV12 = 20,         /*!< Pixelformat value, to describe a 4:2:0 semi planar YUV surface */
    APX_PIXELFORMAT_NV12 = 21,         /*!< Pixelformat value, to describe a 4:2:0 planar YUV surface */
    APX_PIXELFORMAT_NV16 = 22          /*!< Pixelformat value, to describe a 4:2:2 planar YUV surface */
} apxPixelFormat;

/** @struct apx_buffer
 *
 * Opaque type, representing a buffer.
 *
 * This is opaque, in order not to expose at the API the specific types it
 * contains.
 */
struct apx_buffer;

int apx_global_handler(void *data, struct wl_registry *wl_registry,
			uint32_t name, const char *interface,
			uint32_t version);

int apx_init(struct apx *a);
int apx_deinit(struct apx *a);

int apx_authenticate(struct apx *a, struct wl_display *d, unsigned int drm_magic);

struct apx_buffer *apx_buffer_create(struct apx * a, unsigned width,
					unsigned height, apxPixelFormat format);

struct apx_buffer *apx_buffer_create_unbuffered(struct apx * a, unsigned width,
					unsigned height, apxPixelFormat format);

/* For apx-dma-based applications
  * arg p -array of void pointers should have the dma_fd
  * arg l -array of void pointers should have the offset
  */
struct apx_buffer *apx_buffer_wrap(struct apx *a, void *const *p, void *const *l,
			unsigned width, unsigned height, unsigned stride,
			apxPixelFormat format);

int apx_buffer_destroy(struct apx_buffer *b, struct wl_display *d);
struct wl_buffer *apx_buffer_get_wl(struct apx_buffer *b);
int apx_buffer_get_drmfd(struct apx *a);
int apx_buffer_get_dmafd(struct apx_buffer *b);
int apx_buffer_map(struct apx_buffer *b, void **p);
int apx_buffer_unmap(struct apx_buffer *b);
int apx_buffer_get_stride(struct apx_buffer *b, unsigned *stride);

int apx_buffer_get_size(struct apx_buffer *b, unsigned *width,
			unsigned *height);

apxPixelFormat apx_buffer_get_format(struct apx_buffer *b);

int apx_buffer_commit(struct apx *a, struct apx_buffer *b, struct wl_surface *s, int x, int y,
            int width, int height, struct wl_display *d);

int apx_buffer_set_double_buffered(struct apx_buffer *b);
void apx_buffer_set_no_copy(struct apx_buffer *b);

/**
 * Unlock call-back for struct apx_buffer.
 *
 * It is possible to use apx_buffer_set_unlock_callback() to register a call-back
 * function that will be called when the buffer is no longer visible on screen.
 *
 * @param buffer APX buffer the call-back function was registered on.
 * @param data user data passed to apx_buffer_set_unlock_callback() on registration.
 * @retval 1 if the call-back should be removed after processing this call.
 * @retval 0 otherwise.
 *
 * @ingroup buffer_handling
 */
typedef int (*ApxUnlockCallback)(struct apx_buffer* buffer, void *data);

/**
 * Destroy call-back for user data of unlock call-back.
 *
 * If a call-back was registered with apx_buffer_set_unlock_callback() and the
 * call-back is still active upon buffer destruction, it is possible to have the
 * notification user data destroyed automatically by passing a function of this
 * type as the notify parameter.
 *
 * The function will be called with the data pointer passed on call-back
 * registration.
 * @param data user data pointer passed on apx_buffer_set_unlock_callback() call.
 * @ingroup buffer_handling
 */
typedef void (*ApxCallbackDestroyNotify)(void *data);

void apx_buffer_set_unlock_callback(struct apx_buffer *b, ApxUnlockCallback callback,
                                   void *data, ApxCallbackDestroyNotify notify);

void * apx_buffer_get_user_data(struct apx_buffer *b);

#endif /* __APX_H_INCLUDED__ */
