/**
 * \file: libsugc.h
 *
 * \brief : Library to switch user, group, supplementary groups and capabilities
 *          according to a named configuration
 *
 * \author: Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2017 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#ifndef LIBSUGC
#define LIBSUGC

#ifdef __cplusplus
extern "C" {
#endif

/**
 * \brief make sure that external API is visible
 */
#define LIBSUGC_API __attribute__((__visibility__("default")))

/**
 * \defgroup libsugc_api libsugc external API
 */

/**
 * \ingroup libsugc_api
 * \brief error type for libsugc
 */
typedef enum libsugc_error_t
{
    LIBSUGC_OK,                             /**< no error */
    LIBSUGC_CONFIG_ACCESS_FAILED,           /**< accessing the config file failed */
    LIBSUGC_CONFIG_INSECURE,                /**< config file is not owned by root or writable for non-root - an attacker might have altered it */
    LIBSUGC_INVALID_CONFIG_NAME,            /**< the given config name is invalid or can't be found */
    LIBSUGC_INVALID_USER,                   /**< invalid user configured */
    LIBSUGC_INVALID_GROUP,                  /**< invalid group configured */
    LIBSUGC_INVALID_CAPABILITIES,           /**< invalid capabilitiy string configured */
    LIBSUGC_MANDATORY_PARAMETER_MISSING,    /**< mandatory configuration entry missing - i.e. user */
    LIBSUGC_DOUBLE_OR_UNKNOWN_KEY,          /**< unknown or double key found in configuration */
    LIBSUGC_INVALID_PARAMETER,              /**< invalid parameter passed to internal functions */
    LIBSUGC_NO_MEM,                         /**< memory allocation failed */
    LIBSUGC_FAILED,                         /**< operation failed for some reason - please check logs and strace */
} libsugc_error_t;

/**
 * \ingroup libsugc_api
 * \brief Apply the specified configuration for user/group/supplementary groups and capabilities
 *
 * The calling application is supposed to call this function at the very beginning of main,
 * before doing any other initialization. This is required to ensure that the correct
 * user/group etc. settings are used for initialization as well.
 *
 * NOTE: In order to be able to perform all these changes the calling application needs
 * to be started as root with CAP_SETUID capability.
 *
 * This function will read the configuration specified by name from the config (/etc/libsugc.conf)
 * and set user/group/supplementary groups and capabilities accordingly.
 *
 * Configuration entry:\n
 * ...\n
 * [\<CONF_NAME\>]\n
 * User=\<USER_NAME\>\n
 * Group=\<GROUP_NAME\>\n
 * Capabilities=\<CAPABILITY STRING\>\n
 * ...
 *
 * Setting the user is mandatory.\n
 * Group and capabilities are optional.\n
 * In case group is not explicitly set the primary group of the user is used.\n
 * In case capabilities are not explicitly set all capabilities will be dropped.\n
 * Capabilities are described in the format of cap_from_text (see man page of cap_from_text).
 *
 * The supplementary groups are always set to the supplementary groups of the user.\n
 * Extending the list of supplementary groups is currently not supported.
 *
 * In case of any error the fallback configuration will be set and \ref libsugc_error_t error is returned.\n
 * The calling application is supposed to terminate with error in this case (i.e. result != \ref LIBSUGC_OK)
 *
 * If the specified configuration was applied successfully \ref LIBSUGC_OK is returned.
 *
 * \param[in] config_name   name of the desired configuration
 *
 * \return \ref LIBSUGC_OK                      - Success
 * \return else \ref libsugc_error_t error code - Application is supposed to terminate in case of any error
 */
LIBSUGC_API libsugc_error_t libsugc_apply_cfg(const char *config_name);

#ifdef __cplusplus
}
#endif

#endif
