// Copyright 2014 Google Inc. All Rights Reserved.

#ifndef ANDROID_AUTO_PROJECTION_PROTOCOL_ICONTROLLER_CALLBACKS_H
#define ANDROID_AUTO_PROJECTION_PROTOCOL_ICONTROLLER_CALLBACKS_H

#include "common.h"

/**
 * This class represents a general set of callbacks that must be set up for the GAL receiver
 * library to be able to function properly.
 */
class IControllerCallbacks {
public:
    virtual ~IControllerCallbacks() { }
    /**
     * Called when service discovery request is received. This call sends across an icon set
     * and a label that can be used by the native UI to display a button that allows users to
     * switch back to projected mode.
     * @param smallIcon 32x32 png image.
     * @param mediumIcon 64x64 png image.
     * @param largeIcon 128x128 png image.
     * @param label A label that may be displayed alongside the icon.
     * @param deviceName A friendly device name.
     */
    virtual void serviceDiscoveryRequestCallback(const string& smallIcon, const string& mediumIcon,
            const string& largeIcon, const string& label, const string& deviceName) = 0;

    /**
     * Called when an unrecoverable error has been encountered. The recommended course
     * of action is to reset the usb link at this point. A re-establishment of the
     * GAL connection may be attempted after that.
     *
     * The current set of error codes is:
     *  - STATUS_AUTHENTICATION_FAILURE if the SSL handshake fails.
     *  - STATUS_FRAMING_ERROR if an error occurs during communication with the
     *    MD. Examples of errors include (but are not limited to) IO errors and
     *    SSL decryption errors.
     */
    virtual void unrecoverableErrorCallback(MessageStatus err) = 0;

    /**
     * Called when the other end pings us.
     * @param timestamp The (remote) timestamp of the request.
     * @param bugReport Should a bug report be saved away? The implementer can choose
     *        to ignore this if it is sent with high frequency. It is in the end, the
     *        implementers responsibility to ensure that the system cannot be DoS'd
     *        if too many requests for bug reports are received.
     */
    virtual void pingRequestCallback(int64_t timestamp, bool bugReport) = 0;

    /**
     * Called when the other end responds to our ping.
     * @param timestamp The timestamp we wrote on the original ping request.
     */
    virtual void pingResponseCallback(int64_t timestamp) = 0;

    /**
     * Called when a navigation focus request is received from the phone. You must respond
     * to this by calling Controller::setNavigationFocus() (even if there is no change in
     * navigation focus). If navigation focus is given to the mobile device, all native
     * turn by turn guidance systems must be stopped.
     * @param type The type requested (can be NAV_FOCUS_NATIVE or NAV_FOCUS_PROJECTED).
     */
    virtual void navigationFocusCallback(NavFocusType type) = 0;

    /**
     * Called when ByeByeRequest is received from phone. After taking necessary steps,
     * car side should send ByeByeResponse.
     * @param reason The reason for the disconnection request.
     */
    virtual void byeByeRequestCallback(ByeByeReason reason) = 0;

    /**
     * Called when ByeByeResponse is received from phone. Normally this is a reply for
     * ByeByeRequest message sent from car.
     */
    virtual void byeByeResponseCallback() = 0;

    /**
     * Called when a voice session notification is received. Note that this callback only applies
     * to you if you do not always send a PTT short press to us always. If you always send PTT
     * short press to us, you should be able to ignore this call altogether.
     * @param status The status of the voice recongition session.
     */
    virtual void voiceSessionNotificationCallback(VoiceSessionStatus status) = 0;

    /**
     * Called when the source wishes to acquire audio focus.
     * @param request Can be one of AUDIO_FOCUS_GAIN, AUDIO_FOCUS_GAIN_TRANSIENT,
     *        AUDIO_FOCUS_GAIN_TRANSIENT_MAY_DUCK, AUDIO_FOCUS_RELEASE.
     */
    virtual void audioFocusRequestCallback(AudioFocusRequestType request) = 0;
};

#endif // ANDROID_AUTO_PROJECTION_PROTOCOL_ICONTROLLER_CALLBACKS_H
