/*
 * ecnr-service.c
 *
 *
 * Author: Patrick Rey
 * Date: 15.08.2013
 * Version 0.1 :  adaption qwa-service.c as ecnr-service.c
 */



#include <stdlib.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-bindings.h>
#include <string.h>
#include <signal.h> //new
#include <glib/gi18n.h>
#include <glib-object.h>
#include <glib-unix.h> //new
#include <glib.h>
#include <locale.h>




#include "ecnr-object.h"
#include "ecnr-service.h"
#include "ecnr-common-defs.h"
#include "ecnr-alsa.h"
#include "ecnr-configuration.h"
#include "ecnr-engine-handler.h"
#include "ecnr-sse-handler.h"
#include "ecnr-datapool-access.h"
#include "../utest/ecnr-unitest-server.h"
#include "ecnr-ioctrl.h"


/*******************************************************************************
              Constant defintion
*******************************************************************************/
#define ECNR_SERVICE_NAME "org.bosch.ecnr.service"
#define ECNR_ERROR(name) { name, #name }

/*******************************************************************************
              VARIABLE DECLARATIONS
*******************************************************************************/
static struct ecnr_instance_object*     my_obj;
static GObject                          *obj;
GMainLoop                               *ecnr_main_loop;
GMutex                                  *ecnr_service_lock;
static guint                            ecnr_service_sig_source_id[3];
static gboolean                         ecnr_service_init_state = FALSE;
static gboolean                         ecnr_service_info_select_flag = FALSE;
static gint                             ecnr_service_test_suite = 0;
static gchar*                           ecnr_service_info_select_param = NULL;
static gboolean                         ecnr_service_tuning_server_state = FALSE;
#ifdef D_TRIGGER_PERFORMANCE_TEST
static guint                            ecnr_service_performance_test_nb = 0;
#endif
static gboolean                         ecnr_service_pass_through_mode_default_state = FALSE;
static gint16                           ecnr_service_mic_gain_level = 0; //default mic gain level is set 0 dB
static gboolean                         ecnr_service_use_default_dp = FALSE;
//static GString                          *strbuf = NULL; // g_string_new(NULL); /* This will spawn an empty string */




/*******************************************************************************
*
* FUNCTION: ecnr_error_str
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/

static gboolean ecnr_service_cb_info_select
(
    const gchar *option_name,
    const gchar *value,
    gpointer data,
    GError **error
)
{
  ecnr_service_info_select_param = g_strdup (value);
    ecnr_service_info_select_flag = TRUE;

    if (!g_strcmp0(ecnr_service_info_select_param, "datapool"))
    {
    #ifndef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
        pr_message("-----------  ecnr datapool info ------------\n");
        ecnr_datapool_access_print_datapool_infos();
    #else
        pr_message("datapool not supported\n");
    #endif
    }
    else if (!g_strcmp0(ecnr_service_info_select_param, "ecnr-config"))
    {
        pr_message("-----------  ecnr config infos ------------\n");
        ecnr_configuration_print_user_config();
    }
    else if (!g_strcmp0(ecnr_service_info_select_param, "tuning-trigger"))
    {
        pr_message("-----------  ecnr tuning trigger info ------------\n");
        pr_message("In order to setup the tuning mode of ecnr deamon either\n");
        pr_message("for qnx or nuance engines client tools please install following file in rfs:\n");
        pr_message("touch %s\n", DEBUG_TIGGER_FILE);
    }
    else
    {
        ecnr_service_info_select_flag = FALSE;
        pr_message("info paramter not available\n");
    }


  return TRUE;
}



static const GOptionEntry ecnr_service_opts[] =
{
  {
    .long_name = "utest",
    .short_name = 'u',
    .flags = 0,
    .arg = G_OPTION_ARG_INT,
    .arg_data = &ecnr_service_test_suite,
    .description = "run ecnr unitest, option:[suite number 1(suite1),2(suite2),..,255(all suites)]",
    .arg_description = "U"
  },
  {
    .long_name = "infos",
    .short_name = 'i',
    .flags = 0,
    .arg = G_OPTION_ARG_CALLBACK,
    .arg_data = &ecnr_service_cb_info_select,
    .description = "retrieve ecnr infos, options:[datapool, ecnr-config,tuning-trigger]",
    .arg_description = "I"
  },
  { NULL }
};

/* table of ECNR error codes */
static const struct EcnrError ECNR_ERROR_table[] = {

 /* SSE specifical error*/
  ECNR_ERROR(SSE_ERR_OK),
  ECNR_ERROR(SSE_ERR_NO_RESULT),
  ECNR_ERROR(SSE_ERR_SIZE_ONLY),
  ECNR_ERROR(SSE_ERR_ALLOC),
  ECNR_ERROR(SSE_ERR_NULL_POINTER),
  ECNR_ERROR(SSE_ERR_MODULE_NOT_INIT),
  ECNR_ERROR(SSE_ERR_NOT_CREATED),
  ECNR_ERROR(SSE_ERR_MODULE_ALREADY_INIT),
  ECNR_ERROR(SSE_ERR_INVALID_MAIN_STRUCT),
  ECNR_ERROR(SSE_ERR_DEINIT_FAILED),
  ECNR_ERROR(SSE_ERR_ILLEGAL_SET_PARAM),
  ECNR_ERROR(SSE_ERR_ILLEGAL_SET_PARAM_BEFORE_INIT),
  ECNR_ERROR(SSE_ERR_ILLEGAL_SET_PARAM_AFTER_INIT),
  ECNR_ERROR(SSE_ERR_INVALID_PARAM_ID),
  ECNR_ERROR(SSE_ERR_INVALID_CHANNEL_NUMBER),
  ECNR_ERROR(SSE_ERR_INVALID_OPERATION_MODE),
  ECNR_ERROR(SSE_ERR_INVALID_SIZE),
  ECNR_ERROR(SSE_ERR_PARAM),
  ECNR_ERROR(SSE_ERR_PARAM_CNT),
  ECNR_ERROR(SSE_ERR_SAMPLE_RATE),
  ECNR_ERROR(SSE_ERR_FRAMESHIFT),
  ECNR_ERROR(SSE_ERR_INVALID_CHAN_CNT),
  ECNR_ERROR(SSE_ERR_FFT_LENGTH),
  ECNR_ERROR(SSE_ERR_INVALID_FILTER_LEN),
  ECNR_ERROR(SSE_ERR_NULL_FILTER),
  ECNR_ERROR(SSE_ERR_PARAMETER_TABLE),
  ECNR_ERROR(SSE_ERR_NOT_IMP),
  ECNR_ERROR(SSE_ERR_SUBMODULE_DISABLED),
  ECNR_ERROR(SSE_ERR_INVALID_SET_PARAM_ENDIANNESS),
  ECNR_ERROR(SSE_ERR_INIT_MMGR),
  ECNR_ERROR(SSE_ERR_INVALID_EXT_MEM_TYPE),
  ECNR_ERROR(SSE_ERR_INVALID_OPT_LEVEL),
  ECNR_ERROR(SSE_ERR_DBGIO_MSG_QUEUE),
  ECNR_ERROR(SSE_ERR_DBGIO_MSG_TYPE),
  ECNR_ERROR(SSE_ERR_DBGIO_OUTBUF_SIZE),
  ECNR_ERROR(SSE_ERR_INSUFFICIENT_SCRATCH),
  ECNR_ERROR(SSE_ERR_DBGIO_INST_ID),
  ECNR_ERROR(SSE_ERR_INTERNAL_MSG_BUF_TOO_SMALL),
  ECNR_ERROR(SSE_ERR_DBGIO_MSG_FORMAT),
  ECNR_ERROR(SSE_ERR_ASR_POST_PROC),
  ECNR_ERROR(SSE_ERR_INVALID_DATASET_ID),
  ECNR_ERROR(SSE_ERR_INVALID_MODULE_ID),
  ECNR_ERROR(SSE_ERR_INTERNAL_CALLBACK_ERROR),
  ECNR_ERROR(SSE_ERR_INVALID_DATA_CTRL),
  ECNR_ERROR(SSE_ERR_BSD_INVALID_IDENT),
  ECNR_ERROR(SSE_ERR_BSD_INVALID_FORMAT),
  ECNR_ERROR(SSE_ERR_BSD_INVALID_STRUCT_VERSION),
  ECNR_ERROR(SSE_ERR_BSD_INVALID_STRUCT_FORMAT),
  ECNR_ERROR(SSE_ERR_BSD_INVALID_PARAM),
  ECNR_ERROR(SSE_ERR_BSD_INVALID_BUFFER_PARAM),
  ECNR_ERROR(SSE_ERR_BSD_MAX_BSD_CNT_EXCEEDED),
  ECNR_ERROR(SSE_ERR_CONFIG_BSD_MISMATCH),
  ECNR_ERROR(SSE_ERR_BSD_CHECKSUM),
  ECNR_ERROR(SSE_ERR_BSD_READ),
  ECNR_ERROR(SSE_ERR_BSD_WRITE),
  ECNR_ERROR(SSE_ERR_BSD_EOF),
  ECNR_ERROR(SSE_ERR_BSD_VERSION),
  ECNR_ERROR(SSE_ERR_BSD_INVALID_DIM),
  ECNR_ERROR(SSE_ERR_BSD_UIO),
  ECNR_ERROR(SSE_ERR_BSD_INVALID_CONTEXT),
  ECNR_ERROR(SSE_ERR_BSD_DECOMPRESS),
  ECNR_ERROR(SSE_ERR_INVALID_BACKBAND),
  ECNR_ERROR(SSE_ERR_NUM_REF_CHAN),
  ECNR_ERROR(SSE_ERR_AEC_FILTER_LEN),
  ECNR_ERROR(SSE_ERR_LIMITER_DELAY),
  ECNR_ERROR(SSE_ERR_LIMITER_THRESHOLD),
  ECNR_ERROR(SSE_ERR_LIMITER_SUSTAIN_MAX),
  ECNR_ERROR(SSE_ERR_LIMITER_RELEASE_TIME),
  ECNR_ERROR(SSE_ERR_TIMER_NOT_CREATED),
  ECNR_ERROR(SSE_ERR_INVALID_TIMER_HANDLE),
  ECNR_ERROR(SSE_ERR_TIMER_HANDLE_NOT_UNIQUE),
  ECNR_ERROR(SSE_ERR_DIAGNOSTIC_ACTIVE),
  ECNR_ERROR(SSE_ERR_DIAGNOSTIC_NOT_READY),

  /* ECNR specifical error*/
  ECNR_ERROR(ECNR_ERR_CONFIG_FILE_OPEN_FAILED),
  ECNR_ERROR(ECNR_ERR_CONFIG_FILE_EMPTY),
  ECNR_ERROR(ECNR_ERR_ALLOC),
  ECNR_ERROR(ECNR_ERR_CONFIG_FILE_READING_FAILED),
  ECNR_ERROR(ECNR_ERR_NOT_CREATED),
  ECNR_ERROR(ECNR_ERR_CONFIG_NOT_AVAILABLE),
  ECNR_ERROR(ECNR_ERR_NULL_POINTER),
  ECNR_ERROR(ECNR_ERR_DGB_NO_DATA),
  ECNR_ERROR(ECNR_ERR_APPID_NOT_AVAILABLE),
  ECNR_ERROR(ECNR_ERR_CONFID_NOT_AVAILABLE),
  ECNR_ERROR(ECNR_ERR_APPID_INSTANCE_ALREADY_ACTIVE),
  ECNR_ERROR(ECNR_ERR_APPID_INSTANCE_NOT_AVAILABLE),
  ECNR_ERROR(ECNR_ERR_NO_INSTANCE_AVAILABLE),
  ECNR_ERROR(ECNR_ERR_APPID_CONFIGID_NOT_MATCHING),
  ECNR_ERROR(ECNR_ERR_REQ_APPID_NOT_ACTIVE),
  ECNR_ERROR(ECNR_ERR_OBJECT_FI_NOT_SUPPORTED),
  ECNR_ERROR(ECNR_ERR_DEV_ID_NOT_AVAILABLE),
  ECNR_ERROR(ECNR_ERR_DEV_CHNID_NOT_MATCHING),
  ECNR_ERROR(ECNR_ERR_OBJECT_FI_AUDIO_IS_RUNNING),
  ECNR_ERROR(ECNR_ERR_NO_ECNR_OBJECT_AVAILABLE),
  ECNR_ERROR(ECNR_ERR_TBL_IDX_NB_CFG_VALUE_TRANSGRESSION),

  /* Error configuration modul */
  ECNR_ERROR(ECNR_ERR_DP_ERROR_SIZE_OF_DATAPOOL),

  /* Error debug inteface*/
  ECNR_ERROR(ECNR_ERR_DGB_INIT_FAILED),
  ECNR_ERROR(ECNR_ERR_DEBUG_ALREADY_ACTIVE),
  ECNR_ERROR(ECNR_ERR_DEBUG_MODER_NOT_SUPPORTED),
  ECNR_ERROR(ECNR_ERR_DGB_SET_RMV_TRIGGER_FILE_FAILED),
  ECNR_ERROR(ECNR_ERR_NO_AUDIO_STREAMING),
  ECNR_ERROR(ECNR_ERR_AUDIO_THREAD_ALREADY_ACTIVE),

  /* error alsa Module*/
  ECNR_ERROR(ECNR_ERR_ALSA_NO_INSTANCE),

  /* rst modul */
  ECNR_ERROR(ECNR_ERR_RTS_NOT_SUPPORTED),

  /* qwa rcs error  group*/
  ECNR_ERROR(ECNR_QWA_RCS_ERR_INV_POINTER),
  ECNR_ERROR(ECNR_QWA_RCS_ERR_INVALID_BUFFSIZE),

  /* error service module */
  ECNR_ERROR(ECNR_ERR_SERV_NOT_INITIALIZED),

};

#define ECNR_NUM_ERROR_CODES ARRAYSIZE(ECNR_ERROR_table)

/*******************************************************************************
*
* FUNCTION: ecnr_error_str
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
const gchar *ecnr_error_str(int code)//tQwaInt code)
{
  guint i;
  const struct EcnrError *err;
  static const gchar ecnr_unknown_error[] = "Unknown ECNR error";

  for (i = 0, err = ECNR_ERROR_table; i < ECNR_NUM_ERROR_CODES; i++, err++) {
    if (err->code == code)
      return err->str;
  }
  return ecnr_unknown_error;
}


/*******************************************************************************
*
* FUNCTION: ECNR_object_error_quark
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
GQuark ECNR_object_error_quark(void)
{
  static GQuark quark = 0;

  if (!quark)
    quark = g_quark_from_static_string("ecnr-error");

  return quark;
}

/*******************************************************************************
*
* FUNCTION: report_ecnr_error
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
/*
 * Convert a ecnr(sse) status code to a DBus error.  A non-zero QWA status code
 * indicates an error.
 * Returns TRUE if no error occurred, or FALSE if an error occurred.
 */
gboolean report_ecnr_error
(
    int status,
    const gchar * dbus_msg,
    GError ** error
)
{
  gboolean ret = TRUE;

  if (status != ECNR_ERR_OK) {
    ecnr_warning(status, "%s", dbus_msg);

    g_set_error(error, ECNR_object_error_quark(), status,
          "%s (%d): %s", ecnr_error_str(status), status,
          dbus_msg);

    ret = FALSE;
  }

  return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_service_finalize
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_service_finalize(void)
{

    ECNR_PRINT_TIME_MSG("INIT");

    /* stop streaming */
    ecnr_alsa_stop(ECNR_START_STOP_TIMEOUT);

    /* destroy engine instance */
    (void)ecnr_service_do_action_destroy();

    /* leave audio thread*/
    ecnr_alsa_exit();

    /* wait until audio thread has terminated */
    if(my_obj)
    {
        while(my_obj->ecnr_alsa_thread_ref)
            usleep(1000);

        /* remove current service instance */
        (void)ecnr_service_remove_instance(my_obj->ecnr_current_app_id);
    }

    /* remove all alsa related resources */
    ecnr_alsa_finalize();

    /* clean service module*/
    ecnr_service_destroy();

    ECNR_PRINT_TIME_MSG("FINAL");

    return TRUE;
}
/*******************************************************************************
*
* FUNCTION: ecnr_service_handle_quit_signal
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static gboolean ecnr_service_handle_quit_signal (void)
{
    pr_message("Received quit signal");

    g_return_val_if_fail (ecnr_main_loop != NULL, FALSE);

    /* end main loop thread */
    g_main_loop_quit(ecnr_main_loop);

    return TRUE;
}

/*******************************************************************************
*
* FUNCTION: ecnr_service_install_signal_handler
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static void ecnr_service_install_signal_handler (GMainLoop *main_loop)
{
    GSource *source = NULL;
    GMainContext *ctx = g_main_loop_get_context (main_loop);

    pr_message("install the signal cb for SIGTERM, SIGINT and SIGHUP\n");

    source = g_unix_signal_source_new (SIGTERM);
    g_source_set_callback (source,
                          (GSourceFunc)ecnr_service_handle_quit_signal,
                           NULL,
                           NULL);
    ecnr_service_sig_source_id[0] = g_source_attach (source, ctx);

    pr_debug("cb for signal SIGTERM was set");

    GSource *source_1 = NULL;
    source_1 = g_unix_signal_source_new (SIGINT);
    g_source_set_callback (source_1,
                           (GSourceFunc)ecnr_service_handle_quit_signal,
                           NULL,
                           NULL);
    ecnr_service_sig_source_id[1] = g_source_attach (source_1, ctx);
    pr_debug("cb for signal SIGINT was set");


    GSource *source_2 = NULL;
    source_2 = g_unix_signal_source_new (SIGHUP);
    g_source_set_callback (source_2,
                           (GSourceFunc)ecnr_service_handle_quit_signal,
                           NULL,
                           NULL);
    ecnr_service_sig_source_id[2] = g_source_attach (source_2, ctx);
    pr_debug("cb for signal SIGHUP was set");
}

/*******************************************************************************
*
* FUNCTION: main
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/

int main(int argc, char *argv[])
{
    DBusGConnection *connection;
    GError *error;
    DBusGProxy *driver_proxy;
    guint32 request_name_ret;
    gint test_succeed = ECNR_ERR_OK;
    GOptionContext *context;

    pr_message("ecnr_main Entered \n");

    g_type_init();
    g_thread_init(NULL);
    dbus_g_thread_init();


    ecnr_main_loop = g_main_loop_new(NULL, FALSE);
    error = NULL;

    /***********************
      evaluation main keys
    ************************/

    setlocale(LC_ALL, "");
    context = g_option_context_new("");
    g_option_context_set_summary(context, "The ecnr daemon implements "
    "handsfree audio using to QNX/nuance libraries\n and ALSA audio "
    "devices.");
    g_option_context_add_main_entries(context, ecnr_service_opts, NULL);
    if (!g_option_context_parse(context, &argc, &argv, &error))
    {
        g_print("ecnr deamon: %s\n", error->message);
        g_error_free(error);
        g_option_context_free(context);
        _exit(1);
    }

    g_option_context_free(context);


    /****************************************************************
      check for mode selection
      - execute unitest if key 'u' ist set and value is set to "1"
      - print ecnr infos
      - otherwise run ecrn deamon
    *****************************************************************/

    if(ecnr_service_info_select_flag)
    {
        /* print ecnr information and exit immediatly */
        ecnr_service_info_select_flag = FALSE;
        _exit(0);
    }

    /* execute unitest */
    if (ecnr_service_test_suite)
    {
        pr_message("selection unit test ecnr_service_test_suite(%d)\n", ecnr_service_test_suite);

        g_test_init (&argc, &argv, NULL);
        test_succeed = ecnr_unitest_server_run((ecnr_unitset)ecnr_service_test_suite);
        if(test_succeed == 0)
            pr_message("ecnr unitest terminatewd successfully \n");
        else
            pr_message("ecnr unitest failed \n");

        _exit(test_succeed); ; /* unitest terminated */
    }

    /* run ecnr deamon main program */
    connection = dbus_g_bus_get(DBUS_BUS_SYSTEM, &error);
    if (connection == NULL) {
        pr_critical("Failed to open connection to bus: %s",
          error->message);
        g_error_free(error);
        _exit(1);
    }

    pr_debug("dbus_g_bus_get\n");

    obj = g_object_new(ECNR_TYPE_OBJECT, NULL);

    pr_debug("g_object_new\n");

    dbus_g_connection_register_g_object(connection, "/", obj);

    pr_debug("dbus_g_connection_register_g_object \n");

    driver_proxy = dbus_g_proxy_new_for_name(connection,
             DBUS_SERVICE_DBUS,
             DBUS_PATH_DBUS,
             DBUS_INTERFACE_DBUS);

    pr_debug("dbus_g_proxy_new_for_name \n");

    if (!org_freedesktop_DBus_request_name(driver_proxy,
                 ECNR_SERVICE_NAME,
                 0, &request_name_ret, &error))
    {
        g_assert(error != NULL);
        if (error)
            pr_critical("Failed to get name: %s", error->message);
        g_error_free(error);
        _exit(1);
    }

    pr_debug("dbus_g_proxy_new_for_name \n");

    if (!(request_name_ret == DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER))
    {
        pr_critical("Got result code %u from requesting name", request_name_ret);
        _exit(1);
    }

    /******************************************
    initialize alsa module - audio streaming
    ******************************************/
    ecnr_alsa_initialize();

    ECNR_PRINT_TIME_DBG("Start ecnr after power cycle");

#ifdef D_SET_TUNING_PER_TRIGGER_FILE
    /* start debug interface handling to sse per trigger file  */
    if(ecnr_sse_debug_check_enable_file())
    {
        int err = ECNR_ERR_OK;
        err = ecnr_engine_start_debug_session();
        if(!err)
            ecnr_service_tuning_server_state = TRUE;
    }
#endif

    /* install callback for signal handler */
    ecnr_service_install_signal_handler(ecnr_main_loop);

   /* create a mainloop that runs/iterates the default GLib main context
    * (context NULL). When a message has been posted on the
    * bus or bei systemd, the default main context will automatically dispatch the
    * associated resource and call the callback associated with the resource
    * The main loop remains in run state until g_main_loop_quit() is called
   */
    g_main_loop_run(ecnr_main_loop);

    /* finalize ecnr */
    ecnr_service_finalize();

#ifndef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    ecnr_datapool_access_finalize();
#endif

    if(ecnr_service_tuning_server_state)
    {
        ecnr_engine_end_debug_session();
    }

    if (ecnr_main_loop)
        g_main_loop_unref (ecnr_main_loop);


    _exit(0);
}



/*******************************************************************************
+ instance handler static functions
*******************************************************************************/

/*******************************************************************************
 *
 * FUNCTION: ecnr_service_init
 *
 * DESCRIPTION: This function initialize the sse Engine
 *
 * PARAMETER: [check xml file for parameter description]
 *
 * RETURNVALUE: int
 *
 *******************************************************************************/
void ecnr_service_init(void)
{
    pr_message("entered");
    ecnr_service_lock = g_mutex_new();

    ecnr_service_init_state = TRUE;

  ecnr_service_set_pass_through_mode_default_state(FALSE);
  ecnr_service_set_mic_gain_level(0);
    return;
}


/*******************************************************************************
 *
 * FUNCTION: ecnr_service_destroy
 *
 * DESCRIPTION: This function initialize the sse Engine
 *
 * PARAMETER: [check xml file for parameter description]
 *
 * RETURNVALUE: int
 *
 *******************************************************************************/
void ecnr_service_destroy(void)
{
    pr_message("entered");
    g_mutex_free(ecnr_service_lock);

    ecnr_service_init_state = FALSE;

    ecnr_service_set_pass_through_mode_default_state(FALSE);
    ecnr_service_set_mic_gain_level(0);
    return;
}

/*******************************************************************************
 *
 * FUNCTION: ecnr_service_set_passthroughmode_default_state
 *
 * DESCRIPTION: This function set the default state for ecnr pass through mode
 *
 * PARAMETER: default state
 *
 * RETURNVALUE:
 *
 *******************************************************************************/
void ecnr_service_set_pass_through_mode_default_state(gboolean ptm_default_state)
{
  pr_message("entered");
  ecnr_service_pass_through_mode_default_state = ptm_default_state;
}

/*******************************************************************************
 *
 * FUNCTION: ecnr_service_get_passthroughmode_default_state
 *
 * DESCRIPTION: This function get's the default state for ecnr pass through mode
 *
 * PARAMETER:
 *
 * RETURNVALUE: default state
 *
 *******************************************************************************/
gboolean  ecnr_service_get_pass_through_mode_default_state(void)
{
  pr_message("entered");
  return  ecnr_service_pass_through_mode_default_state;
}

/*******************************************************************************
 *
 * FUNCTION: ecnr_service_set_mic_gain_level
 *
 * DESCRIPTION: This function set the mic gain level
 *
 * PARAMETER: default state
 *
 * RETURNVALUE:
 *
 *******************************************************************************/
void ecnr_service_set_mic_gain_level(guint16 mic_gain_level)
{
  pr_message("entered");
  ecnr_service_mic_gain_level = mic_gain_level;
}

/*******************************************************************************
 *
 * FUNCTION: ecnr_service_get_mic_gain_level
 *
 * DESCRIPTION: This function get's the current mic gain level set
 *
 * PARAMETER:
 *
 * RETURNVALUE: default state
 *
 *******************************************************************************/
gint16  ecnr_service_get_mic_gain_level(void)
{
  pr_message("entered");
  return  ecnr_service_mic_gain_level;
}

/*******************************************************************************
*
* FUNCTION: ecnr_service_create_new_instance
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_service_create_new_instance(guchar app_id)
{
    int err = ECNR_ERR_OK;

    pr_debug("entered");

    if(!ecnr_service_init_state)
    {
        /* the ecnr_service_init should be first call prior
           to any ecnr_service module function call
        */
        return ECNR_ERR_SERV_NOT_INITIALIZED;
    }

    g_mutex_lock(ecnr_service_lock);

    if(my_obj)
    {
        err = ECNR_ERR_APPID_INSTANCE_ALREADY_ACTIVE;
        goto error_out;
    }

    if(!app_id)
    {
        err = ECNR_ERR_APPID_NOT_AVAILABLE;
        goto error_out;
    }

    /* create new ecnr instamce */
    my_obj = g_new(struct ecnr_instance_object, 1);

    if(my_obj)
    {
        my_obj->ecnr_is_initialized = FALSE;
        my_obj->ecnr_is_audio_thread_prepared = FALSE;
        my_obj->ecnr_is_audio_thread_started = FALSE;
        my_obj->ecnr_Debug_intf_initialized = FALSE;
        my_obj->ecnr_current_config_id = 0;
        my_obj->ecnr_current_app_id = app_id;
        my_obj->ecnr_alsa_thread_ref = 0;

        pr_message("create instance for application(%d)\n",app_id);
    }
    else
        err = ECNR_ERR_NULL_POINTER;

error_out:
    g_mutex_unlock(ecnr_service_lock);
    pr_debug("leaved with error(%d)\n", err );
    return err;
}



/*******************************************************************************
*
* FUNCTION: ecnr_service_alsa_thread_create
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
int ecnr_service_alsa_thread_create(void)
{
    int err = ECNR_ERR_OK;
    GThread *thread_create_alsa;
    GError *error = NULL;

    if(!ecnr_service_init_state)
    {
        /* the ecnr_service_init should be first call prior
           to any ecnr_service module function call
        */
        return ECNR_ERR_SERV_NOT_INITIALIZED;
    }

    if(!my_obj)
    {
        err = ECNR_ERR_APPID_INSTANCE_ALREADY_ACTIVE;
        pr_debug("no service instance available\n");
    }

    g_mutex_lock(ecnr_service_lock);

    if(!err)
    {
        if(my_obj->ecnr_alsa_thread_ref)
        {
           pr_debug("alsa thread reference still available \n");
           err = ECNR_ERR_AUDIO_THREAD_ALREADY_ACTIVE;
        }

        else if(!my_obj->ecnr_is_audio_thread_prepared && !my_obj->ecnr_is_audio_thread_started)
        {
            pr_debug("create audio streaming thread \n");
            my_obj->ecnr_is_audio_thread_prepared = TRUE;
        }
        else
        {
            pr_debug("audio thread already created\n");
            err = ECNR_ERR_OBJECT_FI_AUDIO_IS_RUNNING;
        }
    }

    if(!err)
    {
        thread_create_alsa = g_thread_try_new ("alsa_create", ecnr_alsa_create_alsa_thread_start, NULL, &error);
        if (!thread_create_alsa)
        {
            my_obj->ecnr_is_audio_thread_prepared = FALSE;
            pr_warning("Can't create ecnr_alsa_create_alsa_thread_start");
            err = ECNR_ERR_NOT_CREATED;
        }
        else
        {
            pr_debug("alsa thread creation was started\n");
        }
        g_thread_unref (thread_create_alsa);
        thread_create_alsa = NULL;
    }

    g_mutex_unlock(ecnr_service_lock);

    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_service_state_audio_prepare
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_service_state_audio_prepare_cb(void)
{
    int err = ECNR_ERR_OK;

    pr_debug("entered");

    if(!ecnr_service_init_state)
    {
        /* the ecnr_service_init should be first call prior
           to any ecnr_service module function call
        */
        return ECNR_ERR_SERV_NOT_INITIALIZED;
    }

    g_mutex_lock(ecnr_service_lock);

    if(!my_obj)
    {
        err = ECNR_ERR_APPID_INSTANCE_ALREADY_ACTIVE;
        pr_debug("no service instance available\n");
    }

    if(!err)
    {
        if(my_obj->ecnr_is_initialized && !my_obj->ecnr_is_audio_thread_prepared)
        {
            my_obj->ecnr_is_audio_thread_prepared = TRUE;
            pr_message("change service state to audio prepared\n");
        }
        else
        {
            pr_warning("service state cannot be changed to prepared\n");
            pr_warning("current state: init(%d), prep(%d), act(%d)\n", \
                        my_obj->ecnr_is_initialized, \
                        my_obj->ecnr_is_audio_thread_prepared, \
                        my_obj->ecnr_is_audio_thread_started);

        }
    }

    g_mutex_unlock(ecnr_service_lock);
    return err ;
}

/*******************************************************************************
*
* FUNCTION: ecnr_service_state_audio_active
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_service_state_audio_active_cb(void)
{
    int err = ECNR_ERR_OK;

    pr_debug("entered");

    if(!ecnr_service_init_state)
    {
        /* the ecnr_service_init should be first call prior
           to any ecnr_service module function call
        */
        return ECNR_ERR_SERV_NOT_INITIALIZED;
    }

    g_mutex_lock(ecnr_service_lock);

    if(!my_obj)
    {
        err = ECNR_ERR_APPID_INSTANCE_ALREADY_ACTIVE;
        pr_debug("no service instance available\n");
    }

    if(!err)
    {
        if(my_obj->ecnr_is_audio_thread_prepared)
        {
            my_obj->ecnr_is_audio_thread_started = TRUE;
            pr_message("change service state to audio active\n");
        }
        else
        {
            pr_warning("service state cannot be changed to active\n");
            pr_warning("current state: init(%d), prep(%d), act(%d)\n", \
                        my_obj->ecnr_is_initialized, \
                        my_obj->ecnr_is_audio_thread_prepared, \
                        my_obj->ecnr_is_audio_thread_started);

        }
    }

    g_mutex_unlock(ecnr_service_lock);
    return err ;
}

/*******************************************************************************
*
* FUNCTION: ecnr_service_state_audio_active
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_service_state_audio_terminate_cb(void)
{
    int err = ECNR_ERR_OK;

    pr_debug("entered");

    if(!ecnr_service_init_state)
    {
        /* the ecnr_service_init should be first call prior
           to any ecnr_service module function call
        */
        return ECNR_ERR_SERV_NOT_INITIALIZED;
    }

    g_mutex_lock(ecnr_service_lock);

    if(!my_obj)
    {
        pr_debug("no service instance available\n");
        err = ECNR_ERR_APPID_INSTANCE_ALREADY_ACTIVE;
    }

    if(!err)
    {
        my_obj->ecnr_is_audio_thread_started = FALSE;
        my_obj->ecnr_is_audio_thread_prepared = FALSE;
        pr_message("change state service audio thread to deactivated\n");
    }

    g_mutex_unlock(ecnr_service_lock);
    return err ;
}

/*******************************************************************************
*
* FUNCTION: ecnr_service_state_audio_active
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_service_state_ist_audio_active(void)
{
    gboolean bret = TRUE;

    pr_debug("entered");

    if(!ecnr_service_init_state)
    {
        /* the ecnr_service_init should be first call prior
           to any ecnr_service module function call
        */
        return FALSE;
    }

    g_mutex_lock(ecnr_service_lock);

    if(!my_obj)
    {
        pr_debug("no service instance available\n");
    }


    if(!my_obj->ecnr_is_audio_thread_prepared && !my_obj->ecnr_is_audio_thread_started)
    {
        pr_debug("No audio started\n");
        bret = FALSE;
    }
    else
        pr_warning("audio thread already started\n");

    g_mutex_unlock(ecnr_service_lock);

    return bret ;
}


/*******************************************************************************
*
* FUNCTION: ecnr_object_create_instance
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_service_remove_instance(guchar app_id)
{
    int err = ECNR_ERR_OK;

    pr_debug("entered");

    if(!ecnr_service_init_state)
    {
        /* the ecnr_service_init should be first call prior
           to any ecnr_service module function call
        */
        return ECNR_ERR_SERV_NOT_INITIALIZED;
    }

    g_mutex_lock(ecnr_service_lock);

    if(!my_obj)
    {
        err = ECNR_ERR_NO_INSTANCE_AVAILABLE;
        goto error_out;
    }

    if(app_id && my_obj->ecnr_current_app_id != app_id )
    {
        err = ECNR_ERR_OK;
        pr_debug("no instance available for this app id(%d)\n", app_id);
        goto error_out;
    }

    /* delte ecnr instamce */
    if(my_obj)
        g_free(my_obj);
    my_obj = NULL;

    pr_message("remove instance from application(%d)\n",app_id);

error_out:
    g_mutex_unlock(ecnr_service_lock);
    pr_debug("leaved with error(%d)\n", err );
    return err;
}



/*******************************************************************************
*
* FUNCTION: ecnr_service_get_this_instance
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
struct ecnr_instance_object* ecnr_service_get_this_instance(guchar app_id)
{
    pr_debug("entered");

    if(!ecnr_service_init_state)
    {
        /* the ecnr_service_init should be first call prior
           to any ecnr_service module function call
        */
        pr_debug("service not initialized\n");
        return (struct ecnr_instance_object*)NULL;;
    }

    if(!my_obj)
    {
        pr_debug("no instance available\n");
        return (struct ecnr_instance_object*)NULL;
    }

    if((my_obj->ecnr_current_app_id != app_id) && (app_id != ECNR_APPID_ADMIN))
    {
        pr_warning("no instance available for the requested appid(%d)\n",app_id);
        return (struct ecnr_instance_object*)NULL; //return ECNR_ERR_APPID_INSTANCE_NOT_AVAILABLE;
    }

    return my_obj;
}

/*******************************************************************************
*
* FUNCTION: ecnr_service_get_curr_instance
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
struct ecnr_instance_object* ecnr_service_get_curr_instance(void)
{
    pr_debug("entered");

    return my_obj;
}

/*******************************************************************************
*
* FUNCTION: ecnr_service_is_instance_active
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_service_is_instance_active(guchar app_id, gboolean* state)
{
    pr_debug("entered");

    if(!ecnr_service_init_state)
    {
        /* the ecnr_service_init should be first call prior
           to any ecnr_service module function call
        */
        return ECNR_ERR_SERV_NOT_INITIALIZED;
    }

    if(!my_obj)
    {
        pr_debug("no instance available\n");
        *state = FALSE;
        return ECNR_ERR_NO_INSTANCE_AVAILABLE;
    }

    if((my_obj->ecnr_current_app_id != app_id) && (app_id != ECNR_APPID_ADMIN))
    {
        pr_debug("no instance available for the requested appid(%d)\n",app_id);
        *state = FALSE;
        return ECNR_ERR_APPID_INSTANCE_NOT_AVAILABLE;
    }

    *state = TRUE;
    return ECNR_ERR_OK;
}


/*******************************************************************************
*
* FUNCTION: ecnr_service_do_action_initialize
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_service_do_action_initialize(gint config_id)
{
    int err = ECNR_ERR_OK;

    pr_message("do initialization for configuration id(%d)\n", config_id);

    if(!ecnr_service_init_state)
    {
        /* the ecnr_service_init should be first call prior
           to any ecnr_service module function call
        */
        return ECNR_ERR_SERV_NOT_INITIALIZED;
    }

    if(!my_obj)
        return ECNR_ERR_NO_INSTANCE_AVAILABLE;

    if (!my_obj->ecnr_is_initialized)
    {
        /* reset the sse engine witha a
         * default configuration file. If a configuration Id was given by teh application
         * then this one it taken instead of.
             */
        err = ecnr_engine_handler_create_session((unsigned short)config_id);
    }
    else
        err = SSE_ERR_MODULE_ALREADY_INIT;

    if (!err)
    {
        my_obj->ecnr_is_initialized = TRUE;
        my_obj->ecnr_current_config_id = config_id;
    }

    return err;
}


/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_frame_info
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_service_check_AppId_ConfigId_Matching(gint ecnrConfigurationId, guchar ecnrAppId)//, guchar CurrAppId )
{
    int ret = ECNR_ERR_OK;

    if(!ecnr_service_init_state)
    {
        /* the ecnr_service_init should be first call prior
           to any ecnr_service module function call
        */
        return ECNR_ERR_SERV_NOT_INITIALIZED;
    }

    /* check validity AppId*/
    ret = ecnr_configuration_check_AppId(ecnrAppId);

    /* check validity configuration Id*/
    if(!ret)
        ret = ecnr_configuration_check_ConfigId(ecnrConfigurationId);

    if(!ret)
        ret = ecnr_configuration_check_match_AppId_and_configId(ecnrConfigurationId,ecnrAppId);

    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_frame_info
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_service_do_action_destroy(void)
{
    int err = ECNR_ERR_OK;

    pr_message("entered");

    if(!ecnr_service_init_state)
    {
        /* the ecnr_service_init should be first call prior
           to any ecnr_service module function call
        */
        return ECNR_ERR_SERV_NOT_INITIALIZED;
    }

    if(!my_obj)
    {
        pr_debug("no instance available\n");
        return err;
    }

    if (my_obj->ecnr_is_initialized)
    {
        /* reset the sse engine witha a
         * default configuration file. If a configuration Id was given by teh application
         * then this one it taken instead of.
             */
        (void)ecnr_engine_handler_destroy();

        /* close io device and associated member variable */
        ecnr_ioctrl_finalize();
    }
    else
        err = SSE_ERR_MODULE_ALREADY_INIT;


    my_obj->ecnr_is_initialized = FALSE;
    my_obj->ecnr_current_config_id = 0;

    return err;
}



/*******************************************************************************
*
* FUNCTION: ecnr_service_alsa_thread_ref
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_service_alsa_thread_ref(void)
{
    int err = ECNR_ERR_OK;

    pr_debug("entered");

    if(!ecnr_service_init_state)
    {
        /* the ecnr_service_init should be first call prior
           to any ecnr_service module function call
        */
        return ECNR_ERR_SERV_NOT_INITIALIZED;
    }

    g_mutex_lock(ecnr_service_lock);

    if(!my_obj)
    {
        err = ECNR_ERR_NO_INSTANCE_AVAILABLE;
    }

    /* increment alsa thread reference */
    if(!err)
    {
        my_obj->ecnr_alsa_thread_ref += 1;
        pr_debug("add alsa thread reference\n");
    }

    g_mutex_unlock(ecnr_service_lock);

    return err;

}

/*******************************************************************************
*
* FUNCTION: ecnr_service_test_performance_start
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: void
*
*******************************************************************************/

void  ecnr_service_set_use_default_dp(void)
{

    pr_message("ENTERED");

    g_mutex_lock(ecnr_service_lock);
    ecnr_service_use_default_dp = TRUE;
    g_mutex_unlock(ecnr_service_lock);

    return;
}


/*******************************************************************************
*
* FUNCTION: ecnr_service_test_performance_start
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: void
*
*******************************************************************************/

void  ecnr_service_set_use_default_dp_exit(void)
{

    pr_message("ENTERED");

    g_mutex_lock(ecnr_service_lock);
    ecnr_service_use_default_dp = FALSE;
    g_mutex_unlock(ecnr_service_lock);

    return;
}

/*******************************************************************************
*
* FUNCTION: ecnr_service_test_performance_start
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: void
*
*******************************************************************************/

gboolean ecnr_service_is_use_default_dp(void)
{
    gboolean state = FALSE;

    pr_message("ENTERED");

    g_mutex_lock(ecnr_service_lock);
    state = ecnr_service_use_default_dp;
    g_mutex_unlock(ecnr_service_lock);

    return state;
}

/*******************************************************************************
*
* FUNCTION: ecnr_service_alsa_thread_unref
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_service_alsa_thread_unref(void)
{
    int err = ECNR_ERR_OK;

    pr_debug("entered");

    if(!ecnr_service_init_state)
    {
        /* the ecnr_service_init should be first call prior
           to any ecnr_service module function call
        */
        return ECNR_ERR_SERV_NOT_INITIALIZED;
    }

    g_mutex_lock(ecnr_service_lock);

    if(!my_obj)
    {
        err = ECNR_ERR_NO_INSTANCE_AVAILABLE;
    }

    /* increment alsa thread reference */
    if(!err)
    {
        my_obj->ecnr_alsa_thread_ref = 0;
        pr_message("remove alsa thread reference\n");
    }

    g_mutex_unlock(ecnr_service_lock);

    return err;

}


#ifdef D_TRIGGER_PERFORMANCE_TEST
/*******************************************************************************
*
* FUNCTION: ecnr_service_test_performance_initialize
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: void
*
*******************************************************************************/
void ecnr_service_test_performance_initialize(void)
{

    ECNR_PRINT_TIME_MSG("entered");
    system("touch  /var/opt/bosch/dynamic/audiomanager/EcnrPerformance.txt");
    ecnr_service_performance_test_nb++;
}


/*******************************************************************************
*
* FUNCTION: ecnr_service_test_performance_start
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: void
*
*******************************************************************************/
void ecnr_service_test_performance_start(void)
{

    ECNR_PRINT_TIME_MSG("entered");

    GString *strbuf = g_string_new(NULL); /* This will spawn an empty string */
    g_string_append(strbuf, "/opt/bosch/audiomanager/test/ecnr-test-script/ecnr_stress_test_proc.sh");
    g_string_append(strbuf, " PERFORMANCE_TEST");
    g_string_append_printf(strbuf, " %d &", ecnr_service_performance_test_nb);

    pr_message("call system(%s)", strbuf->str);

    /*
     * Completely scratching the buffer, including its contents:
     */
    system(strbuf->str);
    g_string_free(strbuf, TRUE);
}



/*******************************************************************************
*
* FUNCTION: ecnr_service_test_performance_stop
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: void
*
*******************************************************************************/
void ecnr_service_test_performance_stop(void)
{
    ECNR_PRINT_TIME_MSG("INIT");
    system("rm  /var/opt/bosch/dynamic/audiomanager/EcnrPerformance.txt");
    ECNR_PRINT_TIME_MSG("FINAL");
}

#endif

#ifdef D_SIMULATE_SPI_CLIENT

/*******************************************************************************
*
* FUNCTION: ecnr_service_test_client_start
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: void
*
*******************************************************************************/
void ecnr_service_test_client_start(void)
{
    ECNR_PRINT_TIME_MSG("INIT");

    GString *alsa_client_spi_in_out = g_string_new(NULL); /* This will spawn an empty string */

    g_string_append(alsa_client_spi_in_out, "/opt/bosch/audiomanager/test/ecnr-test-script/ecnr_stress_test_proc.sh");
    g_string_append(alsa_client_spi_in_out, " ALSA_CLIENT_SIM START");

    pr_message("call system(%s)", alsa_client_spi_in_out->str);
    system(alsa_client_spi_in_out->str);

    g_string_free(alsa_client_spi_in_out, TRUE);

    ECNR_PRINT_TIME_MSG("FINAL");

}


/*******************************************************************************
*
* FUNCTION: ecnr_service_test_client_stop
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: void
*
*******************************************************************************/
void ecnr_service_test_client_stop(void)
{
    ECNR_PRINT_TIME_MSG("INIT");

    GString *alsa_client_spi_in_out = g_string_new(NULL); /* This will spawn an empty string */

    g_string_append(alsa_client_spi_in_out, "/opt/bosch/audiomanager/test/ecnr-test-script/ecnr_stress_test_proc.sh");
    g_string_append(alsa_client_spi_in_out, " ALSA_CLIENT_SIM STOP");

    pr_message("call system(%s)", alsa_client_spi_in_out->str);
    system(alsa_client_spi_in_out->str);

    ECNR_PRINT_TIME_MSG("FINAL");
}

#endif
