/*
/*
 * ecnr-object.c
 *
 *
 * *************************************************
 * qwa-object.c
 *
 * Andy Lowe <alowe@mvista.com>
 *
 * 2010 (c) MontaVista Software, LLC. This file is licensed under
 * the terms of the AFL.
 *
 *************************************************
 * Author : Patrick Rey
 * Date: 27.08.2013
 * description: porting from qwa-alsa.c to G3g
 */

#include <glib/gi18n.h>
#include <glib-object.h>
#include <sched.h>


#include "sse.h"
#include "sse_err.h"
#include "ecnr-object.h"
#include "ecnr-service.h"
#include "ecnr-common-defs.h"
#include "ecnr-alsa.h"
#include "ecnr-configuration.h"
#include "ecnr-audio-rts-handler.h"
#include "ecnr-datapool-access.h"
#include "ecnr-engine-handler.h"
#include "ecnr-qwa-handler.h"
#include "ecnr-sse-handler.h"
#include "ecnr-ioctrl.h"


#define ECNR_S_IMPORT_INTERFACE_DP
#include "ecnr-datapool-default.h"
#include "ecnr-configuration-data.h"

/*
 * Add the ecnr-introspection.h header file to the list of library header files
 * maintained by lint.  It isn't really a library header file, but it is an
 * automatically generated file that we can't modify, so it should be treated
 * like a library header file.
 */
/*lint +libh("ecnr-introspection.h") */


#include "ecnr-introspection.h"

/*
 * Wait for up to one second for the audio thread to respond to a start or
 * stop command.  This typically takes far less than a second since the audio
 * thread is run at a high real-time scheduling priority, but it can take
 * a significant amount of time if console debugging diagnostics are enabled
 * due to the time spent writing to the console.  It can also take an extra
 * long time to enable streaming the first time because some code from shared
 * libraries may have to be read from storage.
 */
//#define ECNR_START_STOP_TIMEOUT 1000000
#define T_BUG


enum{
  PROP_ALSA_DEV_OPEN_ORDER = 1,
  PROP_ALSA_DEV_PREFILL
};


/* ECNR dbug mode  */
enum {
  ECNR_DBUG_DEFAULT=0,
  ECNR_DBUG_SET_TRIGGER_FILE = 1,
  ECNR_DBUG_DELETE_TRIGGER_FILE = 2,
  ECNR_DBUG_FORCE_AUDIO_IO = 3,
  ECNR_DBUG_DATA_ALSA_AVAILABLE= 4,
  ECNR_DBUG_DATA_ALSA_DISABLE= 5,
  ECNR_DBUG_START_SESSION= 6,
  ECNR_DBUG_STOP_SESSION= 7,
  ECNR_DBUG_PRINT_DEV_PROP= 8,
  ECNR_DBUG_PRINT_RTS_CONFIG= 9,
  ECNR_DBUG_PRINT_ECNR_DATA_CONFIG= 10,
  ECNR_DBUG_SWITCH_OFF_CHECK_ALSA_DEV= 11,
  ECNR_DBUG_SWITCH_OFF_CHECK_APPID_CNFGID= 12,
  ECNR_DBUG_TEST_TRIGGER_NOTIFICATION_NEW_DATAPOOL_ALL= 13,
  ECNR_DBUG_TEST_TRIGGER_NOTIFICATION_NEW_DATAPOOL_1= 14,
  ECNR_DBUG_TEST_FORCE_EXIT_ALSA_THREAD= 15,
  ECNR_DBUG_TEST_FORCE_CREATE_ALSA_THREAD= 16,
  ECNR_DBUG_PRINT_CURRENT_AUDI0_STATE=17,
  ECNR_DBUG_PRINT_RCS_CONF_FILES=18,
  ECNR_CALL_NATIVE_START_AUDIO = 19,
  ECNR_DBUG_PRINT_DP_INFOS = 20,
  ECNR_DBUG_SET_IO_CTRL_HF_VR_MODE = 21,
  ECNR_DBUG_RESET_IO_CTRL_HF_VR_MODE = 22,
  ECNR_DBUG_FORCE_USE_DEFAULT_DP = 23,
  ECNR_DBUG_FORCE_USE_DEFAULT_DP_EXIT = 24,
  ECNR_DBUG_LAST_ENTRY = ECNR_DBUG_FORCE_USE_DEFAULT_DP_EXIT
};

enum {
    START_MODE_WAIT_STREAMING_RESULT = 0,
    START_MODE_RETURN_IMMEDIATELY
};


/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_frame_info
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/

gboolean ecnr_object_ecnr_frame_info(EcnrObject * qobj,
           guint * FrameCounter,
           gint * VoicePresent,
           gint * AboveBackgroundNoise,
           gint * PercentClipping,
           gint * InstantaneousSNR,
           gint * SmoothVNR,
           gint * ValidOutputCountdown, GError ** error)
{
    gboolean ret = TRUE;


    /*
    * Suppress lint Info 715 "Symbol not referenced" message regarding
    * the formal parameter 'qobj'.
    */
    /*lint -efunc(715,ecnr_object_ecnr_frame_info) */

    pr_message("no supported by ecnr Engine \n");

    /*due to Info 715: prio2:*/
    *FrameCounter = 0;
    *VoicePresent = 0;
    *AboveBackgroundNoise = 0;
    *PercentClipping = 0;
    *InstantaneousSNR = 0;
    *SmoothVNR = 0;
    *ValidOutputCountdown = 0;
    /*due to Info 715: prio2:*/

    ret = report_ecnr_error(ECNR_ERR_OBJECT_FI_NOT_SUPPORTED, "ecnr_object_ecnr_alsa_set_wide_band", error);

    return ret;
}




#ifndef T_BUG
G_DEFINE_TYPE(EcnrObject, ecnr_object, G_TYPE_OBJECT)
#else
static void     ecnr_object_init       (EcnrObject      *self);
static void     ecnr_object_class_init (EcnrObjectClass *klass);
static gpointer ecnr_object_parent_class = NULL;


/*******************************************************************************
*
* FUNCTION: ecnr_object_class_intern_init
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static void     ecnr_object_class_intern_init (gpointer klass)
{
  pr_debug("ecnr_object_class_intern_init \n");
  ecnr_object_parent_class = g_type_class_peek_parent (klass);
  ecnr_object_class_init ((EcnrObjectClass*) klass);
}



/*******************************************************************************
*
* FUNCTION: ecnr_object_get_type
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/

GType ecnr_object_get_type (void)
{
    static volatile gsize g_define_type_id__volatile = 0;

    pr_debug("ecnr_object_get_type enter \n");

    if (g_once_init_enter (&g_define_type_id__volatile))
    {
        pr_debug("g_once_init_enter enter \n");
        GType g_define_type_id =
        g_type_register_static_simple (G_TYPE_OBJECT,
                                       g_intern_static_string ("EcnrObject"),
                                       sizeof (EcnrObjectClass),
                                       (GClassInitFunc) ecnr_object_class_intern_init,
                                       sizeof (EcnrObject),
                                       (GInstanceInitFunc) ecnr_object_init,
                                       (GTypeFlags)0);


        pr_debug("the type id is %d",g_define_type_id);

        g_once_init_leave (&g_define_type_id__volatile, g_define_type_id);
    }
    pr_debug("ecnr_object_get_type leave \n");
    return g_define_type_id__volatile;
}

#endif




/*******************************************************************************
*
* FUNCTION: ecnr_object_finalize
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static void ecnr_object_finalize(GObject * gobj)
{
    pr_message("");

    ecnr_service_destroy();

    (G_OBJECT_CLASS(ecnr_object_parent_class)->finalize) (gobj);

}

/*******************************************************************************
*
* FUNCTION: ecnr_object_init
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static void ecnr_object_init(EcnrObject * qobj)
{

    pr_debug("ecnr_object_init \n");
    ecnr_service_init();

}

/*******************************************************************************
*
* FUNCTION: ecnr_object_class_init
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static void ecnr_object_class_init(EcnrObjectClass * qobj_class)
{
  GObjectClass *gobj_class = G_OBJECT_CLASS(qobj_class);

  pr_debug("");

  dbus_g_object_type_install_info(ECNR_TYPE_OBJECT,
          &dbus_glib_ecnr_object_object_info);

  gobj_class->finalize = ecnr_object_finalize;

}



/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_get_version
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_get_version(EcnrObject * qobj, GArray ** version,
            gchar ** version_string,
            gchar ** version_comment, GError ** error)
{
  unsigned int i;
    gboolean ret;
    guint val;
    unsigned int sse_Array_Len = 0;
    const int *sse_Version_Array = NULL;
    const char *sse_Version_String = NULL;
    const char *sse_Version_Comment = NULL;
    struct ecnr_instance_object* my_inst;

    pr_debug("enter");

    my_inst = ecnr_service_get_curr_instance();
    if(!my_inst)
    {
        ret = report_ecnr_error(ECNR_ERR_NO_ECNR_OBJECT_AVAILABLE, "ecnr_object_ecnr_get_version", error);
        return ret;
    }

    ecnr_sse_get_version( &sse_Array_Len, &sse_Version_Array, &sse_Version_String, &sse_Version_Comment);

    *version = g_array_sized_new(FALSE, FALSE, sizeof(guint), 4);
    if (sse_Version_Array)
    {
        for (i = 0; i < sse_Array_Len; i++)
        {
            val = sse_Version_Array[i];
            g_array_append_val(*version, val);
        }
    }
    if (sse_Version_String)
        *version_string = g_strdup((gchar *) sse_Version_String);
    else
        *version_string = g_strdup("");
    if (sse_Version_Comment)
        *version_comment = g_strdup((gchar *) sse_Version_Comment);
    else
        *version_comment = g_strdup("");

    pr_debug("Version %s", *version_string);

    return TRUE;
}


/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_get_status_message
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_get_status_message
(
    EcnrObject * qobj,
    guchar ecnrAppId,
    gchar ** status_message,
    GError ** error
)
{
    int status = ECNR_ERR_OK;
    gboolean ret;
    gboolean state = TRUE;

    pr_debug("enter");

    status = ecnr_service_is_instance_active(ecnrAppId, &state);
    if(!state)
    {
        ret = report_ecnr_error(status, "ecnr_object_ecnr_get_status_message", error);
        return ret;
    }
    else
    {
        *status_message = g_malloc0(SSE_MAX_STRING_SIZE + 1);
        if(*status_message)
            status = ecnr_sse_get_Status_Message((char*) *status_message);
        else
            status = ECNR_ERR_NULL_POINTER;

    }

    ret = report_ecnr_error(status, "ecnr_object_ecnr_get_status_message", error);
    if (!ret)
    {
        if(*status_message)
            g_free(*status_message);
        *status_message = NULL;
    }
    else
        pr_debug("%s", *status_message);

  return ret;
}


/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_initialize
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_initialize(EcnrObject * qobj, guchar ecnrAppId, gint ecnrConfigurationId, GError ** error)
{
    int status = (int)ECNR_ERR_OK;
    gboolean ret;

    GError *ptm_error;

    pr_message("Entered with appid(%d) and cfgid (%d)\n", ecnrAppId, ecnrConfigurationId  );
    ECNR_PRINT_TIME_MSG("INIT");

    /* create new ecnr instance*/
    status = ecnr_service_create_new_instance(ecnrAppId);
    if(status != SSE_ERR_OK)
    {
        ret = report_ecnr_error(status, "ecnr_object_ecnr_initialize", error);
        return ret;
    }

    /* check first for AppId*/
    status = ecnr_service_check_AppId_ConfigId_Matching(ecnrConfigurationId, ecnrAppId);

    if (status == ECNR_ERR_OK)
    {
        status = ecnr_service_do_action_initialize(ecnrConfigurationId);
    }

    /* try to remove instance */
    if(status != ECNR_ERR_OK)
        (void)ecnr_service_remove_instance(ecnrAppId);
    else
    {
        #ifdef D_TRIGGER_PERFORMANCE_TEST
        ecnr_service_test_performance_initialize();
        #endif

        if(ecnr_service_get_pass_through_mode_default_state())
            ecnr_object_ecnr_set_pass_through_mode(qobj, ECNR_APPID_TEST, TRUE, &ptm_error);

#if defined (VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI)
        if(ecnrConfigurationId == PHONE_DATASET_1 || ecnrConfigurationId == PHONE_DATASET_2)
        {
          ecnr_object_ecnr_set_send_gain_delta(qobj,ecnrAppId,ecnr_service_get_mic_gain_level(),error);
        }
#else
          ecnr_object_ecnr_set_send_gain_delta(qobj,ecnrAppId,ecnr_service_get_mic_gain_level(),error);
#endif
        /*start audio thread */
        (void)ecnr_service_alsa_thread_create();
    }

    ret = report_ecnr_error(status, "ecnr_object_ecnr_initialize", error);

    pr_message("Leaved with error (%d)", status );
    ECNR_PRINT_TIME_MSG("FINAL");
    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_initialize_sync
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_initialize_sync
(
    EcnrObject * qobj,
    guchar ecnrAppId,
    gint ecnrConfigurationId,
    gchar** InDevices,
    gchar** OutDevices,
    GArray** DeviceProperties,
    GError ** error
)
{
    int         status = (int)ECNR_ERR_OK;
    gboolean    ret = TRUE;

    pr_message("Entered with appid(%d) and cfgid (%d)\n", ecnrAppId, ecnrConfigurationId);
    ECNR_PRINT_TIME_MSG("INIT");

    /* create new ecnr instance*/
    status = ecnr_service_create_new_instance(ecnrAppId);
    if(status != SSE_ERR_OK)
    {
        ret = report_ecnr_error(status, "ecnr_object_ecnr_initialize", error);
        return ret;
    }

    /* check first for AppId*/
    status = ecnr_service_check_AppId_ConfigId_Matching(ecnrConfigurationId, ecnrAppId);

    if (status == ECNR_ERR_OK)
    {
        status = ecnr_service_do_action_initialize(ecnrConfigurationId);
    }

    /* try to remove instance */
    if(status != ECNR_ERR_OK)
        (void)ecnr_service_remove_instance(ecnrAppId);
    else
    {
        #ifdef D_TRIGGER_PERFORMANCE_TEST
        ecnr_service_test_performance_initialize();
        #endif

        if(ecnr_service_get_pass_through_mode_default_state())
            ecnr_object_ecnr_set_pass_through_mode(qobj, ECNR_APPID_TEST, TRUE, error);

#if defined (VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI)
        if(ecnrConfigurationId == PHONE_DATASET_1 || ecnrConfigurationId == PHONE_DATASET_2)
        {
          ecnr_object_ecnr_set_send_gain_delta(qobj,ecnrAppId,ecnr_service_get_mic_gain_level(),error);
        }
#else
          ecnr_object_ecnr_set_send_gain_delta(qobj,ecnrAppId,ecnr_service_get_mic_gain_level(),error);
#endif

       /***************************************************
        *   return configuration data to the caller
        */

        gboolean isSyncDataMatching = FALSE;

        /* transfer device properties list */
        *DeviceProperties = ecnr_configuration_get_device_properties(ecnrAppId,ecnrConfigurationId, &isSyncDataMatching);

        /* transfer input device's list */
        *InDevices = ecnr_configuration_get_in_devices(ecnrAppId, ecnrConfigurationId, &isSyncDataMatching);

        /* transfer output device's list */
        *OutDevices = ecnr_configuration_get_out_devices(ecnrAppId,ecnrConfigurationId, &isSyncDataMatching);

        if(isSyncDataMatching)
            ecnr_configuration_sync_data_clear();

        /*start audio thread */
        (void)ecnr_service_alsa_thread_create();

    }

    ret = report_ecnr_error(status, "ecnr_object_ecnr_initialize_sync", error);

    pr_message("Leaved with error (%d)", status );
    ECNR_PRINT_TIME_MSG("FINAL");
    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_reset
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_reset(EcnrObject * qobj, guchar ecnrAppId, GError ** error)
{
    gboolean ret = TRUE;
    int status = (int)ECNR_ERR_OK;
    gboolean state = TRUE;

    pr_message("Entered with appid(%d)\n", ecnrAppId );

    status = ecnr_service_is_instance_active(ecnrAppId, &state);
    if(!state)
    {
        ret = report_ecnr_error(status, "ecnr_object_ecnr_reset", error);
        return ret;
    }
    else
        status = ecnr_sse_handler_reset();


    ret = report_ecnr_error(status, "ecnr_object_ecnr_reset", error);

    pr_message("Leaved with error (%d)", status );

    return ret;
}


/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_destroy
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_destroy(EcnrObject * qobj, guchar ecnrAppId, GError ** error)
{
    gboolean ret = TRUE;
    int status = (int)ECNR_ERR_OK;
    gboolean state = TRUE;


    pr_message("Entered with appid(%d)\n", ecnrAppId );
    ECNR_PRINT_TIME_MSG("INIT");

    status = ecnr_service_is_instance_active(ecnrAppId, &state);
    if(!state)
    {
        ret = report_ecnr_error(status, "ecnr_object_ecnr_destroy", error);
        return ret;
    }

    /* stop streaming */
    ecnr_alsa_stop(ECNR_START_STOP_TIMEOUT);

    /* destroy engine instance */
    (void)ecnr_service_do_action_destroy();

    /* leave audio thread*/
    ecnr_alsa_exit();

    (void)ecnr_service_remove_instance(ecnrAppId);

    ret = report_ecnr_error(status, "ecnr_object_ecnr_destroy", error);

    pr_message("Leaved with error (%d)", status );
    ECNR_PRINT_TIME_MSG("FINAL");
    return ret;
}


/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_start_audio_ext
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_start_audio_ext
(
    EcnrObject * qobj,
    guchar ecnrAppId,
    guchar StartMode,
    GError ** error
)
{
    gboolean    ret;
    int         status = (int)SSE_ERR_OK;
    struct      ecnr_instance_object* my_inst;
    gboolean    state = TRUE;
    glong       timeout = 0 ;

    pr_message("Entered with appid(%d) with StartMode(%d)\n", ecnrAppId, StartMode);
    ECNR_PRINT_TIME_MSG("INIT");

    status = ecnr_service_is_instance_active(ecnrAppId, &state);
    if(!state)
    {
        ret = report_ecnr_error(status, "ecnr_object_ecnr_start_audio", error);
        return ret;
    }

    my_inst = ecnr_service_get_this_instance(ecnrAppId);
    if(!my_inst)
    {
        ret = report_ecnr_error(ECNR_ERR_NO_INSTANCE_AVAILABLE, "ecnr_object_ecnr_start_audio", error);
        return ret;
    }
    else
    {
        if (my_inst->ecnr_is_initialized)
        {

#ifdef D_SIMULATE_SPI_CLIENT
            //ecnr_service_test_client_start();
#endif

            gboolean started = TRUE;

            /*start audio thread*/
            (void)ecnr_service_alsa_thread_create();

            if(StartMode == START_MODE_WAIT_STREAMING_RESULT)
                timeout = ECNR_START_STOP_TIMEOUT;
            else if (StartMode == START_MODE_RETURN_IMMEDIATELY)
                timeout = 0;
            else // default
                 timeout = ECNR_START_STOP_TIMEOUT;

            started = ecnr_alsa_start(timeout);
            if (!started)
                status = ECNR_ERR_NO_AUDIO_STREAMING; //SSE_ERR_NO_RESULT;
#ifdef D_TRIGGER_PERFORMANCE_TEST
            else
                ecnr_service_test_performance_start();
#endif
        }
        else
            status = SSE_ERR_MODULE_NOT_INIT;
    }

    ret = report_ecnr_error(status, "ecnr_object_ecnr_start_audio", error);

    pr_message("Leaved with error (%d)", status );
    ECNR_PRINT_TIME_MSG("FINAL")
    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_start_audio
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_start_audio(EcnrObject * qobj, guchar ecnrAppId, GError ** error)
{
    gboolean ret;

    pr_message("Entered with appid(%d)\n", ecnrAppId );

    ret = ecnr_object_ecnr_start_audio_ext(qobj, ecnrAppId, START_MODE_WAIT_STREAMING_RESULT, error);

    return ret;
}
/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_stop_audio
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_stop_audio(EcnrObject * qobj, guchar ecnrAppId, GError ** error)
{
    gboolean ret;
    gboolean stopped, state;
    int status = (int)SSE_ERR_OK;

    pr_message("Entered with appid(%d)\n", ecnrAppId );
    ECNR_PRINT_TIME_MSG("INIT");

    status = ecnr_service_is_instance_active(ecnrAppId, &state);
    if(!state)
    {
        ret = report_ecnr_error(status, "ecnr_object_ecnr_stop_audio", error);
        return ret;
    }
    else
    {
#ifdef D_TRIGGER_PERFORMANCE_TEST
        ecnr_service_test_performance_stop();
#endif

#ifdef D_SIMULATE_SPI_CLIENT
        ecnr_service_test_client_stop();
#endif

        stopped = ecnr_alsa_stop(ECNR_START_STOP_TIMEOUT);
        if (!stopped)
        {
            /*new section */
            pr_warning("the stop audio function is timed out\n");

            /* leave audio thread*/
            ecnr_alsa_exit();
            status = ECNR_ERR_OK;
        }
    }

    ret = report_ecnr_error(status, "ecnr_object_ecnr_stop_audio", error);

    pr_message("Leaved with error (%d)", status );
    ECNR_PRINT_TIME_MSG("FINAL");
    return ret;
}



/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_get_clip_stats
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static gboolean ecnr_object_ecnr_get_clip_stats(gint Channel,
                gint DataId, const gchar * Name,
                guint * ClipFrames,
                guint * ClipCount, gint * MaxAmp,
                gint * MinAmp, GError ** error)
{
    tsseClipStats ClipStats;
    unsigned int len;
    int status = (int)ECNR_ERR_OK;
    gboolean ret;
    struct ecnr_instance_object* my_inst;

    pr_debug("enter");

    my_inst = ecnr_service_get_curr_instance();
    if(!my_inst)
    {
        ret = report_ecnr_error(ECNR_ERR_NO_ECNR_OBJECT_AVAILABLE, "ecnr_object_ecnr_get_clip_stats", error);
        return ret;
    }

    len = sizeof(ClipStats);
    status = ecnr_sse_handler_get_data(DataId, Channel, &len, (void*)&ClipStats);


    *ClipFrames = ClipStats.ClipFrames;
    *ClipCount  = ClipStats.ClipCount;
    *MaxAmp     = ClipStats.MaxAmp;
    *MinAmp     = ClipStats.MinAmp;

    ret = report_ecnr_error(status, Name, error);


    if (!status)
        pr_debug("%s (%u, %u, %d, %d)", Name, *ClipFrames, *ClipCount, *MaxAmp, *MinAmp);

    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_mic_clip_stats
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_mic_clip_stats(EcnrObject * qobj, gint Channel,
               guint * ClipFrames,
               guint * ClipCount, gint * MaxAmp,
               gint * MinAmp, GError ** error)
{
    gboolean ret;
    gchar *Name;
    static const gchar *BaseName = "sse_MicInClipStats";
    struct ecnr_instance_object* my_inst;

    pr_debug("enter");

    my_inst = ecnr_service_get_curr_instance();
    if(!my_inst)
    {
        ret = report_ecnr_error(ECNR_ERR_NO_ECNR_OBJECT_AVAILABLE, "ecnr_object_ecnr_mic_clip_stats", error);
        return ret;
    }

    if (Channel >= 0)
        Name = g_strdup_printf("%s%d", BaseName, Channel);
    else
        Name = g_strdup(BaseName);

    ret = ecnr_object_ecnr_get_clip_stats(Channel, sse_MicInClipStats, Name, ClipFrames, ClipCount, MaxAmp, MinAmp, error);

    if(Name)
        g_free(Name);
    Name = NULL;

    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_ref_clip_stats
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_ref_clip_stats(EcnrObject * qobj,
               guint * ClipFrames,
               guint * ClipCount, gint * MaxAmp,
               gint * MinAmp, GError ** error)
{
    gboolean ret;
    struct ecnr_instance_object* my_inst;

    pr_debug("enter");

    my_inst = ecnr_service_get_curr_instance();
    if(!my_inst)
    {
        ret = report_ecnr_error(ECNR_ERR_NO_ECNR_OBJECT_AVAILABLE, "ecnr_object_ecnr_ref_clip_stats", error);
        return ret;
    }

    ret = ecnr_object_ecnr_get_clip_stats(0, sse_RefInClipStats, "sse_RefInClipStats", ClipFrames, ClipCount, MaxAmp, MinAmp, error);

    return ret;
}


/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_recv_clip_stats
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_recv_clip_stats(EcnrObject * qobj,
          guint * ClipFrames,
          guint * ClipCount, gint * MaxAmp,
          gint * MinAmp, GError ** error)
{
    gboolean ret;
    struct ecnr_instance_object* my_inst;

    pr_debug("enter");

    my_inst = ecnr_service_get_curr_instance();
    if(!my_inst)
    {
        ret = report_ecnr_error(ECNR_ERR_NO_ECNR_OBJECT_AVAILABLE, "ecnr_object_ecnr_recv_clip_stats", error);
        return ret;
    }

    ret  = ecnr_object_ecnr_get_clip_stats(0, sse_RecvInClipStats, "sse_RecvInClipStats", ClipFrames, ClipCount, MaxAmp, MinAmp, error);

    return ret;
}


/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_mic_out_clip_stats
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_mic_out_clip_stats(EcnrObject * qobj,
             guint * ClipFrames,
             guint * ClipCount,
             gint * MaxAmp, gint * MinAmp,
             GError ** error)
{
    gboolean ret;
    struct ecnr_instance_object* my_inst;

    pr_debug("enter");

    my_inst = ecnr_service_get_curr_instance();
    if(!my_inst)
    {
        ret = report_ecnr_error(ECNR_ERR_NO_ECNR_OBJECT_AVAILABLE, "ecnr_object_ecnr_mic_out_clip_stats", error);
        return ret;
    }

    ret = ecnr_object_ecnr_get_clip_stats(0, sse_MicOutClipStats,"sse_MicOutClipStats", ClipFrames, ClipCount, MaxAmp, MinAmp, error);

    return ret;
}


/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_recv_out_clip_stats
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_recv_out_clip_stats(EcnrObject * qobj,
              guint * ClipFrames,
              guint * ClipCount,
              gint * MaxAmp, gint * MinAmp,
              GError ** error)
{
    gboolean ret;
    struct ecnr_instance_object* my_inst;

    pr_debug("enter");

    my_inst = ecnr_service_get_curr_instance();
    if(!my_inst)
    {
        ret = report_ecnr_error(ECNR_ERR_NO_ECNR_OBJECT_AVAILABLE, "ecnr_object_ecnr_recv_out_clip_stats", error);
        return ret;
    }

    ret = ecnr_object_ecnr_get_clip_stats(0, sse_RecvOutClipStats, "sse_RecvOutClipStats", ClipFrames, ClipCount, MaxAmp, MinAmp, error);

    return ret;
}


/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_set_configuration
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_set_configuration
(
    EcnrObject * qobj,
    guchar ecnrAppId,
    gint ecnrConfigurationId,
    GError ** error
)
{
    int status = ECNR_ERR_OK;
    gboolean ret;
    gboolean state = TRUE;
    gboolean CreateNewSession = TRUE; //FALSE;
    struct ecnr_instance_object* my_inst;

    pr_message("Entered with appid(%d) and cfgid (%d)\n", ecnrAppId, ecnrConfigurationId  );
    ECNR_PRINT_TIME_MSG("INIT");


    status = ecnr_service_is_instance_active(ecnrAppId, &state);
    if(!state)
    {
        ret = report_ecnr_error(status, "ecnr_object_ecnr_set_configuration", error);
        return ret;
    }

    /* check first for AppId*/
    status = ecnr_service_check_AppId_ConfigId_Matching(ecnrConfigurationId, ecnrAppId);

    if (status == ECNR_ERR_OK)
    {
        /* check whether streaming is currently active */
        if(ecnr_alsa_get_audio_run_state())
        {
            pr_warning("audio streaming is active, ecnrStopAutio should be called before\n");
            status = ECNR_ERR_OBJECT_FI_AUDIO_IS_RUNNING;
        }
        else
        {
            /* retrieve current instance */
            my_inst = ecnr_service_get_curr_instance();

            /* check whether config Id can be configured  */
            if(my_inst)
            {
                if (my_inst->ecnr_is_initialized)
                {
                    if(my_inst->ecnr_current_config_id == ecnrConfigurationId)
                    {
                        pr_warning("sse already initialized with Config %d", ecnrConfigurationId);
                        CreateNewSession = FALSE;
                        status = 0;
                    }
                    else
                        (void)ecnr_service_do_action_destroy();
                }
            }

            if (CreateNewSession)
            {
                status = ecnr_service_do_action_initialize(ecnrConfigurationId);

                /* try to remove instance */
                if(status != ECNR_ERR_OK)
                    (void)ecnr_service_remove_instance(ecnrAppId);
                else
                {
                    /*start audio thread */
                    (void)ecnr_service_alsa_thread_create();
                }
            }
        }
    }

    ret = report_ecnr_error(status, "ecnr_object_ecnr_set_configuration", error);

    pr_message("Leaved with error (%d)", status );
    ECNR_PRINT_TIME_MSG("FINAL");
    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_set_configuration_sync
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_set_configuration_sync
(
    EcnrObject * qobj,
    guchar ecnrAppId,
    gint ecnrConfigurationId,
  gchar** InDevices,
  gchar** OutDevices,
    GArray** DeviceProperties,
    GError ** error
)
{
  int             status = ECNR_ERR_OK;
  gboolean        ret;
  gboolean        state = TRUE;
  gboolean        CreateNewSession = TRUE; //FALSE;
  gboolean        RetSyncInfo = FALSE;
  struct ecnr_instance_object* my_inst;

    pr_message("Entered with appid(%d) and cfgid (%d)\n", ecnrAppId, ecnrConfigurationId  );
    ECNR_PRINT_TIME_MSG("INIT");


    status = ecnr_service_is_instance_active(ecnrAppId, &state);
    if(!state)
    {
        ret = report_ecnr_error(status, "ecnr_object_ecnr_set_configuration_sync", error);
        return ret;
    }

    /* check first for AppId*/
    status = ecnr_service_check_AppId_ConfigId_Matching(ecnrConfigurationId, ecnrAppId);

    if (status == ECNR_ERR_OK)
    {
        /* check whether streaming is currently active */
        if(ecnr_alsa_get_audio_run_state())
        {
            pr_warning("audio streaming is active, ecnrStopAutio should be called before\n");
            status = ECNR_ERR_OBJECT_FI_AUDIO_IS_RUNNING;
        }
        else
        {
            /* retrieve current instance */
            my_inst = ecnr_service_get_curr_instance();

            /* check whether config Id can be configured  */
            if(my_inst)
            {
                if (my_inst->ecnr_is_initialized)
                {
                    RetSyncInfo = TRUE;

                    if(my_inst->ecnr_current_config_id == ecnrConfigurationId)
                    {
                        pr_warning("sse already initialized with Config %d", ecnrConfigurationId);
                        CreateNewSession = FALSE;
                        status = 0;
                    }
                    else
                        (void)ecnr_service_do_action_destroy();
                }
            }

            if (CreateNewSession)
            {
                status = ecnr_service_do_action_initialize(ecnrConfigurationId);

                /* try to remove instance */
                if(status != ECNR_ERR_OK)
                    (void)ecnr_service_remove_instance(ecnrAppId);

                else
                {
                    /*start audio thread */
                    (void)ecnr_service_alsa_thread_create();
                }
            }


            if(RetSyncInfo)
            {
                /***************************************************
                 *   return configuration data to the caller
                 */

                gboolean isSyncDataMatching = FALSE;

                /* transfer device properties list */
                *DeviceProperties = ecnr_configuration_get_device_properties(ecnrAppId,ecnrConfigurationId, &isSyncDataMatching);

                /* transfer input device's list */
                *InDevices = ecnr_configuration_get_in_devices(ecnrAppId, ecnrConfigurationId, &isSyncDataMatching);

                /* transfer output device's list */
                *OutDevices = ecnr_configuration_get_out_devices(ecnrAppId,ecnrConfigurationId, &isSyncDataMatching);

                if(isSyncDataMatching)
                    ecnr_configuration_sync_data_clear();

            }
        }
    }

    ret = report_ecnr_error(status, "ecnr_object_ecnr_set_configuration_sync", error);

    pr_message("Leaved with error (%d)", status );
    ECNR_PRINT_TIME_MSG("FINAL");
    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_get_ecnr_configuration
*
* DESCRIPTION: This function retrives the current active dataset name(opeth file name)
*          by reading configuration data file and
*          current active AppID, ConfigID, EcnrID and CRC of the of the dataset.
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_get_ecnr_configuration
(
    EcnrObject * qobj,
  gchar ** DatasetName,
  GArray ** EcnrProperties,
    GError ** error
)
{
    int        status = ECNR_ERR_OK;
    gboolean    ret;
    guint16      crc = 0;
    guint32      ecnr_properties[4];
    const struct EcnrConfig *pdata = NULL;
    struct ecnr_instance_object* my_inst = NULL;
    guint32         val;
    gint            i;

    pr_debug("enter");

    /* retrieve current instance */
    my_inst = ecnr_service_get_curr_instance();

    if(!my_inst)
    {
        ret = report_ecnr_error(ECNR_ERR_NO_ECNR_OBJECT_AVAILABLE, "ecnr_object_ecnr_get_ecnr_configuration", error);
        return ret;
    }

    if(my_inst->ecnr_is_initialized)
    {
      /* retrieve configuration data for the requested copnfiguration Id */
      pdata = ecnr_configuration_get_config_data(my_inst->ecnr_current_config_id);

        if (!pdata)
        {
            pr_warning("no valid configuration for configId : % d", my_inst->ecnr_current_config_id);
            ret = report_ecnr_error(ECNR_ERR_CONFIG_NOT_AVAILABLE, "ecnr_object_ecnr_get_ecnr_configuration", error);
            return ret;
        }

        if (pdata->opeth_str)
          *DatasetName = g_strdup((gchar *) pdata->opeth_str);
        else
          *DatasetName = g_strdup("");

        pr_debug("ecnr_dataset_name(%s)", pdata->opeth_str);

        if(!ecnr_configuration_get_crc_from_current_active_dataset(pdata->EcnrdPBaseKey, &crc))
          pr_message("value crc(0x%x) of dp(%d)", crc,pdata->EcnrdPBaseKey);

        ecnr_properties[0] = crc;
        ecnr_properties[1] = my_inst->ecnr_current_app_id;
        ecnr_properties[2] = my_inst->ecnr_current_config_id;
        ecnr_properties[3] = pdata->configId;

        *EcnrProperties = g_array_sized_new(FALSE, FALSE, sizeof(guint), 4);

            for (i = 0; i < 4 ; i++)
            {
                val = ecnr_properties[i];
                g_array_append_val(*EcnrProperties, val);

                pr_debug("EcnrProperties[%i] = %d", i, val);
            }
    }

    ret = report_ecnr_error(status, "ecnr_object_ecnr_get_ecnr_configuration", error);

    pr_message("Leaved with error (%d)", status );

    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_start_debug_data
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_start_debug_data(EcnrObject * qobj, guchar ecnrDebugMode, GError ** error)
{
    int status = ECNR_ERR_OK;
    gboolean ret = TRUE;

    pr_message("Entered\n");


    switch (ecnrDebugMode)
    {
        case ECNR_DBUG_DEFAULT:
            status = ECNR_ERR_DEBUG_ALREADY_ACTIVE;
            break;
        case ECNR_DBUG_SET_TRIGGER_FILE:
            if(!ecnr_sse_set_debug_trigger_file())
                status=ECNR_ERR_DGB_SET_RMV_TRIGGER_FILE_FAILED;
            break;
        case ECNR_DBUG_FORCE_AUDIO_IO: //3
            break;
        case ECNR_DBUG_DATA_ALSA_AVAILABLE: //4
            status = ecnr_alsa_enable_sse_debug();
            break;
        case ECNR_DBUG_START_SESSION: //6
            status = ecnr_sse_debug_interface_start_session();
            break;
        case ECNR_DBUG_PRINT_DEV_PROP: //8
            status = ecnr_alsa_print_dev_properties();
            break;
        case ECNR_DBUG_PRINT_RTS_CONFIG: //9
            ecnr_audio_rts_handler_print_config_data();
            break;
        case ECNR_DBUG_PRINT_ECNR_DATA_CONFIG:
            ecnr_configuration_print_config_data();
            break;
        case ECNR_DBUG_TEST_TRIGGER_NOTIFICATION_NEW_DATAPOOL_ALL:
            ecnr_configuration_stub_notify_dp_change();
            break;
        case ECNR_DBUG_TEST_TRIGGER_NOTIFICATION_NEW_DATAPOOL_1:
        #ifndef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
            ecnr_datapool_access_trigger_new_datapool(ECNR_DP_BT_PHONE_NB);
        #endif
            break;
        case ECNR_DBUG_TEST_FORCE_EXIT_ALSA_THREAD:
            ecnr_alsa_exit();
            break;
        case ECNR_DBUG_TEST_FORCE_CREATE_ALSA_THREAD:
            (void)ecnr_service_alsa_thread_create();
            break;
        case ECNR_DBUG_PRINT_CURRENT_AUDI0_STATE:
            ecnr_alsa_request_print_current_audio_state();
            break;
        case ECNR_DBUG_PRINT_RCS_CONF_FILES:
        #ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
            ecnr_qwa_handler_print_rcs_configuration();
        #endif
            break;
        case ECNR_CALL_NATIVE_START_AUDIO:
            ecnr_alsa_start(ECNR_START_STOP_TIMEOUT);
            break;
        case ECNR_DBUG_PRINT_DP_INFOS:
            ecnr_configuration_print_all_dp_infos();
            break;
        case ECNR_DBUG_SET_IO_CTRL_HF_VR_MODE:
            ecnr_ioctrl_set_pin_state(92, TRUE);
            break;
        case ECNR_DBUG_RESET_IO_CTRL_HF_VR_MODE:
            ecnr_ioctrl_set_pin_state(92, FALSE);
            break;
        case ECNR_DBUG_FORCE_USE_DEFAULT_DP:
            ecnr_service_set_use_default_dp();
            break;
        case ECNR_DBUG_FORCE_USE_DEFAULT_DP_EXIT:
            ecnr_service_set_use_default_dp_exit();
            break;
        case ECNR_DBUG_SWITCH_OFF_CHECK_ALSA_DEV:
        case ECNR_DBUG_SWITCH_OFF_CHECK_APPID_CNFGID:
            default:
            status = ECNR_ERR_DEBUG_MODER_NOT_SUPPORTED;
            break;
    }


    ret = report_ecnr_error(status, "ecnr_sse_debug_interface_start_session", error);

    pr_debug("Debug mode %d was requested \n",ecnrDebugMode );

    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_end_debug_data
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/

gboolean ecnr_object_ecnr_end_debug_data(EcnrObject * qobj,guchar ecnrDebugMode, GError ** error)
{
    int status = ECNR_ERR_OK;
    gboolean ret = TRUE;;
    struct ecnr_instance_object* my_inst;

    pr_message("Entered\n");

    my_inst = ecnr_service_get_curr_instance();
    if(!my_inst)
    {
        ret = report_ecnr_error(ECNR_ERR_NO_ECNR_OBJECT_AVAILABLE, "ecnr_object_ecnr_end_debug_data", error);
        return ret;
    }

    switch (ecnrDebugMode)
    {
        case ECNR_DBUG_DEFAULT:
        {
            if(my_inst->ecnr_Debug_intf_initialized)
            {
                status = ecnr_sse_disable_debug_interface();
            }
            break;
        }
        case ECNR_DBUG_DELETE_TRIGGER_FILE: //2
            if(!ecnr_sse_remove_debug_trigger_file())
                status=ECNR_ERR_DGB_SET_RMV_TRIGGER_FILE_FAILED;
            break;
        case ECNR_DBUG_FORCE_AUDIO_IO:
        case ECNR_DBUG_DATA_ALSA_DISABLE: //5
            status = ret = ecnr_alsa_disable_sse_debug();
            break;
        case ECNR_DBUG_STOP_SESSION: //7
            status = ecnr_sse_debug_interface_end_session();
            break;
        case ECNR_DBUG_SWITCH_OFF_CHECK_ALSA_DEV:
        case ECNR_DBUG_SWITCH_OFF_CHECK_APPID_CNFGID:
        default:
            status = ECNR_ERR_DEBUG_MODER_NOT_SUPPORTED;
            break;
    }

    if (!status)
        my_inst->ecnr_Debug_intf_initialized = FALSE;

    ret = report_ecnr_error(status, "ecnr_object_ecnr_end_debug_data", error);

    pr_debug("Debug mode %d is disabled",ecnrDebugMode );

    return ret;
}


/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_send_mute_switch
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_set_send_mute_switch(EcnrObject * qobj, guchar ecnrAppId, guchar ecnrSendMuteSwitch, GError ** error)
{
    int status = ECNR_ERR_OK;
    gboolean ret = TRUE;
    gboolean state = TRUE;
    int lswitch = (int)sseOff; // same value for qwaOff (1)

    pr_message("Entered\n");

    status = ecnr_service_is_instance_active(ecnrAppId, &state);
    if(!state)
    {
        ret = report_ecnr_error(status, "ecnr_object_ecnr_set_send_mute_switch", error);
        return ret;
    }
    else
    {
        if(ecnrSendMuteSwitch)
        lswitch = (int)sseOn; // same value for qwaOn (2)

#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
        status = ecnr_qwa_handler_set_data(qwaSendMuteSwitch, 0, sizeof(lswitch),(void*) &lswitch );
#else
        status = ecnr_sse_handler_set_data(sse_SendMuteSwitch, 0, sizeof(lswitch),(void*) &lswitch );
#endif
    }

    ret = report_ecnr_error(status, "ecnr_object_ecnr_set_send_mute_switch", error);

    pr_message("Leaved with error (%d)", status );

    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_send_gain_delta
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_set_send_gain_delta(EcnrObject * qobj, guchar ecnrAppId, gint16 ecnrLevel, GError ** error)
{
    int status = ECNR_ERR_OK;
    gboolean ret = TRUE;
    gboolean state = TRUE;
    gint8 level = (gint8)ecnrLevel;

    pr_message("Entered\n");

    status = ecnr_service_is_instance_active(ecnrAppId, &state);
    if(!state)
    {
        ret = report_ecnr_error(status, "ecnr_object_ecnr_set_send_gain_delta", error);
        return ret;
    }
    else
    {
        status = ecnr_sse_handler_set_data( sse_SendGainDelta, 0, sizeof(level),(void*) &level );
    }

    ret = report_ecnr_error(status, "ecnr_object_ecnr_set_send_gain_delta", error);

    pr_message("Leaved with error (%d)", status );

    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_set_mic_gain_level
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_set_mic_gain(EcnrObject * qobj, gint16 MicGainLevel, GError ** error)
{
    /*
    * Suppress lint Info 715 "Symbol not referenced" message regarding
    * the formal parameter 'qobj'.
    */
    /*lint -efunc(715,ecnr_object_ecnr_set_mic_gain) */

    int status = ECNR_ERR_OK;
    gboolean ret = TRUE;
    struct ecnr_instance_object* my_inst = NULL;
    gint ecnrConfigurationId;
    guchar ecnrAppId;

    pr_message("Entered\n");

    if(MicGainLevel < -30 || MicGainLevel > 30)
    {
      status = SSE_ERR_ILLEGAL_SET_PARAM;
    }
    else
    {
      ecnr_service_set_mic_gain_level(MicGainLevel);

      /* retrieve current instance */
      my_inst = ecnr_service_get_curr_instance();

      if(my_inst)
      {

        if(my_inst->ecnr_is_initialized)
        {
          ecnrAppId = my_inst->ecnr_current_app_id;
          ecnrConfigurationId = my_inst->ecnr_current_config_id;

#if defined (VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI)
          if(ecnrConfigurationId == PHONE_DATASET_1 || ecnrConfigurationId == PHONE_DATASET_2)
          {
            ecnr_object_ecnr_set_send_gain_delta(qobj,ecnrAppId,MicGainLevel,error);
          }
#else
          ecnr_object_ecnr_set_send_gain_delta(qobj,ecnrAppId,MicGainLevel,error);
#endif
        }
      }
    }

    ret = report_ecnr_error(status, "ecnr_object_ecnr_set_mic_gain_level", error);

    pr_message("Leaved with error (%d)", status );

    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_pass_through_mode
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_set_pass_through_mode(EcnrObject * qobj, guchar ecnrAppId, guchar ecnrPassThroughMode, GError ** error)
{
    int status = ECNR_ERR_OK;
    gboolean ret = TRUE;
    gboolean state = TRUE;
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    int iswitch = (int)qwaOff;
#else
    int lmode = (int)sseNoPassThrough;
#endif

    pr_message("Entered\n");

    if(ecnr_service_get_pass_through_mode_default_state() && (ecnrAppId != ECNR_APPID_TEST ))
      return ECNR_ERR_OK;

    if( ecnrAppId == ECNR_APPID_TEST )
      ecnr_service_set_pass_through_mode_default_state(ecnrPassThroughMode);
    else
      status = ecnr_service_is_instance_active(ecnrAppId, &state);

    if(!state)
    {
        ret = report_ecnr_error(status, "ecnr_object_ecnr_set_pass_through_mode", error);
        return ret;
    }
    else
    {
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
        if(ecnrPassThroughMode)
            iswitch = (int)qwaOn;
        status = ecnr_qwa_handler_set_data(qwaPassThruSwitch, 0, sizeof(iswitch),(void*) &iswitch );
#else
        if(ecnrPassThroughMode)
            lmode = (int)ssePassThroughProcess;
        status = ecnr_sse_handler_set_data(sse_PassThroughMode, 0, sizeof(lmode),(void*) &lmode );
#endif
    }

    ret = report_ecnr_error(status, "ecnr_object_ecnr_set_pass_through_mode", error);

    pr_message("Leaved with error (%d)", status );

    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_nr_floor
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_set_nr_comb_floor(EcnrObject * qobj, guchar ecnrAppId, gint16 ecnrNRCombFloor, GError ** error)
{
    int status = ECNR_ERR_OK;
    gboolean ret = TRUE;
    gboolean state = TRUE;
    gint16 level = ecnrNRCombFloor;

    pr_message("Entered\n");

    status = ecnr_service_is_instance_active(ecnrAppId, &state);
    if(!state)
    {
        ret = report_ecnr_error(status, "ecnr_object_ecnr_set_nr_comb_floor", error);
        return ret;
    }
    else
    {
        status = ecnr_sse_handler_set_data(sse_NRCombFloor, 0, sizeof(level),(void*) &level);
    }

    ret = report_ecnr_error(status, "ecnr_object_ecnr_set_nr_comb_floor", error);

    pr_debug("leave\n");
    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_set_echo_cancellation
*
* DESCRIPTION: This function enable/disable the  EC- echo cancellation feature
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_set_echo_cancellation(EcnrObject * qobj, guchar ecnrAppId, guchar EcnrECState, GError ** error)
{
    int status = ECNR_ERR_OK;
    gboolean ret = TRUE;
    gboolean state = TRUE;
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    int iswitch = (int)qwaOff;
#else
    int lmode = (int)sseOff;
#endif

    pr_message("Entered\n");

    status = ecnr_service_is_instance_active(ecnrAppId, &state);
    if(!state)
    {
        ret = report_ecnr_error(status, "ecnr_object_ecnr_set_echo_cancellation", error);
        return ret;
    }
    else
    {
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
        if(EcnrECState)
            iswitch = (int)qwaOn;
        status = ecnr_qwa_handler_set_data(qwaAECSwitch, 0, sizeof(iswitch),(void*) &iswitch );
#else
        if(EcnrECState)
            lmode = (int)sseOn;
        status = ecnr_sse_handler_set_data(sse_AECSwitch, 0, sizeof(lmode),(void*) &lmode );
#endif
    }

    ret = report_ecnr_error(status, "ecnr_object_ecnr_set_echo_cancellation", error);

    pr_message("Leaved with error (%d)", status );

    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_set_noise_reduction
*
* DESCRIPTION: This function enable/disable the  NR- noise reduction feature
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_set_noise_reduction(EcnrObject * qobj, guchar ecnrAppId, guchar EcnrNRState, GError ** error)
{
    int status = ECNR_ERR_OK;
    gboolean ret = TRUE;
    gboolean state = TRUE;
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    int iswitch = (int)qwaOff;
#else
    int lmode = (int)sseOff;
#endif

    pr_message("Entered\n");

    status = ecnr_service_is_instance_active(ecnrAppId, &state);
    if(!state)
    {
        ret = report_ecnr_error(status, "ecnr_object_ecnr_set_noise_reduction", error);
        return ret;
    }
    else
    {
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
        if(EcnrNRState)
            iswitch = (int)qwaOn;
        status = ecnr_qwa_handler_set_data(qwaNRSwitch, 0, sizeof(iswitch),(void*) &iswitch );
#else
        if(EcnrNRState)
            lmode = (int)sseOn;
        status = ecnr_sse_handler_set_data(sse_NRSwitch, 0, sizeof(lmode),(void*) &lmode );
#endif
    }

    ret = report_ecnr_error(status, "ecnr_object_ecnr_set_noise_reduction", error);

    pr_message("Leaved with error (%d)", status );

    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_alsa_set_device
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_alsa_set_device
(
    EcnrObject * qobj,
    guchar ecnrAppId,
    guint AlsaDeviceSelect,
    gchar * AlsaDeviceName,
    GError ** error
)
{
    int status = ECNR_ERR_OK;
    gboolean ret = TRUE;
    gboolean state = TRUE;
    gboolean set = FALSE;
    enum ecnr_audio dev_id_rep = (enum ecnr_audio)ECNR_MIC_IN0;
    enum ecnr_audio dev_id_set = (enum ecnr_audio)ECNR_MIC_IN0;
    snd_pcm_stream_t sdir = SND_PCM_PRIVATE_DEF_INVALID;

    pr_message("Entered App(%d) set device(%s) to channel(%d)\n", ecnrAppId, AlsaDeviceName,AlsaDeviceSelect);

    status = ecnr_service_is_instance_active(ecnrAppId, &state);
    if(!state)
    {
        ret = report_ecnr_error(status, "ecnr_object_ecnr_alsa_set_device", error);
        return ret;
    }
    else
    {
        dev_id_set = (enum ecnr_audio)AlsaDeviceSelect;

        if(dev_id_set >= NUM_ECNR_AUDIO)
        {
            pr_message("dev_id (%d) out of range (max %d)",dev_id_set, NUM_ECNR_AUDIO);
            status = ECNR_ERR_DEV_ID_NOT_AVAILABLE;
        }

        if(!status)
        {
            set = TRUE;

            /*  set device per string */
            if(    dev_id_set == ECNR_SET_PLB_DEV_PER_STRING
                || dev_id_set == ECNR_SET_CAP_DEV_PER_STRING)
            {
                if (dev_id_set == ECNR_SET_PLB_DEV_PER_STRING)
                    sdir = SND_PCM_STREAM_PLAYBACK;
                else if (dev_id_set == ECNR_SET_CAP_DEV_PER_STRING)
                    sdir = SND_PCM_STREAM_CAPTURE;

                dev_id_rep = ecnr_configuration_get_dev_id(AlsaDeviceName,sdir);
                if(dev_id_rep != NUM_ECNR_AUDIO)
                {
                    dev_id_set = dev_id_rep;
                }
                else
                    set = FALSE;
            }

            /* the multi channel input device alsa device is set*/
            if(     dev_id_set == ECNR_MULTI_CHAN_IN_DEV
                ||  dev_id_set == ECNR_MULTI_CHAN_IN_MIC_0
                ||  dev_id_set == ECNR_MULTI_CHAN_IN_MIC_1
                ||  dev_id_set == ECNR_MULTI_CHAN_IN_REF )
            {
                ecnr_alsa_set_mutli_channel_in_stream_mode(TRUE);
            }

            else if ( dev_id_set == ECNR_MIC_IN0 )
            {
                ecnr_alsa_set_mutli_channel_in_stream_mode(FALSE);
            }

            if(set)
            {
                status = ecnr_configuration_check_match_dev_chid(dev_id_set, AlsaDeviceName);
                if(status == ECNR_ERR_OK)
                {
                    ecnr_alsa_set_dev_name(dev_id_set, AlsaDeviceName);
                    pr_debug("ecnr_alsa_set_dev_name to ecnr audio channel %d and device name %s \n" ,dev_id_set, AlsaDeviceName);
                }
            }
            else
                status = ECNR_ERR_DEV_ID_NOT_AVAILABLE;
        }
    }

    if(status == ECNR_ERR_OK)
        ecnr_alsa_disable_dev_configuration(dev_id_set);

    ret = report_ecnr_error(status, "ecnr_object_ecnr_alsa_set_device", error);

    pr_message("Leaved with error (%d)", status );
    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_alsa_set_device_properties
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
extern  gboolean ecnr_object_ecnr_alsa_set_device_properties
(
    EcnrObject * qobj,
    guchar  ecnrAppId,
    guint   AlsaDeviceSelect,
    guchar   AlsaDeviceProperty,
    guint  PropertyValue,
    GError ** error
)
{
    int status = ECNR_ERR_OK;;
    gboolean ret = TRUE;
    gboolean state = TRUE;

    pr_message("Entered\n");

    status = ecnr_service_is_instance_active(ecnrAppId, &state);
    if(!state)
    {
        ret = report_ecnr_error(status, "ecnr_object_ecnr_alsa_set_device_properties", error);
        return ret;
    }
    else
    {
        switch(AlsaDeviceProperty)
        {
            case PROP_ALSA_DEV_OPEN_ORDER:
                ecnr_alsa_set_device_open_order((enum ecnr_audio)AlsaDeviceSelect, PropertyValue);
                ecnr_alsa_dev_order_ctrl(FALSE);
                break;
            case PROP_ALSA_DEV_PREFILL:
                ecnr_alsa_set_device_prefill_value(PropertyValue);
                ecnr_alsa_prefill_ctrl(FALSE);
                break;
            default:
                break;
        }
    }

    ret = report_ecnr_error(status, "ecnr_object_ecnr_alsa_set_device_properties", error);

    pr_message("Leaved with error (%d)", status );

    return ret;
}
/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_alsa_set_thread_priority
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_alsa_set_thread_priority(EcnrObject * qobj, guchar ecnrAppId, gint priority, GError ** error)
{
    int status = ECNR_ERR_OK;
    gboolean ret = TRUE;
    gboolean state = TRUE;
    guint thread_priority;

    thread_priority =  (guint)priority; // g_value_get_uint(value);
    pr_debug("current Thread Prio is :%u, request Prio ist : %u", thread_priority, ecnr_alsa_get_thread_priority());

    status = ecnr_service_is_instance_active(ecnrAppId, &state);
    if(!state)
    {
        ret = report_ecnr_error(status, "ecnr_object_ecnr_alsa_set_thread_priority", error);
        return ret;
    }
    else
    {
        if (thread_priority > sched_get_priority_max(SCHED_FIFO))
        {
            thread_priority = sched_get_priority_max(SCHED_FIFO);
            pr_debug("set Thread Prio is :%u", thread_priority);

            ecnr_alsa_set_thread_priority(thread_priority);
        }
    }

    ret = report_ecnr_error(status, "ecnr_object_ecnr_alsa_set_thread_priority", error);

    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_alsa_set_wide_band
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_alsa_set_wide_band(EcnrObject * qobj, guchar ecnrAppId, GError ** error)
{
    int status = ECNR_ERR_OK;
    gboolean ret = TRUE;
    gboolean state = TRUE;

    pr_debug("enter");

    status = ecnr_service_is_instance_active(ecnrAppId, &state);
    if(!state)
    {
        ret = report_ecnr_error(status, "ecnr_object_ecnr_alsa_set_wide_band", error);
        return ret;
    }

    status = ECNR_ERR_OBJECT_FI_NOT_SUPPORTED;

    ret = report_ecnr_error(status, "ecnr_object_ecnr_alsa_set_wide_band", error);

    pr_debug("leaved");

    return ret;
}



extern GMainLoop *ecnr_main_loop;

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_terminate
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_terminate(EcnrObject * qobj, GError ** error)
{
  pr_message("");

  if(ecnr_main_loop)
    g_main_loop_quit(ecnr_main_loop);
  return TRUE;
}



