/*
 * node_validation.c
 * ALPS validation node handling for the D-bus Bluetooth Daemon
 *
 * Author: Dean Jenkins <djenkins@mvista.com>
 *
 * 2010 (c) MontaVista Software, LLC. This file is licensed under
 * the terms of the AFL.
 */

#include <glib.h>
#include <glib-object.h>
#include <glib/gprintf.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>

#include "main.h"

#include "marshal.h"

#include "bt_appl/common_def.h"
#include "bt_appl/api/bt_appl_bt_setting_block.h"
#include "bt_appl_dummy.h"
#include "callbacks.h"
#include "utils_async_queue.h"
#include "node_validation.h"

#include "dbus_error.h"
#include "debug.h"

#ifdef USE_VALIDATION_NODE

/************/
/* #defines */
/************/

/**************************/
/* VALIDATION DEFINITIONS */
/**************************/

/***********************/
/* DEVICE DEFINITIONS */
/***********************/

/* OUR VALIDATION OBJECT DEFINITION */
typedef struct {
	/* The parent class object state. */
	GObject parent;

	struct async_queue_list aql_validation;

} ValidationObject;

/* OUR CLASS DEFINITION */
typedef struct {
	/* The parent class state. */
	GObjectClass parent;

	/* put validation signal stuff here, (if any) */
	/* validation signals */
	guint ValidationTestModeComplete_sig_id;

} ValidationObjectClass;

/*
 * MAGIC here to define our validation object functions etc. in GLib
 * declares validation_object_init(), validation_object_class_init(),
 * validation_object_get_type()
 */
G_DEFINE_TYPE(ValidationObject, validation_object, G_TYPE_OBJECT)

/*
 * Forward declaration of the function that will return the GType of
 * the Value implementation. Not used
 */
GType validation_object_get_type(void);

/*
 * Macro for the above. It is common to define macros using the
 * naming convention (seen below) for all GType implementations.
 */
#define VALIDATION_TYPE_OBJECT	(validation_object_get_type())

#define VALIDATION_TYPE_INSTANCE_GET_CLASS(Instance,\
		GTypeObject,StructType)\
		((StructType*) (void *)(((GTypeInstance*) Instance)->g_class))

/* macro to get from the class to object */
#define VALIDATION_OBJECT_GET_CLASS(obj) \
	(VALIDATION_TYPE_INSTANCE_GET_CLASS ((obj), \
	VALIDATION_TYPE_OBJECT, ValidationObjectClass))

/************************/
/* VALIDATION CALLBACKS */
/************************/

/*
 * callback handling, parameters needs to match marshalling signatures
 *
 * callbacks for the auto generated method protoypes
 * Forward declare the prototypes of the callback method handlers for the
 * dbus_glib_validation_object_methods[]
 */

gboolean validation_object_validation_test_mode(ValidationObject *object,
						guint8 enable,
						guint8 test_mode,
						guint8 role,
						GByteArray *addr,
						DBusGMethodInvocation *context);

gboolean validation_object_validation_link_quality(ValidationObject *object,
						guint8 test_mode,
						DBusGMethodInvocation *context);


/* async method handling functions */
static void register_validation_async_queue_list(ValidationObject *object);
#ifdef NOT_USED
static void unregister_validation_async_queue_list(ValidationObject *object);
#endif


/* include auto-generated binding information */
#include "alps-bt-validation-introspection.h"


/***********************************/
/* OBJECT AND CLASS INITIALIZATION */
/***********************************/


/*
 * ValidationObject object initialiser function
 *
 * Used to initialise each object instance via g_object_new()
 */
static void validation_object_init(ValidationObject *object)
{
	DEBUG_FUNC("Called");
	g_assert(object != NULL);

	/* add code to initialise created object contents */
}

/*
 * ValidationObject class initialiser function
 *
 * Used to initialise each class via the first call of g_object_new()
 */
static void validation_object_class_init(ValidationObjectClass *klass)
{
#ifdef NOT_USED
	GObjectClass *gobject_class = G_OBJECT_CLASS(klass);
#endif

	DEBUG_FUNC("Called");
	g_assert(klass != NULL);

	guint signalId;

	/* register new signals here */

	/* Signal: ValidationTestModeComplete */
	signalId = g_signal_new("validation_test_mode_complete", /* str name of the signal */
			/* GType to which signal is bound to */
			G_OBJECT_CLASS_TYPE(klass),
			/*
			 * Combination of GSignalFlags which tell the
			 * signal dispatch machinery how and when to
			 * dispatch this signal. The most common is the
			 * G_SIGNAL_RUN_LAST specification.
			 */
			G_SIGNAL_RUN_LAST,
			/*
			 * Offset into the class structure for the type
			 * function pointer. Since we're implementing a
			 * simple class/type, we'll leave this at zero.
			 */
			0,
			/* GSignalAccumulator to use. We don't need one. */
			NULL,
			/* User-data to pass to the accumulator. */
			NULL,
			/*
			 * Function to use to marshal the signal data into
			 * the parameters of the signal call. Luckily for
			 * us, GLib (GCClosure) already defines just the
			 * function that we want for a signal handler that
			 * we don't expect any return values (void) and
			 * one that will accept one string as parameter
			 * (besides the instance pointer and pointer to
			 * user-data).
			 *
			 * If no such function would exist, you would need
			 * to create a new one (by using glib-genmarshal
			 * tool).
			 */
			marshal_VOID__UCHAR_UCHAR,
			/*
			 * Return GType of the return value. The handler
			 * does not return anything, so we use G_TYPE_NONE
			 * to mark that.
			 */
			G_TYPE_NONE,
			/* Number of parameter GTypes to follow. */
			2,
			/* GType(s) of the parameters. */
			G_TYPE_UCHAR,
			G_TYPE_UCHAR);

	/*
	 * Store the signal Id into the class state, so that we can use
	 * it later.
	 */
	klass->ValidationTestModeComplete_sig_id = signalId;


	DEBUG_HIGH("Binding our GObject with D-bus...");
	/* Uses auto-generated methods and signal structure */
	dbus_g_object_type_install_info(VALIDATION_TYPE_OBJECT,
		&dbus_glib_validation_object_object_info);

}

/* LOCAL GLOBAL */
/* validation object pointer */
static ValidationObject *validation_object = NULL;


/****************************/
/* signal sending functions */
/****************************/

/*
 * exported function
 *
 * Emit the "ValidationTestModeComplete" signal.
 */
gboolean emit_validation_test_mode_complete(u8 noti_type, u8 status)
{
	ValidationObject *object = validation_object;	/* use local global */

	DEBUG_FUNC("Called");

	g_assert(object != NULL);

	/*
	 * Track the object to its class information
	 */
	ValidationObjectClass* klass = VALIDATION_OBJECT_GET_CLASS(object);

	/* This is the simplest way of emitting signals. */
	g_signal_emit( /*
			* Instance of the object that is generating this
			* signal. This will be passed as the first parameter
			* to the signal handler (eventually). But obviously
			* when speaking about D-Bus, a signal caught on the
			* other side of D-Bus will be first processed by
			* the GLib-wrappers (the object proxy) and only then
			* processed by the signal handler.
			*/
			object,
			/*
			 * Signal id for the signal to generate. These are
			 * stored inside the class state structure.
			 */
			klass->ValidationTestModeComplete_sig_id,
			/*
			 * Detail of signal. Since we are not using detailed
			 * signals, we leave this at zero (default).
			 */
			0,
			/*
			 * Data to marshal into the signal.
			 */
			noti_type,
			status);

	return TRUE;
}


/*
 * create a new object
 *
 * Inputs:
 * bus = point to the bus proxy
 * Exported function
 */
gboolean validation_object_creation(DBusGConnection *bus, gchar *node_name)
{
	DEBUG_HIGH("Creating an instance of the ValidationObject"
 				" and calling initialise functions...");

	validation_object = g_object_new(VALIDATION_TYPE_OBJECT, NULL);
	if(validation_object == NULL) {
		DEBUG_ERROR("Failed to create an instance of a ValidationObject");
		return FALSE;
	}

	DEBUG_HIGH("Registering the validation GObject onto the D-Bus");

	/* the object's path does not come from the XML file */
	dbus_g_connection_register_g_object(bus,
		node_name,	/* Object's path for the validation node */
		G_OBJECT(validation_object));

	/* register the async method queue list */
	register_validation_async_queue_list(validation_object);

	return TRUE;
}


/*****************************/
/* METHOD CALLBACK FUNCTIONS */
/*****************************/

/************************/
/* ASYNC METHOD QUEUING */
/************************/

/*
 * initialise the async method queue
 */
static void register_validation_async_queue_list(ValidationObject *object)
{
	DEBUG_FUNC("Called");

	g_assert(object != NULL);

	utils_async_queue_initialise(&object->aql_validation);

	DEBUG_FUNC("Exited");
}

/* FIXME: find somewhere to call this routine */
/*
 * destroy the async method queue by first cleaning up entries
 */
#ifdef NOT_USED
static void unregister_validation_async_queue_list(ValidationObject *object)
{
	DEBUG_FUNC("Called");

	g_assert(object != NULL);

	utils_async_queue_destroy(&object->aql_validation);

	DEBUG_FUNC("Exited");
}
#endif


/************************/
/* Validation interface */
/************************/
/*
 * Method call for "ValidationTestMode"
 *
 * asynchronous method: call phase
 */
gboolean validation_object_validation_test_mode(ValidationObject *object,
						guint8 enable,
						guint8 test_mode,
						guint8 role,
						GByteArray *addr,
						DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;
	BD_ADDRESS *device_address = NULL;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(addr != NULL);

	DEBUG("enable %d, test_mode %d, role %d, address length %d",
					enable, test_mode, role, addr->len);

	/*
	 * need to trap the situation that the input address length
	 * does not match the length of the BD_ADDRESS
	 */
	if (addr->len == sizeof(BD_ADDRESS)) {

		DEBUG("address %02X:%02X:%02X:%02X:%02X:%02X:",
			 addr->data[0], addr->data[1],
			 addr->data[2], addr->data[3],
			 addr->data[4], addr->data[5]);

		/* point to the data in the GByteArray */
		device_address = (BD_ADDRESS *)(void*) &addr->data[0];

		/*
		 * BTHRESULT BT_APPL_TestMode_REQ(IN u8 ucEnable,
		 *				IN u8 ucTestMode,
		 *				IN u8 ucRole,
		 *				IN const BD_ADDRESS aucBD_ADDR);
		 */

#ifndef USE_DUMMY_API
		result = BT_APPL_TestMode_REQ(enable, test_mode, role,
							*device_address);
#else
		result = BT_APPL_TestMode_REQ_DUMMY(enable, test_mode, role,
							*device_address);
#endif
		DEBUG("BT stack function returned");

	} else {
		/* spoof an ALPS error code */
 		result = BT_APPL_ERROR_PARAMETERS;
	}

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_TestMode_REQ",
					&error);

	if (return_code == TRUE) {
		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(
					&object->aql_validation, context,
					ID_validation_test_mode_CFM);
	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "ValidationTestMode"
 *
 * asynchronous method: return phase
 */
gboolean validation_object_validation_test_mode_RTN(u8 status)
{
	DBusGMethodInvocation *context = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(validation_object != NULL);

	context = utils_async_queue_find_list_entry(
					&validation_object->aql_validation,
					ID_validation_test_mode_CFM);

	if (context != NULL) {
		/* found the context info, so send reply */
		dbus_g_method_return(context, status);

		return_code = TRUE;
	} else {
		DEBUG_ERROR("No context so cannot reply on D-bus");
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call for "ValidationLinkQuality"
 *
 * asynchronous method: call phase
 */
gboolean validation_object_validation_link_quality(ValidationObject *object,
						guint8 test_mode,
						DBusGMethodInvocation *context)
{
	GError *error = NULL;
	BTHRESULT result;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);

	/*
	 * BTHRESULT BT_APPL_LinkQuality_REQ(IN u8 ucTestMode);
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_LinkQuality_REQ(test_mode);
#else
	result = BT_APPL_LinkQuality_REQ_DUMMY(test_mode);
#endif
	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
					"BT_APPL_LinkQuality_REQ",
					&error);

	if (return_code == TRUE) {
		/* put the method return in a queue pending the CFM */
		utils_async_queue_add_list_entry(
					&object->aql_validation, context,
					ID_validation_link_quality_CFM);
	} else {
		DEBUG("async call error");

		/* an error occurred so abort the method call */
		dbus_g_method_return_error(context, error);
		g_error_free(error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * exported function
 *
 * Method call "ValidationLinkQuality"
 *
 * asynchronous method: return phase
 */
gboolean validation_object_validation_link_quality_RTN(u8 status,
							s8 rssi,
							u8 link_quality)
{
	DBusGMethodInvocation *context = NULL;
	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	context = utils_async_queue_find_list_entry(
					&validation_object->aql_validation,
					ID_validation_link_quality_CFM);

	if (context != NULL) {
		/* found the context info, so send reply */
		dbus_g_method_return(context, status, (gint16) rssi,
								link_quality);

		return_code = TRUE;
	} else {
		DEBUG_ERROR("No context so cannot reply on D-bus");
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

#endif
