/*
 * handle_phonebook->c
 * phonebook handling for the D-bus Bluetooth Daemon
 *
 * Author: Dean Jenkins <djenkins@mvista.com>
 *
 * 2010 (c) MontaVista Software, LLC. This file is licensed under
 * the terms of the AFL.
 */

#include <glib.h>
#include <dbus/dbus-glib.h>

#include "bt_appl/common_def.h"

#include "main.h"
#include "utils_async_queue.h"
#include "device_info.h"
#include "node_device.h"

#include "handle_phonebook.h"
#include "debug.h"

#include "bt_appl/api/bt_appl_phone_book_block.h"
#include "bt_appl_dummy.h"

#include "dbus_error.h"

/*
 * exported global to hold the phonebook state strings
 *
 * Note these strings are not limited to debug messages
 */
gchar *phonebook_state_str[PBDL_TOTAL_STATES] =
{
	[PBDL_IDLE] = "PBDL_IDLE",
	[PBDL_WAIT_DATA] = "PBDL_WAIT_DATA",
	[PBDL_GOT_DATA] = "PBDL_GOT_DATA",
	[PBDL_SEND_DATA] = "PBDL_SEND_DATA",
	[PBDL_GOT_MAX_RECS] = "PBDL_GOT_MAX_RECS",
	[PBDL_COMPLETE_DATA] = "PBDL_COMPLETE_DATA"
};

/*
 * exported function
 *
 * use this function to switch to the PBDL_WAIT_DATA from any
 * state. Note data will be lost if the state was PBDL_GOT_DATA.
 */
void handle_phonebook_set_wait_data(DeviceObject *object, guint8 pb_type,
						guint32 max_total_recs,
						guint32 current_total_recs,
						guint8 max_recs_per_trans,
						gboolean use_agent)
{
	DEBUG_FUNC("Called");

	DEBUG("Phonebook download state is %s",
			phonebook_state_str[object->pbdl_state]);

	if ((object->pbdl_state == PBDL_GOT_DATA) ||
		(object->pbdl_state == PBDL_GOT_MAX_RECS)){
		/* need to throw away any previously stored data */

		DEBUG_ERROR("Was in \'%s\' state for the new download "
			"start, now deleting data from the previous session...",
			phonebook_state_str[object->pbdl_state]);
	}

	if ((object->pbdl_state == PBDL_GOT_DATA) ||
		(object->pbdl_state == PBDL_WAIT_DATA) ||
		(object->pbdl_state == PBDL_GOT_MAX_RECS)) {

		g_assert(object->phonebook != NULL);

		/* free the index */
		g_assert(object->phonebook->index != NULL);
		g_array_unref(object->phonebook->index);
		object->phonebook->index = NULL;

		/* free the data */
		g_assert(object->phonebook->data != NULL);
		g_byte_array_unref(object->phonebook->data);
		object->phonebook->data = NULL;

	} else {
		/* allocate memory to hold the phonebook info */
		g_assert(object->phonebook == NULL);
		object->phonebook = g_malloc(sizeof(PhonebookInfo));
		g_assert(object->phonebook != NULL);

		object->phonebook->index = NULL;
		object->phonebook->data = NULL;
	}

	/* select the wait for data state */
	object->pbdl_state = PBDL_WAIT_DATA;

	/* store start info in the phonebook variables */
	object->phonebook->max_total_recs = max_total_recs;
	object->phonebook->max_recs_per_trans = max_recs_per_trans;
	object->phonebook->use_agent = use_agent;
	object->phonebook->pb_type = pb_type;

	/* set the initial conditions for the start of the download */
	object->phonebook->total_recs_count = current_total_recs;
	object->phonebook->recs_trans_count = 0;
	object->phonebook->data_present = FALSE;

	g_assert(object->phonebook->index == NULL);
	object->phonebook->index = g_array_new(FALSE, FALSE, sizeof(guint32));
	g_assert(object->phonebook->index != NULL);

	g_assert(object->phonebook->data == NULL);
	object->phonebook->data = g_byte_array_new();
	g_assert(object->phonebook->data != NULL);

	object->phonebook->completed = FALSE;
	object->phonebook->data_type = 0;
	object->phonebook->status = 0;
	object->phonebook->pb_number = 0;

	DEBUG_FUNC("Exited");
}

static void handle_phonebook_send_data(DeviceObject *object)
{
	DEBUG_FUNC("Called");

	DEBUG("Phonebook download state is %s",
			phonebook_state_str[object->pbdl_state]);

	/* Check the download state is valid */
	g_assert((object->pbdl_state == PBDL_SEND_DATA) ||
			(object->pbdl_state == PBDL_COMPLETE_DATA));

	g_assert(object->phonebook != NULL);

	/* transmit the pending records */
	DEBUG("%d records to be sent", object->phonebook->recs_trans_count);

	if (object->phonebook->use_agent == TRUE) {
		DEBUG("using the agent method");

		send_pbdl_agent_data(object);

	} else {
		DEBUG("using the signal solution");

		emit_pb_dl_data_signal(object);

		DEBUG("finished with the phonebook info");

		switch (object->pbdl_state) {
		case PBDL_SEND_DATA:
			/* select the wait for data state */
			object->pbdl_state = PBDL_WAIT_DATA;

			/* reset the record counter */
			object->phonebook->recs_trans_count = 0;

			/* empty the index array, ready for next time */
			g_array_unref(object->phonebook->index);
			object->phonebook->index = g_array_new(FALSE, FALSE,
							sizeof(guint32));
			g_assert(object->phonebook->index != NULL);

			/* empty the data array, ready for next time */
			g_byte_array_unref(object->phonebook->data);
			object->phonebook->data = g_byte_array_new();
			g_assert(object->phonebook->data != NULL);

			break;
		case PBDL_COMPLETE_DATA:
			/* select the idle state */
			object->pbdl_state = PBDL_IDLE;

			/* free the index */
			g_array_unref(object->phonebook->index);

			/* free the data */
			g_byte_array_unref(object->phonebook->data);

			/* free the phonebook info structure */
			g_free(object->phonebook);
			object->phonebook = NULL;

			break;
		default:
			DEBUG_ERROR("Unhandled state %d \'%s\'",
				object->pbdl_state,
				phonebook_state_str[object->pbdl_state]);
		}
	}

	DEBUG_FUNC("Exited");
}

/*
 * exported function
 */
gboolean handle_phonebook_pb_dl_data(BD_ADDRESS device_address,
							guint8 data_type,
							GByteArray *data)
{
	DeviceObject *object = NULL;
	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	object = find_device_object_from_list(device_address);

	if (object == NULL)
	{
	   DEBUG_ERROR("Bluetooth device address: %s not found so "
	         "cannot process the phonebook data",
	         bd_addr_to_str(device_address));
	   g_byte_array_unref(data);
	   return FALSE;
	}

	DEBUG("Phonebook download state is %s",
			phonebook_state_str[object->pbdl_state]);

	if (object->pbdl_state == PBDL_GOT_MAX_RECS)
	{
	   DEBUG("Reached maximum number of records so ignore this data");
	   g_assert(object->phonebook != NULL);
	   DEBUG("Total records processed %u, maximum allowed %u",
	         object->phonebook->total_recs_count,
	         object->phonebook->max_total_recs);
	   g_byte_array_unref(data);
	   DEBUG_FUNC("Exited");
	   return FALSE;
	}

	if ((object->pbdl_state != PBDL_WAIT_DATA) &&
		(object->pbdl_state != PBDL_GOT_DATA))
	{
	   DEBUG_ERROR("Unexpected and unhandled "
	         "BT_APPL_PBDL_Data_IND received");
	   g_byte_array_unref(data);
	   DEBUG_FUNC("Exited");
	   return FALSE;
	}

	/* Check the download state is valid */
	g_assert((object->pbdl_state == PBDL_WAIT_DATA) ||
			(object->pbdl_state == PBDL_GOT_DATA));

	g_assert(object->phonebook != NULL);

	if (object->pbdl_state == PBDL_WAIT_DATA) {
		/* use the first data IND to set the data type */
		object->phonebook->data_type = data_type;
	} else if (object->phonebook->data_type != data_type) {
		DEBUG_ERROR("Processing mixed data types");
	}

	/* select the got data state */
	object->pbdl_state = PBDL_GOT_DATA;

	g_assert(object->phonebook->data != NULL);

	/* put the data into the buffer storage */
	object->phonebook->data = g_byte_array_append(object->phonebook->data,
							data->data, data->len);

	g_assert(object->phonebook->index != NULL);

	/* add the length of the data record into the index */
	object->phonebook->index = g_array_append_val(object->phonebook->index,
								data->len);

	object->phonebook->total_recs_count += 1;

	DEBUG("Total records processed %u, maximum allowed %u",
					object->phonebook->total_recs_count,
					object->phonebook->max_total_recs);

	/* check that the maximum number of records has not been reached */
	if ((object->phonebook->max_total_recs != 0) &&
		(object->phonebook->total_recs_count ==
					object->phonebook->max_total_recs)) {

		BTHRESULT result;

		DEBUG("Reached the total maximum records so issue STOP");
		
		/* Reached total maximum records so ignore new data */
		object->pbdl_state = PBDL_GOT_MAX_RECS;

		/* call the ALPS phonebook stop function to inhibit more data */

		/*
		 * BTHRESULT BT_APPL_PBDL_Stop_REQ(IN BD_ADDRESS aucBD_ADDR,
		 *						IN u8 ucAction);
		 */
#ifndef USE_DUMMY_API
		result = BT_APPL_PBDL_Stop_REQ(*object->device_address, 0);
#else
		result = BT_APPL_PBDL_Stop_REQ_DUMMY(*object->device_address,
									0);
#endif
		DEBUG("BT stack function returned");

		if (result != BT_APPL_SUCCESS) {
			DEBUG_ERROR("%s returned error %d %s",
				"BT_APPL_PBDL_Stop_REQ",
				result,
				lookup_bthresult_str(result));
		}

		/*
		 * Note any error from BT_APPL_PBDL_Stop_REQ is ignored
		 * because nothing can be done if an error is reported.
		 * Also the BT_APPL_PBDL_Stop_CFM is ignored.
		 * The download sequence is terminated by receiving the
		 * BT_APPL_PBDL_*_Complete_Ind.
		 */
	}		

	object->phonebook->recs_trans_count += 1;

	/* has maximum number of records per trans been reached ? */
	if ((object->phonebook->recs_trans_count >=
				object->phonebook->max_recs_per_trans)) {

		gboolean last_send = FALSE;

		if (object->pbdl_state == PBDL_GOT_MAX_RECS) {
			last_send = TRUE;
		}		

		/* Reached maximum records per transaction so send data */
		object->pbdl_state = PBDL_SEND_DATA;

		/* indicate data is present */
		object->phonebook->data_present = TRUE;

		/* set default info for status fields for sending over D-bus */
		object->phonebook->completed = FALSE;
		object->phonebook->status = 0;
		object->phonebook->pb_number = 0;

		/* send the data */
		handle_phonebook_send_data(object);

		if (last_send == TRUE) {
			/* Ensure no more data is handled */
			object->pbdl_state = PBDL_GOT_MAX_RECS;
		}		

	} else {
		DEBUG("%u records pending to be sent",
					object->phonebook->recs_trans_count);
	}


	DEBUG("Phonebook download state is %s",
			phonebook_state_str[object->pbdl_state]);


	/* finshed so need to free the data array */
	g_byte_array_unref(data);

	DEBUG_FUNC("Exited");

	return TRUE;
}

/*
 * exported function
 */
gboolean handle_phonebook_pb_dl_data_complete(BD_ADDRESS device_address,
							guint8 status,
							guint32 pb_number)
{
	DeviceObject *object = NULL;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	object = find_device_object_from_list(device_address);

	if (object == NULL) {
		DEBUG_ERROR("Bluetooth device address: %s not found so "
			"cannot process the phonebook data",
			bd_addr_to_str(device_address));

		return FALSE;
	}

	DEBUG("Phonebook download state is %s",
			phonebook_state_str[object->pbdl_state]);

	if ((object->pbdl_state != PBDL_WAIT_DATA) &&
		(object->pbdl_state != PBDL_GOT_DATA) &&
		(object->pbdl_state != PBDL_GOT_MAX_RECS)) {

		DEBUG_ERROR("Unexpected and unhandled "
			"BT_APPL_PBDL_Data_Complete_IND received");

		return FALSE;
	}

	/* Check the download state is valid */
	g_assert((object->pbdl_state == PBDL_WAIT_DATA) ||
			(object->pbdl_state == PBDL_GOT_DATA) ||
			(object->pbdl_state == PBDL_GOT_MAX_RECS));

	/* select the got data state */
	object->pbdl_state = PBDL_COMPLETE_DATA;

	g_assert(object->phonebook != NULL);

	if (object->phonebook->recs_trans_count > 0) {
		/* indicate data is present */
		object->phonebook->data_present = TRUE;
	} else {
		/* indicate data is not present */
		object->phonebook->data_present = FALSE;
	}

	/* set the status fields for sending over D-bus */
	object->phonebook->completed = TRUE;
	object->phonebook->status = status;
	object->phonebook->pb_number = pb_number;

	/* send status info and any data */
	handle_phonebook_send_data(object);

	DEBUG_FUNC("Exited");

	return TRUE;
}

/*
 * exported function
 */
gboolean handle_phonebook_pb_dl_calendar_complete(BD_ADDRESS device_address,
								guint8 status)
{
	DeviceObject *object = NULL;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	object = find_device_object_from_list(device_address);

	if (object == NULL) {
		DEBUG_ERROR("Bluetooth device address: %s not found so "
			"cannot process the phonebook data",
			bd_addr_to_str(device_address));

		return FALSE;
	}

	DEBUG("Phonebook download state is %s",
			phonebook_state_str[object->pbdl_state]);

	if ((object->pbdl_state != PBDL_WAIT_DATA) &&
		(object->pbdl_state != PBDL_GOT_DATA) &&
		(object->pbdl_state != PBDL_GOT_MAX_RECS)) {

		DEBUG_ERROR("Unexpected and unhandled "
			"BT_APPL_PBDL_Calendar_Complete_IND received");

		return FALSE;
	}

	/* Check the download state is valid */
	g_assert((object->pbdl_state == PBDL_WAIT_DATA) ||
			(object->pbdl_state == PBDL_GOT_DATA) ||
			(object->pbdl_state == PBDL_GOT_MAX_RECS));

	/* select the got data state */
	object->pbdl_state = PBDL_COMPLETE_DATA;

	g_assert(object->phonebook != NULL);

	if (object->phonebook->recs_trans_count > 0) {
		/* indicate data is present */
		object->phonebook->data_present = TRUE;
	} else {
		/* indicate data is not present */
		object->phonebook->data_present = FALSE;
	}

	/* set the status fields for sending over D-bus */
	object->phonebook->completed = TRUE;
	object->phonebook->status = status;
	object->phonebook->pb_number = 0;

	/* send status info and any data */
	handle_phonebook_send_data(object);

	DEBUG_FUNC("Exited");

	return TRUE;
}


/**************************
 * Phonebook agent method *
 **************************/

/* include auto-generated binding information */
#include "alps-bt-pbdl-agent-introspection.h"

/* local global */
static DBusGProxy *pbdl_agent_proxy = NULL;

static void send_pbdl_data_agent_method_reply(DBusGProxy *proxy, GError *error,
							gpointer userdata)
{
	DBusGProxyCall *proxy_call;
	DeviceObject *object = (DeviceObject *) userdata;
	GSList *list_entry;
	PhonebookInfo *old_phonebook;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(object->phonebook_agent_reply_list != NULL);

	if (error != NULL) {
		DEBUG_ERROR("Data agent method reported an error: %s",
			error->message);
		g_error_free(error);
	}

	/* get the first phonebook info struct on the pending list */
	list_entry = object->phonebook_agent_reply_list;
	g_assert(list_entry != NULL);

	/* now free the assoicated data and phoneinfo struct */
	old_phonebook = list_entry->data;
	g_assert(old_phonebook != NULL);

	/* free the index */
	g_array_unref(old_phonebook->index);

	/* free the data */
	g_byte_array_unref(old_phonebook->data);

	/* free the phonebook info structure */
	g_free(old_phonebook);

	/* get the next (if any) pending agent method */
	object->phonebook_agent_reply_list = g_slist_delete_link(
					object->phonebook_agent_reply_list,
					list_entry);

	/* get the next (if any) phonebook info struct on the pending list */
	list_entry = object->phonebook_agent_reply_list;

	if (list_entry != NULL) {
		/* pending entry so get the phoneinfo */
		old_phonebook = list_entry->data;
		g_assert(old_phonebook != NULL);

		DEBUG("sending 1 pending data agent method call out of %d"
			" pending...", g_slist_length(list_entry));

		/*
		 * Warning: the cast (GArray *) from (GByteArray *)
		 * assumes that the GLib structures remain compatible.
		 * Cast needed to match auto-generated marshalling code.
		 */
		/* Non-blocking call */
		proxy_call = com_alps_bt_Pbdl_pb_dl_data_agent_method_async(
					proxy,
					old_phonebook->data_type,
					old_phonebook->data_present,
					old_phonebook->index,
					(GArray *) old_phonebook->data,
					old_phonebook->completed,
					old_phonebook->status,
					old_phonebook->pb_number,
					send_pbdl_data_agent_method_reply,
					object);
	}

	DEBUG_FUNC("Exited");
}

gboolean send_pbdl_agent_data(DeviceObject *object)
{
	DBusGProxyCall *proxy_call;
	PhonebookInfo *old_phonebook;
	GSList *list_entry;

	DEBUG_FUNC("Called");

	g_assert(pbdl_agent_proxy != NULL);
	g_assert(object != NULL);
	g_assert(object->phonebook != NULL);

	/* keep the old phonebook info and prepare a new one */
	old_phonebook = object->phonebook;
	object->phonebook = NULL;

	switch (object->pbdl_state) {
	case PBDL_SEND_DATA:
		/* create a phonebook info struct for the next data */
		handle_phonebook_set_wait_data(object,
					old_phonebook->data_type,
					old_phonebook->max_total_recs,
					old_phonebook->total_recs_count,
					old_phonebook->max_recs_per_trans,
					old_phonebook->use_agent);

		break;
	case PBDL_COMPLETE_DATA:
		/* select the idle state */
		object->pbdl_state = PBDL_IDLE;

		break;
	default:
		DEBUG_ERROR("Unhandled state %d \'%s\'",
			object->pbdl_state,
			phonebook_state_str[object->pbdl_state]);
	}

	/*
	 * need to save the phonebook onto a list so that the method reply
	 * can free it
	 */
	list_entry = object->phonebook_agent_reply_list;

	/* append the old phonebook info to the reply queue list */
	object->phonebook_agent_reply_list = g_slist_append(
					object->phonebook_agent_reply_list,
					old_phonebook);

	/* check for pending entries, otherwise the reply sends the next one */
	if (list_entry == NULL) {

		DEBUG("going to send via agent method...");

		/*
		 * Warning: the cast (GArray *) from (GByteArray *)
		 * assumes that the GLib structures remain compatible.
		 * Cast needed to match auto-generated marshalling code.
		 */
		/* Non-blocking call */
		proxy_call = com_alps_bt_Pbdl_pb_dl_data_agent_method_async(
					pbdl_agent_proxy,
					old_phonebook->data_type,
					old_phonebook->data_present,
					old_phonebook->index,
					(GArray *) old_phonebook->data,
					old_phonebook->completed,
					old_phonebook->status,
					old_phonebook->pb_number,
					send_pbdl_data_agent_method_reply,
					object);

	} else {
		DEBUG("Unable to call the data agent method due to %d"
			" pending method calls", g_slist_length(list_entry));
	}

	DEBUG_FUNC("Exited");

	return TRUE;
}

/*
 * exported function
 */
gboolean create_pbdl_agent_dbus_proxy(DBusGConnection *bus)
{
	DEBUG_FUNC("Called");

	pbdl_agent_proxy = dbus_g_proxy_new_for_name(bus,
						PBDL_AGENT_SERVICE_NAME,
						PBDL_AGENT_SERVICE_OBJECT_PATH,
						PBDL_AGENT_SERVICE_INTERFACE);

	if (pbdl_agent_proxy == NULL) {
		DEBUG_ERROR("Failed to create the pdbl agent dbus proxy");
		return FALSE;
	}

	DEBUG_FUNC("Exited");

	return TRUE;
}
