/*
 * dummy bt_appl handling for the D-bus Bluetooth Daemon
 *
 * Author: Dean Jenkins <djenkins@mvista.com>
 *
 * 2010 (c) MontaVista Software, LLC. This file is licensed under
 * the terms of the AFL.
 */

#include <glib.h>
#include <glib-object.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <string.h>

#include <locale.h>
#include <langinfo.h>

#include "main.h"

#include "bt_appl/common_def.h"
#include "bt_appl/api/bt_appl_bt_setting_block.h"
#include "bt_appl/api/bt_appl_config_mgr_block.h"
#include "bt_appl/api/bt_appl_phone_call_block.h"
#include "bt_appl/api/bt_appl_audio_block.h"
#include "bt_appl/api/bt_appl_phone_book_block.h"
#include "bt_appl/api/bt_appl_messaging_block.h"
#include "bt_appl/api/bt_appl_virtual_device_block.h"
#include "bt_appl_dummy.h"
#include "callbacks.h"
#include "utils_async_queue.h"
#include "utils_test_queue.h"
#include "node_test.h"

#include "dbus_error.h"

#include "debug.h"

#ifdef USE_DUMMY_API

/* local global */
static struct test_queue_list *test_node_queue = NULL;

void dummy_set_test_node_queue(struct test_queue_list *list_info)
{
	test_node_queue = list_info;
}

struct dummy_info {
	gboolean use_dummy;
	gchar *func_name;
	BTHRESULT return_code;
	guint8 state_number;
};

/* local global to hold the dummy function settings */
static struct dummy_info dummy_data[FUNC_total] =
{
	[FUNC_Initialization_REQ] = {
		.use_dummy = TRUE,
		.func_name = "Initialization_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_Shutdown_REQ] = {
		.use_dummy = TRUE,
		.func_name = "Shutdown_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_Set_Configuration_REQ] = {
		.use_dummy = TRUE,
		.func_name = "Set_Configuration_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_Read_Configuration_REQ] = {
		.use_dummy = TRUE,
		.func_name = "Read_Configuration_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_Start_Connection_REQ] = {
		.use_dummy = TRUE,
		.func_name = "Start_Connection_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_Start_Wait_Connection_REQ] = {
		.use_dummy = TRUE,
		.func_name = "Start_Wait_Connection_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_Stop_Wait_Connection_REQ] = {
		.use_dummy = TRUE,
		.func_name = "Stop_Wait_Connection_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_Set_Local_Name_REQ] = {
		.use_dummy = TRUE,
		.func_name = "Set_Local_Name_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_Set_Mode_REQ] = {
		.use_dummy = TRUE,
		.func_name = "Set_Mode_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_Set_SSP_Mode_REQ] = {
		.use_dummy = TRUE,
		.func_name = "Set_SSP_Mode_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_Start_Pairing_REQ] = {
		.use_dummy = TRUE,
		.func_name = "Start_Pairing_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_Stop_Pairing_REQ] = {
		.use_dummy = TRUE,
		.func_name = "Stop_Pairing_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_Set_Serial_Device_Param_REQ] = {
		.use_dummy = TRUE,
		.func_name = "Set_Serial_Device_Param_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_NAP_Create_Device_REQ] = {
		.use_dummy = TRUE,
		.func_name = "NAP_Create_Device_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_NAP_Destroy_Device_REQ] = {
		.use_dummy = TRUE,
		.func_name = "NAP_Destroy_Device_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_Service_Search_REQ] = {
		.use_dummy = TRUE,
		.func_name = "Service_Search_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_SSP_Numeric_Confirm_RES] = {
		.use_dummy = TRUE,
		.func_name = "SSP_Numeric_Confirm_RES",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_Connection_Request_RES] = {
		.use_dummy = TRUE,
		.func_name = "Connection_Request_RES",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_Get_Device_Name_REQ] = {
		.use_dummy = TRUE,
		.func_name = "Get_Device_Name_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_Disconnection_REQ] = {
		.use_dummy = TRUE,
		.func_name = "Disconnection_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_Get_Local_Role_REQ] = {
		.use_dummy = TRUE,
		.func_name = "Get_Local_Role_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_Set_Local_Role_REQ] = {
		.use_dummy = TRUE,
		.func_name = "Set_Local_Role_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_Get_Call_Status_REQ] = {
		.use_dummy = TRUE,
		.func_name = "Get_Call_Status_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_Get_Phone_Status_REQ] = {
		.use_dummy = TRUE,
		.func_name = "Get_Phone_Status_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_HF_Dial_REQ] = {
		.use_dummy = TRUE,
		.func_name = "HF_Dial_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_HF_Redial_REQ] = {
		.use_dummy = TRUE,
		.func_name = "HF_Redial_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_HF_Mem_Dial_REQ] = {
		.use_dummy = TRUE,
		.func_name = "HF_Mem_Dial_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_HF_Terminate_Call_REQ] = {
		.use_dummy = TRUE,
		.func_name = "HF_Terminate_Call_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_HF_Terminate_All_REQ] = {
		.use_dummy = TRUE,
		.func_name = "HF_Terminate_All_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_HF_Audio_Transfer_REQ] = {
		.use_dummy = TRUE,
		.func_name = "HF_Audio_Transfer_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_HF_Send_DTMF_REQ] = {
		.use_dummy = TRUE,
		.func_name = "HF_Send_DTMF_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_HF_Incoming_Call_Act_REQ] = {
		.use_dummy = TRUE,
		.func_name = "HF_Incoming_Call_Act_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_HF_Release_Inactive_Call_REQ] = {
		.use_dummy = TRUE,
		.func_name = "HF_Release_Inactive_Call_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_HF_Release_Active_Call_REQ] = {
		.use_dummy = TRUE,
		.func_name = "HF_Release_Active_Call_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_HF_Swap_Call_REQ] = {
		.use_dummy = TRUE,
		.func_name = "HF_Release_Active_Call_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_HF_Add_Call_REQ] = {
		.use_dummy = TRUE,
		.func_name = "HF_Add_Call_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_HF_Release_Spec_Call_REQ] = {
		.use_dummy = TRUE,
		.func_name = "HF_Release_Spec_Call_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_HF_Hold_Excep_Call_REQ] = {
		.use_dummy = TRUE,
		.func_name = "HF_Hold_Excep_Call_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_HF_Exp_Call_Trans_REQ] = {
		.use_dummy = TRUE,
		.func_name = "HF_Exp_Call_Trans_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_HF_Subscriber_REQ] = {
		.use_dummy = TRUE,
		.func_name = "HF_Subscriber_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_HF_Manufacturer_ID_REQ] = {
		.use_dummy = TRUE,
		.func_name = "HF_Manufacturer_ID_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_HF_Model_ID_REQ] = {
		.use_dummy = TRUE,
		.func_name = "HF_Model_ID_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_HF_Revision_ID_REQ] = {
		.use_dummy = TRUE,
		.func_name = "HF_Revision_ID_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_HF_Voice_Recognition_Activation_REQ] = {
		.use_dummy = TRUE,
		.func_name = "HF_Voice_Recognition_Activation_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_AVP_Ctrl_Cmd_REQ] = {
		.use_dummy = TRUE,
		.func_name = "AVP_Ctrl_Cmd_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_AVP_Get_Capabilities_REQ] = {
		.use_dummy = TRUE,
		.func_name = "AVP_Get_Capabilities_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_AVP_List_Attr_REQ] = {
		.use_dummy = TRUE,
		.func_name = "AVP_List_Attr_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_AVP_List_Val_REQ] = {
		.use_dummy = TRUE,
		.func_name = "AVP_List_Val_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_AVP_Set_Val_REQ] = {
		.use_dummy = TRUE,
		.func_name = "AVP_Set_Val_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_AVP_Config_Notify_REQ] = {
		.use_dummy = TRUE,
		.func_name = "AVP_Config_Notify_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_AVP_Set_Player_REQ] = {
		.use_dummy = TRUE,
		.func_name = "AVP_Set_Player_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_AVP_Get_Metadata_Attr_REQ] = {
		.use_dummy = TRUE,
		.func_name = "AVP_Get_Metadata_Attr_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_AVP_Get_Folder_Item_REQ] = {
		.use_dummy = TRUE,
		.func_name = "AVP_Get_Folder_Item_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_AVP_Get_Media_Player_Item_REQ] = {
		.use_dummy = TRUE,
		.func_name = "AVP_Get_Media_Player_Item_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_AVP_Play_Item_REQ] = {
		.use_dummy = TRUE,
		.func_name = "AVP_Play_Item_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_AVP_Change_Path_REQ] = {
		.use_dummy = TRUE,
		.func_name = "AVP_Change_Path_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_PBDL_Get_Capabilities_REQ] = {
		.use_dummy = TRUE,
		.func_name = "PBDL_Get_Capabilities_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_PBDL_Prepare_REQ] = {
		.use_dummy = TRUE,
		.func_name = "PBDL_Prepare_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_PBDL_Start_REQ] = {
		.use_dummy = TRUE,
		.func_name = "PBDL_Start_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_PBDL_Stop_REQ] = {
		.use_dummy = TRUE,
		.func_name = "PBDL_Stop_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_PBDL_Complete_REQ] = {
		.use_dummy = TRUE,
		.func_name = "PBDL_Complete_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_MAP_Get_Capabilities_REQ] = {
		.use_dummy = TRUE,
		.func_name = "MAP_Get_Capabilities_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_MAP_Start_REQ] = {
		.use_dummy = TRUE,
		.func_name = "MAP_Start_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_MAP_Stop_REQ] = {
		.use_dummy = TRUE,
		.func_name = "MAP_Stop_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_MAP_Chg_Instance_REQ] = {
		.use_dummy = TRUE,
		.func_name = "MAP_Chg_Instance_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_MAP_Update_Inbox_REQ] = {
		.use_dummy = TRUE,
		.func_name = "MAP_Update_Inbox_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_MAP_List_Folder_REQ] = {
		.use_dummy = TRUE,
		.func_name = "MAP_List_Folder_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_MAP_List_Msg_REQ] = {
		.use_dummy = TRUE,
		.func_name = "MAP_List_Msg_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_MAP_Read_Msg_REQ] = {
		.use_dummy = TRUE,
		.func_name = "MAP_Read_Msg_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_MAP_Set_Msg_Status_REQ] = {
		.use_dummy = TRUE,
		.func_name = "MAP_Set_Msg_Status_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_MAP_Abort_REQ] = {
		.use_dummy = TRUE,
		.func_name = "MAP_Abort_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_MAP_Upload_REQ] = {
		.use_dummy = TRUE,
		.func_name = "FUNC_MAP_Upload_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_MAP_Upload_RES] = {
		.use_dummy = TRUE,
		.func_name = "FUNC_MAP_Upload_RES",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_TestMode_REQ] = {
		.use_dummy = TRUE,
		.func_name = "TestMode_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_LinkQuality_REQ] = {
		.use_dummy = TRUE,
		.func_name = "LinkQuality_REQ",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
	[FUNC_register_alps_callbacks] = {
		.use_dummy = TRUE,
		.func_name = "register_alps_callbacks",
		.return_code = BT_APPL_SUCCESS,
		.state_number = 0 },
};

gchar *dummy_get_func_name(enum function_number func_id)
{
	g_assert(func_id < FUNC_total);

	return &dummy_data[func_id].func_name[0];
}

gboolean dummy_get_dummy_state(enum function_number func_id)
{
	g_assert(func_id < FUNC_total);

	return dummy_data[func_id].use_dummy;
}

guint32 dummy_get_return_code(enum function_number func_id)
{
	g_assert(func_id < FUNC_total);

	return dummy_data[func_id].return_code;
}

void dummy_init_scenarios(void)
{
	guint loop;

	/* use the locale defined in the shell environment */
	setlocale(LC_CTYPE, "");

	DEBUG("Current CODESET: %s", nl_langinfo(CODESET));

	/* the locale must have UTF-8 enabled to print the UTF-8 strings */
	if (g_strcmp0(nl_langinfo(CODESET), "UTF-8") != 0) {
		DEBUG_ERROR("Current locale not supporting UTF-8. Please use a UTF-8 locale");
	}

	for (loop = 0; loop < FUNC_total; loop++) {
		DEBUG("FUNC %s: %s RETURN %d: %s", dummy_data[loop].func_name,
			dummy_data[loop].use_dummy ? "DUMMY" : "REAL",
			dummy_data[loop].return_code, lookup_bthresult_str(dummy_data[loop].return_code));
	}
}

void dummy_use_all_dummy_api_functions(gboolean use_dummy_api)
{
	guint loop;

	for (loop = 0; loop < FUNC_total; loop++) {
		/* use dummy or real API functions */
		dummy_data[loop].use_dummy = use_dummy_api;
		dummy_data[loop].return_code = BT_APPL_SUCCESS;
		dummy_data[loop].state_number = 0;
	}

	dummy_init_scenarios();
}

/* change the settings of the dummy function handler */
void dummy_change_dummy_api_settings(enum function_number func_id, gboolean use_dummy_api, BTHRESULT return_code)
{
	g_assert(func_id < FUNC_total);

	dummy_data[func_id].use_dummy = use_dummy_api;
	dummy_data[func_id].return_code = return_code;
}

/* dummy function for registering the ALPS callbacks */
BTHRESULT register_alps_callbacks_DUMMY(void)
{
	BTHRESULT result;

	if (dummy_data[FUNC_register_alps_callbacks].use_dummy == FALSE) {

		DEBUG("Calling REAL FUNCTION");

		if (dummy_data[FUNC_Initialization_REQ].use_dummy == TRUE) {
			DEBUG("WARNING: using real callbacks with dummy "
							"Initialization_REQ");
		}

		/* call the real function rather than the dummy */
		result = register_alps_callbacks();

	} else {

		DEBUG("Using DUMMY FUNCTION");

		if (dummy_data[FUNC_Initialization_REQ].use_dummy == FALSE) {
			DEBUG("WARNING: using dummy callbacks with real "
							"Initialization_REQ");
		}

		result = dummy_data[FUNC_register_alps_callbacks].return_code;
	}

	return result;
}

/******************************************/
/* API functions used by the Manager node */
/******************************************/

BTHRESULT BT_APPL_Initialization_REQ_DUMMY(OUT u16 *pusVersion,
                                     OUT BD_ADDRESS aucBD_ADDR)
{
	BTHRESULT result;

	if (dummy_data[FUNC_Initialization_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_Initialization_REQ(pusVersion, aucBD_ADDR);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		/*************/
		/* test data */
		/*************/
		*pusVersion = 0x0102;

		aucBD_ADDR[0] = 0x0a;
		aucBD_ADDR[1] = 0x1b;
		aucBD_ADDR[2] = 0x2c;
		aucBD_ADDR[3] = 0x3d;
		aucBD_ADDR[4] = 0x4e;
		aucBD_ADDR[5] = 0x5f;

		result = dummy_data[FUNC_Initialization_REQ].return_code;
	}

	return result;
}


BTHRESULT BT_APPL_Shutdown_REQ_DUMMY(void)
{
	BTHRESULT result;

	if (dummy_data[FUNC_Shutdown_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_Shutdown_REQ();

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		result = dummy_data[FUNC_Shutdown_REQ].return_code;
	}

	return result;
}


BTHRESULT BT_APPL_Set_Configuration_REQ_DUMMY(IN u16 usConfigId,IN u16 usConfigDataLen,IN u8 *pucConfigData)
{
	BTHRESULT result;

	if (dummy_data[FUNC_Set_Configuration_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_Set_Configuration_REQ(usConfigId, usConfigDataLen, pucConfigData);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY set request is ID 0x%X, len %u",
			usConfigId, usConfigDataLen);

		DEBUG_DUMP_ARRAY((guint32) usConfigDataLen, pucConfigData);

		result = dummy_data[FUNC_Set_Configuration_REQ].return_code;
	}

	return result;
}


BTHRESULT BT_APPL_Read_Configuration_REQ_DUMMY(IN u16 usConfigId,OUT u16 *pusConfigDataLen,OUT u8 *pucConfigData)
{
	BTHRESULT result;

	if (dummy_data[FUNC_Read_Configuration_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_Read_Configuration_REQ(usConfigId, pusConfigDataLen, pucConfigData);

	} else {
		guint8 test_data[30] = { 0x12, 0x34, 0x56, 0x78, 0x90,
						0xde, 0xad, 0xbe, 0xef };

		DEBUG("Using DUMMY API FUNCTION");

		/*************/
		/* test data */
		/*************/
		switch (usConfigId) {
		case 0x1111:
			*pusConfigDataLen = 1;
			memcpy(pucConfigData, &test_data[0], *pusConfigDataLen);
			break;
		case 0x2222:
			*pusConfigDataLen = 2;
			memcpy(pucConfigData, &test_data[0], *pusConfigDataLen);
			break;
		case 0x4444:
			*pusConfigDataLen = 4;
			memcpy(pucConfigData, &test_data[0], *pusConfigDataLen);
			break;
		case 0x9999:
			*pusConfigDataLen = 16;
			memcpy(pucConfigData, &test_data[0], *pusConfigDataLen);
			break;
		case 0xaaaa:
			/* will cause an abort due to length check */
			*pusConfigDataLen = 20;
			/* do not do the memcpy because it causes a */
			/* buffer overrun and corrupts the stack */
/*			memcpy(pucConfigData, &test_data[0], *pusConfigDataLen);	*/
			break;
		default:
			*pusConfigDataLen = 1;
			*pucConfigData = 0x5a;
			break;
		}

		DEBUG("DUMMY read request is ID 0x%X, len %u",
			usConfigId, *pusConfigDataLen);

		DEBUG_DUMP_ARRAY((guint32) *pusConfigDataLen, pucConfigData);

		result = dummy_data[FUNC_Read_Configuration_REQ].return_code;
	}

	return result;
}

/******************************************/
/* API functions used by the Adapter node */
/******************************************/

BTHRESULT BT_APPL_Start_Connection_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR,IN u8 ucService)
{
	BTHRESULT result;

	if (dummy_data[FUNC_Start_Connection_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_Start_Connection_REQ(aucBD_ADDR, ucService);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		DEBUG("DUMMY service %d", ucService);

		result = dummy_data[FUNC_Start_Connection_REQ].return_code;
	}

	return result;
}


BTHRESULT BT_APPL_Start_Wait_Connection_REQ_DUMMY(void)
{
	BTHRESULT result;

	if (dummy_data[FUNC_Start_Wait_Connection_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_Start_Wait_Connection_REQ();

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		result = dummy_data[FUNC_Start_Wait_Connection_REQ].return_code;
	}

	return result;
}


BTHRESULT BT_APPL_Stop_Wait_Connection_REQ_DUMMY(void)
{
	BTHRESULT result;

	if (dummy_data[FUNC_Stop_Wait_Connection_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_Stop_Wait_Connection_REQ();

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		result = dummy_data[FUNC_Stop_Wait_Connection_REQ].return_code;
	}

	return result;
}


BTHRESULT BT_APPL_Set_Local_Name_REQ_DUMMY(IN u8 ucNameLen,IN const u8 *pucName)
{
	BTHRESULT result;

	if (dummy_data[FUNC_Set_Local_Name_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_Set_Local_Name_REQ(ucNameLen, pucName);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("name len is %d", ucNameLen);

		/* prints UTF-8 string */
		DEBUG("name string is \"%s\"\n", pucName);

		result = dummy_data[FUNC_Set_Local_Name_REQ].return_code;
	}

	return result;
}


BTHRESULT BT_APPL_Set_Mode_REQ_DUMMY(IN u8 ucDiscoverable,IN u8 ucConnectable)
{
	BTHRESULT result;

	if (dummy_data[FUNC_Set_Mode_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_Set_Mode_REQ(ucDiscoverable, ucConnectable);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("Discoverable %d, Connectable %d", ucDiscoverable, ucConnectable);

		result = dummy_data[FUNC_Set_Mode_REQ].return_code;
	}

	return result;
}


BTHRESULT BT_APPL_Set_SSP_Mode_REQ_DUMMY(IN u8 ucSSPMode)
{
	BTHRESULT result;

	if (dummy_data[FUNC_Set_SSP_Mode_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_Set_SSP_Mode_REQ(ucSSPMode);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("ucSSPMode %u", ucSSPMode);

		result = dummy_data[FUNC_Set_SSP_Mode_REQ].return_code;
	}

	return result;
}


BTHRESULT BT_APPL_Start_Pairing_REQ_DUMMY(void)
{
	BTHRESULT result;

	if (dummy_data[FUNC_Start_Pairing_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_Start_Pairing_REQ();

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		result = dummy_data[FUNC_Start_Pairing_REQ].return_code;
	}

	return result;
}


BTHRESULT BT_APPL_Stop_Pairing_REQ_DUMMY(void)
{
	BTHRESULT result;

	if (dummy_data[FUNC_Stop_Pairing_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_Stop_Pairing_REQ();

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		result = dummy_data[FUNC_Stop_Pairing_REQ].return_code;
	}

	return result;
}


BTHRESULT BT_APPL_Set_Serial_Device_Param_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR,
							IN u8 ucService,
							IN u8 ucServiceHandle)
{
	BTHRESULT result;

	if (dummy_data[FUNC_Set_Serial_Device_Param_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_Set_Serial_Device_Param_REQ(aucBD_ADDR,
							ucService,
							ucServiceHandle);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		DEBUG("DUMMY service %u", ucService);
		DEBUG("DUMMY service_handle %u", ucServiceHandle);

		result = dummy_data[FUNC_Set_Serial_Device_Param_REQ].return_code;
	}

	return result;
}


BTHRESULT BT_APPL_Create_Device_REQ_DUMMY(IN BD_ADDRESS aucBD_ADDR, IN u8 ucService)
{
	BTHRESULT result;

	if (dummy_data[FUNC_NAP_Create_Device_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_Create_Device_REQ(aucBD_ADDR, ucService);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		DEBUG("DUMMY service %u", ucService);

		result = dummy_data[FUNC_NAP_Create_Device_REQ].return_code;
	}

	return result;
}


BTHRESULT BT_APPL_Destroy_Device_REQ_DUMMY(IN BD_ADDRESS aucBD_ADDR, IN u8 ucService)
{
	BTHRESULT result;

	if (dummy_data[FUNC_NAP_Destroy_Device_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_Destroy_Device_REQ(aucBD_ADDR, ucService);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		DEBUG("DUMMY service %u", ucService);

		result = dummy_data[FUNC_NAP_Destroy_Device_REQ].return_code;
	}

	return result;
}

BTHRESULT BT_APPL_Service_Search_REQ_DUMMY(IN BD_ADDRESS aucBD_ADDR,
							IN u8 ucServiceType)
{
	BTHRESULT result;

	if (dummy_data[FUNC_Service_Search_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_Service_Search_REQ(aucBD_ADDR, ucServiceType);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		DEBUG("DUMMY service_type%u", ucServiceType);

		result = dummy_data[FUNC_Service_Search_REQ].return_code;
	}

	return result;
}

BTHRESULT BT_APPL_SSP_Numeric_Confirm_RES_DUMMY(IN BD_ADDRESS aucBD_ADDR,
							IN u8 ucAccept)
{
	BTHRESULT result;

	if (dummy_data[FUNC_SSP_Numeric_Confirm_RES].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_SSP_Numeric_Confirm_RES_DUMMY(aucBD_ADDR,
								ucAccept);
	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		DEBUG("DUMMY accept %u", ucAccept);

		result = dummy_data[FUNC_SSP_Numeric_Confirm_RES].return_code;
	}

	return result;
}

BTHRESULT BT_APPL_Connection_Request_RES_DUMMY(IN const BD_ADDRESS aucBD_ADDR,
						IN u8 ucService,
						IN u8 ucAccept)
{
	BTHRESULT result;

	if (dummy_data[FUNC_Connection_Request_RES].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_Connection_Request_RES_DUMMY(aucBD_ADDR,
								ucService,
								ucAccept);
	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		DEBUG("DUMMY service %u, accept %u", ucService, ucAccept);

		result = dummy_data[FUNC_Connection_Request_RES].return_code;
	}

	return result;
}


/*****************************************/
/* API functions used by the Device node */
/*****************************************/

BTHRESULT BT_APPL_Get_Device_Name_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR)
{
	BTHRESULT result;

	if (dummy_data[FUNC_Get_Device_Name_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_Get_Device_Name_REQ(aucBD_ADDR);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		if (dummy_data[FUNC_Get_Device_Name_REQ].return_code != BT_APPL_SUCCESS) {
			if (dummy_data[FUNC_register_alps_callbacks].use_dummy == TRUE) {
				DEBUG("spoofing Callback event handling");
				result = BT_APPL_Get_Device_Name_CFM_Callback_DUMMY(
								aucBD_ADDR,
							      	4,
								(u8 *) "fred");
			} else {
				DEBUG("using REAL Callback event handling");
				result = BT_APPL_SUCCESS;
			}
		} else {
			result = dummy_data[FUNC_Get_Device_Name_REQ].return_code;
		}
	}

	return result;
}


BTHRESULT BT_APPL_Disconnection_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR, IN u8 ucService)
{
	BTHRESULT result;

	if (dummy_data[FUNC_Disconnection_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_Disconnection_REQ(aucBD_ADDR, ucService);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		DEBUG("DUMMY service %d", ucService);

		result = dummy_data[FUNC_Disconnection_REQ].return_code;
	}

	return result;
}


BTHRESULT BT_APPL_Get_Local_Role_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR, OUT u8 *pucRole)
{
	BTHRESULT result;

	if (dummy_data[FUNC_Get_Local_Role_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_Get_Local_Role_REQ(aucBD_ADDR, pucRole);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		result = dummy_data[FUNC_Get_Local_Role_REQ].return_code;
	}

	return result;
}


BTHRESULT BT_APPL_Set_Local_Role_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR, IN u8 ucRole)
{
	BTHRESULT result;

	if (dummy_data[FUNC_Set_Local_Role_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_Set_Local_Role_REQ(aucBD_ADDR, ucRole);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		DEBUG("DUMMY LocalRole %d", ucRole);

		result = dummy_data[FUNC_Set_Local_Role_REQ].return_code;
	}

	return result;
}


BTHRESULT BT_APPL_Get_Call_Status_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR, IN u8 ucSpecCall)
{
	BTHRESULT result;

	if (dummy_data[FUNC_Get_Call_Status_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_Get_Call_Status_REQ(aucBD_ADDR, ucSpecCall);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		DEBUG("DUMMY spec_call %d", ucSpecCall);

		result = dummy_data[FUNC_Get_Call_Status_REQ].return_code;
	}

	return result;
}


BTHRESULT BT_APPL_Get_Phone_Status_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR)
{
	BTHRESULT result;

	if (dummy_data[FUNC_Get_Phone_Status_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_Get_Phone_Status_REQ(aucBD_ADDR);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		result = dummy_data[FUNC_Get_Phone_Status_REQ].return_code;
	}

	return result;
}


BTHRESULT BT_APPL_HF_Dial_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR,
					IN u8 ucLength,
					IN const u8* pucPHNNumber)
{
	BTHRESULT result;

	if (dummy_data[FUNC_HF_Dial_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_HF_Dial_REQ(aucBD_ADDR, ucLength, pucPHNNumber);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		DEBUG("DUMMY length %d number %s", ucLength, pucPHNNumber);

		result = dummy_data[FUNC_HF_Dial_REQ].return_code;
	}

	return result;
}


BTHRESULT BT_APPL_HF_Redial_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR)
{
	BTHRESULT result;

	if (dummy_data[FUNC_HF_Redial_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_HF_Redial_REQ(aucBD_ADDR);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		result = dummy_data[FUNC_HF_Redial_REQ].return_code;
	}

	return result;
}


BTHRESULT BT_APPL_HF_Mem_Dial_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR, IN u16 usMemIndex)
{
	BTHRESULT result;

	if (dummy_data[FUNC_HF_Mem_Dial_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_HF_Mem_Dial_REQ(aucBD_ADDR, usMemIndex);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		DEBUG("DUMMY mem_index %d", usMemIndex);

		result = dummy_data[FUNC_HF_Mem_Dial_REQ].return_code;
	}

	return result;
}


BTHRESULT BT_APPL_HF_Terminate_Call_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR)
{
	BTHRESULT result;

	if (dummy_data[FUNC_HF_Terminate_Call_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_HF_Terminate_Call_REQ(aucBD_ADDR);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		result = dummy_data[FUNC_HF_Terminate_Call_REQ].return_code;
	}

	return result;
}


BTHRESULT BT_APPL_HF_Terminate_All_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR)
{
	BTHRESULT result;

	if (dummy_data[FUNC_HF_Terminate_All_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_HF_Terminate_All_REQ(aucBD_ADDR);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		result = dummy_data[FUNC_HF_Terminate_All_REQ].return_code;
	}

	return result;
}


BTHRESULT BT_APPL_HF_Audio_Transfer_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR,
						IN u8 ucAudioTransDir)
{
	BTHRESULT result;

	if (dummy_data[FUNC_HF_Audio_Transfer_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_HF_Audio_Transfer_REQ(aucBD_ADDR, ucAudioTransDir);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		DEBUG("DUMMY audio_trans_dir %d", ucAudioTransDir);

		result = dummy_data[FUNC_HF_Audio_Transfer_REQ].return_code;
	}

	return result;
}


BTHRESULT BT_APPL_HF_Send_DTMF_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR, IN u8 ucDTMFIndex)
{
	BTHRESULT result;

	if (dummy_data[FUNC_HF_Send_DTMF_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_HF_Send_DTMF_REQ(aucBD_ADDR, ucDTMFIndex);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		DEBUG("DUMMY dtmf_index %d", ucDTMFIndex);

		result = dummy_data[FUNC_HF_Send_DTMF_REQ].return_code;
	}

	return result;
}


BTHRESULT BT_APPL_HF_Incoming_Call_Act_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR,IN u8 ucAction)
{
	BTHRESULT result;

	if (dummy_data[FUNC_HF_Incoming_Call_Act_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_HF_Incoming_Call_Act_REQ(aucBD_ADDR, ucAction);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		DEBUG("DUMMY action %d", ucAction);

		result = dummy_data[FUNC_HF_Incoming_Call_Act_REQ].return_code;
	}

	return result;
}


BTHRESULT BT_APPL_HF_Release_Inactive_Call_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR)
{
	BTHRESULT result;

	if (dummy_data[FUNC_HF_Release_Inactive_Call_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_HF_Release_Inactive_Call_REQ(aucBD_ADDR);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		result = dummy_data[FUNC_HF_Release_Inactive_Call_REQ].return_code;
	}

	return result;
}


BTHRESULT BT_APPL_HF_Release_Active_Call_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR)
{
	BTHRESULT result;

	if (dummy_data[FUNC_HF_Release_Active_Call_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_HF_Release_Active_Call_REQ(aucBD_ADDR);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		result = dummy_data[FUNC_HF_Release_Active_Call_REQ].return_code;
	}

	return result;
}


BTHRESULT BT_APPL_HF_Swap_Call_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR)
{
	BTHRESULT result;

	if (dummy_data[FUNC_HF_Swap_Call_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_HF_Swap_Call_REQ(aucBD_ADDR);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		result = dummy_data[FUNC_HF_Swap_Call_REQ].return_code;
	}

	return result;
}


BTHRESULT BT_APPL_HF_Add_Call_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR)
{
	BTHRESULT result;

	if (dummy_data[FUNC_HF_Add_Call_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_HF_Add_Call_REQ(aucBD_ADDR);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		result = dummy_data[FUNC_HF_Add_Call_REQ].return_code;
	}

	return result;
}


BTHRESULT BT_APPL_HF_Release_Spec_Call_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR,
							IN u8 ucIndex)
{
	BTHRESULT result;

	if (dummy_data[FUNC_HF_Release_Spec_Call_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_HF_Release_Spec_Call_REQ(aucBD_ADDR, ucIndex);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		DEBUG("DUMMY index %d", ucIndex);

		result = dummy_data[FUNC_HF_Release_Spec_Call_REQ].return_code;
	}

	return result;
}


BTHRESULT BT_APPL_HF_Hold_Excep_Call_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR,
							IN u8 ucIndex)
{
	BTHRESULT result;

	if (dummy_data[FUNC_HF_Hold_Excep_Call_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_HF_Hold_Excep_Call_REQ(aucBD_ADDR, ucIndex);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		DEBUG("DUMMY index %d", ucIndex);

		result = dummy_data[FUNC_HF_Hold_Excep_Call_REQ].return_code;
	}

	return result;
}


BTHRESULT BT_APPL_HF_Exp_Call_Trans_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR)
{
	BTHRESULT result;

	if (dummy_data[FUNC_HF_Exp_Call_Trans_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_HF_Exp_Call_Trans_REQ(aucBD_ADDR);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		result = dummy_data[FUNC_HF_Exp_Call_Trans_REQ].return_code;
	}

	return result;
}


BTHRESULT BT_APPL_HF_Subscriber_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR)
{
	BTHRESULT result;

	if (dummy_data[FUNC_HF_Subscriber_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_HF_Subscriber_REQ(aucBD_ADDR);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		result = dummy_data[FUNC_HF_Subscriber_REQ].return_code;
	}

	return result;
}


BTHRESULT BT_APPL_HF_Manufacturer_ID_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR)
{
	BTHRESULT result;

	if (dummy_data[FUNC_HF_Manufacturer_ID_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_HF_Manufacturer_ID_REQ(aucBD_ADDR);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		result = dummy_data[FUNC_HF_Manufacturer_ID_REQ].return_code;
	}

	return result;
}


BTHRESULT BT_APPL_HF_Model_ID_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR)
{
	BTHRESULT result;

	if (dummy_data[FUNC_HF_Model_ID_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_HF_Model_ID_REQ(aucBD_ADDR);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		result = dummy_data[FUNC_HF_Model_ID_REQ].return_code;
	}

	return result;
}


BTHRESULT BT_APPL_HF_Revision_ID_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR)
{
	BTHRESULT result;

	if (dummy_data[FUNC_HF_Revision_ID_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_HF_Revision_ID_REQ(aucBD_ADDR);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		result = dummy_data[FUNC_HF_Revision_ID_REQ].return_code;
	}

	return result;
}

BTHRESULT BT_APPL_HF_Voice_Recognition_Activation_REQ_DUMMY(
						IN const BD_ADDRESS aucBD_ADDR,
						IN u8 ucActivation)
{
	BTHRESULT result;

	if (dummy_data[FUNC_HF_Voice_Recognition_Activation_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_Voice_Recognition_Activation_REQ(aucBD_ADDR,
								ucActivation);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		DEBUG("DUMMY activation %u", ucActivation);

		/* spoof sending of the signal */
		BT_APPL_HF_Voice_Recognition_Activation_IND_Callback_DUMMY(
								aucBD_ADDR,
								ucActivation);

		/* spoof the CFM, status = activation + 1 */
		BT_APPL_HF_Voice_Recognition_Activation_CFM_Callback_DUMMY(
							aucBD_ADDR,
							ucActivation,
							ucActivation + 1);

		result = dummy_data[FUNC_HF_Voice_Recognition_Activation_REQ].return_code;
	}

	return result;
}

BTHRESULT BT_APPL_AVP_Ctrl_Cmd_REQ_DUMMY(IN BD_ADDRESS aucBD_ADDR, IN u8 ucCommand, IN u8 ucStatusFlag)
{
	BTHRESULT result;

	if (dummy_data[FUNC_AVP_Ctrl_Cmd_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_AVP_Ctrl_Cmd_REQ(aucBD_ADDR, ucCommand, ucStatusFlag);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		DEBUG("DUMMY command %u, status_flag %u", ucCommand, ucStatusFlag);

		result = dummy_data[FUNC_AVP_Ctrl_Cmd_REQ].return_code;
	}

	return result;
}

BTHRESULT BT_APPL_AVP_Get_Capabilities_REQ_DUMMY(IN BD_ADDRESS aucBD_ADDR, IN u8 ucCapabilityId)
{
	BTHRESULT result;

	if (dummy_data[FUNC_AVP_Get_Capabilities_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_AVP_Get_Capabilities_REQ(aucBD_ADDR, ucCapabilityId);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		DEBUG("DUMMY capability_id %u", ucCapabilityId);

		result = dummy_data[FUNC_AVP_Get_Capabilities_REQ].return_code;
	}

	return result;
}

BTHRESULT BT_APPL_AVP_List_Attr_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR)
{
	BTHRESULT result;

	if (dummy_data[FUNC_AVP_List_Attr_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_AVP_List_Attr_REQ(aucBD_ADDR);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		result = dummy_data[FUNC_AVP_List_Attr_REQ].return_code;
	}

	return result;
}

BTHRESULT BT_APPL_AVP_List_Val_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR, IN u8 ucSettingAttributeId)
{
	BTHRESULT result;

	if (dummy_data[FUNC_AVP_List_Val_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_AVP_List_Val_REQ(aucBD_ADDR, ucSettingAttributeId);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		DEBUG("setting_attribute_id %u", ucSettingAttributeId);

		result = dummy_data[FUNC_AVP_List_Val_REQ].return_code;
	}

	return result;
}

BTHRESULT BT_APPL_AVP_Set_Val_REQ_DUMMY(IN BD_ADDRESS aucBD_ADDR, IN u8 ucMode, IN u8 ucValue)
{
	BTHRESULT result;

	if (dummy_data[FUNC_AVP_Set_Val_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_AVP_Set_Val_REQ(aucBD_ADDR, ucMode, ucValue);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		DEBUG("mode %u value %u", ucMode, ucValue);

		result = dummy_data[FUNC_AVP_Set_Val_REQ].return_code;
	}

	return result;
}

BTHRESULT BT_APPL_AVP_Config_Notify_REQ_DUMMY(IN u16 usEventId)
{
	BTHRESULT result;

	if (dummy_data[FUNC_AVP_Config_Notify_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_AVP_Config_Notify_REQ(usEventId);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY event_id %u", usEventId);

		result = dummy_data[FUNC_AVP_Config_Notify_REQ].return_code;
	}

	return result;
}

BTHRESULT BT_APPL_AVP_Set_Player_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR, IN u16 usPlayerId, IN u8 ucPlayer)
{
	BTHRESULT result;

	if (dummy_data[FUNC_AVP_Set_Player_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_AVP_Set_Player_REQ(aucBD_ADDR, usPlayerId, ucPlayer);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		DEBUG("DUMMY player_id %u, player %u", usPlayerId, ucPlayer);

		result = dummy_data[FUNC_AVP_Set_Player_REQ].return_code;
	}

	return result;
}

BTHRESULT BT_APPL_AVP_Get_Metadata_Attr_REQ_DUMMY(IN BD_ADDRESS aucBD_ADDR,
							IN u8 ucScope,
							IN u8 aucUid[],
							IN u8 ucNumAttributes,
							IN u32 aulAttributes[])
{
	BTHRESULT result;

	if (dummy_data[FUNC_AVP_Get_Metadata_Attr_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_AVP_Get_Metadata_Attr_REQ(aucBD_ADDR, ucScope,
							&aucUid[0],
							ucNumAttributes,
							&aulAttributes[0]);

	} else {
#ifdef USE_TEST_NODE
		struct avp_get_metadata_attr_cfm_parms_s *cfm_test_data;
#endif

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		DEBUG("DUMMY scope %u, num_attribute %u",
			ucScope, ucNumAttributes);

#ifdef USE_TEST_NODE
		cfm_test_data = (struct avp_get_metadata_attr_cfm_parms_s *)
				utils_test_queue_find_list_entry(
						test_node_queue,
						ID_AVP_Get_Metadata_Attr_CFM,
						&aucBD_ADDR[0]);

		if (cfm_test_data != NULL) {

			DEBUG("spoofing Callback event handling with test data");

			/* call the CFM callback function with test data */
			BT_APPL_AVP_Get_Metadata_Attr_CFM_Callback_DUMMY(
						&aucBD_ADDR[0],
						cfm_test_data->status,
						cfm_test_data->num_attributes,
						cfm_test_data->data->len,
						&cfm_test_data->data->data[0]);

			/* free the GByte Array */
			g_byte_array_unref(cfm_test_data->data);

			/* free the data structure */
			g_free(cfm_test_data);
		}
#endif

		result = dummy_data[FUNC_AVP_Get_Metadata_Attr_REQ].return_code;
	}

	return result;
}

BTHRESULT BT_APPL_AVP_Get_Folder_Item_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR,
						IN u8 ucScope,
						IN u32 ulStartItem,
						IN u32 ulEndItem,
						IN u8 ucNumAttributes,
						IN const u32 aulAttributes[])
{
	BTHRESULT result;

	if (dummy_data[FUNC_AVP_Get_Folder_Item_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_AVP_Get_Folder_Item_REQ(aucBD_ADDR, ucScope,
								ulStartItem,
								ulEndItem,
								ucNumAttributes,
								aulAttributes);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		DEBUG("DUMMY scope %u, start_item %lu, end_item %lu",
			ucScope, ulStartItem, ulEndItem);

		DEBUG("DUMMY num_attribute %u",
			ucNumAttributes);

		result = dummy_data[FUNC_AVP_Get_Folder_Item_REQ].return_code;
	}

	return result;
}

BTHRESULT BT_APPL_AVP_Get_Media_Player_Item_REQ_DUMMY(
						IN const BD_ADDRESS aucBD_ADDR,
						IN u16 usMaxNumber)
{
	BTHRESULT result;

	if (dummy_data[FUNC_AVP_Get_Media_Player_Item_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_AVP_Get_Media_Player_Item_REQ(aucBD_ADDR, usMaxNumber);

	} else {
#ifdef USE_TEST_NODE
		struct avp_get_media_player_item_cfm_parms_s *cfm_test_data;
		struct avp_get_media_player_item_ind_parms_s *ind_test_data;
#endif

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		DEBUG("DUMMY max_number %u", usMaxNumber);

#ifdef USE_TEST_NODE
		cfm_test_data = (struct avp_get_media_player_item_cfm_parms_s *)
				utils_test_queue_find_list_entry(
					test_node_queue,
					ID_AVP_Get_Media_Player_Item_CFM,
					&aucBD_ADDR[0]);

		if (cfm_test_data != NULL) {

			DEBUG("spoofing Callback event handling with CFM test data");

			/* call the CFM callback function with test data */
			BT_APPL_AVP_Get_Media_Player_Item_CFM_Callback_DUMMY(
						&aucBD_ADDR[0],
						cfm_test_data->status,
						cfm_test_data->num_items);

			/* free the data structure */
			g_free(cfm_test_data);

			DEBUG("finished spoofing the event handling");
		}

		ind_test_data = (struct avp_get_media_player_item_ind_parms_s *)
				utils_test_queue_find_list_entry(
					test_node_queue,
					ID_AVP_Get_Media_Player_Item_IND,
					&aucBD_ADDR[0]);

		while(ind_test_data != NULL) {

			DEBUG("spoofing Callback event handling with IND test data");

			DEBUG_DUMP_ARRAY(ind_test_data->data->len,
					&ind_test_data->data->data[0]);

			/* call the CFM callback function with test data */
			BT_APPL_AVP_Get_Media_Player_Item_IND_Callback_DUMMY(
						&aucBD_ADDR[0],
						ind_test_data->status,
						ind_test_data->data->len,
						&ind_test_data->data->data[0]);

			/* free the GByte Array */
			g_byte_array_unref(ind_test_data->data);

			/* free the data structure */
			g_free(ind_test_data);

			DEBUG("finished spoofing the event handling");

			ind_test_data = (struct avp_get_media_player_item_ind_parms_s *)
				utils_test_queue_find_list_entry(
					test_node_queue,
					ID_AVP_Get_Media_Player_Item_IND,
					&aucBD_ADDR[0]);
		}
#endif

		result = dummy_data[FUNC_AVP_Get_Media_Player_Item_REQ].return_code;
	}

	return result;
}

BTHRESULT BT_APPL_AVP_Play_Item_REQ_DUMMY(IN BD_ADDRESS aucBD_ADDR,
						IN u8 ucScope,
						IN const u8 aucUid[])
{
	BTHRESULT result;

	if (dummy_data[FUNC_AVP_Play_Item_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_AVP_Play_Item_REQ(aucBD_ADDR, ucScope, &aucUid[0]);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		DEBUG("DUMMY scope %u", ucScope);

		result = dummy_data[FUNC_AVP_Play_Item_REQ].return_code;
	}

	return result;
}

BTHRESULT BT_APPL_AVP_Change_Path_REQ_DUMMY(IN BD_ADDRESS aucBD_ADDR,
				      IN u8 ucDirection,
				      IN u8 aucFolderUid[])
{
	BTHRESULT result;

	if (dummy_data[FUNC_AVP_Change_Path_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_AVP_Change_Path_REQ(aucBD_ADDR, ucDirection,
							   &aucFolderUid[0]);
	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		DEBUG("DUMMY direction %u", ucDirection);

		result = dummy_data[FUNC_AVP_Change_Path_REQ].return_code;
	}

	return result;
}

BTHRESULT BT_APPL_PBDL_Get_Capabilities_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR)
{
	BTHRESULT result;

	if (dummy_data[FUNC_PBDL_Get_Capabilities_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_PBDL_Get_Capabilities_REQ(aucBD_ADDR);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		if (dummy_data[FUNC_PBDL_Get_Capabilities_REQ].return_code != BT_APPL_SUCCESS) {
			if (dummy_data[FUNC_register_alps_callbacks].use_dummy == TRUE) {
				DEBUG("spoofing Callback event handling");
				result = BT_APPL_PBDL_Get_Capabilities_CFM_Callback_DUMMY(
								aucBD_ADDR,
								0x00,
								0x1F);
			} else {
				DEBUG("using REAL Callback event handling");
				result = BT_APPL_SUCCESS;
			}
		} else {
			result = dummy_data[FUNC_PBDL_Get_Capabilities_REQ].return_code;
		}
	}

	return result;
}


BTHRESULT BT_APPL_PBDL_Prepare_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR)
{
	BTHRESULT result;

	if (dummy_data[FUNC_PBDL_Prepare_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_PBDL_Prepare_REQ(aucBD_ADDR);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		if (dummy_data[FUNC_PBDL_Prepare_REQ].return_code != BT_APPL_SUCCESS) {
			if (dummy_data[FUNC_register_alps_callbacks].use_dummy == TRUE) {
				DEBUG("spoofing Callback event handling");
				result = BT_APPL_PBDL_Prepare_CFM_Callback_DUMMY(
								aucBD_ADDR,
								0x1F);
			} else {
				DEBUG("using REAL Callback event handling");
				result = BT_APPL_SUCCESS;
			}
		} else {
			result = dummy_data[FUNC_PBDL_Prepare_REQ].return_code;
		}
	}

	return result;
}


BTHRESULT BT_APPL_PBDL_Start_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR, IN u32 ulFilter, IN u8 ucPBType, IN u8 ucAction, IN u8 ucVersion)
{
	BTHRESULT result;

	if (dummy_data[FUNC_PBDL_Start_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_PBDL_Start_REQ(aucBD_ADDR, ulFilter, ucPBType, ucAction, ucVersion);

	} else {
#ifdef USE_TEST_NODE
		struct pbdl_start_cfm_parms_s *cfm_test_data;
#endif

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		DEBUG("DUMMY filter 0x%04lX, pb_type %u, action %u, version %u",
				ulFilter, ucPBType, ucAction, ucVersion);

#ifdef USE_TEST_NODE
		cfm_test_data = (struct pbdl_start_cfm_parms_s *)
				utils_test_queue_find_list_entry(
							test_node_queue,
							ID_PBDL_Start_CFM,
							&aucBD_ADDR[0]);

		if (cfm_test_data != NULL) {

			DEBUG("spoofing Callback event handling with test data");

			/* call the CFM callback function with test data */
			BT_APPL_PBDL_Start_CFM_Callback_DUMMY(&aucBD_ADDR[0],
						cfm_test_data->status,
						cfm_test_data->pb_number,
						cfm_test_data->new_missed_call);
		}
#endif

		result = dummy_data[FUNC_PBDL_Start_REQ].return_code;
	}

	return result;
}


BTHRESULT BT_APPL_PBDL_Stop_REQ_DUMMY(IN BD_ADDRESS aucBD_ADDR, IN u8 ucAction)
{
	BTHRESULT result;

	if (dummy_data[FUNC_PBDL_Stop_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_PBDL_Stop_REQ(aucBD_ADDR, ucAction);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		DEBUG("DUMMY action %d", ucAction);

		if (dummy_data[FUNC_PBDL_Stop_REQ].return_code != BT_APPL_SUCCESS) {
			if (dummy_data[FUNC_register_alps_callbacks].use_dummy == TRUE) {
				DEBUG("spoofing Callback event handling");
				result = BT_APPL_PBDL_Stop_CFM_Callback_DUMMY(
								aucBD_ADDR,
								1);
			} else {
				DEBUG("using REAL Callback event handling");
				result = BT_APPL_SUCCESS;
			}
		} else {
			result = dummy_data[FUNC_PBDL_Stop_REQ].return_code;
		}
	}

	return result;
}


BTHRESULT BT_APPL_PBDL_Complete_REQ_DUMMY(IN BD_ADDRESS aucBD_ADDR, IN u8 ucCompMode)
{
	BTHRESULT result;

	if (dummy_data[FUNC_PBDL_Complete_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_PBDL_Complete_REQ(aucBD_ADDR, ucCompMode);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		if (dummy_data[FUNC_PBDL_Complete_REQ].return_code != BT_APPL_SUCCESS) {
			if (dummy_data[FUNC_register_alps_callbacks].use_dummy == TRUE) {
				DEBUG("spoofing Callback event handling");
				result = BT_APPL_PBDL_Complete_CFM_Callback_DUMMY(
								aucBD_ADDR,
								1);
			} else {
				DEBUG("using REAL Callback event handling");
				result = BT_APPL_SUCCESS;
			}	
		} else {
			result = dummy_data[FUNC_PBDL_Complete_REQ].return_code;
		}
	}

	return result;
}

BTHRESULT BT_APPL_MAP_Get_Capabilities_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR)
{
	BTHRESULT result;
	MsgCapability_t msg_capability;

	memset(&msg_capability, 0, sizeof(msg_capability));

	if (dummy_data[FUNC_MAP_Get_Capabilities_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_MAP_Get_Capabilities_REQ(aucBD_ADDR);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		if (dummy_data[FUNC_MAP_Get_Capabilities_REQ].return_code != BT_APPL_SUCCESS) {
			if (dummy_data[FUNC_register_alps_callbacks].use_dummy == TRUE) {
				DEBUG("spoofing Callback event handling");
				result = BT_APPL_MAP_Get_Capabilities_CFM_Callback_DUMMY(
								aucBD_ADDR,
								1,
								&msg_capability);
			} else {
				DEBUG("using REAL Callback event handling");
				result = BT_APPL_SUCCESS;
			}
		} else {
			result = dummy_data[FUNC_MAP_Get_Capabilities_REQ].return_code;
		}
	}

	return result;
}

BTHRESULT BT_APPL_MAP_Start_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR,
		                        IN u8 unInstanceNum,
		                        IN const InstanceInfoPtr_t pstInstanceInfo)
{
	BTHRESULT result;

	if (dummy_data[FUNC_MAP_Start_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_MAP_Start_REQ(aucBD_ADDR, unInstanceNum,
						pstInstanceInfo);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		if (dummy_data[FUNC_MAP_Start_REQ].return_code != BT_APPL_SUCCESS) {
			if (dummy_data[FUNC_register_alps_callbacks].use_dummy == TRUE) {
				DEBUG("spoofing Callback event handling");
				result = BT_APPL_MAP_Start_CFM_Callback_DUMMY(
								aucBD_ADDR,
								1);
			} else {
				DEBUG("using REAL Callback event handling");
				result = BT_APPL_SUCCESS;
			}
		} else {
			result = dummy_data[FUNC_MAP_Start_REQ].return_code;
		}
	}

	return result;
}

BTHRESULT BT_APPL_MAP_Stop_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR)
{
	BTHRESULT result;

	if (dummy_data[FUNC_MAP_Stop_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_MAP_Stop_REQ(aucBD_ADDR);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		if (dummy_data[FUNC_MAP_Stop_REQ].return_code != BT_APPL_SUCCESS) {
			if (dummy_data[FUNC_register_alps_callbacks].use_dummy == TRUE) {
				DEBUG("spoofing Callback event handling");
				result = BT_APPL_MAP_Stop_CFM_Callback_DUMMY(
								aucBD_ADDR,
								1);
			} else {
				DEBUG("using REAL Callback event handling");
				result = BT_APPL_SUCCESS;
			}
		} else {
			result = dummy_data[FUNC_MAP_Stop_REQ].return_code;
		}
	}

	return result;
}

BTHRESULT BT_APPL_MAP_Chg_Instance_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR,
                                       IN u8 ucServiceNameLen,
                                       IN const u8* pucServiceName,
                                       IN u8 ucMsgType)
{
	BTHRESULT result;

	if (dummy_data[FUNC_MAP_Chg_Instance_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_MAP_Chg_Instance_REQ(aucBD_ADDR,
						ucServiceNameLen,
						pucServiceName,
						ucMsgType);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		if (dummy_data[FUNC_MAP_Chg_Instance_REQ].return_code != BT_APPL_SUCCESS) {
			if (dummy_data[FUNC_register_alps_callbacks].use_dummy == TRUE) {
				DEBUG("spoofing Callback event handling");
				result = BT_APPL_MAP_Chg_Instance_CFM_Callback_DUMMY(
								aucBD_ADDR,
								1);
			} else {
				DEBUG("using REAL Callback event handling");
				result = BT_APPL_SUCCESS;
			}
		} else {
			result = dummy_data[FUNC_MAP_Chg_Instance_REQ].return_code;
		}
	}

	return result;
}

BTHRESULT BT_APPL_MAP_Update_Inbox_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR,
                                       IN u8 ucServiceNameLen,
                                       IN const u8* pucServiceName)
{
	BTHRESULT result;

	if (dummy_data[FUNC_MAP_Update_Inbox_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_MAP_Update_Inbox_REQ(aucBD_ADDR,
						ucServiceNameLen,
						pucServiceName);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		if (dummy_data[FUNC_MAP_Update_Inbox_REQ].return_code != BT_APPL_SUCCESS) {
			if (dummy_data[FUNC_register_alps_callbacks].use_dummy == TRUE) {
				DEBUG("spoofing Callback event handling");
				result = BT_APPL_MAP_Update_Inbox_CFM_Callback_DUMMY(
								aucBD_ADDR,
								0,
								NULL,
								1);
			} else {
				DEBUG("using REAL Callback event handling");
				result = BT_APPL_SUCCESS;
			}
		} else {
			result = dummy_data[FUNC_MAP_Update_Inbox_REQ].return_code;
		}
	}

	return result;
}

BTHRESULT BT_APPL_MAP_List_Folder_REQ_DUMMY(IN const BD_ADDRESS aucBD_ADDR,
				IN u8 ucServiceNameLen,
				IN const u8* pucServiceName,
				IN ListFolderParamPtr_t pstParam)
{
	BTHRESULT result;

	if (dummy_data[FUNC_MAP_List_Folder_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_MAP_List_Folder_REQ(aucBD_ADDR,
						ucServiceNameLen,
						pucServiceName,
						pstParam);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		if (dummy_data[FUNC_MAP_List_Folder_REQ].return_code != BT_APPL_SUCCESS) {
			if (dummy_data[FUNC_register_alps_callbacks].use_dummy == TRUE) {
				DEBUG("spoofing Callback event handling");
				result = BT_APPL_MAP_List_Folder_CFM_Callback_DUMMY(
								aucBD_ADDR,
								0,
								NULL,
								1,
								0,
								0);
			} else {
				DEBUG("using REAL Callback event handling");
				result = BT_APPL_SUCCESS;
			}
		} else {
			result = dummy_data[FUNC_MAP_List_Folder_REQ].return_code;
		}
	}

	return result;
}

BTHRESULT BT_APPL_MAP_List_Msg_REQ_DUMMY(IN BD_ADDRESS aucBD_ADDR,
				IN u8 ucServiceNameLen,
				IN u8* pucServiceName,
				IN ListMsgParamPtr_t pstParam)
{
	BTHRESULT result;

	if (dummy_data[FUNC_MAP_List_Msg_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_MAP_List_Msg_REQ(aucBD_ADDR,
						ucServiceNameLen,
						pucServiceName,
						pstParam);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		if (dummy_data[FUNC_MAP_List_Msg_REQ].return_code != BT_APPL_SUCCESS) {
			if (dummy_data[FUNC_register_alps_callbacks].use_dummy == TRUE) {
				DEBUG("spoofing Callback event handling");
				result = BT_APPL_MAP_List_Msg_CFM_Callback_DUMMY(
								aucBD_ADDR,
								ucServiceNameLen,
								pucServiceName,
								1,
								0);
			} else {
				DEBUG("using REAL Callback event handling");
				result = BT_APPL_SUCCESS;
			}
		} else {
			result = dummy_data[FUNC_MAP_List_Msg_REQ].return_code;
		}
	}

	return result;
}

BTHRESULT BT_APPL_MAP_Read_Msg_REQ_DUMMY(IN BD_ADDRESS aucBD_ADDR,
					IN u8 ucServiceNameLen,
					IN u8* pucServiceName,
					IN u8 ucHandleLen,
					IN const u8* pucHandle,
					IN u8 ucAttachment)
{
	BTHRESULT result;
	ReadMsgInfo_t *read_msg = NULL;


	memset(&read_msg, 0, sizeof(read_msg));

	if (dummy_data[FUNC_MAP_Read_Msg_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_MAP_Read_Msg_REQ(aucBD_ADDR,
						ucServiceNameLen,
						pucServiceName,
						ucHandleLen,
						pucHandle,
						ucAttachment);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		if (dummy_data[FUNC_MAP_Read_Msg_REQ].return_code != BT_APPL_SUCCESS) {
			if (dummy_data[FUNC_MAP_Read_Msg_REQ].state_number == 0) {
				DEBUG("Creating test data for msg_info...");

				read_msg = (ReadMsgInfo_t *) g_malloc0(sizeof(ReadMsgInfo_t));
				read_msg->pstMsgInfo = (MsgInfo_t *) g_malloc0(sizeof(MsgInfo_t));
				g_assert(read_msg->pstMsgInfo != NULL);

				read_msg->pstMsgInfo->ucReadStatus = 1;
				read_msg->pstMsgInfo->ucType = 10;

				read_msg->pstMsgInfo->stFolder.pucFolderName = (u8 *) "Test folder",
						read_msg->pstMsgInfo->stFolder.usFolderNameLen =
					strlen((gchar *) read_msg->pstMsgInfo->stFolder.pucFolderName);

				read_msg->pstMsgInfo->stOriginator.pucN  = (u8 *) "N field";
				read_msg->pstMsgInfo->stOriginator.ucNLen =
					strlen((gchar *) read_msg->pstMsgInfo->stOriginator.pucN);

				read_msg->pstMsgInfo->stOriginator.pucFN = (u8 *) "FN field";
				read_msg->pstMsgInfo->stOriginator.ucFNLen =
					strlen((gchar *) read_msg->pstMsgInfo->stOriginator.pucFN);

				read_msg->pstMsgInfo->stOriginator.pucTEL = (u8 *) "Tel field";
				read_msg->pstMsgInfo->stOriginator.ucTELLen =
					strlen((gchar *) read_msg->pstMsgInfo->stOriginator.pucTEL);

				read_msg->pstMsgInfo->stOriginator.pucEMAIL = (u8 *) "E-mail field";
				read_msg->pstMsgInfo->stOriginator.ucEMAILLen =
					strlen((gchar *) read_msg->pstMsgInfo->stOriginator.pucEMAIL);

				/* test 1 recipient vCard */
				read_msg->pstMsgInfo->ucReceiverNum = 1;

				read_msg->pstMsgInfo->stRecipient[0].pucN = (u8 *) "Recipient 1: N";
				read_msg->pstMsgInfo->stRecipient[0].ucNLen =
					strlen((gchar *) read_msg->pstMsgInfo->stRecipient[0].pucN);

				read_msg->pstMsgInfo->stRecipient[0].pucFN= (u8 *) "Recipient 1: FN";
				read_msg->pstMsgInfo->stRecipient[0].ucFNLen =
					strlen((gchar *) read_msg->pstMsgInfo->stRecipient[0].pucFN);

				read_msg->pstMsgInfo->stRecipient[0].pucTEL = (u8 *) "Recipient 1: TEL";
				read_msg->pstMsgInfo->stRecipient[0].ucTELLen =
					strlen((gchar *) read_msg->pstMsgInfo->stRecipient[0].pucTEL);

				read_msg->pstMsgInfo->stRecipient[0].pucEMAIL = (u8 *) "Recipient 1: Email";
				read_msg->pstMsgInfo->stRecipient[0].ucEMAILLen =
					strlen((gchar *) read_msg->pstMsgInfo->stRecipient[0].pucEMAIL);

				read_msg->pstMsgInfo->stRecipient[0].ucBENVLevel =3;

				read_msg->pstMsgInfo->ulMsgLen = 1024;

				/* test 1 attachment */
				read_msg->pstMsgInfo->ucAttachNum = 1;

				read_msg->pstMsgInfo->stAttachmentList[0].ucName = (u8 *) "1st attachment";
				read_msg->pstMsgInfo->stAttachmentList[0].usNameLen =
					strlen((gchar *) read_msg->pstMsgInfo->stAttachmentList[0].ucName);

				read_msg->pstMsgInfo->stAttachmentList[0].ulSize = 2048,

				read_msg->pstMsgInfo->ucLanguage = 2;
				read_msg->pstMsgInfo->ucEncoding = 3;
	
				dummy_data[FUNC_MAP_Read_Msg_REQ].state_number = 1;
			} else {
				dummy_data[FUNC_MAP_Read_Msg_REQ].state_number = 0;
			}

			if (dummy_data[FUNC_register_alps_callbacks].use_dummy == TRUE) {
				DEBUG("spoofing Callback event handling");
				result = BT_APPL_MAP_Read_Msg_CFM_Callback_DUMMY(
							aucBD_ADDR,
							0,
							(u8 *) "TEST_SERVICE",
							1,
							read_msg);
			} else {
				DEBUG("using REAL Callback event handling");
				result = BT_APPL_SUCCESS;
			}
		} else {
			result = dummy_data[FUNC_MAP_Read_Msg_REQ].return_code;
		}
	}

	return result;
}

BTHRESULT BT_APPL_MAP_Set_Msg_Status_REQ_DUMMY(IN BD_ADDRESS aucBD_ADDR,
				IN u8 ucServiceNameLen,
				IN u8* pucServiceName,
				IN u8 ucHandleLen,
				IN const u8* pucHandle,
				IN u8 ucMsgStatus)
{
	BTHRESULT result;

	if (dummy_data[FUNC_MAP_Set_Msg_Status_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_MAP_Set_Msg_Status_REQ(aucBD_ADDR,
						ucServiceNameLen,
						pucServiceName,
						ucHandleLen,
						pucHandle,
						ucMsgStatus);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		if (dummy_data[FUNC_MAP_Set_Msg_Status_REQ].return_code != BT_APPL_SUCCESS) {
			if (dummy_data[FUNC_register_alps_callbacks].use_dummy == TRUE) {

				DEBUG("spoofing Callback event handling");
				result = BT_APPL_MAP_Set_Msg_Status_CFM_Callback_DUMMY(
								aucBD_ADDR,
								0,
								NULL,
								1);
			} else {
				DEBUG("using REAL Callback event handling");
				result = BT_APPL_SUCCESS;
			}
		} else {
			result = dummy_data[FUNC_MAP_Set_Msg_Status_REQ].return_code;
		}
	}

	return result;
}

BTHRESULT BT_APPL_MAP_Abort_REQ_DUMMY(IN BD_ADDRESS aucBD_ADDR,
					IN u8 ucServiceNameLen,
					IN u8* pucServiceName)
{
	BTHRESULT result;

	if (dummy_data[FUNC_MAP_Abort_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_MAP_Abort_REQ(aucBD_ADDR,
						ucServiceNameLen,
						pucServiceName);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		if (dummy_data[FUNC_MAP_Abort_REQ].return_code != BT_APPL_SUCCESS) {
			if (dummy_data[FUNC_register_alps_callbacks].use_dummy == TRUE) {
				DEBUG("spoofing Callback event handling");
				result = BT_APPL_MAP_Abort_CFM_Callback_DUMMY(
								aucBD_ADDR,
								0,
								NULL,
								1);
			} else {
				DEBUG("using REAL Callback event handling");
				result = BT_APPL_SUCCESS;
			}
		} else {
			result = dummy_data[FUNC_MAP_Abort_REQ].return_code;
		}
	}

	return result;
}

BTHRESULT BT_APPL_MAP_Upload_REQ_DUMMY(IN BD_ADDRESS aucBD_ADDR,
					IN u8 ucServiceNameLen,
					IN u8* pucServiceName,
					IN UploadParamPtr_t pstParam)
{
	BTHRESULT result;

	if (dummy_data[FUNC_MAP_Upload_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_MAP_Upload_REQ(aucBD_ADDR,
						ucServiceNameLen,
						pucServiceName,
						pstParam);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		DEBUG("using REAL Callback event handling");
		result = dummy_data[FUNC_MAP_Upload_REQ].return_code;
	}

	return result;
}

BTHRESULT BT_APPL_MAP_Upload_RES_DUMMY(IN BD_ADDRESS aucBD_ADDR,
					IN u8 ucServiceNameLen,
					IN u8* pucServiceName,
					IN u8 ucDataFlag,
					IN u16 usLength,
					IN u8* pucData)
{
	BTHRESULT result;

	if (dummy_data[FUNC_MAP_Upload_RES].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_MAP_Upload_RES_DUMMY(aucBD_ADDR,
						ucServiceNameLen,
						pucServiceName,
						ucDataFlag,
						usLength,
						pucData);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		DEBUG("using REAL Callback event handling");
		result = dummy_data[FUNC_MAP_Upload_RES].return_code;
	}

	return result;
}

/*********************************************/
/* API functions used by the Validation node */
/*********************************************/

BTHRESULT BT_APPL_TestMode_REQ_DUMMY(IN u8 ucEnable, IN u8 ucTestMode, IN u8 ucRole, IN const BD_ADDRESS aucBD_ADDR)
{
	BTHRESULT result;

	if (dummy_data[FUNC_TestMode_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_TestMode_REQ(ucEnable, ucTestMode, ucRole, aucBD_ADDR);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY enable %d, testmode %d, role %d", ucEnable, ucTestMode, ucRole);

		DEBUG("DUMMY address: %02X:%02X:%02X:%02X:%02X:%02X",
			aucBD_ADDR[0], aucBD_ADDR[1], aucBD_ADDR[2],
			aucBD_ADDR[3], aucBD_ADDR[4], aucBD_ADDR[5]);

		if (dummy_data[FUNC_TestMode_REQ].return_code != BT_APPL_SUCCESS) {
			if (dummy_data[FUNC_register_alps_callbacks].use_dummy == TRUE) {
				DEBUG("spoofing Callback event handling");
				result = BT_APPL_TestMode_CFM_Callback_DUMMY(0);
			} else {
				DEBUG("using REAL Callback event handling");
				result = BT_APPL_SUCCESS;
			}
		} else {
			result = dummy_data[FUNC_TestMode_REQ].return_code;
		}
	}

	return result;
}


BTHRESULT BT_APPL_LinkQuality_REQ_DUMMY(IN u8 ucTestMode)
{
	BTHRESULT result;

	if (dummy_data[FUNC_LinkQuality_REQ].use_dummy == FALSE) {

		DEBUG("Calling REAL API FUNCTION");

		/* call the real function rather than the dummy */
		result = BT_APPL_LinkQuality_REQ(ucTestMode);

	} else {

		DEBUG("Using DUMMY API FUNCTION");

		DEBUG("DUMMY testmode %d", ucTestMode);

		if (dummy_data[FUNC_LinkQuality_REQ].return_code != BT_APPL_SUCCESS) {
			if (dummy_data[FUNC_register_alps_callbacks].use_dummy == TRUE) {
				DEBUG("spoofing Callback event handling");
				result = BT_APPL_LinkQuality_CFM_Callback_DUMMY(0, -5, 255);
			} else {
				DEBUG("using REAL Callback event handling");
				result = BT_APPL_SUCCESS;
			}
		} else {
			result = dummy_data[FUNC_LinkQuality_REQ].return_code;
		}
	}

	return result;
}

#endif
