/*
 * device object definitions for the D-bus Bluetooth Daemon
 *
 * Author: Dean Jenkins <djenkins@mvista.com>
 *
 * 2011 (c) MontaVista Software, LLC. This file is licensed under
 * the terms of the AFL.
 */

/* types used by the DeviceObject */
typedef enum {
	PBDL_IDLE = 0,
	PBDL_WAIT_DATA,
	PBDL_GOT_DATA,
	PBDL_SEND_DATA,
	PBDL_GOT_MAX_RECS,
	PBDL_COMPLETE_DATA,

	PBDL_TOTAL_STATES
}  PbDlDownloadState;

typedef struct {
	/*
	 * maximum total number of records per start request
	 * zero means unlimited number of records can be downloaded
	 */
	guint32 max_total_recs;

	/* count the number of records processed */
	guint32 total_recs_count;

	/*
	 * number of records per data transaction
	 * zero means phonebook download is inactive
	 */
	guint8 max_recs_per_trans;

	/* count the number of records per transaction */
	guint8 recs_trans_count;

	/* flag to use either the signal (FALSE) or agent method (TRUE) */
	gboolean use_agent;

	/* indicates the type of the phonebook */
	guint8 pb_type;

	/* indicates the type of the data */
	guint8 data_type;

	/* indicate whether data is present or not */
	gboolean data_present;

	/* holds concatentated phonebook index in guint32 lengths */
	GArray *index;

	/* holds concatentated phonebook data */
	GByteArray *data;

	/* indicate whether data session is completed (status info present) */
	gboolean completed;

	/* completion status */
	guint8 status;

	/* phonebook number */
	guint32 pb_number;

} PhonebookInfo;

typedef enum {
	MAP_LIST_MSG_IDLE = 0,
	MAP_LIST_MSG_WAIT_DATA,
	MAP_LIST_MSG_GOT_DATA,
	MAP_LIST_MSG_SEND_DATA,
	MAP_LIST_MSG_COMPLETE_DATA,

	MAP_LIST_MSG_TOTAL_STATES
} MapListMsgDownloadState;

typedef struct {
	/* number of records per data transaction */
	guint32 max_recs_per_trans;

	/* number of records processed for this data transaction */
	guint32 recs_trans_count;

	/* flag to use either the signal (FALSE) or agent method (TRUE) */
	gboolean use_agent;

	/* keep the service name for validation */
	gchar *service_name;

	GPtrArray *handle_list;
	gchar **subject_list;
	gchar **date_time_list;
	gchar **sender_name_list;
	gchar **sender_address_list;
	gchar **replyto_address_list;
	gchar **recipient_name_list;
	gchar **recipient_address_list;
	GByteArray *type_list;
	GArray *size_list;
	GByteArray *text_list;
	GByteArray *reception_status_list;
	GArray *attach_size_list;
	GByteArray *priority_list;
	GByteArray *read_list;
	GByteArray *sent_list;
	GByteArray *protect_list;

} MapListMsgInfo;

typedef enum {
	AVP_GET_MEDIA_PLAYER_ITEM_IDLE = 0,
	AVP_GET_MEDIA_PLAYER_ITEM_WAIT_DATA,
	AVP_GET_MEDIA_PLAYER_ITEM_SEND_REPLY,

	AVP_GET_MEDIA_PLAYER_ITEM_TOTAL_STATES
}  GetMediaPlayerItemState;

typedef struct {
	u8 status;
	u16 num_items;
	u16 got_items;
	gboolean network_order;
	/* AVRCP data */
	GByteArray *data;

} GetMediaPlayerItemInfo;

typedef enum {
	AVP_GET_FOLDER_ITEM_IDLE = 0,
	AVP_GET_FOLDER_ITEM_WAIT_DATA,
	AVP_GET_FOLDER_ITEM_SEND_REPLY,

	AVP_GET_FOLDER_ITEM_TOTAL_STATES
}  GetFolderItemState;

typedef struct {
	u8 status;
	u8 scope;
	u16 num_items;
	u16 got_items;
	gboolean network_order;
	/* AVRCP data */
	GByteArray *data;

} GetFolderItemInfo;

typedef struct {
	gboolean connected;	/* AVP connected/disconnected */
	gboolean streaming;	/* Media Resume/Media Suspend */
	gboolean codec_valid;	/* TRUE if codec type is valid */
	guint8 codec;		/* A2DP codec */
	guint bitrate;		/* A2DP bitrate */
} A2DPInfo;

/* OUR DEVICE OBJECT DEFINITION */
typedef struct {
	/* The parent class object state. */
	GObject parent;

	/* put device object stuff here */

	/* holds pointer to the device's address */
	BD_ADDRESS *device_address;

	/* current state of the phonebook download procedure */
	PbDlDownloadState pbdl_state;
	/* phonebook info */
	PhonebookInfo *phonebook;
	/* phonebook agent method reply queue list */
	GSList *phonebook_agent_reply_list;

	/* current state of the MAP list message download procedure */
	MapListMsgDownloadState map_list_msg_state;
	/* MAP list message info */
	MapListMsgInfo *map_list_msg_info;

	/* current state of AvpGetMediaPlayerItem */
	GetMediaPlayerItemState get_media_player_item_state;
	/* audiovisual info */
	GetMediaPlayerItemInfo *get_media_player_item_info;

	/* current state of AvpGetFolderItem */
	GetFolderItemState get_folder_item_state;
	/* audiovisual info */
	GetFolderItemInfo *get_folder_item_info;

	/* A2DP state tracking needed for audio streaming */
	A2DPInfo a2dp_info;

	struct async_queue_list aql_device;

} DeviceObject;

/* DBus Types */
#define DBUS_STRUCT_STRING_UCHAR (dbus_g_type_get_struct ("GValueArray", \
	G_TYPE_STRING, G_TYPE_UCHAR, G_TYPE_INVALID))
#define DBUS_STRUCT_UCHAR_UCHAR_STRING (dbus_g_type_get_struct ("GValueArray", \
	G_TYPE_UCHAR, G_TYPE_UCHAR, G_TYPE_STRING, G_TYPE_INVALID))
#define DBUS_STRUCT_STRING_STRING_STRING_STRING \
	(dbus_g_type_get_struct ("GValueArray", G_TYPE_STRING, G_TYPE_STRING, \
	G_TYPE_STRING, G_TYPE_STRING, G_TYPE_INVALID))
#define DBUS_STRUCT_STRING_STRING_STRING_STRING_UCHAR \
	(dbus_g_type_get_struct ("GValueArray", G_TYPE_STRING, G_TYPE_STRING, \
	G_TYPE_STRING, G_TYPE_STRING, G_TYPE_UCHAR, G_TYPE_INVALID))
#define DBUS_STRUCT_STRING_UINT (dbus_g_type_get_struct ("GValueArray", \
	G_TYPE_STRING, G_TYPE_UINT, G_TYPE_INVALID))
#define DBUS_UCHAR_ARRAY_ARRAY (dbus_g_type_get_collection ("GPtrArray", \
	dbus_g_type_get_collection ("GArray", G_TYPE_UCHAR)))

/* exported function from node_adapter.c */
extern DeviceObject *find_device_object_from_list(BD_ADDRESS our_address);

