#include <stdlib.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <fstream>
#include <openssl/rsa.h>

#include <openssl/pem.h>
#include <openssl/x509.h>
#include <openssl/err.h>

#include "util/swu_types.h"
#include "util/swu_filesystem.h"
#include "util/swu_certificate.h"
#include "util/swu_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_I_TTFIS_CMD_PREFIX "SWU_"
#define ETG_I_TRACE_CHANNEL    TR_TTFIS_FCSWUPDATE
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SWUPDATE_UTIL
#include "trcGenProj/Header/swu_certificate.cpp.trc.h"
#endif 

namespace swu {


time_t ASN1_GetTimeT(ASN1_TIME* time) {
   struct tm t;
   const char* str = (const char*) time->data;
   size_t i = 0;
   memset(&t, 0, sizeof(t));
   if (time->type == V_ASN1_UTCTIME) {/* two digit year */
      t.tm_year = (str[i++] - '0') * 10;
      t.tm_year += (str[i++] - '0');
      if (t.tm_year < 70)
         t.tm_year += 100;
   } else if (time->type == V_ASN1_GENERALIZEDTIME) {/* four digit year */
      t.tm_year = (str[i++] - '0') * 1000;
      t.tm_year += (str[i++] - '0') * 100;
      t.tm_year += (str[i++] - '0') * 10;
      t.tm_year += (str[i++] - '0');
      t.tm_year -= 1900;
   }
   t.tm_mon = (str[i++] - '0') * 10;
   t.tm_mon += (str[i++] - '0') - 1; // -1 since January is 0 not 1.
   t.tm_mday = (str[i++] - '0') * 10;
   t.tm_mday += (str[i++] - '0');
   t.tm_hour = (str[i++] - '0') * 10;
   t.tm_hour += (str[i++] - '0');
   t.tm_min = (str[i++] - '0') * 10;
   t.tm_min += (str[i++] - '0');
   t.tm_sec = (str[i++] - '0') * 10;
   t.tm_sec += (str[i++] - '0');
   return mktime(&t);
}

bool validateCertificate(const std::string &CACert, const std::string &CodeCert) {
   const char * ptrCertBeginStr = strstr(CodeCert.c_str(), Constants::Cert::CERTIFICATE_BEGIN_STR);
   const char * ptrCertEndStr = strstr(CodeCert.c_str(), Constants::Cert::CERTIFICATE_END_STR);
   if ((0 == ptrCertBeginStr) || (0 == ptrCertEndStr)) {
      ETG_TRACE_ERR(("%s: Can't find CodeCert begin/end certificate", __func__));
      printf("%s: Can't find CodeCertbegin/end certificate\n", __func__);
      return false;
   }
   OpenSSL_add_all_algorithms();
   ERR_load_BIO_strings();
   ERR_load_crypto_strings();

   BIO *bioCert = BIO_new_mem_buf((void *) ptrCertBeginStr, (int) (ptrCertEndStr - ptrCertBeginStr + strlen(Constants::Cert::CERTIFICATE_END_STR)));
   if (0 == bioCert) {
      ETG_TRACE_ERR(("%s: Can't create openssl BIO with certificate", __func__));
      printf("%s: Can't create openssl BIO with certificate\n", __func__);
      return false;
   }
   X509 * cert = PEM_read_bio_X509(bioCert, 0, 0, 0);
   BIO_free_all(bioCert);
   if (0 == cert) {
      ETG_TRACE_ERR(("%s: failed to init X509 cert", __func__));
      printf("%s: failed to init X509 cert\n", __func__);
      return false;
   }
   X509_STORE * store = X509_STORE_new();
   if (0 == store) {
      ETG_TRACE_ERR(("%s: failed to init X509_STORE", __func__));
      printf("%s: failed to init X509_STORE\n", __func__);
      return false;
   }
   char fname[PATH_MAX] = "/tmp/fileXXXXXX";
   int handle = mkstemp(fname); /* Create and open temp file */
   if (0 >= handle) {
      ETG_TRACE_ERR(("%s: Can't create temporary file", __func__));
      printf("%s: Can't create temporary file\n", __func__);
      return false;
   }
   if (CACert.size() != write(handle, CACert.c_str(), CACert.size())) {
      ETG_TRACE_ERR(("%s: Can't write temporary file", __func__));
      printf("%s: Can't write temporary file\n", __func__);
      return false;
   }
   close(handle);
   if (1 != X509_STORE_load_locations(store, fname, NULL)) {
      ETG_TRACE_ERR(("%s: Can't load CACertificate from temp file", __func__));
      printf(("%s: Can't load CACertificate from temp file\n", __func__));
      return false;
   }
   unlink(fname); /* Remove temp file */

   X509_STORE_CTX * vrfy_ctx = X509_STORE_CTX_new();
   if (0 == vrfy_ctx) {
      ETG_TRACE_ERR(("%s: failed to create X509_STORE_CTX", __func__));
      printf("%s: failed to create X509_STORE_CTX\n", __func__);
      return false;
   }
   if (1 != X509_STORE_CTX_init(vrfy_ctx, store, cert, NULL)) {
      ETG_TRACE_ERR(("%s: Can't init X509_STORE_CTX", __func__));
      printf("%s: Can't init X509_STORE_CTX", __func__);
      return false;
   }
   time_t time = ASN1_GetTimeT(X509_get_notBefore(cert));
   X509_STORE_CTX_set_time(vrfy_ctx, 0, time);
   X509_STORE_CTX_set_flags(vrfy_ctx, X509_V_FLAG_USE_CHECK_TIME);
   int ret = X509_verify_cert(vrfy_ctx);
   ETG_TRACE_USR3(("validateCertificate: %s", X509_verify_cert_error_string(vrfy_ctx->error)));
   X509_STORE_CTX_free(vrfy_ctx);

   if (1 != ret) {
      BIO *bioOut = BIO_new(BIO_s_mem());
      if (0 == bioOut) {
         ETG_TRACE_ERR(("%s: Can't create openssl BIO for diag", __func__));
         printf("%s: Can't create openssl BIO for diag\n", __func__);
         return false;
      }
      /*  get the offending certificate causing the failure */
      X509_NAME * certsubject = X509_get_subject_name(cert);
      X509_NAME_print_ex(bioOut, certsubject, 0, XN_FLAG_MULTILINE);
      char *ptr = NULL;
      BIO_get_mem_data(bioOut, &ptr);
      ETG_TRACE_ERR(("validateCertificate: %s", ptr));
      BIO_free_all(bioOut);
   }
   X509_STORE_free(store);
   X509_free(cert);
   return (1 == ret) ? true : false;
}
bool extractPublicKeyFromCertificate(const std::string &certificate, std::string &key) {
   const char * ptrCertBeginStr = strstr(certificate.c_str(), Constants::Cert::CERTIFICATE_BEGIN_STR);
   const char * ptrCertEndStr = strstr(certificate.c_str(), Constants::Cert::CERTIFICATE_END_STR);
   if ((0 == ptrCertBeginStr) || (0 == ptrCertEndStr)) {
      ETG_TRACE_ERR(("%s: Can't find begin/end certificate", __func__));
      return false;
   }
   BIO *bioCert = BIO_new_mem_buf((void *) ptrCertBeginStr, (int) (ptrCertEndStr - ptrCertBeginStr + strlen(Constants::Cert::CERTIFICATE_END_STR)));
   if (0 == bioCert) {
      ETG_TRACE_ERR(("%s: Can't create openssl BIO with certificate", __func__));
      return false;
   }
   X509 *cert = PEM_read_bio_X509(bioCert, 0, 0, 0);
   BIO_free_all(bioCert);
   if (0 == cert) {
      ETG_TRACE_ERR(("%s: Can't create openssl X509 from cert", __func__));
      return false;
   }
   EVP_PKEY *pkey = X509_get_pubkey(cert);
   X509_free(cert);
   if (0 == pkey) {
      ETG_TRACE_ERR(("%s: Can't extract openssl EVP_PKEY from X509", __func__));
      return false;
   }
   char buff[80];
   switch (pkey->type) {
   case EVP_PKEY_RSA:
      sprintf(buff, "%d bit RSA Key\n", EVP_PKEY_bits(pkey));
      break;
   case EVP_PKEY_DSA:
      sprintf(buff, "%d bit DSA Key\n", EVP_PKEY_bits(pkey));
      break;
   default:
      sprintf(buff, "%d bit non-RSA/DSA Key\n", EVP_PKEY_bits(pkey));
      break;
   }
   ETG_TRACE_USR3(("extractPublicKeyFromCertificate: %s", buff));
   BIO *bioKey = BIO_new(BIO_s_mem());
   if (0 == bioKey) {
      ETG_TRACE_ERR(("%s: Can't create openssl BIO for key", __func__));
      return false;
   }
   if (0 == PEM_write_bio_PUBKEY(bioKey, pkey)) {
      ETG_TRACE_ERR(("%s: Can't write openssl EVP_PKEY", __func__));
      return false;
   }
   EVP_PKEY_free(pkey);
   char *ptr = NULL;
   long length = BIO_get_mem_data(bioKey, &ptr);
   if (length > Constants::Mmc::MTD_CERTIFICATE_BUFFER_SIZE) {
      ETG_TRACE_ERR(("%s: EVP_PKEY is too long", __func__));
      return false;
   }
   key.assign(ptr, length);
   BIO_free_all(bioKey);
   return true;
}

bool checkSignature(const unsigned char *msg, size_t msg_len, const unsigned char *signature, size_t signature_len, const std::string & key, bool & verify_result) {
// To understnad how openssl verify works have a look at following:
// http://wiki.openssl.org/index.php/EVP_Signing_and_Verifying
   verify_result = false;

   BIO *bio = BIO_new_mem_buf((void *) key.c_str(), (int) key.size());
   if (!bio) {
      ETG_TRACE_ERR(("Could not build bio from key."));
      return false;
   }
   EVP_PKEY *evp_pub_key;
   RSA *pub_key = PEM_read_bio_RSA_PUBKEY(bio, NULL, NULL, NULL);
   if (!pub_key) {
      ETG_TRACE_ERR(("Could not load public key"));
      return false;
   }
   if (0 == (evp_pub_key = EVP_PKEY_new())) {
      ETG_TRACE_ERR(("Could not create EVP key object"));
      RSA_free(pub_key);
      return false;
   }
   if (1 != EVP_PKEY_set1_RSA(evp_pub_key, pub_key)) {
      ETG_TRACE_ERR(("Could not bring RSA public key into EVP key"));
      RSA_free(pub_key);
      EVP_PKEY_free(evp_pub_key);
      return false;
   }
   RSA_free(pub_key);
// Verify
   EVP_MD_CTX *mdctx;
   if (0 == (mdctx = EVP_MD_CTX_create())) {
      ETG_TRACE_ERR(("Could not create EVP MD Context object"));
      EVP_PKEY_free(evp_pub_key);
      return false;
   }
   if (1 != EVP_DigestVerifyInit(mdctx, NULL, EVP_sha256(), NULL, evp_pub_key)) {
      ETG_TRACE_ERR(("Could not init verify"));
      EVP_PKEY_free(evp_pub_key);
      EVP_MD_CTX_destroy(mdctx);
      return false;
   }
   if (1 != EVP_DigestVerifyUpdate(mdctx, msg, msg_len)) {
      ETG_TRACE_ERR(("Could not update verify context"));
      EVP_PKEY_free(evp_pub_key);
      EVP_MD_CTX_destroy(mdctx);
      return false;
   }
   unsigned char signatureCopy[signature_len];
   memcpy(signatureCopy, signature, signature_len);
   if (1 == EVP_DigestVerifyFinal(mdctx, signatureCopy, (unsigned int) signature_len)) {
      verify_result = true;
   } else {
      ETG_TRACE_ERR(("EVP_DigestVerifyFinal failed"));
      verify_result = false;
   }
   EVP_PKEY_free(evp_pub_key);      // Clean up
   EVP_MD_CTX_destroy(mdctx);
   return true;
}

bool checkSignatureAsserting(const unsigned char *msg, size_t msg_len, const unsigned char *signature, size_t signature_len, const std::string &key) {
   bool result;
   assert(checkSignature(msg, msg_len, signature, signature_len, key, result));
   return result;
}

void TestSignature::TestSignature::vInit() {
   ETG_I_REGISTER_FILE();
}
ETG_I_CMD_DEFINE((testCheckSignature, "testCheckSignature %60s %60s %60s", ETG_I_STRING, ETG_I_STRING, ETG_I_STRING))
void TestSignature::testCheckSignature(const char *messageFile, const char *signatureFile, const char *keyFile) {
   ETG_TRACE_USR4(("swu_filesystem testCheckSignature message file %s", messageFile));
   ETG_TRACE_USR4(("swu_filesystem testCheckSignature signature file %s", signatureFile));
   ETG_TRACE_USR4(("swu_filesystem testCheckSignature key file %s", keyFile));
   string message;
   string signature;
   string key;
   if (!loadFile(messageFile, message)) {
      ETG_TRACE_ERR(("Could not load message file %s", messageFile));
      return;
   }
   if (!loadFile(signatureFile, signature)) {
      ETG_TRACE_ERR(("Could not load signature file %s", signatureFile));
      return;
   }
   if (!loadFile(keyFile, key)) {
      ETG_TRACE_ERR(("Could not load key file %s", keyFile));
      return;
   }
   bool verifyResult;
   if (!checkSignature((const unsigned char *) message.c_str(), message.size(), (const unsigned char *) signature.c_str(), signature.length(), key, verifyResult)) {
      ETG_TRACE_ERR(("Could not trace signature"));
      return;
   }
   if (verifyResult) {
      ETG_TRACE_COMP(("swu_filesystem testCheckSignature checksum valid"));
   } else {
      ETG_TRACE_COMP(("swu_filesystem testCheckSignature checksum not valid"));
   }
}

ETG_I_CMD_DEFINE((testCheckSignatureWithSystemkey, "testCheckSignatureWithSystemkey %60s %60s", ETG_I_STRING, ETG_I_STRING))
void TestSignature::testCheckSignatureWithSystemkey(const char *messageFile, const char *signatureFile) {
   ETG_TRACE_USR4(("swu_filesystem testCheckSignature message file %s", messageFile));
   ETG_TRACE_USR4(("swu_filesystem testCheckSignature signature file %s", signatureFile));
   string message;
   string signature;
   if (!loadFile(messageFile, message)) {
      ETG_TRACE_ERR(("Could not load message file %s", messageFile));
      return;
   }
   if (!loadFile(signatureFile, signature)) {
      ETG_TRACE_ERR(("Could not load signature file %s", signatureFile));
      return;
   }
   bool verifyResult;
   std::string certificate, &certificateRef = certificate;
   bool certificateSet, &certificateSetRef = certificateSet;
#if 0 // peha todo
   if (true != RawCertificate::getCertificate(certificateSetRef, certificateRef)) {
      ETG_TRACE_ERR(("%s: read certificate failed", __func__));
      return;
   }
#endif
   if (false == certificateSetRef) {
      ETG_TRACE_USR3(("%s Certificate is not set, returning true without signature check", __func__));
      return;
   }
   std::string key, &keyRef = key;
   if (false == extractPublicKeyFromCertificate(certificateRef, keyRef)) {
      ETG_TRACE_ERR(("%s: extract key public from certificate failed", __func__));
      return;
   }
   if (!checkSignature((const unsigned char *) message.c_str(), message.size(), (const unsigned char *) signature.c_str(), signature.length(), keyRef, verifyResult)) {
      ETG_TRACE_ERR(("Could not trace signature"));
      return;
   }
   if (verifyResult) {
      ETG_TRACE_COMP(("swu_filesystem testCheckSignature checksum valid"));
   } else {
      ETG_TRACE_COMP(("swu_filesystem testCheckSignature checksum not valid"));
   }
}


}
