/**************************************************************** +MHDR * .MA *
    
 .FILE            [clDestroyMe.h]

 .SW-COMPONENT    [SXL\EXH]

 .DESCRIPTION                                                          
  This file specifies the base class tclDestroyMe. It extends the capabilities 
  of the exception handling mechanismen (EXH) by adding the possibility to 
  define classes which wouldn't be forgotten in case of an exception. That is,
  the constructors of decendants of this class would be called in case of an 
  exception, even if the end of their scope was not properly reached because 
  of throwing an exception.
   
 .AUTOR           [ CM-DI/ENS51 F. Wiedemann                            ]          
 .COPYRIGHT       [ (c) 2001 Robert Bosch GmbH, Hildesheim           ]

 .HISTORY 
   
  Initial revision.
   
**************************************************************** -MHDR * .ME */

#ifndef CLDESTROYME_H
#define CLDESTROYME_H

#ifdef VARIANT_S_FTR_ENABLE_EXH_EXCEPTIONS

// Base class to register stack object at exception handling (general type)
class tclRegistrarType
{
public:
   tclRegistrarType()
      : u8RegistrarType((tU8) enNotRegistered)
   {}

   typedef enum{
      enNotRegistered,
      enRegisteredByDestroyMe,
      enRegisteredByUnwindRegistrar
   } tRegistrarType;

   void vSetRegistrarType(tRegistrarType enType)
   {
      u8RegistrarType = (tU8)enType;
   }
   
   tRegistrarType enGetRegistrarType(void) const
   {
      return (tRegistrarType) u8RegistrarType;
   }

private:
   tU8 u8RegistrarType;
};

// Base class to register stack object with exception handling.
// Definition for registration in use place, using macro EXH_REG().
class tclStackUnwindBase : public exh_tclStackAble<tclStackUnwindBase>
{
public:
   tclStackUnwindBase();
   tclStackUnwindBase(const tclStackUnwindBase&);
   tclStackUnwindBase& operator=(const tclStackUnwindBase&);

   virtual ~tclStackUnwindBase(){}
protected:
   exh_tclTryContext* poGetTryContext() const;
   exh_tclTryContext* poGetTryContext(exh_tclExceptionContext* pECT) const;
   void vRegister(exh_tclTryContext* poTryContext, tclRegistrarType::tRegistrarType enType);
   void vUnRegister(exh_tclTryContext* poTryContext, tclRegistrarType::tRegistrarType enType);
   tBool bValidateType(tclRegistrarType::tRegistrarType enType) const;

   tBool bIsRegisteredForType(tclRegistrarType::tRegistrarType enType) const;
   tBool bIsNotRegistered() const;

   void vSetRegistrar(tclRegistrarType::tRegistrarType enType);
   void vResetRegistrar();
private:
   tclRegistrarType oRegistrar;
};


// Base class to declare a class for stack unwinding. This base has to be
// used at declaration time. (You should not use this any more. Use EXH_REG instead!)
class tclDestroyMe : virtual public tclStackUnwindBase 
{
   friend class exh_test_tclExceptionFlow;

public:

   static const tU8 U8_IS_UNDEFINED;
   static const tU8 U8_IS_HEAP_OBJECT;
   static const tU8 U8_IS_STACK_OBJECT;
   static const tU8 U8_IS_MEMBER_OBJECT;

   tU8 u8GetState() const {return u8State;}

   tU32 u32GetObjectSize() const {return u32ObjectSize;};

   tclDestroyMe(tU32 u32ParamObjectSize);
   tclDestroyMe(const tclDestroyMe& otherObject);
   virtual ~tclDestroyMe();

   tclDestroyMe& operator= (const tclDestroyMe& /*other*/);

   tBool bIsValidObject() const
   {
      return (u8State != U8_IS_UNDEFINED)? TRUE : FALSE;
   }

protected:
   tBool bIsRegistered() const 
   {
      return (u8State == U8_IS_STACK_OBJECT)? TRUE : FALSE;
   }

   tBool bIsAutonomous() const 
   {
      return (    u8State == U8_IS_HEAP_OBJECT 
               || u8State == U8_IS_STACK_OBJECT )? TRUE : FALSE;
   }

   exh_tclObjectDesc oObjDesc;
   tU32 u32ObjectSize;
   tU8 u8State;
private:
   tclDestroyMe(); //lint !e1704
};

// helper template for instantiation-time registration to exception handling
// via macro EXH_REG(). Use EXH_REG() macro instead of this template directly.
template <class TC>
class tclStackUnwindRegistrar : public TC, virtual public tclStackUnwindBase
{
public:
   tclStackUnwindRegistrar()
   {
      // Register to exception handling
      vRegister(poGetTryContext(), tclRegistrarType::enRegisteredByUnwindRegistrar);
   }

   template < class T1 > tclStackUnwindRegistrar(T1 p1)
      : TC(p1)
   {
      // Register to exception handling
      vRegister(poGetTryContext(), tclRegistrarType::enRegisteredByUnwindRegistrar);
   }

   template < class T1, class T2 > tclStackUnwindRegistrar(T1 p1, T2 p2)
      : TC(p1, p2)
   {
      // Register to exception handling
      vRegister(poGetTryContext(), tclRegistrarType::enRegisteredByUnwindRegistrar);
   }

   template < class T1, class T2, class T3 > tclStackUnwindRegistrar(T1 p1, T2 p2, T3 p3)
      : TC(p1, p2, p3)
   {
      // Register to exception handling
      vRegister(poGetTryContext(), tclRegistrarType::enRegisteredByUnwindRegistrar);
   }

   template < class T1, 
              class T2, 
              class T3,
              class T4 > tclStackUnwindRegistrar(T1 p1, 
                                                 T2 p2, 
                                                 T3 p3,
                                                 T4 p4)
      : TC(p1, p2, p3, p4)
   {
      // Register to exception handling
      vRegister(poGetTryContext(), tclRegistrarType::enRegisteredByUnwindRegistrar);
   }

   template < class T1, 
              class T2, 
              class T3,
              class T4,
              class T5 > tclStackUnwindRegistrar(T1 p1, 
                                                 T2 p2, 
                                                 T3 p3,
                                                 T4 p4,
                                                 T5 p5)
      : TC(p1, p2, p3, p4, p5)
   {
      // Register to exception handling
      vRegister(poGetTryContext(), tclRegistrarType::enRegisteredByUnwindRegistrar);
   }


   virtual ~tclStackUnwindRegistrar()
   {
      // Unregister from exception handling
      vUnRegister(poGetTryContext(), tclRegistrarType::enRegisteredByUnwindRegistrar);
   }
};


#define CONSTRUCTDESTROYME(class) tclDestroyMe(sizeof(class))

// registration of stack objects to exception handling
#define EXH_REG(c) tclStackUnwindRegistrar< c >
/*** example usage for EXH_REG() **********************************************
 * 
 * void MyFunction()
 * {
 *    EXH_REG(tclMyClass) oMyObject(Parameter); 
 *    ...
 * }
 *
 *****************************************************************************/ 
#else
class tclDestroyMe 
{
public:
   tclDestroyMe(tU32 /*u32ObjectSize*/){};
   virtual ~tclDestroyMe(){};
};

template <class TC>
class tclStackUnwindRegistrar : public TC
{
};

#define CONSTRUCTDESTROYME(class) tclDestroyMe(sizeof(class))

// registration of stack objects to exception handling
#define EXH_REG(c) c
/*** example usage for EXH_REG() **********************************************
 * 
 * void MyFunction()
 * {
 *    EXH_REG(tclMyClass) oMyObject(Parameter); 
 *    ...
 * }
 *
 *****************************************************************************/ 

#endif
#endif 
