/* NeuQuant Neural-Net Quantization Algorithm Interface
* ----------------------------------------------------
*
* Copyright (c) 1994 Anthony Dekker
*
* NEUQUANT Neural-Net quantization algorithm by Anthony Dekker, 1994.
* See "Kohonen neural networks for optimal colour quantization"
* in "Network: Computation in Neural Systems" Vol. 5 (1994) pp 351-367.
* for a discussion of the algorithm.
* See also  http://members.ozemail.com.au/~dekker/NEUQUANT.HTML
*
* Any party obtaining a copy of these files from the author, directly or
* indirectly, is granted, free of charge, a full and unrestricted irrevocable,
* world-wide, paid up, royalty-free, nonexclusive right and license to deal
* in this software and documentation files (the "Software"), including without
* limitation the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons who receive
* copies from any such party to do so, with the only requirement being
* that this copyright notice remain intact.
*/

// Source code from members.ozemail.com.au / ~dekker / NEUQUANT.H has been modified for integration by
// Socionext Embedded Software Austria (SESA)

#if !defined(NEU_QUANT_H)
#define NEU_QUANT_H

#include <CanderaPlatform/OS/CanderaTypes.h>

namespace Candera { namespace ClutBitmapConverter {

    class NeuQuant {

    public:
        /* Main Learning Loop
        ------------------ */
        void Learn();

        /* Initialise network in range (0,0,0) to (255,255,255) and set parameters
        ----------------------------------------------------------------------- */
        void InitNet(UInt8 *thepic, Int32 len, Int32 sample);

        /* Unbias network to give byte values 0..255 and record position i to prepare for sort
        ----------------------------------------------------------------------------------- */
        void UnbiasNet();    /* can edit this function to do output of color map */

        /* Output color map
        ----------------- */
        void WriteColorMap(UInt8* clut) const;

        NeuQuant(Int32 colorCount);
        ~NeuQuant();

        /* Program Skeleton
        ----------------
        [select samplefac in range 1..30]
        pic = (UInt8*) malloc(3*width*height);
        [read image from input file into pic]
        InitNet(pic,3*width*height,samplefac);
        Learn();
        UnbiasNet();
        [write output image header, using WriteColorMap(f),
        possibly editing the loops in that function]
        inxbuild();
        [write output image using InxSearch(b,g,r)]     */

    private:
        Int32 m_netSize;   /* number of colors used */

        /* For 256 colors, fixed arrays need 8kb, plus space for the image
        ---------------------------------------------------------------- */

        /* four primes near 500 - assume no image has a length so large */
        /* that it is divisible by all four primes */
        static const Int32 c_prime1 = 499;
        static const Int32 c_prime2 = 491;
        static const Int32 c_prime3 = 487;
        static const Int32 c_prime4 = 503;

        static const Int32  c_minPictureBytes = 3 * c_prime4; /* minimum size for input image */

        /* Network Definitions
        ------------------- */

        Int32 m_maxNetPos;
        UInt32 m_netBiasShift;    /* bias for color values */
        Int32 m_nCycles;         /* no. of learning cycles */

        /* defs for freq and bias */
        UInt32 m_intBiasShift;    /* bias for fractions */
        Int32 m_intBias;
        UInt32 m_gammaShift;
        Int32 m_gamma;
        UInt32 m_betaShift;
        Int32 m_beta;            /* beta = 1/1024 */
        Int32 m_betaGamma;

        /* defs for decreasing radius factor */
        Int32 m_initRad;             /* for 256 cols, radius starts */
        UInt32 m_radiusBiasShift;     /* at 32.0 biased by 6 bits */
        Int32 m_radiusBias;
        Int32 m_initRadius;          /* and decreases by a */
        Int32 m_radiusDec;           /* factor of 1/30 each cycle */

        /* defs for decreasing alpha factor */
        UInt32 m_alphaBiasShift;      /* alpha starts at 1.0 */
        Int32 m_initAlpha;
        Int32 m_alphaDec;               /* biased by 10 bits */

        /* radbias and alpharadbias used for radpower calculation */
        UInt32 m_radBiasShift;
        Int32 m_radBias;
        UInt32 m_alphaRadBiasShift;
        Int32 m_alphaRadBias;

        /* Types and Global Variables
        -------------------------- */

        UInt8* m_thePicture;  /* the input image itself */
        Int32 m_lengthCount;            /* lengthcount = H*W*3 */

        Int32 m_sampleFac;              /* sampling factor 1..30 */


        typedef struct Pixel {
            Int32 item[4];       /* BGRc */
        } Pixel;

        Pixel* m_network;             /* the network itself */

        Int32* m_netIndex;              /* for network lookup - really 256 */

        Int32* m_bias;                  /* bias and freq arrays for learning */
        Int32* m_freq;
        Int32* m_radPower;              /* radpower for precomputation */

        /* Insertion sort of network and building of netindex[0..255] (to do after unbias)
        ------------------------------------------------------------------------------- */
        //void inxbuild();

        /* Search for BGR values 0..255 (after net is unbiased) and return color index
        ---------------------------------------------------------------------------- */
        //int InxSearch(int b,  int g,  int r);

        void AlterNeighbours(Int32 rad, Int32 i, Int32  b, Int32  g, Int32  r);

        void AlterSingle(Int32 alpha, Int32 i, Int32 b, Int32 g, Int32 r);

        Int32 Contest(Int32 b, Int32 g, Int32 r);

    };
}
}

#endif
