/*
 * node_manager.c
 * Manager node handling for the D-bus Bluetooth Daemon
 *
 * Author: Dean Jenkins <djenkins@mvista.com>
 *
 * 2010 (c) MontaVista Software, LLC. This file is licensed under
 * the terms of the AFL.
 */

#ifdef USE_GPIO_RESET_CTRL
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdio.h>
#endif
#include <string.h>
#include <sched.h>
#include <stdlib.h>

#include <glib.h>
#include <glib-object.h>
#include <glib/gprintf.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>

#include "node_manager.h"
#include "main.h"

#include "bt_appl/common_def.h"
#include "bt_appl/bt_appl_version.h"
#include "bt_appl/api/bt_appl_bt_setting_block.h"
#include "bt_appl/api/bt_appl_config_mgr_block.h"
#include "bt_appl_dummy.h"
#include "callbacks.h"

#include "dbus_error.h"
#include "debug.h"

/****************/
/* LOCAL GLOBAL */
/****************/

static guint16 initialised_alps_version = 0;
static guint8 initialised_alps_test_version = 0;

/***********************/
/* MANAGER DEFINITIONS */
/***********************/

/* OUR MANAGER OBJECT DEFINITION */
typedef struct {
	/* The parent class object state. */
	GObject parent;

} ManagerObject;

/* OUR CLASS DEFINITION */
typedef struct {
	/* The parent class state. */
	GObjectClass parent;

	/* put manager signal stuff here, (if any) */

} ManagerObjectClass;

/*
 * MAGIC here to define our manager object functions etc. in GLib
 * declares manager_object_init(), manager_object_class_init(),
 * manager_object_get_type()
 */
G_DEFINE_TYPE(ManagerObject, manager_object, G_TYPE_OBJECT)

/*
 * Forward declaration of the function that will return the GType of
 * the Value implementation. Not used
 */
GType manager_object_get_type(void);

/*
 * Macro for the above. It is common to define macros using the
 * naming convention (seen below) for all GType implementations.
 */
#define MANAGER_TYPE_OBJECT	(manager_object_get_type())

/*********************/
/* MANAGER CALLBACKS */
/*********************/

/*
 * callback handling, parameters needs to match marshalling signatures
 *
 * callbacks for the auto generated method protoypes
 * Forward declare the prototypes of the callback method handlers for the
 * dbus_glib_adapter_object_methods[]
 */

gboolean manager_object_initialization(ManagerObject *object,
					gchar **dbus_bt_daemon_version_out,
					guint16 *alps_version_out,
					GByteArray **local_address_out,
					GError **error);

gboolean manager_object_shutdown(ManagerObject *object, GError **error);

gboolean manager_object_set_manager_configuration(ManagerObject *object,
						guint16 config_id_in,
						guint16 config_length_in,
						guint32 data_u32_in,
						GError **error);

gboolean manager_object_read_manager_configuration(ManagerObject *object,
						guint16 config_id_in,
						guint16 *config_length_out,
						guint32 *data_u32_out,
						GError **error);

gboolean manager_object_set_configuration_word(ManagerObject *object,
						guint32 config_id_in,
						guint32 config_length_in,
						guint32 data_u32_in,
						GError **error);

gboolean manager_object_read_configuration_word(ManagerObject *object,
						guint32 config_id_in,
						guint32 *config_length_out,
						guint32 *data_u32_out,
						GError **error);

gboolean manager_object_set_configuration_raw(ManagerObject *object,
						guint32 config_id_in,
						GByteArray *raw_bytes_in,
						GError **error);

gboolean manager_object_read_configuration_raw(ManagerObject *object,
						guint32 config_id_in,
						GByteArray **raw_bytes_out,
						GError **error);

gboolean manager_object_set_debug_levels(ManagerObject *object,
						guint8 btd_log_level_in,
						guint8 fusion_log_level_in,
						GError **error);

gboolean manager_object_read_debug_levels(ManagerObject *object,
						guint8 *btd_log_level_out,
						guint8 *fusion_log_level_out,
						GError **error);

/* include auto-generated binding information */
#include "alps-bt-manager-introspection.h"


/***********************************/
/* OBJECT AND CLASS INITIALIZATION */
/***********************************/


/*
 * ManagerObject object initialiser function
 *
 * Used to initialise each object instance via g_object_new()
 */
static void manager_object_init(ManagerObject *object)
{
	DEBUG_FUNC("Called");
	g_assert(object != NULL);

	/* add code to initialise created object contents */
}

/*
 * ManagerObject class initialiser function
 *
 * Used to initialise each class via the first call of g_object_new()
 */
static void manager_object_class_init(ManagerObjectClass *klass)
{
#ifdef NOT_USED
	GObjectClass *gobject_class = G_OBJECT_CLASS(klass);
#endif

	DEBUG_FUNC("Called");
	g_assert(klass != NULL);

	/* create new signals here, if needed */

	DEBUG_HIGH("Binding our GObject with D-bus...");
	/* Uses auto-generated methods and signal structure */
	dbus_g_object_type_install_info(MANAGER_TYPE_OBJECT,
		&dbus_glib_manager_object_object_info);

}


/* LOCAL GLOBAL */
/* manager object pointer */
static ManagerObject *manager_object = NULL;


/*
 * create a new object
 *
 * Inputs:
 * bus = point to the bus proxy
 * Exported function
 */
gboolean manager_object_creation(DBusGConnection *bus, gchar *node_name)
{
	DEBUG_HIGH("Creating an instance of the ManagerObject"
 				" and calling initialise functions...");

	manager_object = g_object_new(MANAGER_TYPE_OBJECT, NULL);
	if(manager_object == NULL) {
		DEBUG_ERROR("Failed to create an instance of a ManagerObject");
		return FALSE;
	}

	DEBUG_HIGH("Registering the manager GObject onto the D-Bus");

	/* the object's path does not come from the XML file */
	dbus_g_connection_register_g_object(bus,
		node_name,	/* Object's path for the manager node */
		G_OBJECT(manager_object));

	return TRUE;
}


/*****************************/
/* METHOD CALLBACK FUNCTIONS */
/*****************************/

static void alps_bt_set_scheduling_priority(guint thread_priority)
{
	gint err;
	struct sched_param sched_param;

	err = sched_getparam(0, &sched_param);
	if (err < 0) {
		DEBUG("sched_getparam error %d", err);
		return;
	}

	sched_param.sched_priority = thread_priority;
	err = sched_setscheduler(0, SCHED_RR, &sched_param);
	if (!err) {
		DEBUG("Priority of Alps initialization thread set to "
			"SCHED_RR %d", sched_param.sched_priority);
	} else {
		DEBUG("sched_setscheduler error %d when attempting to set "
			"priority SCHED_RR %d",
			   err, sched_param.sched_priority);
	}
}

struct alps_bt_thread_creation_args {
	BTHRESULT result;	/* return value */
	guint thread_priority;	/* input */
	guint16 version;	/* output */
	BD_ADDRESS bd_addr;	/* output */
};

static gpointer alps_bt_creation_thread(gpointer data)
{
	struct alps_bt_thread_creation_args *bt_args;

	bt_args = (struct alps_bt_thread_creation_args *) data;

	alps_bt_set_scheduling_priority(bt_args->thread_priority);

#ifndef USE_DUMMY_API
	/*
	 * Call BTHRESULT BT_APPL_Initialization_REQ(OUT u16 *pusVersion,
	 *					 OUT BD_ADDRESS aucBD_ADDR);
	 */
	bt_args->result = BT_APPL_Initialization_REQ(&bt_args->version,
							&bt_args->bd_addr[0]);
#else
	/*
	 * Call BTHRESULT BT_APPL_Initialization_REQ_DUMMY(OUT u16 *pusVersion,
	 *					 OUT BD_ADDRESS aucBD_ADDR);
	 */
	bt_args->result = BT_APPL_Initialization_REQ_DUMMY(&bt_args->version,
							&bt_args->bd_addr[0]);
#endif

	return (gpointer) bt_args;
}

#ifdef USE_GPIO_RESET_CTRL
/*
 * GPIO reset line control for the Bluetooth chip
 *
 * This function is only appropriate for the target hardware
 */
void reset_cycle_bluetooth_chip(void)
{
	FILE *fd;
	guint16 error_code = 0;
	struct stat sts;

	DEBUG("Performing reset cycle of the Bluetooth chip...");

	DEBUG_HIGH("is the GPIO line already exported ?");

	if (!((stat("/sys/class/gpio/gpio36", &sts) != -1) &&
						S_ISDIR(sts.st_mode))) {

		/* no directory so try to enable the GPIO */
		DEBUG_HIGH("enabling the GPIO reset line");

		fd = fopen("/sys/class/gpio/export", "wb");

		if (fd == NULL) {
			error_code = 1;
			DEBUG("Reset cycle failed with code: %d", error_code);
			return;
		}

		fwrite("36", sizeof(char), 3, fd);

		fclose(fd);
	}

	DEBUG_HIGH("setting GPIO direction to output");

	fd = fopen("/sys/class/gpio/gpio36/direction", "wb");

	if (fd == NULL) {
		error_code = 2;
		DEBUG("Reset cycle failed with code: %d", error_code);
		return;
	}

	fwrite("out", sizeof(char), 3, fd);

	fclose(fd);
#if 1 //Testing for Audio issue, Enabling now requested by volker
	DEBUG_HIGH("setting GPIO reset line to High for testing");

	   fd = fopen("/sys/class/gpio/gpio36/value", "wb");

	   if (fd == NULL) {
	      error_code = 3;
	      DEBUG("Reset cycle failed with code: %d", error_code);
	      return;
	   }

	   fwrite("1", sizeof(char), 1, fd);

	   fclose(fd);

	   DEBUG_HIGH("keep reset High for 100ms then performs usal intialization by doing one more reset");
	   usleep(100000);
#endif

	DEBUG_HIGH("setting GPIO reset line to low");

	fd = fopen("/sys/class/gpio/gpio36/value", "wb");

	if (fd == NULL) {
		error_code = 3;
		DEBUG("Reset cycle failed with code: %d", error_code);
		return;
	}

	fwrite("0", sizeof(char), 1, fd);

	fclose(fd);

	/* FIXME: Reduce this period to a minimum */
	DEBUG_HIGH("keep reset low for 100ms");
	usleep(100000);

	DEBUG_HIGH("setting GPIO reset line to high");

	fd = fopen("/sys/class/gpio/gpio36/value", "wb");

	if (fd == NULL) {
		error_code = 4;
		DEBUG("Reset cycle failed with code: %d", error_code);
		return;
	}

	fwrite("1", sizeof(char), 1, fd);

	fclose(fd);

	DEBUG_FUNC("Exited");

	return;
}
#endif

BTHRESULT perform_bluetooth_initialisation(guint16 *alps_version_out,
						BD_ADDRESS temp_address,
						gboolean do_not_reset_bt_chip,
						gint attempt)
{
	BTHRESULT result;
	guint16 local_version;

#ifdef USE_GPIO_RESET_CTRL
	/*
	 * Before calling the initialization function
	 * perform a reset cycle of the Bluetooth chip
	 *
	 * afterwards, call the initialization function anyway
	 * so that error handling is performed
	 */

	if (do_not_reset_bt_chip == FALSE) {
		reset_cycle_bluetooth_chip();
	} else {
		DEBUG("Not performing chip reset cycle");
	}
#endif

	{
		struct alps_bt_thread_creation_args *bt_args;
		GThread *thread;
		GError *thread_error = NULL;

		bt_args = g_new0(struct alps_bt_thread_creation_args, 1);
		/* should use a low real-time priority */
		bt_args->thread_priority = get_alps_thread_priority();	
		thread =
		    g_thread_create(alps_bt_creation_thread, bt_args, TRUE,
				    &thread_error);
		if (!thread) {
			/* fatal error -- terminate the daemon */
			DEBUG_ERROR("Can't create alps_bt_creation_thread: %s",
				 	thread_error->message);
			g_error_free(thread_error);
			exit(1);
		} else {
			bt_args = (struct alps_bt_thread_creation_args *)
				g_thread_join(thread);
		}
		result = bt_args->result;
		local_version = bt_args->version;
		memcpy(temp_address, bt_args->bd_addr, sizeof(BD_ADDRESS));
		g_free(bt_args);
	}

	/* do test for alps test build libraries */
	initialised_alps_test_version = (local_version & 0xF000) >> 12;

	if (initialised_alps_test_version != 0) {
		DEBUG("DETECTED: ALPS test build libraires: %u",
						initialised_alps_test_version);

		DEBUG("raw version number: 0x%04X", local_version);
	}

	/* suppress the test build # */
	local_version = local_version & 0x0FFF;
	initialised_alps_version = local_version;

	if (test_alps_version_check() == TRUE) {
		/*
		 * check that the daemon is compatible
	         * with the detected ALPS BT stack version
	         */
		DEBUG("ALPS BT stack version is %u.%02u",
			local_version/256, local_version%256);

		if ((local_version >= MIN_ALPS_LIB) &&
			local_version <= MAX_ALPS_LIB) {
			DEBUG("ALPS BT version OK >= %u.%02u and <= %u.%02u",
							MIN_ALPS_LIB/256,
							MIN_ALPS_LIB%256,
							MAX_ALPS_LIB/256,
							MAX_ALPS_LIB%256);
		} else {
			DEBUG_ERROR("This daemon is incompatible with "
				"the %u.%02u libraries. Is compatible with "
				">= %u.%02u and <= %u.%02u",
						local_version/256,
						local_version%256,
						MIN_ALPS_LIB/256,
						MIN_ALPS_LIB%256,
						MAX_ALPS_LIB/256,
						MAX_ALPS_LIB%256);

			if (result == BT_APPL_SUCCESS) {
				/* spoof an ALPS failure, report back failure */
				result = BT_APPL_ERROR_PARAMETERS;
			}
		}
	} else {
		/*
		 * The ALPS libraries check is disabled
		 * so report ALPS version
		 */
		if (initialised_alps_test_version == 0) {
			g_printf("ALPS BT stack version is %u.%02u\n",
				local_version/256, local_version%256);
		} else {
			g_printf("ALPS BT stack version is %u.%02u: "
				"test build %u\n",
				local_version/256,
				local_version%256,
				initialised_alps_test_version);
		}
	}

	*alps_version_out = local_version;

	if(result == BT_APPL_SUCCESS) {
		DEBUG("BT stack function returned %d, %s",
			result, lookup_bthresult_str(result));
	} else {
		DEBUG_ERROR("ALPS BT stack initialization failed: "
				"attempt %d: %d, %s",
				attempt, result, lookup_bthresult_str(result));
	}

	return result;
}

/* prevent re-initialisation if already initialised */
static gboolean already_initialised = FALSE;
static BD_ADDRESS temp_address = { 0 };	/* contains initialised address */

/* exported function */
guint16 get_alps_stack_version(void)
{
	return initialised_alps_version;
}

/* exported function */
guint8 get_alps_stack_test_version(void)
{
	return initialised_alps_test_version;
}

/*
 * Method call for "Initialization"
 */
gboolean manager_object_initialization(ManagerObject *object,
					gchar **dbus_bt_daemon_version_out,
					guint16 *alps_version_out,
					GByteArray **local_address_out,
					GError **error)
{
	gint init_fail_retry_count = read_max_init_count();

	BTHRESULT result = BT_APPL_SUCCESS;

	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(dbus_bt_daemon_version_out != NULL);
	g_assert(alps_version_out != NULL);
	g_assert(local_address_out != NULL);

	/* initialise the ALPS version variable */	
	*alps_version_out = 0x0;

	/* create a new byte array for the BT address */
	*local_address_out = g_byte_array_new();

	/* indicate the version of the D-bus Bluetooth Daemon */
	/* FIXME: hope it is correctly freed */
	*dbus_bt_daemon_version_out =
				(gchar *) g_malloc0(sizeof(D_BUS_BT_VERSION));

	g_sprintf(*dbus_bt_daemon_version_out, D_BUS_BT_VERSION);

	DEBUG("%s", *dbus_bt_daemon_version_out);

	/* intercept already initialised system */
	if (already_initialised == TRUE) {
		/* spoof success result with alps version and local address */

		DEBUG_WARN("ALPS stack is already initialised");

		/* save the BT address into the byte array */
		*local_address_out = g_byte_array_append(*local_address_out,
							&temp_address[0],
							sizeof(temp_address));

		*alps_version_out = initialised_alps_version;

		return_code = report_any_error_to_dbus(result,
						"BT_APPL_Initialization_REQ",
						error);

		DEBUG_FUNC("Exited spoofed success result");

		return return_code;
	}

	/* do loop to retry on failure */
	do {
		result = perform_bluetooth_initialisation(alps_version_out,
				&temp_address[0],
				FALSE,
				read_max_init_count() - init_fail_retry_count + 1);

		if (result == BT_APPL_ERROR_OPERATION_FAILED) {
			/* try doing a shutdown before reinitialising */
			DEBUG("Initialisation failed so shutdown and re-try");

			/* Call to BTHRESULT BT_APPL_Shutdown_REQ(void); */
#ifndef USE_DUMMY_API
			BT_APPL_Shutdown_REQ();
#else
			BT_APPL_Shutdown_REQ_DUMMY();
#endif
		}

		init_fail_retry_count--;

	} while ((init_fail_retry_count != 0) &&
		(result != BT_APPL_SUCCESS));

	/* end of re-try do loop */

	if (result == BT_APPL_SUCCESS) {
		DEBUG_HIGH("Rememeber initialisation was successful");
		already_initialised = TRUE;
	} else {
		DEBUG_ERROR("FATAL ERROR: ALPS BT stack failed to initialize");
	}

	/* save the BT address into the byte array */
	*local_address_out = g_byte_array_append(*local_address_out,
							&temp_address[0],
							sizeof(temp_address));

	return_code = report_any_error_to_dbus(result,
						"BT_APPL_Initialization_REQ",
						error);

	if (return_code == TRUE) {
#ifndef USE_DUMMY_API
		result = register_alps_callbacks();
#else
		result = register_alps_callbacks_DUMMY();
#endif	
		if (result != BT_APPL_SUCCESS) {
			/* do a shutdown as clallback registration failed */
			DEBUG("Callback registration failed so shutdown");

			/* Call to BTHRESULT BT_APPL_Shutdown_REQ(void); */
#ifndef USE_DUMMY_API
			BT_APPL_Shutdown_REQ();
#else
			BT_APPL_Shutdown_REQ_DUMMY();
#endif

			/* inidicate initialisation is cancelled */
			already_initialised = FALSE;
		}

		return_code = report_any_error_to_dbus(result,
						"BT_APPL_Register_Callback_REQ",
						error);
	}

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call for "Shutdown"
 */
gboolean manager_object_shutdown(ManagerObject *object, GError **error)
{
	BTHRESULT result;

	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);

	/* Call to BTHRESULT BT_APPL_Shutdown_REQ(void); */
#ifndef USE_DUMMY_API
	result = BT_APPL_Shutdown_REQ();
#else
	result = BT_APPL_Shutdown_REQ_DUMMY();
#endif

	DEBUG("BT stack function returned");

	if (result != BT_APPL_ERROR_CONNECTION_ALREADY_EXISTED) {
		/* Don't clear the initialisation flag on a shutdown error */

		DEBUG_HIGH("Forget initialisation was successful");
		already_initialised = FALSE;
	}

	return_code = report_any_error_to_dbus(result,	"BT_APPL_Shutdown_REQ",
									error);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call for "SetManagerConfiguration"
 */
gboolean manager_object_set_manager_configuration(ManagerObject *object,
						guint16 config_id_in,
						guint16 config_length_in,
						guint32 data_u32_in,
						GError **error)
{
	BTHRESULT result;

	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);

	DEBUG_WARN("SetManagerConfiguration D-bus method is DEPRECIATED, "
			"use SetConfigurationWord or SetConfigurationRaw");

	DEBUG("set request is ID 0x%04X, len %u, data 0x%08X",
		config_id_in, config_length_in, data_u32_in);

	/*
	 * Call BTHRESULT BT_APPL_Set_Configuration_REQ(IN u16 usConfigId,
	 *						IN u16 usConfigDataLen,
	 *						IN u8 *pucConfigData);
	 *
	 * FIXME: resolve usage of (u8 *)
	 * WARNING: Casting the u32 data parameter to a u8 data pointer,
	 * endian issue ?
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_Set_Configuration_REQ(config_id_in, config_length_in,
							(u8 *) &data_u32_in);
#else
	result = BT_APPL_Set_Configuration_REQ_DUMMY(config_id_in,
							config_length_in,
							(u8 *) &data_u32_in);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
						"BT_APPL_Set_Configuration_REQ",
						error);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call for "ReadManagerConfiguration"
 */
gboolean manager_object_read_manager_configuration(ManagerObject *object,
						guint16 config_id_in,
						guint16 *config_length_out,
						guint32 *data_u32_out,
						GError **error)
{
	BTHRESULT result;

	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(config_length_out != NULL);
	g_assert(data_u32_out != NULL);

	DEBUG_WARN("ReadManagerConfiguration D-bus method is DEPRECIATED, "
			"use ReadConfigurationWord or ReadConfigurationRaw");

	/* initialise parameters */
	*config_length_out = 0;
	*data_u32_out = 0;

	DEBUG("read request is ID 0x%X", config_id_in);

	/*
	 * Call to BTHRESULT BT_APPL_Read_Configuration_REQ(IN u16 usConfigId,
	 *					OUT u16 *pusConfigDataLen,
	 *					OUT u8 *pucConfigData);
	 *
	 * FIXME: resolve usage of (u8 *)
	 * WARNING: Casting the u32 data parameter to a u8 data pointer,
	 * endian issue ?
	 *
	 * Cannot prevent the ALPS stack overflowing data_u32_out
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_Read_Configuration_REQ(config_id_in,
							config_length_out,
							(u8 *) data_u32_out);
#else
	result = BT_APPL_Read_Configuration_REQ_DUMMY(config_id_in,
							config_length_out,
							(u8 *) data_u32_out);
#endif

	DEBUG("BT stack function returned");

	DEBUG("read request is ID 0x%04X, len %u, data 0x%08X",
		config_id_in, *config_length_out, *data_u32_out);

	return_code = report_any_error_to_dbus(result,
						"BT_APPL_Set_Configuration_REQ",
						error);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call for "SetConfigurationWord"
 */
gboolean manager_object_set_configuration_word(ManagerObject *object,
						guint32 config_id_in,
						guint32 config_length_in,
						guint32 data_u32_in,
						GError **error)
{
	BTHRESULT result;

	gboolean return_code = FALSE;
	guint8 alps_write_buffer[4];

	DEBUG_FUNC("Called");

	g_assert(object != NULL);

	DEBUG("set request is ID 0x%04X, len %u, data 0x%08X",
		config_id_in, config_length_in, data_u32_in);

	/*
	 * Call BTHRESULT BT_APPL_Set_Configuration_REQ(IN u16 usConfigId,
	 *						IN u16 usConfigDataLen,
	 *						IN u8 *pucConfigData);
	 *
	 * (u8 *) format is big endian so need to do a conversion
	 */

	/*
	 * ALPS API is non-portable
	 * now do host endian (little endian for ARM and x86) to big endian
	 */
	switch (config_length_in) {
	case 1:
		alps_write_buffer[0] = (guint8) (data_u32_in &0xFF);
		break;
	case 2:
		alps_write_buffer[0] = (guint8) ((data_u32_in >> 8) &0xFF);
		alps_write_buffer[1] = (guint8) (data_u32_in &0xFF);
		break;
	case 4:
		alps_write_buffer[0] = (guint8) ((data_u32_in >> 24) &0xFF);
		alps_write_buffer[1] = (guint8) ((data_u32_in >> 16) &0xFF);
		alps_write_buffer[2] = (guint8) ((data_u32_in >> 8) &0xFF);
		alps_write_buffer[3] = (guint8) (data_u32_in &0xFF);
		break;
	default:
		DEBUG_ERROR("Passed an invalid length %u", config_length_in);
	}

#ifndef USE_DUMMY_API
	result = BT_APPL_Set_Configuration_REQ((u16) config_id_in,
						(u16) config_length_in,
						(u8 *) &alps_write_buffer[0]);
#else
	result = BT_APPL_Set_Configuration_REQ_DUMMY((u16) config_id_in,
						(u16) config_length_in,
						(u8 *) &alps_write_buffer[0]);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
						"BT_APPL_Set_Configuration_REQ",
						error);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call for "ReadConfigurationWord"
 */
gboolean manager_object_read_configuration_word(ManagerObject *object,
						guint32 config_id_in,
						guint32 *config_length_out,
						guint32 *data_u32_out,
						GError **error)
{
	BTHRESULT result;

	gboolean return_code = FALSE;
	guint16 alps_read_length;
	guint8 alps_read_buffer[4];

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(config_length_out != NULL);
	g_assert(data_u32_out != NULL);

	/* initialise parameters */
	*config_length_out = 0;
	*data_u32_out = 0;

	DEBUG("read request is ID 0x%X", config_id_in);

	/*
	 * Call to BTHRESULT BT_APPL_Read_Configuration_REQ(IN u16 usConfigId,
	 *					OUT u16 *pusConfigDataLen,
	 *					OUT u8 *pucConfigData);
	 *
	 * (u8 *) format is big endian so need to do a conversion
	 *
	 * Cannot prevent the ALPS stack overflowing data_u32_out
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_Read_Configuration_REQ(config_id_in,
						&alps_read_length,
						(u8 *) &alps_read_buffer[0]);
#else
	result = BT_APPL_Read_Configuration_REQ_DUMMY(config_id_in,
						&alps_read_length,
						(u8 *) &alps_read_buffer[0]);
#endif

	/* Unfortunately the ALPS stack can overflow data_u32_out */
	/* Therefore, die if an overflow occurred */
	g_assert(alps_read_length <= sizeof(guint32));

	*config_length_out = (guint8) alps_read_length;

	DEBUG("BT stack function returned");

	/*
	 * ALPS API is non-portable
	 * now do big endian to host endian (little endian for ARM and x86)
	 */
	switch (alps_read_length) {
	case 1:
		*data_u32_out = (guint32) alps_read_buffer[0];
		break;
	case 2:
		*data_u32_out = (guint32) ((alps_read_buffer[0] << 8) |
					(alps_read_buffer[1]));
		break;
	case 4:
		*data_u32_out = (guint32) ((alps_read_buffer[0] << 24) |
					(alps_read_buffer[1] << 16) |
					(alps_read_buffer[2] << 8) |
					(alps_read_buffer[3]));
		break;
	default:
		DEBUG_ERROR("Passed an invalid length %u", alps_read_length);
	}

	DEBUG("read request is ID 0x%04X, len %u, data 0x%08X",
		config_id_in, *config_length_out, *data_u32_out);

	return_code = report_any_error_to_dbus(result,
						"BT_APPL_Set_Configuration_REQ",
						error);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call for "SetConfigurationRaw"
 */
gboolean manager_object_set_configuration_raw(ManagerObject *object,
						guint32 config_id_in,
						GByteArray *raw_bytes_in,
						GError **error)
{
	BTHRESULT result;

	gboolean return_code = FALSE;

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(raw_bytes_in != NULL);

	DEBUG("set request is ID 0x%04X, len %u",
		config_id_in, raw_bytes_in->len);

	DEBUG_DUMP_ARRAY((guint32) raw_bytes_in->len, &raw_bytes_in->data[0]);

	/*
	 * Call BTHRESULT BT_APPL_Set_Configuration_REQ(IN u16 usConfigId,
	 *						IN u16 usConfigDataLen,
	 *						IN u8 *pucConfigData);
	 *
	 * In raw mode avoid the endianness issue by using a byte array
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_Set_Configuration_REQ((u16) config_id_in,
						(u16) raw_bytes_in->len,
						(u8 *) &raw_bytes_in->data[0]);
#else
	result = BT_APPL_Set_Configuration_REQ_DUMMY((u16) config_id_in,
						(u16) raw_bytes_in->len,
						(u8 *) &raw_bytes_in->data[0]);
#endif

	DEBUG("BT stack function returned");

	return_code = report_any_error_to_dbus(result,
						"BT_APPL_Set_Configuration_REQ",
						error);

	DEBUG_FUNC("Exited");

	return return_code;
}

#define MAX_READ_LENGTH (16)
/*
 * Method call for "ReadConfigurationRaw"
 */
gboolean manager_object_read_configuration_raw(ManagerObject *object,
						guint32 config_id_in,
						GByteArray **raw_bytes_out,
						GError **error)
{
	BTHRESULT result;

	gboolean return_code = FALSE;

	/* use a local buffer as don't know the length of data in advance */
	guint16 alps_read_length = 0;
	guint8 alps_read_buffer[MAX_READ_LENGTH];

	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	g_assert(raw_bytes_out != NULL);

	DEBUG("read request is ID 0x%X", config_id_in);

	/*
	 * Call to BTHRESULT BT_APPL_Read_Configuration_REQ(IN u16 usConfigId,
	 *					OUT u16 *pusConfigDataLen,
	 *					OUT u8 *pucConfigData);
	 *
	 * In raw mode avoid the endianness issue by using a byte array
	 *
	 * However, cannot prevent the ALPS stack overflowing the buffer
	 */
#ifndef USE_DUMMY_API
	result = BT_APPL_Read_Configuration_REQ(config_id_in,
						&alps_read_length,
						(u8 *) &alps_read_buffer[0]);
#else
	result = BT_APPL_Read_Configuration_REQ_DUMMY(config_id_in,
						&alps_read_length,
						(u8 *) &alps_read_buffer[0]);
#endif

	/* Unfortunately the ALPS stack can overflow alps_read_buffer */
	/* Therefore, die if an overflow occurred */
	g_assert(alps_read_length <= MAX_READ_LENGTH);

	DEBUG("BT stack function returned");

	/* create an empty byte array */
	*raw_bytes_out = g_byte_array_new();

	/* copy the temp buffer into the gbyte array for the method reply */
	*raw_bytes_out = g_byte_array_append(*raw_bytes_out,
						alps_read_buffer,
						alps_read_length);

	DEBUG("read request is ID 0x%04X, len %u",
		config_id_in, (*raw_bytes_out)->len);

	DEBUG_DUMP_ARRAY((guint32) (*raw_bytes_out)->len,
				&(*raw_bytes_out)->data[0]);

	return_code = report_any_error_to_dbus(result,
						"BT_APPL_Set_Configuration_REQ",
						error);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * Method call for "SetDebugLevels"
 */
gboolean manager_object_set_debug_levels(ManagerObject *object,
						guint8 btd_log_level_in,
						guint8 fusion_log_level_in,
						GError **error)
{
	DEBUG_FUNC("Called");

	g_assert(object != NULL);
	(void)error;

	btd_debug_level = (btd_log_level_in <= BTD_DBG_MAX_LEV) ?
				(gint) btd_log_level_in: BTD_DBG_MAX_LEV;

	DEBUG("Set: BTD debug level: %u, Fusion debug level: %u",
		btd_debug_level, fusion_log_level_in);

	DEBUG_WARN("Setting the Fusion debug log level is not implemented");

	DEBUG_FUNC("Exited");

	return TRUE;
}

/*
 * Method call for "ReadDebugLevels"
 */
gboolean manager_object_read_debug_levels(ManagerObject *object,
						guint8 *btd_log_level_out,
						guint8 *fusion_log_level_out,
						GError **error)
{
	DEBUG_FUNC("Called");

	g_assert(object != NULL);

	(void) error;
	*btd_log_level_out = get_btd_debug_level();
	*fusion_log_level_out = get_alps_debug_level();

	DEBUG("Read: BTD debug level: %u, Fusion debug level: %u",
		*btd_log_level_out, *fusion_log_level_out);

	DEBUG_FUNC("Exited");

	return TRUE;
}
