/*
 * handle_spp_service.c
 * spp service handling for the D-bus Bluetooth Daemon
 *
 * Author: Dean Jenkins <djenkins@mvista.com>
 *
 * 2011 (c) MontaVista Software, LLC. This file is licensed under
 * the terms of the AFL.
 */

#include <glib.h>
#include <dbus/dbus-glib.h>

#include "bt_appl/common_def.h"

#include "main.h"
#include "callbacks.h"
#include "node_adapter.h"
#include "handle_spp_service.h"
#include "debug.h"

/* ALPS continue flag codes */
#define	CONTINUE_1ST		(2)
#define	CONTINUE_CONTINUE	(1)
#define	CONTINUE_FINAL		(0)

enum capabilities_state {
	SPP_CAPABILITIES_IDLE,
	SPP_CAPABILITIES_GOT_DATA,
	SPP_CAPABILITIES_TOTAL_STATES
};

/*****************/
/* LOCAL GLOBALS */
/*****************/

static GPtrArray *all_spp_capabilities = NULL;

static enum capabilities_state spp_capabilities_state = SPP_CAPABILITIES_IDLE;

static BD_ADDRESS spp_bd_addr;

#ifdef DBG_ENABLED
static gchar *spp_capabilities_state_str[SPP_CAPABILITIES_TOTAL_STATES] =
{
	[SPP_CAPABILITIES_IDLE] = "SPP_CAPABILITIES_IDLE",
	[SPP_CAPABILITIES_GOT_DATA] = "SPP_CAPABILITIES_GOT_DATA"
};
#endif

/*
 * exported function
 *
 * state machine to handle the SPP capabilities
 */
void handle_spp_capabilities(callback_data *data)
{
	gen_spp_capabilities_ind_t *parms =
				&data->parms.gen_spp_capabilities_ind_parms;
	GValue *value;

	g_assert(data->id == BT_APPL_SPP_CAPABILITIES_IND_CB);

	g_assert(spp_capabilities_state < SPP_CAPABILITIES_TOTAL_STATES);

	DEBUG_FUNC("Called, continue_flag %u", parms->continue_flag);
	DEBUG("spp_capabilities_state %s",
			spp_capabilities_state_str[spp_capabilities_state]);

	if ((spp_capabilities_state == SPP_CAPABILITIES_GOT_DATA) &&
		(parms->continue_flag == CONTINUE_1ST)) {

		gpointer free_ptr;

		/*
		 * a final continue_flag failed to appear so abort previous
		 * stored data
		 */
		DEBUG_ERROR("Aborting previous SppCapabilities information");

		g_assert(all_spp_capabilities != NULL);
		free_ptr = g_ptr_array_free(all_spp_capabilities, TRUE);
		g_assert(free_ptr == NULL);

		all_spp_capabilities = NULL;

		spp_capabilities_state = SPP_CAPABILITIES_IDLE;
	}

	if (spp_capabilities_state == SPP_CAPABILITIES_IDLE) {
		/*
		 * In all continue_flag states
		 * allocate a GPtrArray for the array of structs
		 */
		g_assert(all_spp_capabilities == NULL);
		all_spp_capabilities = g_ptr_array_new();
		g_assert(all_spp_capabilities != NULL);

		/* save BD address to check all calls are for this BD address */
		memcpy(spp_bd_addr, parms->bd_addr, sizeof(BD_ADDRESS));

		spp_capabilities_state = SPP_CAPABILITIES_GOT_DATA;
	} else {
		guint8 loop;
		/*
		 * check the BD addresses are consistent.
		 * If a mismatch occurs then suggests
		 * simultaneous SPP BT_APPL_SPP_Capabilities_IND
		 * call sessions are occurring and this
		 * is not supported
		 */
		for (loop = 0; loop < sizeof(BD_ADDRESS); loop++) {
			g_assert(spp_bd_addr[loop] == parms->bd_addr[loop]);
		}
	}

	g_assert(spp_capabilities_state == SPP_CAPABILITIES_GOT_DATA);

	switch (parms->continue_flag) {
	case CONTINUE_1ST:
	case CONTINUE_CONTINUE:
		/* allocate a GValue struct and fill in the data */
		value = g_new0(GValue, 1);
		g_assert(value != NULL);

		g_value_init(value, DBUS_STRUCT_AUCHAR_AUCHAR_UCHAR_STRING);
		g_value_take_boxed(value, dbus_g_type_specialized_construct(
					DBUS_STRUCT_AUCHAR_AUCHAR_UCHAR_STRING));

		dbus_g_type_struct_set(value,
					0, parms->uuid_index,
					1, parms->uuid_data,
					2, parms->service_handle,
					3, parms->service_name,
					G_MAXUINT);

		g_ptr_array_add(all_spp_capabilities, g_value_get_boxed(value));
		g_free(value);
		break;
	case CONTINUE_FINAL:
		{
			/***************/
			/* send signal */
			/***************/

			DEBUG("Calling the signal send function");
			emit_nap_spp_capabilities(&spp_bd_addr,
						all_spp_capabilities);

			gpointer free_ptr;
			free_ptr = g_ptr_array_free(all_spp_capabilities, TRUE);
			g_assert(free_ptr == NULL);
			all_spp_capabilities = NULL;
			spp_capabilities_state = SPP_CAPABILITIES_IDLE;
		}
		break;
	default:
		DEBUG_ERROR("Unhandled continue_flag value: %u",
			parms->continue_flag);
		break;
	}

	/* free UUID arrays (might not be populated) */
	g_array_unref(parms->uuid_index);
	parms->uuid_index = NULL;

	g_byte_array_unref(parms->uuid_data);
	parms->uuid_data = NULL;

	/* free the service name (if allocated) */
	g_free(parms->service_name);
	parms->service_name = NULL;

	DEBUG_FUNC("Exited");
}

