/*
 * debug handling for the D-bus Bluetooth Daemon
 *
 * Author: Dean Jenkins <djenkins@mvista.com>
 *
 * 2010 (c) MontaVista Software, LLC. This file is licensed under
 * the terms of the AFL.
 */

#include <sys/types.h>
#include <unistd.h>
#include <glib.h>
#include <glib/gprintf.h>
#include <glib-object.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <stdarg.h>
#include <time.h>

#include "bt_appl/common_def.h"

#include "main.h"

#include "debug.h"

/* the debug log level */
gint btd_debug_level = BTD_DBG_MAX_LEV;

/* exported function */
guint8 get_btd_debug_level(void)
{
	guint8 ret;

	if (btd_debug_level >= 0) {
		if (btd_debug_level < G_MAXUINT8) {
			ret = (guint8) btd_debug_level;
		} else {
			ret = G_MAXUINT8;
		}
	} else {
		ret = 0;
	}

	return ret;
}

#ifdef DBG_ENABLED

/* size of debug buffer--allocated on the stack */
#define BTD_DBG_BUF_SIZE 1024

/*
 * Conditionally log a debug message if the debug logging level is greater
 * than or equal to the level of the message. Each log message is time stamped.
 * 'func' is the name of the function logging the message, and 'fmt' is a
 * printf-style format string.
 */
void btd_debug_log(gint level, const gchar *func, const gchar *fmt, ...)
{
	if (btd_debug_level >= level) {
		va_list args;
		struct timespec timestamp;
		gchar str[BTD_DBG_BUF_SIZE];
		gint idx = 0;
		gint len = BTD_DBG_BUF_SIZE;
		gint ret;
		const gchar *prg_name = g_get_prgname();

		/* get the current time and use it a timestamp */
		clock_gettime(CLOCK_MONOTONIC, &timestamp); 

		if (prg_name != NULL) {
			ret = g_snprintf(&str[idx], len, "(%s:%lu): ",
					prg_name, (gulong) getpid());
		} else {
			ret = g_snprintf(&str[idx], len, "(process:%lu): ",
					(gulong) getpid());
		}

		if ((ret > 0) && (ret < len)) {
			len -= ret;
			idx += ret;
		}
 
		ret = g_snprintf(&str[idx], len, "[%5u.%06lu] "PROG_ID,
				(guint32) timestamp.tv_sec,
				timestamp.tv_nsec/1000);

		if ((ret > 0) && (ret < len)) {
			len -= ret;
			idx += ret;
		}

		switch (level) {
		case BTD_DBG_ERR_LEV:
			ret = g_snprintf(&str[idx], len, "ERROR: ");
			break;
		case BTD_DBG_WARN_LEV:
			ret = g_snprintf(&str[idx], len, "WARNING: ");
			break;
		case BTD_DBG_INFO_LEV:
		case BTD_DBG_FUNC_LEV:
		case BTD_DBG_HIGH_LEV:
			ret = g_snprintf(&str[idx], len, "DEBUG: ");
			break;
		default:
			ret = g_snprintf(&str[idx], len, "UNKNOWN: ");
			break;
		}

		if ((ret > 0) && (ret < len)) {
			len -= ret;
			idx += ret;
		}

		ret = g_snprintf(&str[idx], len, "%s: ", func);

		if ((ret > 0) && (ret < len)) {
			len -= ret;
			idx += ret;
		}

		va_start(args, fmt);
		ret = g_vsnprintf(&str[idx], len, fmt, args);
		va_end(args);

		if ((ret > 0) && (ret < len)) {
			len -= ret;
			idx += ret;
		}

		ret = g_snprintf(&str[idx], len, "\n");

		if(level == BTD_DBG_ERR_LEV) {
			/* use STDERR */
			g_printerr("%s", str);
		} else {
			/* use STDOUT */
			g_print("%s", str);
		}
	}
}

#define PER_ROW	(10)

/* dump the contents of an array as debug output */
void debug_dump_array(guint32 len, guint8 *data)
{
	gchar dump_str[PER_ROW*12];
	guint16 x;
	guint16 y;
	guint16 pos;

	/* Dump the content of the data array in rows */
	for (x = 0; x*PER_ROW < len; x++) {
		pos = 0;
		for (y = 0; (x*PER_ROW+y < len) && (y < PER_ROW); y++) {
			pos += g_sprintf(&dump_str[pos], "%u:0x%02X ",
					x*PER_ROW+y, data[x*PER_ROW+y]);
		}
		dump_str[pos] = 0;
		DEBUG_HIGH("%s", &dump_str[0]);
	}
}

static gchar bd_addr_str[18];

gchar *bd_addr_to_str(const BD_ADDRESS bd_addr)
{
	bd_addr_str[0] = 0;
	if (bd_addr) {
		g_sprintf(bd_addr_str, "%02x:%02x:%02x:%02x:%02x:%02x",
			  bd_addr[0], bd_addr[1], bd_addr[2],
			  bd_addr[3], bd_addr[4], bd_addr[5]);
	}

	return bd_addr_str;
}

#endif

