/*
 * btd_audio.c
 *
 * Andy Lowe <alowe@mvista.com>
 *
 * 2011 (c) MontaVista Software, LLC. This file is licensed under
 * the terms of the AFL.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <sys/param.h>
#include <glib/gi18n.h>
#include <glib-object.h>
#include <sched.h>
#include <signal.h>

#include "bt_appl/common_def.h"
#include "bt_appl/api/bt_appl_audio_block.h"

#include "main.h"
#include "debug.h"
#include "dbus_error.h"

#include "btd_audio.h"

/* Return TRUE if two bdaddrs match, or FALSE otherwise */
#define bdaddrs_match(bdaddr1, bdaddr2) \
	 (memcmp((bdaddr1), (bdaddr2), sizeof(BD_ADDRESS)) == 0)

#define bdaddr_zero(bdaddr) \
	(memset(bdaddr, 0, sizeof(BD_ADDRESS)))

#ifdef DBG_ENABLED
gchar *aud_states[] = { "AUD_IDLE", "AUD_IDLE_BUF", "AUD_IDLE_DISC",
	"AUD_WAITING", "AUD_BUF", "AUD_DISC", "AUD_PIPING", "AUD_DRAIN",
	"AUD_EXIT" };
#endif

/* structure for audio data buffers */
struct aud_buf {
	guint16 len;
	guint8 data[0];
};

#define AUD_BUF_SIZE PIPE_BUF
#define AUD_BUF_DATA_MAX (AUD_BUF_SIZE - sizeof(struct aud_buf))

/*
 * Calculate the audio state from the boolean state variables.
 * The caller must hold the lock on the audio structure.
 * The state is updated in the audio structure as well as being returned.
 */
guint btd_audio_state(struct btd_audio *aud)
{
	if (aud->exit)
		aud->state = AUD_EXIT;
	else if (!aud->started && !aud->streaming)
		aud->state = AUD_IDLE;
	else if (!aud->started && aud->streaming) {
		if (!aud->discarding)
			aud->state = AUD_IDLE_BUF;
		else
			aud->state = AUD_IDLE_DISC;
	} else if (aud->started && !aud->streaming) {
		if (!aud->piping)
			aud->state = AUD_WAITING;
		else
			aud->state = AUD_DRAIN;
	} else {
		if (aud->piping)
			aud->state = AUD_PIPING;
		else if (!aud->discarding)
			aud->state = AUD_BUF;
		else
			aud->state = AUD_DISC;
	}

	return aud->state;
}

/*
 * Routines for communication between the DBus thread and the audio thread.
 */

BTHRESULT btd_audio_start(struct btd_audio *aud, const BD_ADDRESS bdaddr,
	guint8 codec, guint bitrate)
{
	BTHRESULT result = BT_APPL_SUCCESS;

	g_mutex_lock(aud->lock);
	if (!aud->started) {
		if (!aud->streaming || bdaddrs_match(aud->bdaddr, bdaddr)) {
			if (!aud->streaming) {
				memcpy(aud->bdaddr, bdaddr, sizeof(BD_ADDRESS));
				aud->codec = codec;
				aud->bitrate = bitrate;
			}
			aud->started = TRUE;
			btd_audio_state(aud);
			g_cond_broadcast(aud->update);
		} else	{
			DEBUG("Already streaming from different device");
			result = BT_APPL_ERROR_STATE_ERROR;
		}
	} else {
		/*
		 * Audio has already been started.  If the streaming parameters
		 * match, then return with no error.  If the streaming
		 * parameters don't match, then return an error because we
		 * can't change the streaming parameters after audio has been
		 * started.
		 */
		if (!bdaddrs_match(aud->bdaddr, bdaddr)
			|| (aud->codec != codec)
			|| (aud->bitrate != bitrate))
		{
			DEBUG("Audio has already been started with different "
				"device, codec, or bitrate");
			result = BT_APPL_ERROR_STATE_ERROR;
		}
	}
	g_mutex_unlock(aud->lock);

	return result;
}

void btd_audio_stop(struct btd_audio *aud)
{
	g_mutex_lock(aud->lock);
	if (aud->started || (aud->streaming && !aud->discarding)) {
		/*
		 * We need to change states here if we're in any state
		 * where aud->started is true or if we're in the IDLE_BUF
		 * state (streaming but not discarding).
		 */
		if (!aud->streaming) {
			bdaddr_zero(aud->bdaddr);
			aud->codec = CODEC_SBC;
			aud->bitrate = 0;
			aud->discarding = FALSE;
		} else
			aud->discarding = TRUE;
		aud->piping = FALSE;
		aud->started = FALSE;
		btd_audio_state(aud);
		g_cond_broadcast(aud->update);
	}
	g_mutex_unlock(aud->lock);
}

void btd_audio_exit(struct btd_audio *aud)
{
	g_mutex_lock(aud->lock);
	if (aud->exit == FALSE) {
		aud->exit = TRUE;
		btd_audio_state(aud);
		g_cond_broadcast(aud->update);
	}
	g_mutex_unlock(aud->lock);
}

void btd_audio_media_resume(struct btd_audio *aud, const BD_ADDRESS bdaddr,
	guint8 codec, guint bitrate)
{
	g_mutex_lock(aud->lock);
	if (!aud->streaming) {
		if (!aud->started || bdaddrs_match(aud->bdaddr, bdaddr)) {
			if (!aud->started) {
				memcpy(aud->bdaddr, bdaddr, sizeof(BD_ADDRESS));
				aud->codec = codec;
				aud->bitrate = bitrate;
			}
			aud->streaming = TRUE;
			btd_audio_state(aud);
			g_cond_broadcast(aud->update);
		} else {
			DEBUG("Ignoring media resume for different device");
		}
	} else {
		DEBUG("Ignoring media resume when not suspended");
	}
	g_mutex_unlock(aud->lock);
}

void btd_audio_media_suspend(struct btd_audio *aud, const BD_ADDRESS bdaddr)
{
	g_mutex_lock(aud->lock);
	if (aud->streaming) {
		if (bdaddrs_match(aud->bdaddr, bdaddr)) {
			if (!aud->started) {
				bdaddr_zero(aud->bdaddr);
				aud->codec = CODEC_SBC;
				aud->bitrate = 0;
			}
			aud->discarding = FALSE;
			aud->streaming = FALSE;
			btd_audio_state(aud);
			g_cond_broadcast(aud->update);
		} else {
			DEBUG("Ignoring media suspend for different device");
		}
	} else {
		DEBUG("Ignoring media suspend when already suspended");
	}
	g_mutex_unlock(aud->lock);
}

void btd_audio_avp_disconnect(struct btd_audio *aud, const BD_ADDRESS bdaddr)
{
	g_mutex_lock(aud->lock);
	if (aud->started || aud->streaming) {
		if (bdaddrs_match(aud->bdaddr, bdaddr)) {
			bdaddr_zero(aud->bdaddr);
			aud->codec = CODEC_SBC;
			aud->bitrate = 0;
			aud->piping = FALSE;
			aud->discarding = FALSE;
			aud->streaming = FALSE;
			aud->started = FALSE;
			btd_audio_state(aud);
			g_cond_broadcast(aud->update);
			DEBUG("Stopping audio due to AVP disconnect");
		} else {
			DEBUG("Ignoring AVP disconnect for different device");
		}
	}
	g_mutex_unlock(aud->lock);
}

/*
 * Get the address of the Bluetooth remote device currently designated as
 * the audio streaming source along with its codec type, codec bitrate, and
 * the cumulative total of encoded audio bytes streamed since last in the
 * AUD_IDLE state.
 * The bdaddr argument points to a six-byte buffer for returning the
 * device address.
 */
void btd_audio_get_status(struct btd_audio *aud, BD_ADDRESS bdaddr,
	guint8 *codec, guint *bitrate, guint *bytes_streamed)
{
	g_mutex_lock(aud->lock);
	memcpy(bdaddr, aud->bdaddr, sizeof(BD_ADDRESS));
	*codec = aud->codec;
	*bitrate = aud->bitrate;
	*bytes_streamed = aud->bytes_streamed;
	g_mutex_unlock(aud->lock);
}

void btd_audio_set_pipe_name(struct btd_audio *aud, guint8 codec,
	const gchar *pipe_name)
{

	if (codec < MAX_CODECS) {
		g_mutex_lock(aud->lock);
		g_free(aud->pipe_name[codec]);
		aud->pipe_name[codec] = g_strdup(pipe_name);
		g_mutex_unlock(aud->lock);
	}
}

/*
 * Get the pipe name assigned to a codec.
 * The caller must free the returned string with g_free().
 */
gchar *btd_audio_get_pipe_name(struct btd_audio *aud, guint8 codec)
{
	gchar *name = NULL;

	if (codec < MAX_CODECS) {
		g_mutex_lock(aud->lock);
		name = g_strdup(aud->pipe_name[codec]);
		g_mutex_unlock(aud->lock);
	}

	return name;
}

void btd_audio_set_thread_priority(struct btd_audio *aud, guint thread_priority)
{
	g_mutex_lock(aud->lock);
	aud->thread_priority = thread_priority;
	g_mutex_unlock(aud->lock);
}

guint btd_audio_get_thread_priority(struct btd_audio *aud)
{
	guint thread_priority;

	g_mutex_lock(aud->lock);
	thread_priority = aud->thread_priority;
	g_mutex_unlock(aud->lock);

	return thread_priority;
}

static void btd_audio_set_scheduling_priority(struct btd_audio *aud)
{
	gint err;
	struct sched_param sched_param;

	err = sched_getparam(0, &sched_param);
	if (err < 0) {
		DEBUG("sched_getparam error %d", err);
		return;
	}

	sched_param.sched_priority = btd_audio_get_thread_priority(aud);
	err = sched_setscheduler(0, SCHED_FIFO, &sched_param);
	if (!err) {
		DEBUG("Priority of btd_audio_thread set to SCHED_FIFO %d",
			   sched_param.sched_priority);
	} else {
		DEBUG("sched_setscheduler error %d when attempting to set "
			   "priority SCHED_FIFO %d",
			   err, sched_param.sched_priority);
	}
}

#define POLL_MIN 20000	/* minimum polling interval in microseconds */
#define POLL_MAX 50000	/* maximum polling interval in microseconds */
#define MAX_BQ_LEN 32	/* maximum number of audio buffers in the queue */

/*
 * The audio handling thread.
 */
static gpointer btd_audio_thread(gpointer data)
{
	struct btd_audio *aud = data;
	guint state, prev_state;
	GTimeVal gtime;
	gint pipe_fd = -1;
	gint poll_interval = POLL_MIN;
	struct aud_buf *buf;	/* audio buffer */
	GQueue *bq;		/* queue of audio buffers */
	struct sigaction act;

	/* ignore the SIGPIPE signal */
	memset(&act, 0, sizeof(act));
	act.sa_handler = SIG_IGN;
	sigaction(SIGPIPE, &act, NULL);

	state = AUD_EXIT;
	bq = g_queue_new();

	while (!aud->exit) {
		g_mutex_lock(aud->lock);
		prev_state = state;
		state = aud->state;
		g_mutex_unlock(aud->lock);

		/* Check for state change */
		if (state != prev_state) {
			DEBUG("Audio streaming state is %s", aud_states[state]);

			/*
			 * Set the scheduling priority of this thread
			 * whenever we enter or exit the IDLE state.
			 */
			if ((state == AUD_IDLE) || (prev_state == AUD_IDLE))
				btd_audio_set_scheduling_priority(aud);

			/*
			 * Reset the number of bytes streamed whenever we enter
			 * the IDLE state.
			 */
			if (state == AUD_IDLE) {
				guint bytes_streamed;

				g_mutex_lock(aud->lock);
				bytes_streamed = aud->bytes_streamed;
				aud->bytes_streamed = 0;
				g_mutex_unlock(aud->lock);
				DEBUG("Resetting bytes_streamed from %d to 0",
					bytes_streamed);
			}

			/*
			 * Calculate the polling interval whenever we exit the
			 * IDLE state.
			 */
			if (prev_state == AUD_IDLE) {
				g_mutex_lock(aud->lock);
				if (aud->bitrate > 0) {
					poll_interval =
						(PIPE_BUF*8*1000)/aud->bitrate;
					if (poll_interval > POLL_MAX)
						poll_interval = POLL_MAX;
					else if (poll_interval < POLL_MIN)
						poll_interval = POLL_MIN;
				} else
					poll_interval = POLL_MIN;
				g_mutex_unlock(aud->lock);
				DEBUG("Audio stream polling interval is %d "
					"microseconds", poll_interval);
			}

			if ((pipe_fd >= 0)
				&& ((state == AUD_IDLE)
				|| (state == AUD_IDLE_BUF)
				|| (state == AUD_IDLE_DISC)
				|| (state == AUD_WAITING)
				|| (state == AUD_BUF)
				|| (state == AUD_DISC)))
			{
				gint err;

				/* close pipe */
				DEBUG("Closing pipe");
				err = close(pipe_fd);
				if (err) {
					DEBUG("Error %d when closing pipe",
						errno);
					err = close(pipe_fd);
					if (err) {
						DEBUG("Pipe may not be closed");
					} else {
						DEBUG("Pipe is closed");
					}
				}
				pipe_fd = -1;
			}

			/* flush the audio buffer queue */
			if ((g_queue_get_length(bq) > 0)
				&& ((state == AUD_IDLE)
				|| (state == AUD_IDLE_DISC)
				|| (state == AUD_WAITING)
				|| (state == AUD_DISC)))
			{
				DEBUG("Flushing buffered audio data");
				while ((buf = g_queue_pop_head(bq)) != NULL)
					g_slice_free1(AUD_BUF_SIZE, buf);
			}
		}

		if ((pipe_fd < 0)
			&& ((state == AUD_BUF) || (state == AUD_DISC)))
		{
			guint8 codec = aud->codec;

			/* attempt to open pipe */
			pipe_fd = open(aud->pipe_name[codec],
				O_WRONLY | O_NONBLOCK);
			if (pipe_fd >= 0) {
				DEBUG("Opened pipe %s", aud->pipe_name[codec]);
				g_mutex_lock(aud->lock);
				aud->piping = TRUE;
				aud->discarding = FALSE;
				btd_audio_state(aud);
				g_mutex_unlock(aud->lock);
				continue;
			}
		}

		if ((pipe_fd >= 0)
			&& ((state == AUD_PIPING) || (state == AUD_DRAIN)))
		{
			gboolean pipe_closed = FALSE;

			/* write to pipe */
			while ((buf = g_queue_pop_head(bq)) != NULL) {
				gint err;

				err = write(pipe_fd, buf->data, buf->len);
				if (err < 0) {
					/*
					 * The write was unsuccessful, so put
					 * the buffer back at the head of the
					 * queue.
					 */
					g_queue_push_head(bq, buf);

					err = errno;
					if (err == EAGAIN) {
						/* the pipe is full */
					} else if (err == EPIPE) {
						/* the reader closed the pipe */
						DEBUG("Pipe closed by reader");
						g_mutex_lock(aud->lock);
						aud->piping = FALSE;
						if (aud->streaming)
							aud->discarding = TRUE;
						btd_audio_state(aud);
						g_mutex_unlock(aud->lock);
						pipe_closed = TRUE;
					} else {
						DEBUG("Pipe write error %d",
							err);
					}
					break;
				} else {
					/* successful write */
					g_mutex_lock(aud->lock);
					aud->bytes_streamed += buf->len;
					g_mutex_unlock(aud->lock);
					g_slice_free1(AUD_BUF_SIZE, buf);
				}
			}
			if (pipe_closed)
				continue;

			if ((state == AUD_DRAIN) && !g_queue_get_length(bq)) {
				DEBUG("Audio buffers drained");
				g_mutex_lock(aud->lock);
				aud->piping = FALSE;
				btd_audio_state(aud);
				g_mutex_unlock(aud->lock);
				continue;
			}

		}

		if ((state == AUD_IDLE_BUF) || (state == AUD_IDLE_DISC)
			|| (state == AUD_BUF) || (state == AUD_DISC)
			|| (state == AUD_PIPING))
		{
			BTHRESULT result;
			guint8 bufstat;
			gboolean new_data = FALSE;

			if (g_queue_get_length(bq) == MAX_BQ_LEN) {
				/* buffer queue is full */
				DEBUG("Audio buffer overrun");
				g_mutex_lock(aud->lock);
				aud->discarding = TRUE;
				aud->piping = FALSE;
				btd_audio_state(aud);
				g_mutex_unlock(aud->lock);
				continue;
			}

			while (g_queue_get_length(bq) < MAX_BQ_LEN) {
				/* allocate buffer */
				buf = g_slice_alloc(AUD_BUF_SIZE);
				buf->len = 0;

				/* read from input stream */
				result = BT_APPL_AVP_Get_Audio_Stream_REQ(aud->bdaddr,
					AUD_BUF_DATA_MAX, buf->data, &buf->len);

				if (result &&
					(result != BT_APPL_ERROR_OPERATION_FAILED))
				{
					DEBUG("BT_APPL_AVP_Get_Audio_Stream_REQ"
						"returned %s",
						lookup_bthresult_str(result));
				}
				if (result || (buf->len == 0)) {
					/*
					 * Nothing was read, so discard the
					 * buffer and stop reading.
					 */
					g_slice_free1(AUD_BUF_SIZE, buf);
					break;
				}

				/* We read audio data */
				if ((state == AUD_IDLE_DISC)
					|| (state == AUD_DISC))
				{
					/*
					 * We're discarding audio data, so
					 * throw it away
					 */
					g_slice_free1(AUD_BUF_SIZE, buf);
				} else {
					/* Save the audio data in the queue */
					g_queue_push_tail(bq, buf);
					new_data = TRUE;
				}

				bufstat = 0;
				result = BT_APPL_AVP_Audio_Buffer_Status_REQ(
					aud->bdaddr, &bufstat);
				if (result) {
					DEBUG("BT_APPL_AVP_Audio_Buffer_Status_"
						"REQ returned %s",
						lookup_bthresult_str(result));
				}
			}

			/*
			 * If we read new audio data and we're piping, loop
			 * without delay to write the data to the pipe.
			 */
			if (new_data && (state == AUD_PIPING))
				continue;
		}

		g_get_current_time(&gtime);
		g_time_val_add(&gtime, poll_interval);

		g_mutex_lock(aud->lock);
		if ((state == AUD_IDLE) || (state == AUD_WAITING))
		{
			g_cond_wait(aud->update, aud->lock);
		} else {
			g_cond_timed_wait(aud->update, aud->lock, &gtime);
		}
		g_mutex_unlock(aud->lock);
	}

	if (pipe_fd >= 0)
		close(pipe_fd);

	/* flush the audio buffer queue */
	while ((buf = g_queue_pop_head(bq)) != NULL)
		g_slice_free1(AUD_BUF_SIZE, buf);

	/* free the audio buffer queue */
	g_queue_free(bq);

	return 0;
}

/*
 * Routines to initialize and finalize the audio handling thread.
 */

void btd_audio_finalize(struct btd_audio *aud)
{
	btd_audio_exit(aud);
	g_thread_join(aud->thread);

	/* free the pipe names */
	btd_audio_set_pipe_name(aud, CODEC_SBC, NULL);
	btd_audio_set_pipe_name(aud, CODEC_MP3, NULL);
	btd_audio_set_pipe_name(aud, CODEC_AAC, NULL);

	g_mutex_free(aud->lock);
	g_cond_free(aud->update);
}

gboolean btd_audio_initialize(struct btd_audio *aud)
{
	GError *error = NULL;

	/* initialize the audio state */
	memset(aud, 0, sizeof(*aud));
	btd_audio_state(aud);

	aud->lock = g_mutex_new();
	aud->update = g_cond_new();

	/* set the pipe names */
	btd_audio_set_pipe_name(aud, CODEC_SBC, PIPE_NAME_SBC);
	btd_audio_set_pipe_name(aud, CODEC_MP3, PIPE_NAME_MP3);
	btd_audio_set_pipe_name(aud, CODEC_AAC, PIPE_NAME_AAC);

	/* create the pipes if they don't exist */
	mkfifo(PIPE_NAME_SBC, 0666);
	mkfifo(PIPE_NAME_MP3, 0666);
	mkfifo(PIPE_NAME_AAC, 0666);

	/* set default scheduling priority for the audio thread */
	btd_audio_set_thread_priority(aud, get_audio_thread_priority());

	aud->thread = g_thread_create(btd_audio_thread, aud, TRUE, &error);
	if (!aud->thread) {
		DEBUG_ERROR("Can't create btd_audio_thread: %s", error->message);
		g_error_free(error);
		return FALSE;
	}

	return TRUE;
}
