/*
 * btd_audio.h
 *
 * Andy Lowe <alowe@mvista.com>
 *
 * 2011 (c) MontaVista Software, LLC. This file is licensed under
 * the terms of the AFL.
 */

#ifndef __BTD_AUDIO_H__
#define __BTD_AUDIO_H__

/*
 * Define macros for the codec formats.  These definitions match the
 * values used by the ALPS API, so don't change them.
 */
#define CODEC_SBC 0
#define CODEC_MP3 1
#define CODEC_AAC 2
#define MAX_CODECS 3

/* Define pipe names for each codec type */
#define PIPE_NAME_SBC	"/dev/btsbc"
#define PIPE_NAME_MP3	"/dev/btmp3"
#define PIPE_NAME_AAC	"/dev/btaac"

/* Define macros for the audio state */
#define AUD_IDLE	0 /* !started && !streaming                           */
#define AUD_IDLE_BUF	1 /* !started &&  streaming && !discarding            */
#define AUD_IDLE_DISC	2 /* !started &&  streaming &&  discarding            */
#define AUD_WAITING	3 /*  started && !streaming                && !piping */
#define AUD_BUF		4 /*  started &&  streaming && !discarding && !piping */
#define AUD_DISC	5 /*  started &&  streaming &&  discarding && !piping */
#define AUD_PIPING	6 /*  started &&  streaming                &&  piping */
#define AUD_DRAIN	7 /*  started && !streaming                &&  piping */
#define AUD_EXIT	8

struct btd_audio {
	GThread *thread;
	GMutex *lock;
	GCond *update;

	guint state;			/* the audio state */
	gboolean exit;			/* TRUE if audio thread is exiting */
	BD_ADDRESS bdaddr;		/* audio device Bluetooth address */
	guint codec;			/* codec used by current device */
	guint bitrate;			/* bitrate in kbps for current device */
	gboolean started;		/* TRUE if audio has been started via
					   StartAudioPipe */
	gboolean streaming;		/* TRUE if there is an incoming audio
					   stream */
	gboolean discarding;		/* TRUE if incoming audio stream samples
					   are being discarded */
	gboolean piping;		/* TRUE if the pipe is open */
	guint bytes_streamed;		/* cumulative number of encoded audio
					   bytes streamed since last in the
					   AUD_IDLE state */
	gchar *pipe_name[MAX_CODECS];	/* pipe name used for each codec */
	guint thread_priority;		/* scheduling priority of audio thread */
};

extern BTHRESULT btd_audio_start(struct btd_audio *aud, const BD_ADDRESS bdaddr,
	guint8 codec, guint bitrate);
extern void btd_audio_stop(struct btd_audio *aud);
extern void btd_audio_exit(struct btd_audio *aud);
extern void btd_audio_media_resume(struct btd_audio *aud,
	const BD_ADDRESS bdaddr, guint8 codec, guint bitrate);
extern void btd_audio_media_suspend(struct btd_audio *aud,
	const BD_ADDRESS bdaddr);
extern void btd_audio_avp_disconnect(struct btd_audio *aud,
	const BD_ADDRESS bdaddr);
extern void btd_audio_get_status(struct btd_audio *aud,
	BD_ADDRESS bdaddr, guint8 *codec, guint *bitrate,
	guint *bytes_streamed);
extern void btd_audio_set_pipe_name(struct btd_audio *aud, guint8 codec,
	const gchar *pipe_name);
extern gchar *btd_audio_get_pipe_name(struct btd_audio *aud, guint8 codec);
extern void btd_audio_set_thread_priority(struct btd_audio *aud,
	guint thread_priority);
extern guint btd_audio_get_thread_priority(struct btd_audio *aud);
extern void btd_audio_finalize(struct btd_audio *aud);
extern gboolean btd_audio_initialize(struct btd_audio *aud);

#endif
