/* -*-c++-*-
mapengine_oss

* osgEarth - Dynamic map generation toolkit for OpenSceneGraph
* Copyright 2008-2010 Pelican Mapping
* http://osgearth.org
*
* osgEarth is free software; you can redistribute it and/or modify
* it under the terms of the GNU Lesser General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public License
* along with this program. If not, see <http://www.gnu.org/licenses/>

History:
Copied from https://github.com/gwaldron/osgearth/blob/osgearth_2.0_T2011-02-24/src/osgEarthUtil/SkyNode
List of changes:
1. Optimized sky calculation to use an icosphere instead of a UVSphere
2. Optimized star creation to read binary data instead of strings
*/

#ifndef MAPENGINE_OSS_SKYNODE_
#define MAPENGINE_OSS_SKYNODE_

#include <osg/MatrixTransform>
#include <osg/Uniform>
#include <osg/Group>
#include <osg/Depth>
#include <osg/CoordinateSystemNode>
#include "oss/sky/ISky.h"
#include "oss/Export.h"

namespace mapengine {
namespace oss {

/**
* Class that provides information about astronomical objects at a given time.
*/
class MAPENGINE_OSS_EXPORT EphemerisProvider : public osg::Referenced
{
public:
   /** Gets the moon position in geocentric coordinates at the given time */
   virtual osg::Vec3d getMoonPosition( int year, int month, int date, double hoursUTC ) = 0;

   /** Gets the sun position in geocentric coordinates at the given time */
   virtual osg::Vec3d getSunPosition( int year, int month, int date, double hoursUTC ) = 0;

   /** Gets the sun position in polar coordinates at the given time */
   virtual osg::Vec2d getSunPositionWGS84(int year, int month, int date, double hoursUTC) = 0;
protected:
   virtual ~EphemerisProvider(){}
};

/**
* The default EphemerisProvider, provides positions based on freely available models
*/
class MAPENGINE_OSS_EXPORT DefaultEphemerisProvider : public EphemerisProvider
{
public:
   /** Gets the moon position in geocentric coordinates at the given time */
   virtual osg::Vec3d getMoonPosition( int year, int month, int date, double hoursUTC );

   /** Gets the sun position in geocentric coordinates at the given time */
   virtual osg::Vec3d getSunPosition( int year, int month, int date, double hoursUTC );

   /** Gets the sun position in poral coordinates at the given time */
   virtual osg::Vec2d getSunPositionWGS84(int year, int month, int date, double hoursUTC);

protected:
   virtual ~DefaultEphemerisProvider(){}
};

/**
* A sky model
*/
class MAPENGINE_OSS_EXPORT me_SkyNode : public osg::Group, public mapengine::oss::me_ISky
{
public:
   /** Creates a new sky node based on the provided map. */
   me_SkyNode(osg::Node* parent, float minStarMagnitude=-1.0f );

   int getRenderBinAtmosphereInside() const { return _renderBinAtmosphereInside; }
   void setRenderBinAtmosphereInside(int val) { _renderBinAtmosphereInside = val; }
   int getRenderBinAtmosphereOutside() const { return _renderBinAtmosphereOutside; }
   void setRenderBinAtmosphereOutside(int val) { _renderBinAtmosphereOutside = val; }
   bool getSkyToOccludeStars() const { return _skyToOccludeStars; }
   void setSkyToOccludeStars(bool val) { _skyToOccludeStars = val; }
public:
   /** Gets/Sets the EphemerisProvider */
   EphemerisProvider* getEphemerisProvider() const;
   void setEphemerisProvider(EphemerisProvider* ephemerisProvider );

   /** Gets a position from the right ascension, declination and range */
   static osg::Vec3d getPositionFromRADecl( double ra, double decl, double range );

   void initialize(const osg::EllipsoidModel* em, const std::string& starFile = "");

public:
   /** Attached this sky node to a view (placing a sky light). */
   void attach( osg::View* view, int lightNum =0 );

   /** Gets the date time for the sky position  */
   void getDateTime( int &year, int &month, int &date, double &hoursUTC );

   /** Sets the sky's position based on a julian date. */
   virtual void setDateTime( int year, int month, int date, double hoursUTC );
   void setDateTime( bool ff );

   /** The minimum brightness for non-sunlit areas. */
   void setAmbientBrightness( float value );
   float getAmbientBrightness() const;

   /** Whether atmosphere is visible */
   void setAtmosphereVisible(bool value);
   bool getAtmosphereVisible() const;

   /** Whether the stars are visible */
   void setStarsVisible(bool value);
   bool getStarsVisible() const;

   virtual void traverse( osg::NodeVisitor& nv );
   virtual osg::BoundingSphere computeBound() const;

   /** Gets the sun's position as a unit vector */
   virtual void getSunPosition( osg::Vec3& pos );

   /** Gets the sun's position as a polar coordinate */
   virtual void getSunPosition(osg::Vec2d& pos);

protected:
   /** dtor */
   virtual ~me_SkyNode() {}

   /** Sets the sun's position as a unit vector. */
   void setSunPosition( const osg::Vec3& pos );

   /** Sets the moon position as a geocentric coordinate */
   void setMoonPosition( const osg::Vec3d& pos );

   /** Sets the sun's position as a latitude and longitude. */
   void setSunPosition( double lat_degrees, double lon_degrees );

   osg::ref_ptr<osg::Program> generateAtmosphereShaderProgram(const char* programName, const char* vertexShaderExtensionCode, const char* fragmentShaderExtensionCode);

   struct StarData
   {
      std::string name;
      double right_ascension;
      double declination;
      double magnitude;

      StarData(): right_ascension(0.0), declination(0.0), magnitude(0.0) {}
      StarData( std::stringstream &ss );
   };

   osg::Vec3f _lightPos;
   osg::ref_ptr<osg::Light> _light;
   osg::ref_ptr<osg::Uniform> _lightPosUniform;

   osg::ref_ptr<osg::MatrixTransform> _sunXform;
   osg::ref_ptr<osg::MatrixTransform> _moonXform;
   osg::ref_ptr<osg::MatrixTransform> _starsXform;

   osg::ref_ptr<osg::Group> _cullContainer;

   int _year, _month, _date; double _hoursUTC;
   float _innerRadius, _outerRadius, _sunDistance, _starRadius, _minStarMagnitude;
   osg::ref_ptr<osg::Node> _sun, _stars, _atmosphere, _moon;

   osg::ref_ptr< const osg::EllipsoidModel > _ellipsoidModel;

   void initializeEllipsoidModel(const osg::EllipsoidModel* em);

   virtual osg::Geode* makeAtmosphere( const osg::EllipsoidModel* );
   void makeSun();
   void makeMoon();

   void makeStars(const std::string& starFile);
   osg::Node* buildStarGeometry(const std::vector<StarData>& stars);
   void getDefaultStars(std::vector<StarData>& out_stars);
   bool parseStarFile(const std::string& starFile, std::vector<StarData>& out_stars);

   osg::ref_ptr< EphemerisProvider > _ephemerisProvider;

   osg::ref_ptr<osg::Depth> _depthSettingsAtmosphere;
   int _renderBinAtmosphereInside;
   int _renderBinAtmosphereOutside;
   bool _skyToOccludeStars;
};

class me_AnimateSunCallback : public osg::NodeCallback
{
   void operator()( osg::Node* node, osg::NodeVisitor* /*nv*/ )
   {
      me_SkyNode* skyNode = static_cast<me_SkyNode*>(node);
      double hours = fmod( osg::Timer::instance()->time_s()/4.0, 24.0 );
      skyNode->setDateTime( 2011, 6, 6, hours );
   }
protected:
   virtual ~me_AnimateSunCallback(){}
};

}
}

#endif // MAPENGINE_OSS_SKYNODE_
