/* -*-c++-*-
mapengine_oss

* osgEarth - Dynamic map generation toolkit for OpenSceneGraph
* Copyright 2008-2010 Pelican Mapping
* http://osgearth.org
*
* osgEarth is free software; you can redistribute it and/or modify
* it under the terms of the GNU Lesser General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public License
* along with this program. If not, see <http://www.gnu.org/licenses/>.

History:
Copied from https://github.com/gwaldron/osgearth/blob/osgearth_2.0_T2011-02-24/src/osgEarthUtil/SkyNode.cpp
List of changes:
1. Optimized sky calculation to use an icosphere instead of a UVSphere
2. Optimized star creation to read binary data instead of strings
3. Made 'initialize' function public to allow atmosphere creation to be overloaded
4. Modified star vertex shader to fix flashing stars at low pitch and low scale
*/

#include <sstream>
#include <time.h>
#include <osg/MatrixTransform>
#include <osg/ShapeDrawable>
#include <osg/PointSprite>
#include <osg/BlendFunc>
#include <osg/FrontFace>
#include <osg/CullFace>
#include <osg/Program>
#include <osg/Camera>
#include <osg/Point>
#include <osg/Shape>
#include <osg/Depth>
#include <osg/Quat>
#include <osg/Geometry>
#include <osg/Geode>
#include <osg/StateSet>
#include <osg/Billboard>
#include <osg/Texture2D>
#include <osgUtil/CullVisitor>
#include <osgDB/ReadFile>
#include "oss/sky/SkyNode.h"
#include "oss/sky/StarData.h"
#include "oss/OSSUtils.h"
#include "oss/sky/IcoSphereCreator.h"

using namespace mapengine::oss;

#ifdef OSG_GLES2_AVAILABLE
   #define GLSL_VERSION_STR "100"
#else
   #define GLSL_VERSION_STR "110" 
#endif

#define BIN_STARS       -100003
#define BIN_SUN         -100002
#define BIN_MOON        -100001
#define BIN_ATMOSPHERE  -100000

namespace {
struct Stringify
{
   operator std::string () const
   {
      std::string result;
      result = buf.str();
      return result;
   }

   template<typename T>
   Stringify& operator << (const T& val) { buf << val; return (*this); }

   Stringify& operator << (const Stringify& val) { buf << (std::string)val; return (*this); }

protected:
   std::stringstream buf;
};

// utility: functor for traversing a target node
template<typename T> struct TraverseFunctor {
   T* _target;
   TraverseFunctor(T* target) : _target(target) { }
   void operator()(osg::NodeVisitor& nv) { _target->T::traverse(nv); }
};

// utility: node that traverses another node via a functor
template<typename T>
struct TraverseNode : public osg::Node {
   TraverseFunctor<T> _tf;
   TraverseNode(TraverseFunctor<T>& tf) : _tf(tf) { }
   TraverseNode(T* target) : _tf(TraverseFunctor<T>(target)) { }
   void traverse(osg::NodeVisitor& nv) { _tf(nv); }
   osg::BoundingSphere computeBound() const { return _tf._target->getBound(); }
};

// makes a disc geometry that we'll use to render the sun/moon
osg::Geometry* s_makeDiscGeometry(double radius)
{
   int segments = 48;
   float deltaAngle = 360.0f/static_cast<float>(segments);

   osg::Geometry* geom = new osg::Geometry();
   geom->setUseVertexBufferObjects(true);

   osg::Vec3Array* verts = new osg::Vec3Array();
   verts->reserve( 1 + segments );
   geom->setVertexArray( verts );

   osg::DrawElementsUShort* el = new osg::DrawElementsUShort( GL_TRIANGLES );
   el->reserve( 1 + 2*segments );
   geom->addPrimitiveSet( el );

   verts->push_back( osg::Vec3(0,0,0) ); // center point

   for( int i=0; i<segments; ++i )
   {
      double angle = osg::DegreesToRadians( deltaAngle * static_cast<float>(i) );
      float x = static_cast<float>(radius * cos( angle ));
      float y = static_cast<float>(radius * sin( angle ));
      verts->push_back( osg::Vec3(x, y, 0.0f) );

      int i_plus_1 = i < segments-1? i+1 : 0;
      el->push_back( 0 );
      el->push_back( static_cast<GLushort>(1 + i_plus_1 ));
      el->push_back( static_cast<GLushort>(1 + i ));
   }

   return geom;
}
}

// Astronomical Math (http://www.stjarnhimlen.se/comp/ppcomp.html)
namespace
{

#define d2r(X) osg::DegreesToRadians(X)
#define r2d(X) osg::RadiansToDegrees(X)
#define nrad(X) { while( X > TWO_PI ) X -= TWO_PI; while( X < 0.0 ) X += TWO_PI; }
#define nrad2(X) { while( X <= -osg::PI ) X += TWO_PI; while( X > osg::PI ) X -= TWO_PI; }

static const double TWO_PI = (2.0*osg::PI);
static const double JD2000 = 2451545.0;

double getJulianDate( int year, int month, int date )
{
   if ( month <= 2 )
   {
      month += 12;
      year -= 1;
   }
   
   int aVal = int(year/100);
   int bVal = 2-aVal+(aVal/4);
   int cVal = int(365.25*(year+4716));
   int dVal = int(30.6001*(month+1));
   return bVal + cVal + dVal + date - 1524.5;
}

struct Sun
{
   Sun() {}

   // https://www.cfa.harvard.edu/~wsoon/JuanRamirez09-d/Chang09-OptimalTiltAngleforSolarCollector.pdf
   osg::Vec3d getPosition(int year, int month, int date, double hoursUTC ) const
   {
      double JD = getJulianDate(year, month, date);
      double JD1 = (JD - JD2000);                         // julian time since JD2000 epoch
      double JC = JD1/36525.0;                            // julian century

      double mu = 282.937348 + 0.00004707624*JD1 + 0.0004569*(JC*JC);

      double epsilon = 280.466457 + 0.985647358*JD1 + 0.000304*(JC*JC);

      // orbit eccentricity:
      double E = 0.01670862 - 0.00004204 * JC;

      // mean anomaly of the perihelion
      double M = epsilon - mu;

      // perihelion anomaly:
      double v =
         M + 
         360.0*E*sin(d2r(M))/osg::PI + 
         900.0*(E*E)*sin(d2r(2*M))/4*osg::PI - 
         180.0*(E*E*E)*sin(d2r(M))/4.0*osg::PI;

      // longitude of the sun in ecliptic coordinates:
      double sun_lon = d2r(v - 360.0 + mu); // lambda
      nrad2(sun_lon);

      // angle between the ecliptic plane and the equatorial plane
      double zeta = d2r(23.4392); // zeta

      // latitude of the sun on the ecliptic plane:
      //double omega = d2r(0.0);

      // latitude of the sun with respect to the equatorial plane (solar declination):
      double sun_lat = asin( sin(sun_lon)*sin(zeta) );
      nrad2(sun_lat);

      // finally, adjust for the time of day (rotation of the earth)
      double time_r = hoursUTC/24.0; // 0..1
      nrad(sun_lon); // clamp to 0..TWO_PI
      double sun_r = sun_lon/TWO_PI; // convert to 0..1

      // rotational difference between UTC and current time
      double diff_r = sun_r - time_r;
      double diff_lon = TWO_PI * diff_r;

      // apparent sun longitude.
      double app_sun_lon = sun_lon - diff_lon + osg::PI;
      nrad2(app_sun_lon);

      return osg::Vec3d(
         cos(sun_lat) * cos(-app_sun_lon),
         cos(sun_lat) * sin(-app_sun_lon),
         sin(sun_lat) );
   }

   osg::Vec2d getPositionWGS84(int year, int month, int date, double hoursUTC) const
   {
      double JD = getJulianDate(year, month, date);
      double JD1 = (JD - JD2000);                         // julian time since JD2000 epoch
      double JC = JD1 / 36525.0;                            // julian century

      double mu = 282.937348 + 0.00004707624*JD1 + 0.0004569*(JC*JC);

      double epsilon = 280.466457 + 0.985647358*JD1 + 0.000304*(JC*JC);

      // orbit eccentricity:
      double E = 0.01670862 - 0.00004204 * JC;

      // mean anomaly of the perihelion
      double M = epsilon - mu;

      // perihelion anomaly:
      double v =
         M +
         360.0*E*sin(d2r(M)) / osg::PI +
         900.0*(E*E)*sin(d2r(2 * M)) / 4 * osg::PI -
         180.0*(E*E*E)*sin(d2r(M)) / 4.0*osg::PI;

      // longitude of the sun in ecliptic coordinates:
      double sun_lon = d2r(v - 360.0 + mu); // lambda
      nrad2(sun_lon);

      // angle between the ecliptic plane and the equatorial plane
      double zeta = d2r(23.4392); // zeta

      // latitude of the sun on the ecliptic plane:
      //double omega = d2r(0.0);

      // latitude of the sun with respect to the equatorial plane (solar declination):
      double sun_lat = asin(sin(sun_lon)*sin(zeta));
      nrad2(sun_lat);

      // finally, adjust for the time of day (rotation of the earth)
      double time_r = hoursUTC / 24.0; // 0..1
      nrad(sun_lon); // clamp to 0..TWO_PI
      double sun_r = sun_lon / TWO_PI; // convert to 0..1

      // rotational difference between UTC and current time
      double diff_r = sun_r - time_r;
      double diff_lon = TWO_PI * diff_r;

      // apparent sun longitude.
      double app_sun_lon = sun_lon - diff_lon + osg::PI;
      nrad2(app_sun_lon);

      return osg::Vec2d(app_sun_lon, sun_lat);
   }
};

struct Moon
{
   Moon() {}

   static std::string radiansToHoursMinutesSeconds(double ra)
   {
      while (ra < 0) ra += (osg::PI * 2.0);
      //Get the total number of hours
      double hours = (ra / (osg::PI * 2.0) ) * 24.0;
      double minutes = hours - (int)hours;
      hours -= minutes;
      minutes *= 60.0;
      double seconds = minutes - (int)minutes;
      seconds *= 60.0;
      std::stringstream buf;
      buf << (int)hours << ":" << (int)minutes << ":" << (int)seconds;
      return buf.str();
   }

   // From http://www.stjarnhimlen.se/comp/ppcomp.html
   osg::Vec3d getPosition(int year, int month, int date, double hoursUTC ) const
   {
      //double julianDate = getJulianDate( year, month, date );
      //julianDate += hoursUTC /24.0;
      double d = 367*year - 7 * ( year + (month+9)/12 ) / 4 + 275*month/9 + date - 730530;
      d += (hoursUTC / 24.0);                     

      double ecl = osg::DegreesToRadians(23.4393 - 3.563E-7 * d);

      double N = osg::DegreesToRadians(125.1228 - 0.0529538083 * d);
      double i = osg::DegreesToRadians(5.1454);
      double w = osg::DegreesToRadians(318.0634 + 0.1643573223 * d);
      double a = 60.2666;//  (Earth radii)
      double e = 0.054900;
      double M = osg::DegreesToRadians(115.3654 + 13.0649929509 * d);

      double E = M + e*(180.0/osg::PI) * sin(M) * ( 1.0 + e * cos(M) );

      double xv = a * ( cos(E) - e );
      double yv = a * ( sqrt(1.0 - e*e) * sin(E) );

      double v = atan2( yv, xv );
      double r = sqrt( xv*xv + yv*yv );

      //Compute the geocentric (Earth-centered) position of the moon in the ecliptic coordinate system
      double xh = r * ( cos(N) * cos(v+w) - sin(N) * sin(v+w) * cos(i) );
      double yh = r * ( sin(N) * cos(v+w) + cos(N) * sin(v+w) * cos(i) );
      double zh = r * ( sin(v+w) * sin(i) );

      // calculate the ecliptic latitude and longitude here
      double lonEcl = atan2 (yh, xh);
      double latEcl = atan2(zh, sqrt(xh*xh + yh*yh));

      double xg = r * cos(lonEcl) * cos(latEcl);
      double yg = r * sin(lonEcl) * cos(latEcl);
      double zg = r * sin(latEcl);

      double xe = xg;
      double ye = yg * cos(ecl) -zg * sin(ecl);
      double ze = yg * sin(ecl) +zg * cos(ecl);

      double RA    = atan2(ye, xe);
      double Dec = atan2(ze, sqrt(xe*xe + ye*ye));

      //Just use the average distance from the earth            
      double rg = 6378137.0 + 384400000.0;

      // finally, adjust for the time of day (rotation of the earth)
      double time_r = hoursUTC/24.0; // 0..1            
      double moon_r = RA/TWO_PI; // convert to 0..1

      // rotational difference between UTC and current time
      double diff_r = moon_r - time_r;
      double diff_lon = TWO_PI * diff_r;

      RA -= diff_lon;

      nrad2(RA);

      return me_SkyNode::getPositionFromRADecl( RA, Dec, rg );
   }
};
}

namespace
{
// Atmospheric Scattering and Sun Shaders
// Adapted from code that is
// Copyright (c) 2004 Sean O'Neil

static char s_versionString[] =
#ifdef OSG_GLES2_AVAILABLE
   "#version 100 \n";
#else
   "#version 110 \n";
#endif

static char s_mathUtils[] =
   "float fastpow( in float x, in float y ) \n"
   "{ \n"
   "    return x/(x+y-y*x); \n"
   "} \n";

static char s_atmosphereVertexDeclarations[] =
   "uniform mat4 osg_ViewMatrixInverse;     // camera position \n"
   "uniform vec3 atmos_v3LightPos;        // The direction vector to the light source \n"
   "uniform vec3 atmos_v3InvWavelength;   // 1 / pow(wavelength,4) for the rgb channels \n"
   "uniform float atmos_fOuterRadius;     // Outer atmosphere radius \n"
   "uniform float atmos_fOuterRadius2;    // fOuterRadius^2 \n"		
   "uniform float atmos_fInnerRadius;     // Inner planetary radius \n"
   "uniform float atmos_fInnerRadius2;    // fInnerRadius^2 \n"
   "uniform float atmos_fKrESun;          // Kr * ESun \n"	
   "uniform float atmos_fKmESun;          // Km * ESun \n"		
   "uniform float atmos_fKr4PI;           // Kr * 4 * PI \n"	
   "uniform float atmos_fKm4PI;           // Km * 4 * PI \n"		
   "uniform float atmos_fScale;           // 1 / (fOuterRadius - fInnerRadius) \n"	
   "uniform float atmos_fScaleDepth;      // The scale depth \n"
   "uniform float atmos_fScaleOverScaleDepth;     // fScale / fScaleDepth \n"	
   "uniform int atmos_nSamples; \n"	
   "uniform float atmos_fSamples; \n"

   "varying vec3 atmos_v3Direction; \n"
   "varying vec3 atmos_mieColor; \n"
   "varying vec3 atmos_rayleighColor; \n"

   "vec3 vVec; \n"
   "float atmos_fCameraHeight;    // The camera's current height \n"		
   "float atmos_fCameraHeight2;   // fCameraHeight^2 \n";

static char s_atmosphereVertexShared[] =
   "float atmos_scale(float fCos) \n"	
   "{ \n"
   "    float x = 1.0 - fCos; \n"
   "    return atmos_fScaleDepth * exp(-0.00287 + x*(0.459 + x*(3.83 + x*(-6.80 + x*5.25)))); \n"
   "} \n"

   "void SkyFromSpace(void) \n"
   "{ \n"
   "    // Get the ray from the camera to the vertex and its length (which is the far point of the ray passing through the atmosphere) \n"
   "    vec3 v3Pos = gl_Vertex.xyz; \n"
   "    vec3 v3Ray = v3Pos - vVec; \n"
   "    float fFar = length(v3Ray); \n"
   "    v3Ray /= fFar; \n"

   "    // Calculate the closest intersection of the ray with the outer atmosphere \n"
   "    // (which is the near point of the ray passing through the atmosphere) \n"
   "    float B = 2.0 * dot(vVec, v3Ray); \n"
   "    float C = atmos_fCameraHeight2 - atmos_fOuterRadius2; \n"
   "    float fDet = max(0.0, B*B - 4.0 * C); \n"	
   "    float fNear = 0.5 * (-B - sqrt(fDet)); \n"		

   "    // Calculate the ray's starting position, then calculate its atmos_ing offset \n"
   "    vec3 v3Start = vVec + v3Ray * fNear; \n"			
   "    fFar -= fNear; \n"	
   "    float fStartAngle = dot(v3Ray, v3Start) / atmos_fOuterRadius; \n"			
   "    float fStartDepth = exp(-1.0 / atmos_fScaleDepth); \n"
   "    float fStartOffset = fStartDepth*atmos_scale(fStartAngle); \n"		

   "    // Initialize the atmos_ing loop variables \n"	
   "    float fSampleLength = fFar / atmos_fSamples; \n"		
   "    float fScaledLength = fSampleLength * atmos_fScale; \n"					
   "    vec3 v3SampleRay = v3Ray * fSampleLength; \n"	
   "    vec3 v3SamplePoint = v3Start + v3SampleRay * 0.5; \n"	

   "    // Now loop through the sample rays \n"
   "    vec3 v3FrontColor = vec3(0.0, 0.0, 0.0); \n"
   "    vec3 v3Attenuate; \n"  
   "    for(int i=0; i<atmos_nSamples; i++) \n"		
   "    { \n"
   "        float fHeight = length(v3SamplePoint); \n"			
   "        float fDepth = exp(atmos_fScaleOverScaleDepth * (atmos_fInnerRadius - fHeight)); \n"
   "        float fLightAngle = dot(atmos_v3LightPos, v3SamplePoint) / fHeight; \n"		
   "        float fCameraAngle = dot(v3Ray, v3SamplePoint) / fHeight; \n"			
   "        float fscatter = (fStartOffset + fDepth*(atmos_scale(fLightAngle) - atmos_scale(fCameraAngle))); \n"	
   "        v3Attenuate = exp(-fscatter * (atmos_v3InvWavelength * atmos_fKr4PI + atmos_fKm4PI)); \n"	
   "        v3FrontColor += v3Attenuate * (fDepth * fScaledLength); \n"					
   "        v3SamplePoint += v3SampleRay; \n"		
   "    } \n"		

   "    // Finally, scale the Mie and Rayleigh colors and set up the varying \n"			
   "    // variables for the pixel shader \n"	
   "    atmos_mieColor      = v3FrontColor * atmos_fKmESun; \n"				
   "    atmos_rayleighColor = v3FrontColor * (atmos_v3InvWavelength * atmos_fKrESun); \n"						
   "    atmos_v3Direction = vVec  - v3Pos; \n"			
   "} \n"		

   "void SkyFromAtmosphere(void) \n"		
   "{ \n"
   "  // Get the ray from the camera to the vertex, and its length (which is the far \n"
   "  // point of the ray passing through the atmosphere) \n"		
   "  vec3 v3Pos = gl_Vertex.xyz; \n"	
   "  vec3 v3Ray = v3Pos - vVec; \n"			
   "  float fFar = length(v3Ray); \n"					
   "  v3Ray /= fFar; \n"				

   "  // Calculate the ray's starting position, then calculate its atmos_ing offset \n"
   "  vec3 v3Start = vVec; \n"
   "  float fHeight = length(v3Start); \n"		
   "  float fDepth = exp(atmos_fScaleOverScaleDepth * (atmos_fInnerRadius - atmos_fCameraHeight)); \n"
   "  float fStartAngle = dot(v3Ray, v3Start) / fHeight; \n"	
   "  float fStartOffset = fDepth*atmos_scale(fStartAngle); \n"

   "  // Initialize the atmos_ing loop variables \n"		
   "  float fSampleLength = fFar / atmos_fSamples; \n"			
   "  float fScaledLength = fSampleLength * atmos_fScale; \n"				
   "  vec3 v3SampleRay = v3Ray * fSampleLength; \n"		
   "  vec3 v3SamplePoint = v3Start + v3SampleRay * 0.5; \n"

   "  // Now loop through the sample rays \n"		
   "  vec3 v3FrontColor = vec3(0.0, 0.0, 0.0); \n"		
   "  vec3 v3Attenuate; \n"  
   "  for(int i=0; i<atmos_nSamples; i++) \n"			
   "  { \n"	
   "    float fHeight = length(v3SamplePoint); \n"	
   "    float fDepth = exp(atmos_fScaleOverScaleDepth * (atmos_fInnerRadius - fHeight)); \n"
   "    float fLightAngle = dot(atmos_v3LightPos, v3SamplePoint) / fHeight; \n"
   "    float fCameraAngle = dot(v3Ray, v3SamplePoint) / fHeight; \n"	
   "    float fscatter = (fStartOffset + fDepth*(atmos_scale(fLightAngle) - atmos_scale(fCameraAngle))); \n"	
   "    v3Attenuate = exp(-fscatter * (atmos_v3InvWavelength * atmos_fKr4PI + atmos_fKm4PI)); \n"	
   "    v3FrontColor += v3Attenuate * (fDepth * fScaledLength); \n"		
   "    v3SamplePoint += v3SampleRay; \n"		
   "  } \n"

   "  // Finally, scale the Mie and Rayleigh colors and set up the varying \n"
   "  // variables for the pixel shader \n"					
   "  atmos_mieColor      = v3FrontColor * atmos_fKmESun; \n"			
   "  atmos_rayleighColor = v3FrontColor * (atmos_v3InvWavelength * atmos_fKrESun); \n"				
   "  atmos_v3Direction = vVec - v3Pos; \n"				
   "} \n";

static char s_atmosphereVertexExtShader[] = {
    "void mainExt() {\n"
    "}"
};
static char s_atmosphereVertexMain[] =
   "void main(void) \n"
   "{ \n"
   "  // Get camera position and height \n"
   "  vVec = osg_ViewMatrixInverse[3].xyz; \n"
   "  atmos_fCameraHeight = length(vVec); \n"
   "  atmos_fCameraHeight2 = atmos_fCameraHeight*atmos_fCameraHeight; \n"

   "  vec4 vertex = gl_ModelViewProjectionMatrix * gl_Vertex; \n"
   "  float NDCz = vertex.z / vertex.w; \n"
   "  if ((NDCz >= -1.0) && (NDCz <= 1.0)) vertex.z = vertex.w * 0.999999; // If inside the frustum, push the vertex to the end of the view-frustum to let it depth-test against everything drawn before it \n"
   "  gl_Position = vertex; \n"

   "  if(atmos_fCameraHeight >= atmos_fOuterRadius) { \n"
   "      SkyFromSpace(); \n"
   "  } \n"
   "  else { \n"
   "      SkyFromAtmosphere(); \n"
   "  } \n"
   "  mainExt();\n"
   "} \n";

static char s_atmosphereFragmentDeclarations[] =
   "uniform vec3 atmos_v3LightPos; \n"							
   "uniform float atmos_g; \n"				
   "uniform float atmos_g2; \n"
   "uniform float atmos_fWeather; \n"

   "varying vec3 atmos_v3Direction; \n"	
   "varying vec3 atmos_mieColor; \n"
   "varying vec3 atmos_rayleighColor; \n"

   "const float fExposure = 4.0; \n";

static char s_atmosphereFragmentExtShader[] = {
    "\n"
    "void mainExt(vec3 color) {\n"
    "    gl_FragColor.rgb = color.rgb*atmos_fWeather; \n"
    "    gl_FragColor.a = (color.r+color.g+color.b) * 2.0; \n"
    "}\n"
    "\n"
};
static char s_atmosphereFragmentMain[] =
   "void main(void) \n"			
   "{ \n"				
   "    float fCos = dot(atmos_v3LightPos, atmos_v3Direction) / length(atmos_v3Direction); \n"
   "    float fRayleighPhase = 1.0; \n" // 0.75 * (1.0 + fCos*fCos); \n"
   "    float fMiePhase = 1.5 * ((1.0 - atmos_g2) / (2.0 + atmos_g2)) * (1.0 + fCos*fCos) / fastpow(1.0 + atmos_g2 - 2.0*atmos_g*fCos, 1.5); \n"
   "    vec3 f4Color = fRayleighPhase * atmos_rayleighColor + fMiePhase * atmos_mieColor; \n"
   "    vec3 color = 1.0 - exp(f4Color * -fExposure); \n"
   "    mainExt(color); \n"
   "} \n";

static char s_sunVertexSource[] = 
   "varying vec3 atmos_v3Direction; \n"

   "void main() \n"
   "{ \n"
   "    vec3 v3Pos = gl_Vertex.xyz; \n"
   "    gl_Position = gl_ModelViewProjectionMatrix * gl_Vertex; \n"
   "    atmos_v3Direction = vec3(0.0,0.0,1.0) - v3Pos; \n"
   "    atmos_v3Direction = atmos_v3Direction/length(atmos_v3Direction); \n"
   "} \n";

static char s_sunFragmentSource[] =
   "uniform float sunAlpha; \n"
   "varying vec3 atmos_v3Direction; \n"

   "void main( void ) \n"
   "{ \n"
   "   float fCos = -atmos_v3Direction[2]; \n"         
   "   float fMiePhase = 0.050387596899224826 * (1.0 + fCos*fCos) / fastpow(1.9024999999999999 - -1.8999999999999999*fCos, 1.5); \n"
   "   gl_FragColor.rgb = fMiePhase*vec3(.3,.3,.2); \n"
   "   gl_FragColor.a = sunAlpha*gl_FragColor.r; \n"
   "} \n";

static char s_moonVertexSource[] = 
   "uniform mat4 osg_ModelViewProjectionMatrix;"
   "varying vec4 moon_TexCoord;\n"
   "void main() \n"
   "{ \n"
   "    moon_TexCoord = gl_MultiTexCoord0; \n"
   "    gl_Position = osg_ModelViewProjectionMatrix * gl_Vertex; \n"
   "} \n";

static char s_moonFragmentSource[] =
   "varying vec4 moon_TexCoord;\n"
   "uniform sampler2D moonTex;\n"
   "void main( void ) \n"
   "{ \n"
   "   gl_FragColor = texture2D(moonTex, moon_TexCoord.st);\n"
   "} \n";
}

namespace
{
static std::string s_createStarVertexSource()
{
   float glslVersion = 1.1f;

   return Stringify()
      << "#version " << (glslVersion < 1.2f ? GLSL_VERSION_STR : "120") << "\n"

      << "float remap( float val, float vmin, float vmax, float r0, float r1 ) \n"
      << "{ \n"
      << "    float vr = (clamp(val, vmin, vmax)-vmin)/(vmax-vmin); \n"
      << "    return r0 + vr * (r1-r0); \n"
      << "} \n"

      << "uniform vec3 atmos_v3LightPos; \n"
      << "uniform mat4 osg_ViewMatrixInverse; \n"
      << "varying float visibility; \n"
      << "varying vec4 osg_FrontColor; \n"
      << "void main() \n"
      << "{ \n"
      << "    osg_FrontColor = gl_Color; \n"
      << "    gl_PointSize = gl_Color.r * " << (glslVersion < 1.2f ? "2.0" : "14.0") << ";\n"
      << "    vec4 vertex = gl_ModelViewProjectionMatrix * gl_Vertex; \n"
      << "    // Push all stars to the end of the view-frustum to let it falls inside the frustum and behind everything else.\n"
      << "    // view-frustum check is not performed as all the stars can be drawn as they are GL_POINTS. \n"
      << "    // This is done in shader instead of pushing the farplane to a high value \n"
      << "    vertex.z = vertex.w * 0.999999;\n"
      << "    gl_Position = vertex; \n"

      << "    vec3 eye = osg_ViewMatrixInverse[3].xyz; \n"
      << "    float hae = length(eye) - 6378137.0; \n"
      // "highness": visibility increases with altitude
      << "    float highness = remap( hae, 25000.0, 150000.0, 0.0, 1.0 ); \n"
      << "    eye = normalize(eye); \n"
      // "darkness": visibility increase as the sun goes around the other side of the earth
      << "    float darkness = 1.0-remap(dot(eye,atmos_v3LightPos), -0.25, 0.0, 0.0, 1.0); \n"
      << "    visibility = clamp(highness + darkness, 0.0, 1.0); \n"
      << "} \n";
}

static std::string s_createStarFragmentSource()
{
   float glslVersion = 1.1f;

   if ( glslVersion < 1.2f )
   {
      return Stringify()
         << "#version " << GLSL_VERSION_STR << "\n"
#ifdef OSG_GLES2_AVAILABLE
         << "precision mediump float;\n"
#endif  
         << "varying float visibility; \n"
         << "varying vec4 osg_FrontColor; \n"
         << "void main( void ) \n"
         << "{ \n"
         << "    gl_FragColor = osg_FrontColor * visibility; \n"
         << "} \n";
   }
   else
   {
      return Stringify()
         << "#version 120 \n"
#ifdef OSG_GLES2_AVAILABLE
         << "precision mediump float;\n"
#endif
         << "varying float visibility; \n"
         << "varying vec4 osg_FrontColor; \n"
         << "void main( void ) \n"
         << "{ \n"
         << "    float b1 = 1.0-(2.0*abs(gl_PointCoord.s-0.5)); \n"
         << "    float b2 = 1.0-(2.0*abs(gl_PointCoord.t-0.5)); \n"
         << "    float i = b1*b1 * b2*b2; \n" //b1*b1*b1 * b2*b2*b2; \n"
         << "    gl_FragColor = osg_FrontColor * i * visibility; \n"
         << "} \n";
   }
}
}

osg::Vec3d DefaultEphemerisProvider::getSunPosition( int year, int month, int date, double hoursUTC )
{
   Sun sun;
   return sun.getPosition( year, month, date, hoursUTC );
}

osg::Vec2d DefaultEphemerisProvider::getSunPositionWGS84(int year, int month, int date, double hoursUTC)
{
   Sun sun;
   return sun.getPositionWGS84(year, month, date, hoursUTC);
}

osg::Vec3d DefaultEphemerisProvider::getMoonPosition( int year, int month, int date, double hoursUTC )
{
   Moon moon;
   return moon.getPosition( year, month, date, hoursUTC );
}

me_SkyNode::me_SkyNode( osg::Node* /*parent*/, float minStarMagnitude )
 : _year(2011)
 , _month(1)
 , _date(1)
 , _hoursUTC(0.0)
 , _innerRadius(0.0f)
 , _outerRadius(0.0f)
 , _sunDistance(0.0f)
 , _starRadius(0.0f)
 , _minStarMagnitude(minStarMagnitude)
 , _renderBinAtmosphereInside(BIN_ATMOSPHERE)
 , _renderBinAtmosphereOutside(BIN_ATMOSPHERE)
 , _skyToOccludeStars(false)
{
   setName("SkyNode");
}

void me_SkyNode::initializeEllipsoidModel(const osg::EllipsoidModel* em) 
{
   _ephemerisProvider = new DefaultEphemerisProvider();

   _lightPos = osg::Vec3f(0.0f, 1.0f, 0.0f);

   _light = new osg::Light(0);
   _light->setPosition(osg::Vec4f(0.0f, 0.0f, 1.0, 0.0f));
   _light->setAmbient(osg::Vec4f(0.03f, 0.03f, 0.03f, 1.0f));
   _light->setDiffuse(osg::Vec4f(1.0f, 1.0f, 1.0f, 1.0f));
   _light->setSpecular(osg::Vec4f(1.0f, 1.0f, 1.0f, 1.0f));

   // containers for sky elements.
   _cullContainer = new osg::Group();

   _lightPosUniform = new osg::Uniform(osg::Uniform::FLOAT_VEC3, "atmos_v3LightPos");
   _lightPosUniform->set(_lightPos / _lightPos.length());

   // set up the astronomical parameters:
   _ellipsoidModel = em;
   _innerRadius = static_cast<float>(_ellipsoidModel->getRadiusPolar());

   _outerRadius = _innerRadius * 1.03f; //1.038, 1.048; 1.035
   _sunDistance = _innerRadius * 12000.0f;
   _innerRadius -= 0.02f * _innerRadius;  //0.009, 0.008

}
void me_SkyNode::initialize( const osg::EllipsoidModel* em, const std::string& starFile )
{
   initializeEllipsoidModel(em);
   // make the sky elements (don't change the order here)
   makeAtmosphere( _ellipsoidModel.get() );
   makeSun();
   //makeMoon();
   makeStars(starFile);
   setDateTime( 2011, 3, 6, 12 );
}

osg::BoundingSphere me_SkyNode::computeBound() const
{
   return osg::BoundingSphere();
}

void me_SkyNode::traverse( osg::NodeVisitor& nv )
{
   if (nv.getVisitorType() == osg::NodeVisitor::CULL_VISITOR)
   {
      osgUtil::CullVisitor* cv = static_cast<osgUtil::CullVisitor*>( &nv );   //lint  !e1774

      // If there's a custom projection matrix clamper installed, remove it temporarily.
      // We don't want it mucking with our sky elements.
      osg::ref_ptr<osg::CullSettings::ClampProjectionMatrixCallback> cb = cv->getClampProjectionMatrixCallback();
      cv->setClampProjectionMatrixCallback( 0L );

      osg::RefMatrix* mvMat = cv->getModelViewMatrix();
      osg::Matrixd invMat = osg::Matrixd::inverse(*mvMat);
      osg::Vec3d trans = invMat.getTrans();

      if (trans.length() >= _outerRadius) // the length of the translation component is the camera height (see shader code)
      {
         _depthSettingsAtmosphere->setFunction(getSkyToOccludeStars() ? osg::Depth::LESS : osg::Depth::ALWAYS);
         _atmosphere->getOrCreateStateSet()->setRenderBinDetails( getRenderBinAtmosphereOutside(), "RenderBin" );
         if (getSkyToOccludeStars())
         {
            _stars->getOrCreateStateSet()->setRenderBinDetails(getRenderBinAtmosphereOutside() + 1, "RenderBin");
         }
      }
      else
      {
         _depthSettingsAtmosphere->setFunction(osg::Depth::LESS);
         _atmosphere->getOrCreateStateSet()->setRenderBinDetails( getRenderBinAtmosphereInside(), "RenderBin" );
         if (getSkyToOccludeStars())
         {
            _stars->getOrCreateStateSet()->setRenderBinDetails(getRenderBinAtmosphereInside() + 1, "RenderBin");
         }
      }

      _cullContainer->accept( nv );

      // restore a custom clamper.
      if ( cb.valid() ) cv->setClampProjectionMatrixCallback( cb.get() );
   }
   else
   {
      osg::Group::traverse( nv );
   }
}

EphemerisProvider*
me_SkyNode::getEphemerisProvider() const
{
   return _ephemerisProvider;
}

void me_SkyNode::setEphemerisProvider(EphemerisProvider* ephemerisProvider)
{
   if (_ephemerisProvider != ephemerisProvider)
   {
      _ephemerisProvider = ephemerisProvider;
      setDateTime(_year, _month, _date, _hoursUTC);
   }
}

void me_SkyNode::attach( osg::View* view, int lightNum )
{
   if ( !view || !_light.valid() )
      return;

   _light->setLightNum( lightNum );
   view->setLight( _light.get() );
   view->setLightingMode( osg::View::SKY_LIGHT );
   view->getCamera()->setClearColor( osg::Vec4(0,0,0,1) );
   view->getCamera()->getOrCreateStateSet()->addUniform( _lightPosUniform.get() );
}

float me_SkyNode::getAmbientBrightness() const
{
   return _light->getAmbient().r();
}

void me_SkyNode::setAmbientBrightness( float value )
{
   value = osg::clampBetween( value, 0.0f, 1.0f );
   _light->setAmbient( osg::Vec4f(value, value, value, 1.0f) );
}

void me_SkyNode::getSunPosition( osg::Vec3& pos )
{
   pos = _lightPos / _lightPos.length();
}

void me_SkyNode::getSunPosition(osg::Vec2d& pos)
{
   int year = 0;
   int month = 0;
   int date = 0;
   double hoursUTC = 0.0;
   getDateTime(year, month, date, hoursUTC);

   pos = _ephemerisProvider->getSunPositionWGS84(year, month, date, hoursUTC);
}

void me_SkyNode::setSunPosition( const osg::Vec3& pos )
{
   _lightPos = pos;

   _light->setPosition( osg::Vec4(pos, 0.0f) );

   _lightPosUniform->set( pos/pos.length() );

   _sunXform->setMatrix( osg::Matrix::translate(
      _sunDistance * pos.x(),
      _sunDistance * pos.y(),
      _sunDistance * pos.z() ) );
}

void me_SkyNode::setSunPosition( double lat_degrees, double long_degrees )
{
   if (_ellipsoidModel.valid())
   {
      double x, y, z;
      _ellipsoidModel->convertLatLongHeightToXYZ(
         osg::RadiansToDegrees(lat_degrees),
         osg::RadiansToDegrees(long_degrees),
         0, 
         x, y, z);
      osg::Vec3d up  = _ellipsoidModel->computeLocalUpVector(x, y, z);
      setSunPosition( up );
   }
}

void me_SkyNode::setMoonPosition( const osg::Vec3d& pos )
{
   if (_moonXform.valid()) _moonXform->setMatrix(osg::Matrix::translate( pos.x(), pos.y(), pos.z() ));
}

void me_SkyNode::getDateTime( int &year, int &month, int &date, double &hoursUTC )
{    
   year = _year;
   month = _month;
   date = _date;
   hoursUTC = _hoursUTC;
}

void me_SkyNode::setDateTime( int year, int month, int date, double hoursUTC )
{
   _year = year;
   _month = month;
   _date = date;
   _hoursUTC = hoursUTC;

   if ( _ellipsoidModel.valid() )
   {
      osg::Vec3d sunPosition;
      osg::Vec3d moonPosition;

      if (_ephemerisProvider)
      {
         sunPosition = _ephemerisProvider->getSunPosition( year, month, date, hoursUTC );
         moonPosition = _ephemerisProvider->getMoonPosition( year, month, date, hoursUTC );
      }

      sunPosition.normalize();
      setSunPosition( sunPosition );
      setMoonPosition( moonPosition );       

      // position the stars:
      double time_r = hoursUTC/24.0; // 0..1
      double rot_z = -osg::PI + TWO_PI*time_r;

      _starsXform->setMatrix(osg::Matrix::rotate( -rot_z, 0, 0, 1 ));
      _year = year;
      _month = month;
      _date = date;
      _hoursUTC = hoursUTC;
   }
}

void me_SkyNode::setDateTime( bool ff )
{
   if (ff)
      setDateTime(_year, _month, _date, _hoursUTC + .125);
   else
      setDateTime(_year, _month, _date, _hoursUTC - .125);
}

void me_SkyNode::setStarsVisible(bool value)
{
   _stars->setNodeMask(value ? ~0 : 0);
}

void me_SkyNode::setAtmosphereVisible(bool value)
{
   _atmosphere->setNodeMask(value ? ~0 : 0);
}

bool me_SkyNode::getStarsVisible() const
{
   return _stars->getNodeMask() != 0 ? true : false;
}

bool me_SkyNode::getAtmosphereVisible() const
{
   return _atmosphere->getNodeMask() != 0 ? true : false;
}

osg::ref_ptr<osg::Program> me_SkyNode::generateAtmosphereShaderProgram(const char* programName, const char* vertexShaderExtensionCode, const char* fragmentShaderExtensionCode)
{
   osg::Program* program = new osg::Program();
   program->setName(programName);

   osg::Shader* vs = new osg::Shader(osg::Shader::VERTEX, Stringify()
      << s_versionString
      << s_mathUtils
      << s_atmosphereVertexDeclarations
      << s_atmosphereVertexShared
      << vertexShaderExtensionCode
      << s_atmosphereVertexMain);
   program->addShader(vs);

   osg::Shader* fs = new osg::Shader(osg::Shader::FRAGMENT, Stringify()
      << s_versionString
#ifdef OSG_GLES2_AVAILABLE
      << "precision highp float;\n"
#endif
      << s_mathUtils
      << s_atmosphereFragmentDeclarations
      << fragmentShaderExtensionCode
      << s_atmosphereFragmentMain);
   program->addShader(fs);

   return program;
}

osg::Geode* me_SkyNode::makeAtmosphere(const osg::EllipsoidModel* /*em*/)
{
   // create some skeleton geometry to shade
   osg::Geode* geode = new osg::Geode();

   me_IcoSphere icosphere(static_cast<unsigned int>(_outerRadius));
   unsigned int recursionLevel = 4;
   icosphere.create(recursionLevel);

   osg::ref_ptr<osg::Vec3Array> spherePoints = icosphere._vertices;
   osg::ref_ptr<osg::UShortArray> indices = icosphere._indices;

   unsigned int numOfGeometries = 16;
   unsigned int numOfVertsPerGeometry = static_cast<unsigned int>(indices->size())/numOfGeometries;
   numOfVertsPerGeometry -= numOfVertsPerGeometry%3;
   unsigned int numOfVertInLastGeom = static_cast<unsigned int>(indices->size()) - (numOfVertsPerGeometry * (numOfGeometries-1));

   for(unsigned int i = 0; i < numOfGeometries ; ++i)
   {
      osg::Geometry* geom = new osg::Geometry();
      geom->setUseVertexBufferObjects(true);

      geom->setVertexArray( spherePoints.get() );
      geom->addPrimitiveSet(new osg::DrawElementsUShort(osg::PrimitiveSet::TRIANGLES,i != numOfGeometries-1 ? numOfVertsPerGeometry : numOfVertInLastGeom,&(*indices)[i*numOfVertsPerGeometry]));
      geode->addDrawable(geom);
   }

   // configure the state set
   osg::StateSet* set = geode->getOrCreateStateSet();
   set->setMode( GL_LIGHTING, osg::StateAttribute::OFF );
   set->setAttributeAndModes( new osg::CullFace(osg::CullFace::BACK), osg::StateAttribute::ON );
   set->setAttributeAndModes( new osg::BlendFunc( GL_ONE, GL_ONE ), osg::StateAttribute::ON );

   // create and add the shader
   osg::ref_ptr<osg::Program> program = generateAtmosphereShaderProgram("AtmosphereShader", s_atmosphereVertexExtShader, s_atmosphereFragmentExtShader);
   set->setAttributeAndModes( program.get(), osg::StateAttribute::ON );

   // apply the uniforms
   float r_wl   = ::powf( .65f, 4.0f );
   float g_wl = ::powf( .57f, 4.0f );
   float b_wl  = ::powf( .475f, 4.0f );
   osg::Vec3 RGB_wl( 1.0f/r_wl, 1.0f/g_wl, 1.0f/b_wl );
   float Kr = 0.003f;//0.003f;
   float Kr4PI = static_cast<float>(Kr * 4.0f * osg::PI);
   float Km = 0.0005f; //0.0005
   float Km4PI = static_cast<float>(Km * 4.0f * osg::PI);
   float ESun = 25.0f;//22.0f;
   float MPhase = -.095f;//-.095f;
   float RayleighScaleDepth = 0.25f;
   int   Samples = 2;
   float Weather = 1.0f;
   float Scale = 1.0f / (_outerRadius - _innerRadius);

   set->getOrCreateUniform( "atmos_v3InvWavelength", osg::Uniform::FLOAT_VEC3 )->set( RGB_wl );
   set->getOrCreateUniform( "atmos_fInnerRadius",    osg::Uniform::FLOAT )->set( _innerRadius );
   set->getOrCreateUniform( "atmos_fInnerRadius2",   osg::Uniform::FLOAT )->set( _innerRadius * _innerRadius );
   set->getOrCreateUniform( "atmos_fOuterRadius",    osg::Uniform::FLOAT )->set( _outerRadius );
   set->getOrCreateUniform( "atmos_fOuterRadius2",   osg::Uniform::FLOAT )->set( _outerRadius * _outerRadius );
   set->getOrCreateUniform( "atmos_fKrESun",         osg::Uniform::FLOAT )->set( Kr * ESun );
   set->getOrCreateUniform( "atmos_fKmESun",         osg::Uniform::FLOAT )->set( Km * ESun );
   set->getOrCreateUniform( "atmos_fKr4PI",          osg::Uniform::FLOAT )->set( Kr4PI );
   set->getOrCreateUniform( "atmos_fKm4PI",          osg::Uniform::FLOAT )->set( Km4PI );
   set->getOrCreateUniform( "atmos_fScale",          osg::Uniform::FLOAT )->set( Scale );
   set->getOrCreateUniform( "atmos_fScaleDepth",     osg::Uniform::FLOAT )->set( RayleighScaleDepth );
   set->getOrCreateUniform( "atmos_fScaleOverScaleDepth", osg::Uniform::FLOAT )->set( Scale / RayleighScaleDepth );
   set->getOrCreateUniform( "atmos_g",               osg::Uniform::FLOAT )->set( MPhase );
   set->getOrCreateUniform( "atmos_g2",              osg::Uniform::FLOAT )->set( MPhase * MPhase );
   set->getOrCreateUniform( "atmos_nSamples",        osg::Uniform::INT )->set( Samples );
   set->getOrCreateUniform( "atmos_fSamples",        osg::Uniform::FLOAT )->set( static_cast<float>(Samples) );
   set->getOrCreateUniform( "atmos_fWeather",        osg::Uniform::FLOAT )->set( Weather );


   // A nested camera isolates the projection matrix calculations so the node won't 
   // affect the clip planes in the rest of the scene.
   osg::Camera* cam = new osg::Camera();
   cam->setName("SkyAtmosCam");
   cam->setRenderOrder( osg::Camera::NESTED_RENDER );
   cam->setComputeNearFarMode( osg::CullSettings::COMPUTE_NEAR_FAR_USING_BOUNDING_VOLUMES );
   cam->addChild(geode);

   _depthSettingsAtmosphere = new osg::Depth( osg::Depth::LESS, 0, 1, getSkyToOccludeStars() );
   cam->getOrCreateStateSet()->setAttributeAndModes(_depthSettingsAtmosphere, osg::StateAttribute::ON);

   _atmosphere = cam;

   _cullContainer->addChild(_atmosphere.get());

   return geode;
}

void me_SkyNode::makeSun()
{
   osg::Billboard* sun = new osg::Billboard();
   sun->setMode( osg::Billboard::POINT_ROT_EYE );
   sun->setNormal( osg::Vec3(0, 0, 1) );

   float sunRadius = _innerRadius * 100.0f;

   sun->addDrawable( s_makeDiscGeometry( sunRadius*80.0f ) ); 

   osg::StateSet* set = sun->getOrCreateStateSet();
   set->setMode( GL_BLEND, 1 );

   set->getOrCreateUniform( "sunAlpha", osg::Uniform::FLOAT )->set( 1.0f );

   // configure the stateset
   set->setMode( GL_LIGHTING, osg::StateAttribute::OFF );
   set->setMode( GL_CULL_FACE, osg::StateAttribute::OFF );
   set->setAttributeAndModes( new osg::Depth(osg::Depth::ALWAYS, 0, 1, false), osg::StateAttribute::ON );
   // set->setAttributeAndModes( new osg::BlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA), osg::StateAttribute::ON );

   // create shader
   osg::Program* program = new osg::Program();
   program->setName("SunShader");
   osg::Shader* vs = new osg::Shader( osg::Shader::VERTEX, Stringify()
      << s_versionString
      << s_sunVertexSource );
   program->addShader( vs );
   osg::Shader* fs = new osg::Shader( osg::Shader::FRAGMENT, Stringify()
      << s_versionString
#ifdef OSG_GLES2_AVAILABLE
      << "precision mediump float;\n"
#endif
      << s_mathUtils
      << s_sunFragmentSource );
   program->addShader( fs );
   set->setAttributeAndModes( program, osg::StateAttribute::ON );

   // A nested camera isolates the projection matrix calculations so the node won't 
   // affect the clip planes in the rest of the scene.
   osg::Camera* cam = new osg::Camera();
   cam->getOrCreateStateSet()->setRenderBinDetails( BIN_SUN, "RenderBin" );
   cam->setRenderOrder( osg::Camera::NESTED_RENDER );
   cam->setComputeNearFarMode( osg::CullSettings::COMPUTE_NEAR_FAR_USING_BOUNDING_VOLUMES );
   cam->addChild( sun );

   _sun = cam;

   // make the sun's transform
   _sunXform = new osg::MatrixTransform();
   _sunXform->setMatrix( osg::Matrix::translate(
      _sunDistance * _light->getPosition().x(),
      _sunDistance * _light->getPosition().y(),
      _sunDistance * _light->getPosition().z() ) );
   _sunXform->addChild( _sun.get() );

   _cullContainer->addChild( _sunXform.get() );
}

void me_SkyNode::makeMoon()
{
   osg::Image* image = osgDB::readImageFile( "moon.png" );
   if (!image) return; //can't create moon without moon texture anyway

   osg::ref_ptr< osg::EllipsoidModel > em = new osg::EllipsoidModel( 1738140.0, 1735970.0 );   
   osg::Geode* moon = new osg::Geode;
   moon->getOrCreateStateSet()->setAttributeAndModes( new osg::Program(), osg::StateAttribute::OFF | osg::StateAttribute::PROTECTED );

   //If the icosphere approach is used for the moon, then atan2() will have to be used in the fragment shader to fix the tex coords running along the pole
   //so the UV sphere approach will be used for the moon instead of icosphere 

   moon->addDrawable(createEllipsoidGeometry( em.get(), em->getRadiusEquator(), 1, 1, true ));
   osg::Texture2D * texture = new osg::Texture2D( image );
   texture->setFilter(osg::Texture::MIN_FILTER,osg::Texture::LINEAR);
   texture->setFilter(osg::Texture::MAG_FILTER,osg::Texture::LINEAR);
   texture->setUnRefImageDataAfterApply(true);
   texture->setResizeNonPowerOfTwoHint(false);
   osg::Vec4Array* colors = new osg::Vec4Array(1);    
   (*colors)[0] = osg::Vec4(1, 1, 1, 1 );

   for(unsigned int i = 0; i < moon->getNumDrawables(); ++i)
   {
      osg::Geometry* geom = static_cast<osg::Geometry*>(moon->getDrawable(i));   //lint  !e1774
      geom->setColorArray( colors, osg::Array::BIND_OVERALL );
   }

   moon->getOrCreateStateSet()->setTextureAttribute( 0, texture, osg::StateAttribute::OFF | osg::StateAttribute::PROTECTED);

   // configure the stateset
   osg::StateSet* set = moon->getOrCreateStateSet();
   set->setMode( GL_LIGHTING, osg::StateAttribute::ON );
   set->setAttributeAndModes( new osg::CullFace( osg::CullFace::BACK ), osg::StateAttribute::ON);
   set->setRenderBinDetails( BIN_MOON, "RenderBin" );
   set->setAttributeAndModes( new osg::Depth(osg::Depth::ALWAYS, 0, 1, false), osg::StateAttribute::ON );
   set->setAttributeAndModes( new osg::BlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA), osg::StateAttribute::ON );

#ifdef OSG_GLES2_AVAILABLE
   set->addUniform(new osg::Uniform("moonTex", 0));

   // create shaders
   osg::Program* program = new osg::Program();
   program->setName("MoonShader");
   osg::Shader* vs = new osg::Shader( osg::Shader::VERTEX, Stringify()
      << s_versionString
      << "precision mediump float;\n"
      << s_moonVertexSource );
   program->addShader( vs );
   osg::Shader* fs = new osg::Shader( osg::Shader::FRAGMENT, Stringify()
      << s_versionString
      << "precision mediump float;\n"
      << s_moonFragmentSource );
   program->addShader( fs );
   set->setAttributeAndModes( program, osg::StateAttribute::ON | osg::StateAttribute::PROTECTED );
#endif

   // A nested camera isolates the projection matrix calculations so the node won't 
   // affect the clip planes in the rest of the scene.
   osg::Camera* cam = new osg::Camera();
   cam->getOrCreateStateSet()->setRenderBinDetails( BIN_MOON, "RenderBin" );
   cam->setRenderOrder( osg::Camera::NESTED_RENDER );
   cam->setComputeNearFarMode( osg::CullSettings::COMPUTE_NEAR_FAR_USING_BOUNDING_VOLUMES );
   cam->addChild( moon );

   _moon = cam;

   // make the moon's transform
   Moon moonModel;
   osg::Vec3d pos = moonModel.getPosition( 2011, 2, 1, 0 );
   _moonXform = new osg::MatrixTransform(); 
   _moonXform->setMatrix( osg::Matrix::translate( pos ) ); 
   _moonXform->addChild( moon );

   _cullContainer->addChild( _moonXform.get() );
}

me_SkyNode::StarData::StarData(std::stringstream &ss) : right_ascension(0.0), declination(0.0), magnitude(0.0)
{
   std::getline( ss, name, ',' );
   std::string buff;
   std::getline( ss, buff, ',' );
   std::stringstream(buff) >> right_ascension;
   std::getline( ss, buff, ',' );
   std::stringstream(buff) >> declination;
   std::getline( ss, buff, '\n' );
   std::stringstream(buff) >> magnitude;
}

void me_SkyNode::makeStars(const std::string& starFile)
{
   _starRadius = 20000.0f * (_sunDistance > 0.0f ? _sunDistance : _outerRadius);

   std::vector<StarData> starsData;
   if( starFile.empty() || parseStarFile(starFile, starsData) == false )
   {
      getDefaultStars(starsData);
   }
   
   _stars = buildStarGeometry(starsData);

   // make the moon's transform
   _starsXform = new osg::MatrixTransform();
   _starsXform->addChild( _stars.get() );

   _cullContainer->addChild( _starsXform.get() );
}

osg::Node* me_SkyNode::buildStarGeometry(const std::vector<StarData>& stars)
{
   osg::Geode* starsGeode = new osg::Geode;
   double minMag = DBL_MAX, maxMag = DBL_MIN;

   osg::Vec3Array* coords = new osg::Vec3Array(static_cast<unsigned int>(stars.size()));
   unsigned int ind = 0;
   std::vector<StarData>::const_iterator p;
   for( p = stars.begin(); p != stars.end(); p++ )
   {
      (*coords)[ind++] = getPositionFromRADecl( p->right_ascension, p->declination, _starRadius );

      if ( p->magnitude < minMag ) minMag = p->magnitude;
      if ( p->magnitude > maxMag ) maxMag = p->magnitude;
   }

   osg::Vec4Array* colors = new osg::Vec4Array(static_cast<unsigned int>(stars.size()));
   colors->setBinding(osg::Array::BIND_PER_VERTEX);
   ind = 0;
   for( p = stars.begin(); p != stars.end(); p++ )
   {
      float c = static_cast<float>( (p->magnitude-minMag) / (maxMag-minMag) );

      // make the stars a little bit brighter
      c += 0.6f;

      (*colors)[ind++].set(c,c,c,1.0f);
   }

   osg::Geometry* geometry = new osg::Geometry;
   geometry->setUseVertexBufferObjects(true);

   geometry->setVertexArray( coords );
   geometry->setColorArray( colors );
   geometry->addPrimitiveSet( new osg::DrawArrays(osg::PrimitiveSet::POINTS, 0, static_cast<GLsizei>(coords->size())));

   osg::StateSet* sset = geometry->getOrCreateStateSet();

#if !defined (OSG_GLES2_AVAILABLE)
   sset->setTextureAttributeAndModes( 0, new osg::PointSprite(), osg::StateAttribute::OFF );
   sset->setMode( GL_VERTEX_PROGRAM_POINT_SIZE, osg::StateAttribute::ON );
#endif

   osg::Program* program = new osg::Program;
   program->setName("StarShader");
   program->addShader( new osg::Shader(osg::Shader::VERTEX, s_createStarVertexSource()) );
   program->addShader( new osg::Shader(osg::Shader::FRAGMENT, s_createStarFragmentSource()) );
   sset->setAttributeAndModes( program, osg::StateAttribute::ON );

   sset->setRenderBinDetails( BIN_STARS, "RenderBin");
   sset->setAttributeAndModes(new osg::Depth(getSkyToOccludeStars() ? osg::Depth::LESS : osg::Depth::ALWAYS, 0, 1, false), osg::StateAttribute::ON);
   sset->setMode(GL_BLEND, 1);

   starsGeode->addDrawable( geometry );

   // A separate camera isolates the projection matrix calculations.
   osg::Camera* cam = new osg::Camera();
   cam->getOrCreateStateSet()->setRenderBinDetails( BIN_STARS, "RenderBin" );
   cam->setRenderOrder( osg::Camera::NESTED_RENDER );
   cam->setComputeNearFarMode( osg::CullSettings::COMPUTE_NEAR_FAR_USING_BOUNDING_VOLUMES );
   cam->addChild( starsGeode );
   return cam;
}

void me_SkyNode::getDefaultStars(std::vector<StarData>& out_stars)
{
   out_stars.clear();
   const unsigned int numOfStars = sizeof(s_defaultStarData) / (sizeof(double) * 3);
   out_stars.resize(numOfStars);
   StarData* outStarData = &out_stars[0];
   unsigned int numOfOutStars = 0;
   for (unsigned int i = 0; i < numOfStars; ++i)
   {
      if (s_defaultStarData[3*i + 2] >= _minStarMagnitude)
      {
         (*outStarData).right_ascension = s_defaultStarData[3*i];
         (*outStarData).declination = s_defaultStarData[3*i + 1];
         (*outStarData).magnitude = s_defaultStarData[3*i + 2];
         ++numOfOutStars;
         ++outStarData;
      }
   }
   out_stars.resize(numOfOutStars);
}

bool me_SkyNode::parseStarFile(const std::string& starFile, std::vector<StarData>& out_stars)
{
   out_stars.clear();

   std::fstream in(starFile.c_str());
   if (!in)
   {
      return false ;
   }

   while (!in.eof())
   {
      std::string line;

      std::getline(in, line);
      if (in.eof())
         break;

      if (line.empty() || line[0] == '#') 
         continue;

      std::stringstream ss(line);
      out_stars.push_back(StarData(ss));

      if (out_stars[out_stars.size() - 1].magnitude < _minStarMagnitude)
         out_stars.pop_back();
   }

   in.close();

   return true;
}

osg::Vec3d me_SkyNode::getPositionFromRADecl( double ra, double decl, double range )
{
   return osg::Vec3d(0,range,0) * 
      osg::Matrix::rotate( decl, 1, 0, 0 ) * 
      osg::Matrix::rotate( ra - osg::PI_2, 0, 0, 1 );
}
