/* -*-c++-*-
mapengine_oss

Copyright (C) 1998-2006 Robert Osfield
Copyright (C) 2015 Robert Bosch Car Multimedia GmbH

This library is open source and may be redistributed and/or modified under
the terms of the OpenSceneGraph Public License (OSGPL) version 0.0 or
(at your option) any later version.  The full license is in LICENSE file
included with this distribution, and on the openscenegraph.org website.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
OpenSceneGraph Public License for more details.

History:
Copied from https://github.com/openscenegraph/osg/blob/OpenSceneGraph-3.2.0/src/osgTerrain/GeometryTechnique.cpp
List of changes:
1. Modified terrain technique to use morphing between detail levels
*/

#include <osg/io_utils>
#include <osgDB/Registry>
#include <osgTerrain/Terrain>
#include <osgTerrain/TerrainTile>
#include "oss/terrain/MorphTerrainTechnique.h"

using namespace mapengine::oss;

osg::ref_ptr<osg::DrawElements> me_MorphTerrainTechnique::_drawElements = static_cast<osg::DrawElements*>(new osg::DrawElementsUShort(GL_TRIANGLES));

me_MorphTerrainTechnique::me_MorphTerrainTechnique()
{
}

void me_MorphTerrainTechnique::generateGeometry(BufferData& buffer, osgTerrain::Locator* masterLocator, const osg::Vec3d& centerModel)
{
   osgTerrain::Terrain* terrain = _terrainTile->getTerrain();
   osgTerrain::Layer* elevationLayer = _terrainTile->getElevationLayer();
   if (!terrain || !elevationLayer)
   {
      return;
   }

   // setup relation transform -> geode -> geometry
   buffer._geode = new osg::Geode;
   if (buffer._transform.valid()) buffer._transform->addChild(buffer._geode.get());
   buffer._geometry = new osg::Geometry;
   buffer._geode->addDrawable(buffer._geometry.get());
   osg::Geometry* geometry = buffer._geometry.get();

   // set needed values
   unsigned int numColumns = elevationLayer->getNumColumns();
   unsigned int numRows = elevationLayer->getNumRows();
   float scaleHeight = terrain->getVerticalScale();

   // construct the MorphTerrainTechniqueVNG which will
   // manage the generation and the vertices and normals
   me_MorphTerrainTechniqueVNG VNG(masterLocator, centerModel, numRows, numColumns, scaleHeight);

   // setup geometry
   geometry->setUseDisplayList(false);
   geometry->setUseVertexBufferObjects(true);
   geometry->setVertexArray(VNG._vertices.get());
   geometry->setTexCoordArray(0, VNG._texcoords.get());
   geometry->setNormalArray(VNG._normals.get(), osg::Array::BIND_PER_VERTEX);
   geometry->setVertexAttribArray(4, VNG._ups.get(), osg::Array::BIND_PER_VERTEX);

   // setup vertex arrays for morphing
   geometry->setVertexAttribArray(5, VNG._verticesMorph.get(), osg::Array::BIND_PER_VERTEX);
   geometry->setVertexAttribArray(6, VNG._texcoordsMorph.get(), osg::Array::BIND_PER_VERTEX);
   geometry->setVertexAttribArray(7, VNG._normalsMorph.get(), osg::Array::BIND_PER_VERTEX);
   geometry->setVertexAttribArray(9, VNG._upsMorph.get(), osg::Array::BIND_PER_VERTEX);

   // populate vertex and tex coord arrays
   VNG.populateCenter(elevationLayer);

   // populate normals
   VNG.computeNormals();

   // create primitive set if not yet available
   if (_drawElements->getNumIndices() == 0)
   {
      _drawElements = static_cast<osg::DrawElements*>(new osg::DrawElementsUShort(GL_TRIANGLES));

      // add indices to index array
      unsigned int i, j;
      for(j=0; j<numRows-1; ++j)
      {
         for(i=0; i<numColumns-1; ++i)
         {
            // remap indices to final vertex positions
            int i00 = VNG.vertex_index(i,   j);
            int i01 = VNG.vertex_index(i,   j+1);
            int i10 = VNG.vertex_index(i+1, j);
            int i11 = VNG.vertex_index(i+1, j+1);

            unsigned int numValid = 0;
            if (i00>=0) ++numValid;
            if (i01>=0) ++numValid;
            if (i10>=0) ++numValid;
            if (i11>=0) ++numValid;

            if (numValid==4)
            {
               _drawElements->addElement(i01);
               _drawElements->addElement(i00);
               _drawElements->addElement(i11);

               _drawElements->addElement(i00);
               _drawElements->addElement(i10);
               _drawElements->addElement(i11);
            }
            else if (numValid==3)
            {
               if (i00>=0) _drawElements->addElement(i00);
               if (i01>=0) _drawElements->addElement(i01);
               if (i11>=0) _drawElements->addElement(i11);
               if (i10>=0) _drawElements->addElement(i10);
            }
         }
      }
   }

   // finally add drawelemets
   geometry->addPrimitiveSet(_drawElements);
}

void me_MorphTerrainTechnique::applyColorLayers(BufferData& buffer)
{
   GeometryTechnique::applyColorLayers(buffer);
#ifdef OSG_GLES2_AVAILABLE
   osg::StateSet* stateset = buffer._geode->getOrCreateStateSet();
#endif

   //// force activate unref after apply feature for textures
   //for(unsigned int i = 0; i < stateset->getTextureAttributeList().size(); ++i)
   //{
   //  osg::Texture* texture = dynamic_cast<osg::Texture*>(stateset->getTextureAttribute(i, osg::StateAttribute::TEXTURE));
   //  if (texture)
   //  {
   //    texture->setUnRefImageDataAfterApply(true);
   //  }
   //}

   // the color layer is holding an additional reference
   // to the image, so force to unload once texture is
   // being uploaded to GPU (do not access image afterwards!)
   _terrainTile->setColorLayer(0, NULL);

#ifdef OSG_GLES2_AVAILABLE
   stateset->setTextureMode(0, GL_TEXTURE_2D, osg::StateAttribute::OFF);
#endif

}

me_MorphTerrainTechniqueVNG::me_MorphTerrainTechniqueVNG(osgTerrain::Locator* masterLocator, const osg::Vec3d& centerModel, int numRows, int numColumns, float scaleHeight):
_masterLocator(masterLocator),
_centerModel(centerModel),
_numRows(numRows),
_numColumns(numColumns),
_scaleHeight(scaleHeight)
{
   int numVertices = numColumns*numRows;

   _indices.resize((_numRows+2)*(_numColumns+2),0);

   _vertices = new osg::Vec3Array;
   _vertices->reserve(numVertices);

   _texcoords = new osg::Vec2Array;
   _texcoords->reserve(numVertices);

   _normals = new osg::Vec3Array;
   _normals->resize(numVertices);

   _ups = new osg::Vec3Array;
   _ups->reserve(numVertices);

   _verticesMorph = new osg::Vec3Array;
   _verticesMorph->resize(numVertices);

   _texcoordsMorph = new osg::Vec2Array;
   _texcoordsMorph->resize(numVertices);

   _normalsMorph = new osg::Vec3Array;
   _normalsMorph->resize(numVertices);

   _upsMorph = new osg::Vec3Array;
   _upsMorph->resize(numVertices);
}

me_MorphTerrainTechniqueVNG::~me_MorphTerrainTechniqueVNG()
{
   _masterLocator = NULL;
}
void me_MorphTerrainTechniqueVNG::setVertex(int c, int r, const osg::Vec3d& v, const osg::Vec2d& t, const osg::Vec3d& up)
{
   int& i = index(c,r);
   if (i==0)
   {
      i = static_cast<int>(_vertices->size()) + 1;
      _vertices->push_back(v);
      _texcoords->push_back(t);
      //_normals->push_back(n);
      _ups->push_back(up);

      if (c%2 == 0 && r%2 == 0)
      {
         // we have a "morphTo" vector here!
         (*_verticesMorph)[i-1] = v;
         (*_texcoordsMorph)[i-1] = t;
         (*_upsMorph)[i-1] = up;
         //(*_normalsMorph)[i-1] = n;
         if (c != 0)
         {
            (*_verticesMorph)[vertex_index(c-1, r)] = v;
            (*_texcoordsMorph)[vertex_index(c-1, r)] = t;
            (*_upsMorph)[vertex_index(c-1, r)] = up;
            //(*_normalsMorph)[vertex_index(c-1, r)] = n;
         }
         if (r != 0)
         {
            (*_verticesMorph)[vertex_index(c, r-1)] = v;
            (*_texcoordsMorph)[vertex_index(c, r-1)] = t;
            (*_upsMorph)[vertex_index(c, r-1)] = up;
            //(*_normalsMorph)[vertex_index(c, r-1)] = n;
         }
         if (c != 0 && r != 0)
         {
            (*_verticesMorph)[vertex_index(c-1, r-1)] = v;
            (*_texcoordsMorph)[vertex_index(c-1, r-1)] = t;
            (*_upsMorph)[vertex_index(c-1, r-1)] = up;
            //(*_normalsMorph)[vertex_index(c-1, r-1)] = n;
         }
      }
   }
   else
   {
      // average the vertex positions
      // TODO this won't work with morphing...
      (*_vertices)[i-1] = ((*_vertices)[i-1] + v)*0.5f;
      //(*_normals)[i-1] = n;
   }
}

bool me_MorphTerrainTechniqueVNG::vertex(int c, int r, osg::Vec3& v) const
{
   int i = index(c,r);
   if (i==0) return false;
   else v = (*_vertices)[i-1];
   return true;
}

void me_MorphTerrainTechniqueVNG::populateCenter(osgTerrain::Layer* elevationLayer)
{
   //ME_DEBUG << "[MorphTerrainTechniqueVNG] populateCenter(" << elevationLayer << ")" << std::endl;

   for(int j=0; j<_numRows; ++j)
   {
      for(int i=0; i<_numColumns; ++i)
      {
         osg::Vec3d ndc;
         if( _numColumns > 2 && _numRows > 2)
         {
            ndc = osg::Vec3d( ((double)i)/(double)(_numColumns-1), ((double)j)/(double)(_numRows-1), 0.0);

            float value = 0.0f;
            bool validValue = elevationLayer->getValidValue(i,j,value);

            //ndc.z() = value*_scaleHeight;

            if (validValue)
            {
               osg::Vec3d model;
               _masterLocator->convertLocalToModel(ndc, model);

               // compute the local normal
               //osg::Vec3d ndc_one = ndc; ndc_one.z() += 1.0;
               //osg::Vec3d model_one;
               //_masterLocator->convertLocalToModel(ndc_one, model_one);
               //model_one = model_one - model;
               //model_one.normalize();

               //osg::Vec3d ndc_up(0., 0., 1.);
               //osg::Vec3d model_up;
               //_masterLocator->convertLocalToModel(ndc_up, model_up);

               ndc.z() = value*_scaleHeight;
               osg::Vec3d up;
               _masterLocator->convertLocalToModel(ndc, up);

               // set vertex, normal and tex coords for the vertex
               setVertex(i, j, osg::Vec3d(up-_centerModel), osg::Vec2d(ndc.x(), ndc.y()), osg::Vec3d(model-_centerModel));
            }
         }
      }
   }
}

void me_MorphTerrainTechniqueVNG::computeNormals() const
{
   // compute normals for the center section
   for(int j=0; j<_numRows; ++j)
   {
      for(int i=0; i<_numColumns; ++i)
      {
         int vi = vertex_index(i, j);
         if (vi>=0) computeNormal(i, j, (*_normals)[vi]);
         else OSG_NOTICE<<"Not computing normal, vi="<<vi<<std::endl;
      }
   }
}

bool me_MorphTerrainTechniqueVNG::computeNormal(int c, int r, osg::Vec3& n) const
{
#if 1
   return computeNormalWithNoDiagonals(c,r,n);
#else
   return computeNormalWithDiagonals(c,r,n);
#endif
}

bool me_MorphTerrainTechniqueVNG::computeNormalWithNoDiagonals(int c, int r, osg::Vec3& n) const
{
   osg::Vec3 center;
   bool center_valid  = vertex(c, r,  center);
   if (!center_valid) return false;

   osg::Vec3 left, right, top,  bottom;
   bool left_valid  = vertex(c-1, r,  left);
   bool right_valid = vertex(c+1, r,   right);
   bool bottom_valid = vertex(c,   r-1, bottom);
   bool top_valid = vertex(c,   r+1, top);

   osg::Vec3 dx(0.0f,0.0f,0.0f);
   osg::Vec3 dy(0.0f,0.0f,0.0f);
   osg::Vec3 zero(0.0f,0.0f,0.0f);
   if (left_valid)
   {
      dx = center-left;
   }
   if (right_valid)
   {
      dx = right-center;
   }
   if (bottom_valid)
   {
      dy += center-bottom;
   }
   if (top_valid)
   {
      dy += top-center;
   }

   if (dx==zero || dy==zero) return false;

   n = dx ^ dy;
   return n.normalize() != 0.0f;
}

bool me_MorphTerrainTechniqueVNG::computeNormalWithDiagonals(int c, int r, osg::Vec3& n) const
{
   osg::Vec3 center;
   bool center_valid  = vertex(c, r,  center);
   if (!center_valid) return false;

   osg::Vec3 top_left, top_right, bottom_left, bottom_right;
   bool top_left_valid  = vertex(c-1, r+1,  top_left);
   bool top_right_valid  = vertex(c+1, r+1,  top_right);
   bool bottom_left_valid  = vertex(c-1, r-1,  bottom_left);
   bool bottom_right_valid  = vertex(c+1, r-1,  bottom_right);

   osg::Vec3 left, right, top,  bottom;
   bool left_valid  = vertex(c-1, r,  left);
   bool right_valid = vertex(c+1, r,   right);
   bool bottom_valid = vertex(c,   r-1, bottom);
   bool top_valid = vertex(c,   r+1, top);

   osg::Vec3 dx(0.0f,0.0f,0.0f);
   osg::Vec3 dy(0.0f,0.0f,0.0f);
   osg::Vec3 zero(0.0f,0.0f,0.0f);
   const float ratio = 0.5f;
   if (left_valid)
   {
      dx = center-left;
      if (top_left_valid) dy += (top_left-left)*ratio;
      if (bottom_left_valid) dy += (left-bottom_left)*ratio;
   }
   if (right_valid)
   {
      dx = right-center;
      if (top_right_valid) dy += (top_right-right)*ratio;
      if (bottom_right_valid) dy += (right-bottom_right)*ratio;
   }
   if (bottom_valid)
   {
      dy += center-bottom;
      if (bottom_left_valid) dx += (bottom-bottom_left)*ratio;
      if (bottom_right_valid) dx += (bottom_right-bottom)*ratio;
   }
   if (top_valid)
   {
      dy += top-center;
      if (top_left_valid) dx += (top-top_left)*ratio;
      if (top_right_valid) dx += (top_right-top)*ratio;
   }

   if (dx==zero || dy==zero) return false;

   n = dx ^ dy;
   return n.normalize() != 0.0f;
}
