/* -*-c++-*-
mapengine_oss

Copyright (C) 1998-2006 Robert Osfield
Copyright (C) 2015 Robert Bosch Car Multimedia GmbH

This library is open source and may be redistributed and/or modified under
the terms of the OpenSceneGraph Public License (OSGPL) version 0.0 or
(at your option) any later version.  The full license is in LICENSE file
included with this distribution, and on the openscenegraph.org website.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
OpenSceneGraph Public License for more details.

History:
Copied from https://github.com/openscenegraph/osg/blob/OpenSceneGraph-3.2.0/src/osgDB/DatabasePager.cpp
List of changes:
1. Support different cache limits in cache-management.
2. Support request sorting by priority first
*/

#ifndef MAPENGINE_TYPES_DATABASEPAGERBASE_
#define MAPENGINE_TYPES_DATABASEPAGERBASE_ 1

#include <osgDB/Registry>
#include <osgDB/DatabasePager>


struct osgDB::DatabasePager::DatabasePagerCompileCompletedCallback : public osgUtil::IncrementalCompileOperation::CompileCompletedCallback
{
   DatabasePagerCompileCompletedCallback(osgDB::DatabasePager* pager, osgDB::DatabasePager::DatabaseRequest* databaseRequest):
_pager(pager),
_databaseRequest(databaseRequest) {}

virtual bool compileCompleted(osgUtil::IncrementalCompileOperation::CompileSet* /*compileSet*/)
{
   _pager->compileCompleted(_databaseRequest.get());
   return true;
}

osgDB::DatabasePager*                               _pager;
osg::ref_ptr<osgDB::DatabasePager::DatabaseRequest> _databaseRequest;

protected:
   virtual ~DatabasePagerCompileCompletedCallback(){}
};

class osgDB::DatabasePager::FindCompileableGLObjectsVisitor : public osgUtil::StateToCompile
{
public:
   FindCompileableGLObjectsVisitor(const DatabasePager* pager):
      osgUtil::StateToCompile(osgUtil::GLObjectsVisitor::COMPILE_DISPLAY_LISTS|osgUtil::GLObjectsVisitor::COMPILE_STATE_ATTRIBUTES),
         _pager(pager),
         _changeAutoUnRef(false), _valueAutoUnRef(false),
         _changeAnisotropy(false), _valueAnisotropy(1.0)
      {
         _assignPBOToImages = _pager->_assignPBOToImages;

         _changeAutoUnRef = _pager->_changeAutoUnRef;
         _valueAutoUnRef = _pager->_valueAutoUnRef;
         _changeAnisotropy = _pager->_changeAnisotropy;
         _valueAnisotropy = _pager->_valueAnisotropy;

         switch(_pager->_drawablePolicy)
         {
         case DatabasePager::DO_NOT_MODIFY_DRAWABLE_SETTINGS:
            // do nothing, leave settings as they came in from loaded database.
            // OSG_NOTICE<<"DO_NOT_MODIFY_DRAWABLE_SETTINGS"<<std::endl;
            break;
         case DatabasePager::USE_DISPLAY_LISTS:
            _mode = _mode | osgUtil::GLObjectsVisitor::SWITCH_ON_DISPLAY_LISTS;
            _mode = _mode | osgUtil::GLObjectsVisitor::SWITCH_OFF_VERTEX_BUFFER_OBJECTS;
            _mode = _mode & ~osgUtil::GLObjectsVisitor::SWITCH_ON_VERTEX_BUFFER_OBJECTS;
            break;
         case DatabasePager::USE_VERTEX_BUFFER_OBJECTS:
            _mode = _mode | osgUtil::GLObjectsVisitor::SWITCH_ON_VERTEX_BUFFER_OBJECTS;
            break;
         case DatabasePager::USE_VERTEX_ARRAYS:
            _mode = _mode & ~osgUtil::GLObjectsVisitor::SWITCH_ON_DISPLAY_LISTS;
            _mode = _mode & ~osgUtil::GLObjectsVisitor::SWITCH_ON_VERTEX_BUFFER_OBJECTS;
            _mode = _mode | osgUtil::GLObjectsVisitor::SWITCH_OFF_DISPLAY_LISTS;
            _mode = _mode | osgUtil::GLObjectsVisitor::SWITCH_OFF_VERTEX_BUFFER_OBJECTS;
            break;
         }

         if (osgDB::Registry::instance()->getBuildKdTreesHint()==osgDB::Options::BUILD_KDTREES &&
            osgDB::Registry::instance()->getKdTreeBuilder())
         {
            _kdTreeBuilder = osgDB::Registry::instance()->getKdTreeBuilder()->clone();
         }
      }

      META_NodeVisitor("osgDB","FindCompileableGLObjectsVisitor")

         bool requiresCompilation() const { return !empty(); }

      virtual void apply(osg::Geode& geode)
      {
         StateToCompile::apply(geode);

         if (_kdTreeBuilder.valid())
         {
            geode.accept(*_kdTreeBuilder);
         }
      }

      void apply(osg::Texture& texture)
      {
         StateToCompile::apply(texture);

         if (_changeAutoUnRef)
         {
            texture.setUnRefImageDataAfterApply(_valueAutoUnRef);
         }

         if ((_changeAnisotropy && texture.getMaxAnisotropy() != _valueAnisotropy))
         {
            texture.setMaxAnisotropy(_valueAnisotropy);
         }
      }

      const DatabasePager*                    _pager;
      bool                                    _changeAutoUnRef;
      bool                                    _valueAutoUnRef;
      bool                                    _changeAnisotropy;
      float                                   _valueAnisotropy;
      osg::ref_ptr<osg::KdTreeBuilder>        _kdTreeBuilder;

protected:
   virtual ~FindCompileableGLObjectsVisitor(){}
//   FindCompileableGLObjectsVisitor& operator = (const FindCompileableGLObjectsVisitor&) { return *this; }
};

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//  SortFileRequestFunctor
//
//Anish: Our BDR requests are currently first sorted by priority
struct osgDB::DatabasePager::SortFileRequestFunctor
{
   bool operator() (const osg::ref_ptr<DatabasePager::DatabaseRequest>& lhs,const osg::ref_ptr<DatabasePager::DatabaseRequest>& rhs) const
   {
      if (lhs->_priorityLastRequest > rhs->_priorityLastRequest) return true;
      else if (lhs->_priorityLastRequest < rhs->_priorityLastRequest) return false;
      else if (lhs->_timestampLastRequest < rhs->_timestampLastRequest) return false;
      else return (lhs->_frameNumberLastRequest > rhs->_frameNumberLastRequest);
   }
};


#endif //MAPENGINE_TYPES_DATABASEPAGERBASE_
