/* -*-c++-*-
mapengine_oss

Copyright (C) 1998-2006 Robert Osfield
Copyright (C) 2015 Robert Bosch Car Multimedia GmbH

This library is open source and may be redistributed and/or modified under
the terms of the OpenSceneGraph Public License (OSGPL) version 0.0 or
(at your option) any later version.  The full license is in LICENSE file
included with this distribution, and on the openscenegraph.org website.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
OpenSceneGraph Public License for more details.

History:
Copied from https://github.com/openscenegraph/osg/blob/OpenSceneGraph-3.2.0/include/osgDB/DatabasePager
List of changes:
1. Support several different kinds of requests instead of just 'file' and 'http' requests.
2. Checking if a specific request-type is supported by the DBPager thread looking for requests to execute.
3. Support deletion of objects in a specific type of DBPager thread, instead of only using one particular thread.
4. Support different cache limits in cache-management.
5. Disable incremental-compile of GL objects completely.
*/

#ifndef MAPENGINE_TYPES_DATABASEPAGER_
#define MAPENGINE_TYPES_DATABASEPAGER_ 1

#include <osgDB/DatabasePager>
#include <oss/dbpager/DatabasePagerConfig.h>
#include <oss/Export.h>

#define DB_PAGER_THREAD_CONFIG "DBPagerThreadConfig"
const std::string DATABASE_REQUEST_OPTIONS_STRING("ReqOptions");
const std::string DATABASE_REQUEST_FRAMENUMBER_STRING("FrameNumber");

namespace mapengine {
namespace oss {

class MAPENGINE_OSS_EXPORT me_DatabasePager : public osgDB::DatabasePager
{
public:

   class MAPENGINE_OSS_EXPORT ThreadConfig : public osg::Referenced
   {
   public:
      ThreadConfig()
      {
         memset(_numberOfExclusiveThreads, 0, sizeof(_numberOfExclusiveThreads));
         memset(_totalNumberOfThreads, 0, sizeof(_totalNumberOfThreads));
         memset(_threadGroupOfWorkload, mapengine::oss::DatabasePagerConfig::THREAD_GROUP_INVALID, sizeof(_threadGroupOfWorkload));
      }
      unsigned int                                     _numberOfExclusiveThreads[mapengine::oss::DatabasePagerConfig::WORKLOAD_TYPE_COUNT];
      unsigned int                                     _totalNumberOfThreads[mapengine::oss::DatabasePagerConfig::WORKLOAD_TYPE_COUNT];
      mapengine::oss::DatabasePagerConfig::ThreadGroup _threadGroupOfWorkload[mapengine::oss::DatabasePagerConfig::WORKLOAD_TYPE_COUNT];

      /** determine the total number of threads available for asynchronous single request */
      void determineTotalThreadsForAsyncSingleRequest(unsigned int numFMChannels);

   protected:
      virtual ~ThreadConfig(){}
   };

   me_DatabasePager();

   /////////////////////////////////////////////////////////////////////////////////////////////////////////////////
   //
   //  SortFileRequestFunctor
   //
   struct SortFileRequestFunctorPerFrame
   {
      bool operator() (const osg::ref_ptr<DatabasePager::DatabaseRequest>& lhs,const osg::ref_ptr<DatabasePager::DatabaseRequest>& rhs) const
      {
         if (lhs->_timestampLastRequest>rhs->_timestampLastRequest) return true;
         else if (lhs->_timestampLastRequest<rhs->_timestampLastRequest) return false;
         else return (lhs->_priorityLastRequest>rhs->_priorityLastRequest);
      }
   };

   class MAPENGINE_OSS_EXPORT me_DatabaseThread : public osgDB::DatabasePager::DatabaseThread
   {
   public:
      me_DatabaseThread(DatabasePager* pager, Mode mode, const std::string& name, const std::vector<mapengine::oss::DatabasePagerConfig::ThreadWorkloadType>& supportedWorkloads);
      virtual void run();
      virtual int cancel();

      unsigned int getSupportedWorkloadsFlag() const { return _supportedWorkloadsFlag; }
      void setSupportedWorkloadsFlag(unsigned int val) { _supportedWorkloadsFlag = val; }

      void block() { _threadActiveBlock.block(); }

   protected:
      virtual ~me_DatabaseThread(){}

      unsigned int _supportedWorkloadsFlag;
      OpenThreads::Block _threadActiveBlock;
   };

   struct MAPENGINE_OSS_EXPORT me_DatabaseRequest : public osgDB::DatabasePager::DatabaseRequest
   {
      me_DatabaseRequest(mapengine::oss::DatabasePagerConfig::ThreadWorkloadType workloadType, const std::string& fileName,
         const osg::FrameStamp* frameStamp, float prio);

      virtual bool isPerFrameRequest() const
      {
         bool perFrame = false;
         switch(_workloadType)
         {
         case mapengine::oss::DatabasePagerConfig::SYNCHRONOUS_PER_FRAME_REQUEST:
         case mapengine::oss::DatabasePagerConfig::HTTP_PER_FRAME_REQUEST:
            perFrame = true;
            break;
         default:
            break;
         }
         return perFrame;
      }

      unsigned int getWorkloadFlag() const { return (1 << _workloadType); }
      static unsigned int getFlag(const std::vector<mapengine::oss::DatabasePagerConfig::ThreadWorkloadType>& workloads);
      static void getRequestsTypeString(const std::vector<mapengine::oss::DatabasePagerConfig::ThreadWorkloadType>& workloads, std::string& requestsTypeString);
      bool isObsolete(unsigned int frameNumber, unsigned int frameNumberOfLastRequestReset) const;
      mapengine::oss::DatabasePagerConfig::ThreadWorkloadType _workloadType;

   protected:
      virtual ~me_DatabaseRequest(){}
   };

   /** Functor used to provide additional checks for deletion of DB pager requests */
   class MAPENGINE_OSS_EXPORT me_RequestIdentifierFunctor
   {
   public:
      me_RequestIdentifierFunctor() {}
      virtual bool operator () (const osg::ref_ptr<mapengine::oss::me_DatabasePager::me_DatabaseRequest>& dbRequest) = 0;

   protected:
      virtual ~me_RequestIdentifierFunctor() {};
   };

   enum DatabaseRequestOptions
   {
      E_CHECK_FOR_OBSOLESCENCE = 1 << 0,
      E_COULD_BE_OBSOLETE = 1 << 1
   };

   typedef std::list< osg::ref_ptr<mapengine::oss::me_DatabasePager::me_DatabaseRequest> > DatabaseRequestList;

   class MAPENGINE_OSS_EXPORT me_ReadQueue : public osgDB::DatabasePager::ReadQueue
   {
   public:
      me_ReadQueue(DatabasePager* pager, const std::string& name) : osgDB::DatabasePager::ReadQueue(pager, name) {}

      void clearRequests(DatabaseRequestList& requestList, mapengine::oss::DatabasePagerConfig::ThreadWorkloadType workloadType,
         const std::vector<std::string>& extensions = std::vector<std::string>(), me_RequestIdentifierFunctor* requestIdentifier = NULL);

      void clearAllData();

   };

   class MAPENGINE_OSS_EXPORT me_DeletionCallback : public osg::Referenced
   {
   public:
      me_DeletionCallback() : _pager(0) {}
      me_DeletionCallback(const me_DatabasePager* pager) : _pager(pager) {}

      me_DeletionCallback(const me_DeletionCallback& srcDeletionCallback, const osg::CopyOp&) : _pager(srcDeletionCallback._pager) {}

      /** do customized deletion code.*/
      virtual void deletion(osg::ref_ptr<me_DatabasePager::me_ReadQueue> readQueue) const = 0;

      const me_DatabasePager* _pager;
   };

   friend class me_DatabaseThread;

   /** Modify original addDatabaseThread() to create our own customDatabaseThreads */
   virtual unsigned int addDatabaseThread(DatabaseThread::Mode mode, const std::string& name);

   virtual void requestData(const std::string& fileName, osg::NodePath& nodePath,
                            float priority, const osg::FrameStamp* framestamp,
                            osg::ref_ptr<osg::Referenced>& databaseRequest,
                            mapengine::oss::DatabasePagerConfig::ThreadWorkloadType workloadType,
                            const osg::Referenced* options);
   
   /** Modify original requestNodeFile() to be able to post "http request" directly into the proper queue */
   virtual void requestNodeFile(const std::string& fileName, osg::NodePath& nodePath,
                                float priority, const osg::FrameStamp* framestamp,
                                osg::ref_ptr<osg::Referenced>& databaseRequest,
                                const osg::Referenced* options);

   virtual unsigned int getFileRequestListSize() const;

   /** Cancel the database pager thread(s).*/
   virtual int cancel();

   /** Set whether the database pager thread should be paused or not.*/
   /** This should've been a virtual function in the base class. But we have to over-ride it anyway.*/
   void setDatabasePagerThreadPause(bool pause);

   /**
   * Pauses database pager threads and additionally blocks the calling thread
   * until all already running requests in pager threads have been processed.
   */
   void pauseDatabasePagerThreadsAndWaitForCompletion();


   /**Clear the ReadQueue.*/
   void clearAllReadQueue();
   virtual void deletion(osg::ref_ptr<mapengine::oss::me_DatabasePager::me_ReadQueue> readQueue, osgDB::DatabasePager::ObjectList& deleteList) const;

   /** Iterate through the active PagedLOD nodes children removing
          * children which havn't been visited since specified expiryTime.
          * note, should be only be called from the update thread. */
   virtual void removeExpiredSubgraphs(const osg::FrameStamp &frameStamp);

   void addToChildrenToDeleteList(osgDB::DatabasePager::ObjectList& objectsToDeleteList, 
      mapengine::oss::DatabasePagerConfig::ThreadWorkloadType workloadType) const;

   std::map< unsigned int, osg::ref_ptr<me_DatabasePager::me_ReadQueue> > _requestQueues;

   unsigned int getFrameNumberOfLastRequestReset();

   void setFrameNumberOfLastRequestReset(unsigned int val);

   unsigned int getTargetDesiredMaximumNumberOfPagedLOD() const { return _targetDesiredMaximumNumberOfPagedLOD; }
   void setTargetDesiredMaximumNumberOfPagedLOD(unsigned int val) 
   {
      _targetDesiredMaximumNumberOfPagedLOD = val;
      osgDB::DatabasePager::setTargetMaximumNumberOfPageLOD(_targetDesiredMaximumNumberOfPagedLOD);
   }

   osg::ref_ptr<mapengine::oss::me_DatabasePager::me_DeletionCallback> getDeletionCallback() const { return _deletionCallback; }
   void setDeletionCallback(osg::ref_ptr<mapengine::oss::me_DatabasePager::me_DeletionCallback> val) { _deletionCallback = val; }

   /** Merge the changes to the scene graph by calling calling removeExpiredSubgraphs then addLoadedDataToSceneGraph.
   * Note, must only be called from single thread update phase. */
   virtual void updateSceneGraph(const osg::FrameStamp& frameStamp);

   /**
   * Cancels the pending DB pager requests based on workload type, an optionally empty list of extensions and an optional RequestIdentifierFunctor.
   * In the absence of optional parameters, all the requests of the given workload type are deleted.Returns a list of cancelled requests.
   */
   void clearRequests(DatabaseRequestList& requestList, mapengine::oss::DatabasePagerConfig::ThreadWorkloadType workloadType,
      const std::vector<std::string>& extensions = std::vector<std::string>(), me_RequestIdentifierFunctor* requestIdentifier = NULL);

   unsigned int getNumOfRequests(mapengine::oss::DatabasePagerConfig::ThreadWorkloadType workloadType) const;

protected:
   unsigned int                               _numberOfExclusiveThreadsCreated[mapengine::oss::DatabasePagerConfig::WORKLOAD_TYPE_COUNT];
   unsigned int                               _totalNumberOfThreadsCreated[mapengine::oss::DatabasePagerConfig::WORKLOAD_TYPE_COUNT];



   virtual ~me_DatabasePager();

   virtual void takeFirstRequest(osg::ref_ptr<me_DatabasePager::me_ReadQueue>& requestQueue, 
      osg::ref_ptr<DatabaseRequest>& databaseRequest, const DatabaseThread* dbPagerThread) const;

   bool isWorkloadCurrentlyAllowed(mapengine::oss::DatabasePagerConfig::ThreadWorkloadType workloadType, const unsigned int supportedWorkloadsFlag) const;

   osg::ref_ptr<me_DatabasePager::me_ReadQueue> getRequestQueueByFlag(unsigned int workloadsFlag) const;

   virtual bool isRequestObsolete(const DatabaseRequest* databaseRequest) const;

   void getWorkloadsToBeSupported(std::vector<mapengine::oss::DatabasePagerConfig::ThreadWorkloadType>& supportedWorkloads);

   void createRequestQueues();

   /** Add the loaded data to the scene graph.*/
   void customAddLoadedDataToSceneGraph(const osg::FrameStamp &frameStamp);

   unsigned int _frameNumberOfLastRequestReset;

   unsigned int _targetDesiredMaximumNumberOfPagedLOD;

   unsigned int _supportedWorkloads;

   osg::ref_ptr<me_DeletionCallback> _deletionCallback;
};


inline bool me_DatabasePager::isWorkloadCurrentlyAllowed( mapengine::oss::DatabasePagerConfig::ThreadWorkloadType workloadType, const unsigned int supportedWorkloadsFlag ) const
{
   return ((1 << workloadType) & supportedWorkloadsFlag) ? true : false;
}

}
}

#endif //MAPENGINE_TYPES_DATABASEPAGER_
