/*
Copyright (c) 2010 The Khronos Group Inc.

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and/or associated documentation files (the
"Materials"), to deal in the Materials without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Materials, and to
permit persons to whom the Materials are furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be included
unaltered in all copies or substantial portions of the Materials.
Any additions, deletions, or changes to the original source files
must be clearly indicated in accompanying documentation.

If only executable code is distributed, then the accompanying
documentation must state that "this software is based in part on the
work of the Khronos Group."

THE MATERIALS ARE PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
MATERIALS OR THE USE OR OTHER DEALINGS IN THE MATERIALS.

History:
Copied from https://github.com/KhronosGroup/KTX/blob/master/lib/loader.c
List of changes:
1. Decodes images to etc2 format.
2. Does not upload to VRAM or make any gl calls.
*/



#include <assert.h>
#include <stdlib.h>
#include "ktx.h"
#include "ktxint.h"
#include "ktxdecode.h"

namespace libktx {

/*
 * @private
 * @~English
 * @name Supported Sized Format Macros
 *
 * These macros describe values that may be used with the sizedFormats
 * variable.
 */
/**@{*/
#define _NON_LEGACY_FORMATS 0x1 /*< @private @internal non-legacy sized formats are supported. */
#define _LEGACY_FORMATS 0x2  /*< @private @internal legacy sized formats are supported. */

/*
 * @private
 * @~English
 * @brief all sized formats are supported
 */
#define _ALL_SIZED_FORMATS (_NON_LEGACY_FORMATS | _LEGACY_FORMATS)
#define _NO_SIZED_FORMATS 0 /*< @private @internal no sized formats are supported. */

static GLint sizedFormats = _ALL_SIZED_FORMATS;
static GLboolean supportsSwizzle = GL_TRUE;
/**
 * @private
 * @~English
 * @brief indicates which R16 & RG16 formats are supported by the current context.
 */
static GLint R16Formats = _KTX_ALL_R16_FORMATS;
/**
 * @private
 * @~English
 * @brief indicates if the current context supports sRGB textures.
 */
static GLboolean supportsSRGB = GL_TRUE;

KTX_error_code
ktxDecodeImage(std::istream& fin, KTX_dimensions* pDimensions,  GLboolean* pIsMipmapped, unsigned char** decImageData,
               GLenum* format, GLenum* internalFormat, GLenum* type)
{
   KTX_header			header;
   KTX_texinfo			texinfo;
   char*				data = NULL;
   khronos_uint32_t	dataSize = 0;
   khronos_uint32_t    faceLodSize;
   khronos_uint32_t    faceLodSizeRounded;
   khronos_uint32_t	level;
   khronos_uint32_t	face;
   GLenum				glFormat, glInternalFormat;
   KTX_error_code		errorCode = KTX_SUCCESS;

   fin.seekg(0, std::ios::end);
   khronos_uint32_t fileLength = fin.tellg();
   fin.seekg(0, std::ios::beg);

   //read in the data for the header and store it
   fin.read((char*)&header, sizeof(KTX_header));
   if (!fin.good()) {
      return KTX_UNEXPECTED_END_OF_FILE;
   }

   errorCode = _ktxCheckHeader(&header, &texinfo);
   if (errorCode != KTX_SUCCESS) {
      return errorCode;
   }

   /* skip key/value metadata */
   fin.ignore(header.bytesOfKeyValueData);

   if (texinfo.glTarget == GL_TEXTURE_CUBE_MAP) {
      texinfo.glTarget = GL_TEXTURE_CUBE_MAP_POSITIVE_X;
   }

   glInternalFormat = header.glInternalFormat;
   glFormat = header.glFormat;

   for (level = 0; level < /*header.numberOfMipmapLevels*/1; ++level)
   {
      GLsizei pixelWidth  = MAX(1, header.pixelWidth  >> level);
      GLsizei pixelHeight = MAX(1, header.pixelHeight >> level);
      GLsizei pixelDepth  = MAX(1, header.pixelDepth  >> level);

      fin.read((char*)&faceLodSize, sizeof(faceLodSize));

      if(!fin.good())
      {
         errorCode = KTX_UNEXPECTED_END_OF_FILE;
         goto cleanup;
      }

      if (header.endianness == KTX_ENDIAN_REF_REV) {
         _ktxSwapEndian32(&faceLodSize, 1);
      }
      faceLodSizeRounded = (faceLodSize + 3) & ~(khronos_uint32_t)3;
      if (!data) {
         /* allocate memory sufficient for the first level */
         data = (char*)malloc(faceLodSizeRounded);
         if (!data) {
            errorCode = KTX_OUT_OF_MEMORY;
            goto cleanup;
         }
         dataSize = faceLodSizeRounded;
      }
      else if (dataSize < faceLodSizeRounded) {
         /* subsequent levels cannot be larger than the first level */
         errorCode = KTX_INVALID_VALUE;
         goto cleanup;
      }

      for (face = 0; face < header.numberOfFaces; ++face)
      {
         fin.read(data, faceLodSizeRounded);
         if(!fin.good())
         {
            errorCode = KTX_UNEXPECTED_END_OF_FILE;
            goto cleanup;
         }

         /* Perform endianness conversion on texture data */
         if (header.endianness == KTX_ENDIAN_REF_REV && header.glTypeSize == 2) {
            _ktxSwapEndian16((khronos_uint16_t*)data, faceLodSize / 2);
         }
         else if (header.endianness == KTX_ENDIAN_REF_REV && header.glTypeSize == 4) {
            _ktxSwapEndian32((khronos_uint32_t*)data, faceLodSize / 4);
         }

#if SUPPORT_SOFTWARE_ETC_UNPACK
         // Renderion is returning INVALID_VALUE. Oops!!
         if (texinfo.compressed
            && texinfo.textureDimensions == 2
            /*&& (glInternalFormat == GL_ETC1_RGB8_OES || (glInternalFormat >= GL_COMPRESSED_R11_EAC && glInternalFormat <= GL_COMPRESSED_SRGB8_ALPHA8_ETC2_EAC))*/)
         {
             errorCode = _ktxUnpackETC((GLubyte*)data, glInternalFormat, pixelWidth, pixelHeight,
               decImageData, format, internalFormat, type,
               R16Formats, supportsSRGB);
            if (errorCode != KTX_SUCCESS) {
               goto cleanup;
            }
            if (!sizedFormats & _NON_LEGACY_FORMATS) {
               if (*internalFormat == GL_RGB8)
                  *internalFormat = GL_RGB;
               else if (*internalFormat == GL_RGBA8)
                  *internalFormat = GL_RGBA;
            }

         }
#endif
      }
   }

cleanup:
   free(data);

   if (errorCode == KTX_SUCCESS)
   {
      if (pDimensions) {
         pDimensions->width = header.pixelWidth;
         pDimensions->height = header.pixelHeight;
         pDimensions->depth = header.pixelDepth;
      }
      if (pIsMipmapped) {
         if (texinfo.generateMipmaps || header.numberOfMipmapLevels > 1)
            *pIsMipmapped = GL_TRUE;
         else
            *pIsMipmapped = GL_FALSE;
      }
   }
   return errorCode;
}

}
