/* -*-c++-*-
mapengine_oss

Copyright (C) 1998-2006 Robert Osfield
Copyright (C) 2015 Robert Bosch Car Multimedia GmbH

This library is open source and may be redistributed and/or modified under
the terms of the OpenSceneGraph Public License (OSGPL) version 0.0 or
(at your option) any later version.  The full license is in LICENSE file
included with this distribution, and on the openscenegraph.org website.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
OpenSceneGraph Public License for more details.

History:
Copied from https://github.com/openscenegraph/osg/blob/OpenSceneGraph-3.2.0/src/osgUtil/Optimizer.cpp
List of changes:
1. Support to replace a single texture in an atlas.
*/

#include <osgDB/Registry>
#include <osgDB/ReaderWriter>
#include <osgDB/WriteFile>
#include "oss/ThreadUtils.h"
#include "TextureBatchingAtlas.h"

using namespace mapengine::oss;

bool me_TextureBatchingStrategyAtlas::addImage( unsigned int imageId, const std::vector<unsigned char>& imageData, const TextureParameters& textureParam, unsigned int frameNumber )
{
   bool objectAdded = false;
   OpenThreads::ScopedLock<OpenThreads::ReentrantMutex> lock(_mutex);
   
   for (AtlasList::iterator aitr = _atlasList.begin(); (aitr != _atlasList.end()) && !objectAdded; ++aitr)
   {
      if (static_cast<WallpaperAtlas*>(aitr->get())->getID() == textureParam.atlasID)//lint !e1774
      {
         for (SourceList::iterator sitr = aitr->get()->_sourceList.begin(); (sitr != aitr->get()->_sourceList.end()) && !objectAdded; ++sitr)
         {  
            AtlasSource* currentSource = static_cast<AtlasSource*>( sitr->get() );  //lint !e1774
            if( currentSource->getID() == imageId  )
            {  
               currentSource->setFrameNumber(frameNumber);
               objectAdded = true;
            }
         }
      }
   }

   if( !objectAdded)
   {
      if( isImagePNG(imageData))
      {
         //support for only png image
         osgDB::ReaderWriter* reader = 0;
         reader = osgDB::Registry::instance()->getReaderWriterForExtension("png");
         Streambuf sb( reinterpret_cast<char*>(const_cast<unsigned char*>(&imageData[0])), static_cast<int>(imageData.size()));
         std::istream is(&sb);
         osgDB::ReaderWriter::ReadResult result = reader->readImage(is);

         osg::Texture2D* texture = createTextureWrapper(result.takeImage(), textureParam); // take not get!
         osg::ref_ptr<osg::Array> texCoords;
         objectAdded = (0 != addTextureObject(imageId, texture, frameNumber, textureParam, texCoords));  //atlas width = atlas height
      }
      else
      {
         OSG_WARN << "Unable to add image to atlas (unsupported format)" << std::endl;
      }
   }
   if (!objectAdded)
   {
      OSG_WARN << "Failed to add image to atlas with ID: " << textureParam.atlasID << std::endl;
   }
   return objectAdded;
}

bool me_TextureBatchingStrategyAtlas::isTexturePresent( unsigned int textureID,
                                                        unsigned int atlasID,
                                                        osg::ref_ptr<osg::Texture>& batchedTexture,
                                                        osg::ref_ptr<osg::Array>& texCoords)
{
   OpenThreads::ScopedLock<OpenThreads::ReentrantMutex> lock(_mutex);
   batchedTexture = 0;
   texCoords = 0;
   me_TextureBatchingStrategyAtlas::AtlasSource* sourceFound = findSource(textureID, atlasID);
   if (sourceFound)
   {
      batchedTexture = sourceFound->_atlas->_texture;
      texCoords = sourceFound->getTextureCoords();
   }
   return (0 != sourceFound);
}

void me_TextureBatchingStrategyAtlas::addDefaultsToTexCoordArray( osg::Array* tcArray ) const
{
   osg::Vec2Array* tc = static_cast<osg::Vec2Array*>(tcArray);//lint !e1774
   tc->push_back(osg::Vec2(0.0F, 0.0F));
   tc->push_back(osg::Vec2(1.0F, 0.0F));
   tc->push_back(osg::Vec2(1.0F, 1.0F));
   tc->push_back(osg::Vec2(0.0F, 1.0F));
}

void me_TextureBatchingStrategyAtlas::replaceEndWithDefaultsForTexCoordArray(osg::Array* tcArray) const
{
   osg::Vec2Array* texCoordArray = static_cast<osg::Vec2Array*>(tcArray); //lint !e1774
   if(texCoordArray)
   {
      for(unsigned int i=0; i < 4 ; ++i)
      {
         texCoordArray->pop_back();
      }

      addDefaultsToTexCoordArray(tcArray);
   }
}

osg::Array* me_TextureBatchingStrategyAtlas::createTexCoordArray() const
{
    return new osg::Vec2Array;
}

void me_TextureBatchingStrategyAtlas::addToTexCoordArray( osg::Array* tcArray, const osg::ref_ptr<osg::Array>& texCoordsToAdd ) const
{
   osg::Vec2Array* tcv2 = static_cast<osg::Vec2Array*>(tcArray); //lint !e1774
   osg::Vec2Array* tcToAddv2 = static_cast<osg::Vec2Array*>(texCoordsToAdd.get()); //lint !e1774
   tcv2->insert(tcv2->end(), tcToAddv2->begin(), tcToAddv2->begin() + 4 );
}

void me_TextureBatchingStrategyAtlas::addToTexCoordArray(std::vector<osg::Vec3>& tcArray, const osg::ref_ptr<osg::Array>& texCoordsToAdd) const
{
   osg::Vec2Array* tcToAddv2 = static_cast<osg::Vec2Array*>(texCoordsToAdd.get()); //lint !e1774
   for (int i = 0; i < 4; ++i)
   {
      tcArray.push_back(osg::Vec3((*tcToAddv2)[i], 0.0f));
   }
}

void me_TextureBatchingStrategyAtlas::clearTexCoordArray( osg::Array* tcArray ) const
{
   static_cast<osg::Vec2Array*>(tcArray)->clear(); //lint !e1774
}

bool me_TextureBatchingStrategyAtlas::isTexturePresent(unsigned int textureID,
                                                       unsigned int atlasID,
                                                       osg::ref_ptr<osg::Texture>& batchedTexture,
                                                       osg::ref_ptr<osg::Array>& texCoords,
                                                       unsigned int frameNumber)
{
   OpenThreads::ScopedLock<OpenThreads::ReentrantMutex> lock(_mutex);
   batchedTexture = 0;
   texCoords = 0;
   me_TextureBatchingStrategyAtlas::AtlasSource* sourceFound = findSource(textureID, atlasID);
   if (sourceFound)
   {
      sourceFound->setFrameNumber(frameNumber);
      batchedTexture = sourceFound->_atlas->_texture;
      texCoords = sourceFound->getTextureCoords();
   }
   return (0 != sourceFound);
}

void me_TextureBatchingStrategyAtlas::dumpBatches(const std::string& dumpPath)
{
   OpenThreads::ScopedLock<OpenThreads::ReentrantMutex> lock(_mutex);

   // A filename will have 3 parts- path, index and, wallpaperindex + extension.
   for (std::map<unsigned int, std::vector<WallpaperAtlas*> >::iterator atlasIterator = _atlasMap.begin();
      atlasIterator != _atlasMap.end(); ++atlasIterator)
   {
      std::stringstream fileNameWithIndex;
      fileNameWithIndex << dumpPath << "/atlas_" << atlasIterator->first << "_";
      unsigned int atlasWallPaperSize = static_cast<unsigned int>(atlasIterator->second.size());
      for (unsigned int wallPaperIndex = 0; wallPaperIndex < atlasWallPaperSize; wallPaperIndex++)
      {
         std::stringstream fileNameComplete;
         fileNameComplete << fileNameWithIndex.str() << wallPaperIndex << ".png";
         // We write to a png file for every image that we parse
         osg::ref_ptr<osg::Image> imageToWrite = ((atlasIterator->second)[wallPaperIndex])->_image.get();
         if (imageToWrite)
         {
            osgDB::writeImageFile(*imageToWrite, fileNameComplete.str());
         }
      }
   }
}

void me_TextureBatchingStrategyAtlas::addSourceToList(me_TextureBatchingStrategyAtlas::AtlasSource* source)
{
   if ((source != NULL) && !getSource(source->_texture))
   {
      _sourceList.push_back(source);
   }
}

osg::Texture* me_TextureBatchingStrategyAtlas::addTextureObject(unsigned int textureID, osg::Texture2D* inputTexture, unsigned int frameNumber,
                                                                const TextureParameters& textureParam, osg::ref_ptr<osg::Array>& texCoords)
{
   OpenThreads::ScopedLock<OpenThreads::ReentrantMutex> lock(_mutex);
   osg::Texture* batchedTexture = addSource(inputTexture, textureID, frameNumber, textureParam, texCoords);
   return batchedTexture;
}//lint !e429

void me_TextureBatchingStrategyAtlas::storeTextureCoordinates(osg::ref_ptr<AtlasSource> source)
{
   if(source.valid())
   {
      osg::Matrix textureMatrix = getTextureMatrix(source->_texture);
      osg::ref_ptr<osg::Vec2Array> texCoords = new osg::Vec2Array; 
      if(texCoords.valid())
      {
         addDefaultsToTexCoordArray(texCoords.get());

         //compute the actual texture co-ordinates
         for(unsigned int i = 0; i < 4 ; i++)
         {
            (*texCoords)[i].x() = static_cast<float>(((*texCoords)[i].x()  * textureMatrix(0,0)) + ( (*texCoords)[i].y()  * textureMatrix(1,0) ) + textureMatrix(3,0));
            (*texCoords)[i].y() = static_cast<float>(((*texCoords)[i].x()  * textureMatrix(0,1)) + ( (*texCoords)[i].y()  * textureMatrix(1,1) ) + textureMatrix(3,1));
         }

         source->setTextureCoords(texCoords);
      }
   }
}

bool me_TextureBatchingStrategyAtlas::rebuildAtlas(osg::ref_ptr<AtlasSource> source, osg::ref_ptr<WallpaperAtlas> atlas, unsigned int frameNumber)
{
   // since this method is only called in locked state we remove the mutex here to avoid recursive mutex deadlocks

   //few parts of the code copied from osgUtil::Optimizer::TextureAtlasBuilder::buildAtlas()
   AtlasList::iterator aitr = _atlasList.end();
   //get the source and atlas iterator
   //need the iterator for completeRow(aitr - _atlasList.begin()) provided by osg
   {
      for(AtlasList::iterator itr = _atlasList.begin(); itr != _atlasList.end(); ++itr)
      {
         if( (*itr) == atlas)
         {
            aitr = itr;
            break;
         }
      }
   }
   bool addedSourceToAtlas = false;
   if (!source->_atlas && source->suitableForAtlas(atlas->_maximumAtlasWidth, atlas->_maximumAtlasHeight,atlas->_margin))
   {
      bool dirtyImmediately = (getThreadID() == getImmediateDirtyThreadId());
      //check where does the source fit in the atlas
      if(atlas->_image) 
      {
         if( atlas->_image->getPixelFormat() == source->_image->getPixelFormat() &&  atlas->_image->getPacking() == source->_image->getPacking() )
         {
            OSG_INFO<<"checking source "<<source->_image->getFileName()<<" to see it it'll fit in atlas "<<atlas<<std::endl;
            Atlas::FitsIn fitsIn = atlas->doesSourceFit(source);
            if (fitsIn == Atlas::FITS_IN_CURRENT_ROW)
            {
               addedSourceToAtlas = true;
               {
                  // addSource() in the atlas is what determines the position of the source image in the wallpaper atlas
                  atlas->addSource(source); // Add in the currentRow.

                  atlas->copyOnlyThisSource(source, dirtyImmediately);
                  source->setFrameNumber(frameNumber);
               }
            }
            else if(fitsIn == Atlas::IN_NEXT_ROW)
            {
               //need the atlas iterator only for this function
               completeRow(static_cast<unsigned int>(aitr - _atlasList.begin())); //Fill Empty spaces
               addedSourceToAtlas = true;
               //if(!source->_atlas) // prevents multiple copies in _atlasList->sourceList
               {
                  // addSource() in the atlas is what determines the position of the source image in the wallpaper atlas
                  atlas->addSource(source); // Add the source in the new row.
                  atlas->copyOnlyThisSource(source, dirtyImmediately);
                  source->setFrameNumber(frameNumber);
               }
            }
            else
            {
               atlas->setAtlasFull(true);
            }
         }
         else
         {
            addedSourceToAtlas = false;
         }
      }
      //allocate memory for the atlas->_image
      else
      {         
         {  
            //addSource()in the atlas is what determines the position of the source image in the wallpaper atlas
            //atlas->_images gets its properties from the first source added to it         
            atlas->addSource(source);
         }
         source->setFrameNumber(frameNumber);

         //allocate memory for the atlas image
         GLenum pixelFormat = (atlas)->_image->getPixelFormat();
         GLenum dataType = (atlas)->_image->getDataType();
         GLenum packing = (atlas)->_image->getPacking();
         (atlas)->_image->allocateImage( atlas->_maximumAtlasWidth, atlas->_maximumAtlasHeight ,1, pixelFormat, dataType,packing);
         {
            //clear memory
            unsigned int size = (atlas)->_image->getTotalSizeInBytes();
            unsigned char* str = (atlas)->_image->data();
            memset(str, 0, size);
         }

         atlas->copyOnlyThisSource(source, dirtyImmediately);
         addedSourceToAtlas = true;
      }
   }
   return addedSourceToAtlas;
}

bool me_TextureBatchingStrategyAtlas::replaceSource( Source* oldSrc , Source* newSrc, bool dirtyImmediately )
{
   //replace source will be successful only if the old texture and the new texture are of the same size

   //check with max width, height and margin of source
   Atlas* atlas = oldSrc->_atlas;
   if( (oldSrc->_image->s() == newSrc->_image->s() && oldSrc->_image->t() == newSrc->_image->t() ))
   { 
      newSrc->_atlas = oldSrc->_atlas;
      newSrc->_x = oldSrc->_x;
      newSrc->_y = oldSrc->_y;
      atlas->_sourceList.push_back(newSrc);
      atlas->copyOnlyThisSource(newSrc, dirtyImmediately);

      //remove the oldSrc from the source lists
      for(SourceList::iterator sitr = _sourceList.begin(); sitr != _sourceList.end(); ++sitr)
      {
         if( oldSrc == (*sitr) )
         {
            _sourceList.erase(sitr);
            break;
         }
      }

      for(SourceList::iterator sitr = atlas->_sourceList.begin(); sitr != atlas->_sourceList.end(); ++sitr)
      {
         if( oldSrc == (*sitr) )
         {
            atlas->_sourceList.erase(sitr);
            break;
         }
      }
      return true;
   }
   return false;
}

void me_TextureBatchingStrategyAtlas::WallpaperAtlas::copyOnlyThisSource(Source* source, bool dirtyImmediately)
{
   { //code copied from osgUtil::Optimizer::TextureAtlasBuilder::Atlas::copySources() with minor changes

      Atlas* atlas = source->_atlas;
      if (atlas == this)
      {
         OSG_INFO<<"Copying image "<<source->_image->getFileName()<<" to "<<source->_x<<" ,"<<source->_y<<std::endl;
         OSG_INFO<<"        image size "<<source->_image->s()<<","<<source->_image->t()<<std::endl;

         const osg::Image* sourceImage = source->_image.get();
         osg::Image* atlasImage = atlas->_image.get();
         //assert(sourceImage->getPacking() == atlasImage->getPacking()); //Test if packings are equals.
         unsigned int rowSize = sourceImage->getRowSizeInBytes();
         unsigned int pixelSizeInBits = sourceImage->getPixelSizeInBits();
         unsigned int pixelSizeInBytes = pixelSizeInBits/8;
         unsigned int marginSizeInBytes = pixelSizeInBytes*_margin;

         //assert(atlas->_width  == static_cast<int>(atlasImage->s()));
         //assert(atlas->_height == static_cast<int>(atlasImage->t()));
         //assert(source->_x + static_cast<int>(source->_image->s())+_margin <= static_cast<int>(atlas->_image->s()));        // "+_margin" and not "+2*_margin" because _x already takes the margin into account
         //assert(source->_y + static_cast<int>(source->_image->t())+_margin <= static_cast<int>(atlas->_image->t()));
         //assert(source->_x >= _margin);
         //assert(source->_y >= _margin);
         int x = source->_x;
         int y = source->_y;

         int t;
         for(t=0; t<sourceImage->t(); ++t, ++y)
         {
            unsigned char* destPtr = atlasImage->data(x, y);
            const unsigned char* sourcePtr = sourceImage->data(0, t);
            if ((NULL != destPtr) && (NULL != sourcePtr))
            {
               memcpy(destPtr, sourcePtr, rowSize);
            }
         }

         // copy top row margin
         y = source->_y + sourceImage->t();
         int m;
         for(m=0; m<_margin; ++m, ++y)
         {
            unsigned char* destPtr = atlasImage->data(x, y);
            if (NULL != destPtr)
            {
               memset(destPtr, 0, rowSize);
            }
         }
         // copy bottom row margin
         y = source->_y-1;
         for(m=0; m<_margin; ++m, --y)
         {
            unsigned char* destPtr = atlasImage->data(x, y);
            if (NULL != destPtr)
            {
               memset(destPtr, 0, rowSize);
            }
         }


         // copy left column margin
         y = source->_y;
         for(t=0; t<sourceImage->t(); ++t, ++y)
         {
            x = source->_x-1;
            for(m=0; m<_margin; ++m, --x)
            {
               unsigned char* destPtr = atlasImage->data(x, y);
               if (NULL != destPtr)
               {
                  memset(destPtr, 0, pixelSizeInBytes);
               }
            }
         }            

         // copy right column margin
         y = source->_y;
         for(t=0; t<sourceImage->t(); ++t, ++y)
         {
            x = source->_x + sourceImage->s();
            for(m=0; m<_margin; ++m, ++x)
            {
               unsigned char* destPtr = atlasImage->data(x, y);
               if (NULL != destPtr)
               {
                  memset(destPtr, 0, pixelSizeInBytes);
               }
            }
         }            

         // copy top left corner margin
         y = source->_y + sourceImage->t();
         for(m=0; m<_margin; ++m, ++y)
         {
            unsigned char* destPtr = atlasImage->data(source->_x - _margin, y);
            if (NULL != destPtr)
            {
               memset(destPtr, 0, marginSizeInBytes);
            }
         }

         // copy top right corner margin
         y = source->_y + sourceImage->t();
         for(m=0; m<_margin; ++m, ++y)
         {
            unsigned char* destPtr = atlasImage->data(source->_x + sourceImage->s(), y);
            if (NULL != destPtr)
            {
               memset(destPtr, 0, marginSizeInBytes);
            }
         }

         // copy bottom left corner margin
         y = source->_y - 1;
         for(m=0; m<_margin; ++m, --y)
         {
            unsigned char* destPtr = atlasImage->data(source->_x - _margin, y);
            if (NULL != destPtr)
            {
               memset(destPtr, 0, marginSizeInBytes);
            }
         }

         // copy bottom right corner margin
         y = source->_y - 1;
         for(m=0; m<_margin; ++m, --y)
         {
            unsigned char* destPtr = atlasImage->data(source->_x + sourceImage->s(), y);
            if (NULL != destPtr)
            {
               memset(destPtr, 0, marginSizeInBytes);
            }
         }
      }
   }

   if (dirtyImmediately)
   {
      _image->dirty();
   }
   else
   {
      setDirtyUpdatePending(true);
   }
}

osg::Texture* me_TextureBatchingStrategyAtlas::addTextureObject( unsigned int textureID,
                                                                 osg::Image* inputImage,
                                                                 unsigned int frameNumber,
                                                                 const me_TextureBatchingStrategyBase::TextureParameters& textureParam,
                                                                 osg::ref_ptr<osg::Array>& texCoords )
{
   OpenThreads::ScopedLock<OpenThreads::ReentrantMutex> lock(_mutex);
   osg::Texture* batchedTexture = 0;
   osg::Texture2D* sourceTexture = createTextureWrapper(inputImage, textureParam);
   if (sourceTexture)
   {
      batchedTexture = addSource(sourceTexture, textureID, frameNumber, textureParam, texCoords);
      sourceTexture = 0; //lint !e423
   }
   return batchedTexture;
}

osg::Texture* me_TextureBatchingStrategyAtlas::replaceTextureObject( unsigned int textureID,
                                                                     osg::Texture2D* inputTexture,
                                                                     unsigned int frameNumber,
                                                                     const TextureParameters& textureParam,
                                                                     osg::ref_ptr<osg::Array>& texCoords )
{
   OpenThreads::ScopedLock<OpenThreads::ReentrantMutex> lock(_mutex);
   osg::Texture* atlasTexture = 0;
   bool atlasPresent = false;
   osg::ref_ptr<WallpaperAtlas> atlas = NULL;

   //check if the atlas with the given atlasID already exists
   for(AtlasList::iterator aitr = _atlasList.begin(); aitr != _atlasList.end(); ++aitr)
   {
      WallpaperAtlas* currentAtlas= static_cast<WallpaperAtlas*>( aitr->get() );//lint !e1774
      if(currentAtlas->getID() == textureParam.atlasID /*&& !(osg::Texture::REPEAT == textureParam.wrapMode) */)
      {
         atlas = static_cast<WallpaperAtlas*>( aitr->get() );//lint !e1774
         atlasPresent = true;
         break;
      }
   }

   if( atlasPresent && atlas.valid())
   {
      //a valid atlas with the given atlasID is present
      osg::ref_ptr<AtlasSource> existingSource = atlas->getExistingSource(textureID);
      if (existingSource.valid())
      {
         osg::ref_ptr<AtlasSource> newSource = new AtlasSource(inputTexture, textureID);
         newSource->setFrameNumber(frameNumber);
         if (replaceSource(existingSource, newSource, getThreadID() == getImmediateDirtyThreadId()))
         {
            storeTextureCoordinates(newSource);
            texCoords = newSource->getTextureCoords();
            atlasTexture = atlas->_texture;
         }
      }
   }

   if (!atlasTexture)
   {
      // The texture doesn't exist yet, so create it
      atlasTexture = addTextureObject(textureID, inputTexture, frameNumber, textureParam, texCoords);
   }
   
   return atlasTexture;
}

osg::Texture* me_TextureBatchingStrategyAtlas::replaceTextureObject( unsigned int textureID,
                                                                     osg::Image* inputImage, 
                                                                     unsigned int frameNumber, 
                                                                     const me_TextureBatchingStrategyBase::TextureParameters& textureParam, 
                                                                     osg::ref_ptr<osg::Array>& texCoords )
{
   osg::Texture* batchedTexture = 0;
   osg::Texture2D* sourceTexture = createTextureWrapper(inputImage, textureParam);
   if (sourceTexture)
   {
      batchedTexture = replaceTextureObject(textureID, sourceTexture, frameNumber, textureParam, texCoords);
      sourceTexture = 0; //lint !e423
   }
   return batchedTexture;
}

void me_TextureBatchingStrategyAtlas::removeAllTextures( unsigned int batchID )
{
   OpenThreads::ScopedLock<OpenThreads::ReentrantMutex> lock(_mutex);
   AtlasList::iterator aitr = _atlasList.begin();
   while(aitr != _atlasList.end())
   {      
      WallpaperAtlas* atlas = static_cast<WallpaperAtlas*>(aitr->get()); //lint !e1774
      ++aitr;
      if(atlas->getID() == batchID)
      {
         _sourceList.erase(remove_if(_sourceList.begin(), _sourceList.end(), SourceErase(atlas)), _sourceList.end());
         aitr = _atlasList.erase(--aitr);
      }
   }   
   _atlasMap.erase(batchID);
}

osg::Texture* me_TextureBatchingStrategyAtlas::createSingleTexture( osg::Image* image ) const
{
   return new osg::Texture2D(image);
}

osg::Texture* me_TextureBatchingStrategyAtlas::addSource(osg::Texture2D* texture,
                                                         unsigned int textureID,
                                                         unsigned int frameNumber,
                                                         const me_TextureBatchingStrategyBase::TextureParameters& textureParam,
                                                         osg::ref_ptr<osg::Array>& texCoords)
{
   bool atlasReady = false;
   osg::ref_ptr<AtlasSource> curSource;
   osg::ref_ptr<WallpaperAtlas> atlas;
   if (texture)
   {
      if (texture->getImage() && (texture->getImage()->getPixelFormat() != GL_RGBA) && !texture->getImage()->isCompressed())
      {
         texture->setImage(mapengine::oss::convertToRGBA(texture->getImage()));
      }
      curSource = new AtlasSource(texture, textureID);
   }
   if (curSource.valid())
   {
      bool atlasPresent = false;

      //check if the atlas with the given atlasID already exists
      for (AtlasList::iterator aitr = _atlasList.begin(); aitr != _atlasList.end(); ++aitr)
      {
         WallpaperAtlas* currentAtlas = static_cast<WallpaperAtlas*>(aitr->get());//lint !e1774
         if (currentAtlas->getID() == textureParam.atlasID /*&& !(osg::Texture::REPEAT == textureParam.wrapMode) */)
         {
            atlas = static_cast<WallpaperAtlas*>(aitr->get());//lint !e1774
            atlasPresent = true;
            break;
         }
      }

      if (atlasPresent && atlas.valid()) //a valid atlas with the given atlasID is present
      {
         atlasReady = rebuildAtlas(curSource, atlas, frameNumber);
         if (!atlasReady)
         {
            {
               unsigned int i = 1;
               while (!atlasReady && (i < _atlasMap[textureParam.atlasID].size()))
               {
                  WallpaperAtlas* localAtlas = _atlasMap[textureParam.atlasID][i];
                  //check if re-caching is possible
                  atlasReady = rebuildAtlas(curSource, localAtlas, frameNumber);
                  ++i;
               }
            }
            //if the source could not be put in any of the existing atlas, create a new one
            if (!atlasReady)
            {
               //std::cout<< "given atlas is full, creating new one"<<std::endl;
               //if (textureParam.atlasHeight >= curSource->_texture->getTextureHeight() )
               {
                  osg::ref_ptr<WallpaperAtlas> texAtlas = createAtlasAndAddSource(curSource, textureParam, atlas->getSubID() + 1, frameNumber);
                  atlasReady = texAtlas.valid();
               }
            }
         }
      }
      else
      {
         //if the texture is used in repeat mode, then the size of atlas == size of the texture
         //          if(osg::Texture::REPEAT == textureParam.wrapMode)
         //          {
         //             textureParam.atlasWidth = curSource->_image->s();
         //             textureParam.atlasHeight = curSource->_image->t();
         //          }
         atlas = createAtlasAndAddSource(curSource, textureParam, 0, frameNumber);
         atlasReady = atlas.valid();
      }
   }
   if (atlasReady)
   {
      addSourceToList(curSource);
      storeTextureCoordinates(curSource);
      texCoords = curSource->getTextureCoords();
      return atlas->_texture;
   }
   else
   {
      return 0;
   }
}

osg::ref_ptr<me_TextureBatchingStrategyAtlas::WallpaperAtlas> me_TextureBatchingStrategyAtlas::createAtlasAndAddSource(osg::ref_ptr<AtlasSource> atlasSource,
   const me_TextureBatchingStrategyBase::TextureParameters& textureParam, unsigned int subId, unsigned int frameNumber)
{
   bool atlasReady = false;
   osg::ref_ptr<WallpaperAtlas> texAtlas = createAtlas(textureParam, atlasSource->_texture->getTextureWidth(), atlasSource->_texture->getTextureHeight());
   if (texAtlas.valid())
   {
      _atlasList.push_back(texAtlas);
      texAtlas->setID(textureParam.atlasID);
      texAtlas->setSubID(subId);
      _atlasMap[textureParam.atlasID].push_back(texAtlas);
      atlasReady = rebuildAtlas(atlasSource, texAtlas, frameNumber);
   }

   return atlasReady ? texAtlas : 0;
}

void me_TextureBatchingStrategyAtlas::setTexCoordPointer( osg::State& state, osg::Array* tcArray, unsigned int vertOffset )
{
   osg::Vec2Array* tcv2 = static_cast<osg::Vec2Array*>(tcArray); //lint !e1774
   if(!tcv2->empty()) state.setTexCoordPointer( 0, 2, GL_FLOAT, 0, (&(tcv2->front())+ vertOffset) );
}

me_TextureBatchingStrategyAtlas::AtlasSource* me_TextureBatchingStrategyAtlas::WallpaperAtlas::getExistingSource( unsigned int textureID )
{
   me_TextureBatchingStrategyAtlas::AtlasSource* existingSource = 0;
   for(Optimizer::TextureAtlasBuilder::SourceList::iterator sitr = _sourceList.begin(); sitr != _sourceList.end(); ++sitr)
   {
      me_TextureBatchingStrategyAtlas::AtlasSource* currSource = static_cast<me_TextureBatchingStrategyAtlas::AtlasSource*>(sitr->get()); //lint !e1774
      if( currSource->getID() == textureID)//lint !e1774
      {
         existingSource = currSource;//lint !e1774
         break;
      }
   }
   return existingSource;
}

bool me_TextureBatchingStrategyAtlas::getSourceTextureDimensions(unsigned int textureID, unsigned int atlasID, unsigned int& width, unsigned int& height)
{
   OpenThreads::ScopedLock<OpenThreads::ReentrantMutex> lock(_mutex);
   me_TextureBatchingStrategyAtlas::AtlasSource* sourceFound = findSource(textureID, atlasID);
   if (sourceFound)
   {
      width = sourceFound->_texture->getImage()->s();
      height = sourceFound->_texture->getImage()->t();
   }
   return (0 != sourceFound);
}

void me_TextureBatchingStrategyAtlas::WallpaperAtlas::updateContentDirtyFlag()
{
   if (_isDirtyUpdatePending)
   {
      _isDirtyUpdatePending = false;
      _image->dirty();
   }
}

void me_TextureBatchingStrategyAtlas::updateAllDirtyContentFlags()
{
   OpenThreads::ScopedLock<OpenThreads::ReentrantMutex> lock(_mutex);
   for (AtlasList::iterator aitr = _atlasList.begin(); aitr != _atlasList.end(); ++aitr)
   {
      WallpaperAtlas* currentAtlas = static_cast<WallpaperAtlas*>(aitr->get());//lint !e1774
      currentAtlas->updateContentDirtyFlag();
   }
}

me_TextureBatchingStrategyAtlas::WallpaperAtlas* me_TextureBatchingStrategyAtlas::createAtlas(const me_TextureBatchingStrategyBase::TextureParameters& textureParam,
   int firstTextureWidth, int firstTextureHeight)
{
   int largerDimensionWidth = osg::maximum(static_cast<int>(textureParam.atlasWidth), firstTextureWidth);
   int largerDimensionHeight = osg::maximum(static_cast<int>(textureParam.atlasHeight), firstTextureHeight);
   int margin = (textureParam.atlasWidth > (firstTextureWidth + textureParam.textureMargin)) ? static_cast<int>(textureParam.textureMargin) : 0;
   margin = osg::minimum((textureParam.atlasHeight > (firstTextureHeight + textureParam.textureMargin)) ? static_cast<int>(textureParam.textureMargin) : static_cast<int>(0), margin);
   if ((static_cast<int>(textureParam.atlasWidth + textureParam.textureMargin) <= firstTextureWidth) || (static_cast<int>(textureParam.atlasHeight + textureParam.textureMargin) <= firstTextureHeight))
   {
      OSG_WARN << "image being added to atlas is as big or bigger than the desired atlas size. TexSize: " << firstTextureWidth << "x" << firstTextureHeight << "AtlasSize: "
         << textureParam.atlasWidth << "x" << textureParam.atlasHeight << "Margin: " << textureParam.textureMargin;
   }
   return new WallpaperAtlas(largerDimensionWidth, largerDimensionHeight, margin);
}

osg::Texture2D* me_TextureBatchingStrategyAtlas::createTextureWrapper(osg::Image* inputImage, const me_TextureBatchingStrategyBase::TextureParameters& textureParam) const
{
   osg::Texture2D* texture = new osg::Texture2D(inputImage);
   if (texture)
   {
      // set the desired attribs to the source texture as OSG Optimizer sets the same to the atlas
      texture->setWrap(osg::Texture::WRAP_R, textureParam.wrapMode);
      texture->setWrap(osg::Texture::WRAP_S, textureParam.wrapMode);
      texture->setWrap(osg::Texture::WRAP_T, textureParam.wrapMode);
      texture->setFilter(osg::Texture::MIN_FILTER, textureParam.disableMipmap ? osg::Texture::LINEAR : osg::Texture::LINEAR_MIPMAP_LINEAR);
      texture->setFilter(osg::Texture::MAG_FILTER, osg::Texture::LINEAR);
      texture->setTextureWidth(inputImage->s());
      texture->setTextureHeight(inputImage->t());
   }
   return texture;
}

me_TextureBatchingStrategyAtlas::AtlasSource* me_TextureBatchingStrategyAtlas::findSource(unsigned int textureID, unsigned int atlasID)
{
   AtlasSource* sourceFound = 0;
   //iterate through the source list of the given atlasID to see if the texture with the given textureID exists
   {
      OpenThreads::ScopedLock<OpenThreads::ReentrantMutex> lock(_mutex);
      for (AtlasList::iterator aitr = _atlasList.begin(); (aitr != _atlasList.end()) && (!sourceFound); ++aitr)
      {
         if (static_cast<WallpaperAtlas*>(aitr->get())->getID() == atlasID)//lint !e1774
         {
            for (SourceList::iterator sitr = aitr->get()->_sourceList.begin(); (sitr != aitr->get()->_sourceList.end()) && (!sourceFound); ++sitr)
            {
               AtlasSource* currentSource = static_cast<AtlasSource*>(sitr->get());//lint !e1774
               if (currentSource->getID() == textureID)
               {
                  sourceFound = currentSource;
               }
            }
         }
         //keep searching other atlases as the atlas id is not necessarily unique
      }
   }
   if (sourceFound)
   {
      OSG_INFO << "Found texture " << textureID << " in atlas " << atlasID << std::endl;
   }
   else
   {
      OSG_INFO << "Did not find texture " << textureID << " in atlas " << atlasID << std::endl;
   }

   return sourceFound;
}
