////////////////////////////////////////////////////////////////////////////////
// File: tHashMap.h
//
// Author: Heinemann
//       - modified by Stresing to have hashfunction calculate hash also for non integer keys : change not working
//		 - modified: lines 245, 260: initialize vector with single parameter, instead of second parameter = 0 (clang complains matching ctors)
//
// Creation Date: 03.04.2006
//
// Dependencies:           none
//
// Purpose: hash map
//
// The class tHashMap is adapted from Ulrich Breymann's
// book "Designing Components with the C++STL"
//
////////////////////////////////////////////////////////////////////////////////

#ifndef T_HASHMAP_H_
#define T_HASHMAP_H_

// implicit data structures

#include <vector>
#include <deque>
#include <algorithm>


class tHashFunction {
   public:
    /////////////////////////////////////////////////////////////////////////
    // Function: operator=()
    //
    // Description:
    //
    // Inputs: S:         (in) new hash map content
    //
    // Returns:
    /////////////////////////////////////////////////////////////////////////
    tHashFunction() {};

    /////////////////////////////////////////////////////////////////////////
    // Function: operator=()
    //
    // Description:
    //
    // Inputs: S:         (in) new hash map content
    //
    // Returns:
    /////////////////////////////////////////////////////////////////////////
    tS32 operator()(tS32 p) const
    {
       return p & 255;
    }

    static tS32 TableSize()
    {
      return 256;
    }
};


// hash map class
template<class Key, class T>
class tHashMap
{
  public:
    typedef size_t size_type;
    typedef std::pair<const Key,T> value_type;

    // define more readable denominations
    typedef std::deque<value_type> list_type;
    typedef std::vector<list_type*> vector_type;

    /* The template parameter Key stands for the type of the key; T
       stands for the class of data associated to a key; and hashFun
       is the placeholder for the data type of the function objects
       used for address calculation. Below, a function object for
       address calculation is proposed, but any other one can be used
       as well. In analogy to map, value_type is the type of the
       elements that are stored in an tHashMap object. value_type is a
       pair consisting of a constant key and the associated data. */

    class iterator;
    // maintain compatibility with the STL:
    typedef iterator const_iterator;
    friend class iterator;

    /* The nested class iterator closely cooperates with tHashMap, so that
       both are mutually declared as friend. iterator is only supposed
       to allow forward traversal and therefore its category is
       defined as the standard forward_iterator_tag. An iterator object
       allows you to visit all elements of an tHashMap object one after
       the other. Neither an order nor a sorting is defined for the
       elements. The visiting order of the iterator is given by the
       implicit data structure (see below, operator++()). */

    class iterator {
      friend class tHashMap<Key, T>;
      private:
       typename list_type::iterator current;
       typedef std::forward_iterator_tag iterator_category;
       size_type Address;
       const vector_type *pVec;

      public:
       iterator()
          : Address(0), pVec(0) {
       }

       iterator(typename list_type::iterator LI,
                size_type A,  const vector_type *C)
       : current(LI), Address(A), pVec(C) {
       }

       /* The following operators allow you to check an tHashMap iterator
          in the condition part of if or while as to whether it is at
          all defined: */

       operator const void* () const {
           return pVec;
       }  

       bool operator!() const {
           return pVec == 0;
       }

       /* The operator for dereferencing occurs both in the const
          variation and in the non-const variation. Thus,
          dereferencing of an undefined iterator is punished with a
          program abort, which is a clear message to you to check the
          program that uses the iterator. */

       const value_type& operator*() const {
          OSAL_vAssert(pVec);
          return *current;
       }  

       value_type& operator*() {
          OSAL_vAssert(pVec);
          return *current;
       }

       /* The non-const variation is required to be able to modify
          data independently from the key. Modification of the key
          must be excluded because it requires a new address
          calculation. Constancy is guaranteed by the const
          declaration in the type definition of value_type.

          How does the tHashMap iterator move from one element to the
          other with operator++()? First, current is incremented:
         */

       iterator& operator++() {
          ++current;

          /* If after this, current points to a list element, a
             reference to the iterator is returned (see below: return
             *this. Otherwise, the end of the list is reached. */

          if(current == (*pVec)[Address]->end()) {

            /* At this point, one address after the other is checked
               in the vector, until either a list entry is found or
               the end of the vector is reached. In the latter case,
               the iterator becomes invalid, because it can only move
               forward. In order to exclude further use, pVec is set
               to 0: */

             while(++Address < pVec->size())
                if((*pVec)[Address]) {
                   current = (*pVec)[Address]->begin();
                   break;
                }

             if(Address == pVec->size()) // end of vector reached
                pVec = 0;
          }
          return *this;
       }

       iterator operator++(int) {
           iterator temp = *this;
           operator++();
           return temp;
       }

       /* The last two methods compare two tHashMap iterators. Two
          undefined or invalidated iterators are always considered as
          equal: */

       bool operator==(const iterator& x) const {
           return  pVec && x.pVec && (current == x.current)
                || (!pVec && !x.pVec);
       }  

       bool operator!=(const iterator& x) const {
          return !operator==(x);
       }
    }; // iterator

  public:
    /////////////////////////////////////////////////////////////////////////
    // Function: begin()
    //
    // Description: constructor
    //
    // Inputs: f:         (in) hash function
    //
    // Returns:
    /////////////////////////////////////////////////////////////////////////
    iterator begin() const {
        size_type adr = 0;
        while(adr < v.size()) {
           if(!v[adr])    // found nothing?
             ++adr;       // continue search
           else
             return iterator(v[adr]->begin(), adr, &v);
        }
        return iterator();
    }

    /////////////////////////////////////////////////////////////////////////
    // Function: end()
    //
    // Description: constructor
    //
    // Inputs: f:         (in) hash function
    //
    // Returns:
    /////////////////////////////////////////////////////////////////////////
    iterator end() const {
        return iterator();
    }

    /////////////////////////////////////////////////////////////////////////
    // Function: tHashMap()
    //
    // Description: constructor
    //
    // Inputs: f:         (in) hash function
    //
    // Returns:
    /////////////////////////////////////////////////////////////////////////
    tHashMap()
    : v(tHashFunction::TableSize()), count(0) {
    }

    /////////////////////////////////////////////////////////////////////////
    // Function: tHashMap()
    //
    // Description: constructor
    //
    // Inputs: S:         (in) hash map that is used for initialization
    //
    // Returns:
    /////////////////////////////////////////////////////////////////////////
    tHashMap(const tHashMap& S) {
        hf = S.hf;
        // provide deep copy
        v = vector_type(S.v.size());
        count = 0;
        // begin(), end(), insert(): see below
        iterator t = S.begin();
        while(t != S.end())
          insert(*t++); 
    }

    /////////////////////////////////////////////////////////////////////////
    // Function: tHashMap~()
    //
    // Description: destructor
    //
    // Inputs:
    //
    // Returns:
    /////////////////////////////////////////////////////////////////////////
    ~tHashMap() {
        clear();                       // see below
    }

    /////////////////////////////////////////////////////////////////////////
    // Function: operator=()
    //
    // Description:
    //
    // Inputs: S:         (in) new hash map content
    //
    // Returns:
    /////////////////////////////////////////////////////////////////////////
    tHashMap& operator=(const tHashMap& S) {
        if(this != &S) {
          tHashMap temp(S);
          swap(temp); // see below
         }
        return *this;
    }

    /* clear() uses delete to call the destructor of each list
       referred to by a vector element. Subsequently, the vector
       element is marked as unoccupied.*/

    /////////////////////////////////////////////////////////////////////////
    // Function: clear()
    //
    // Description:
    //
    // Inputs:
    //
    // Returns:
    /////////////////////////////////////////////////////////////////////////
    void clear() {
         for(size_t i = 0; i < v.size(); ++i)
            if(v[i]) {                 // does list exist?
                OSAL_DELETE v[i];
                v[i] = 0;
            }
         count = 0;
    }

    /* In the following find() and insert() functions, the sought
       address within the vector v is calculated directly by means of
       the hash function object. If the vector element contains a
       pointer to a list, the list is searched in find() by means of
       the list iterator temp until an element with the correct key is
       found or the list has been completely processed: */

    /////////////////////////////////////////////////////////////////////////
    // Function: find()
    //
    // Description:
    //
    // Inputs: k:         (in) key
    //
    // Returns: iterator
    /////////////////////////////////////////////////////////////////////////
    iterator find(const Key& k) const
    {
        size_type address = hf(k);     // calculate address


        if(!v[address])
           return iterator();          // not existent
        typename list_type::iterator temp =  v[address]->begin();

        // find k in the list
        while(temp != v[address]->end())
          if((*temp).first == k)
            return iterator(temp,address,&v); //found
          else ++temp;

        return iterator();
    }

    /* A map stores pairs of keys and associated data, where the first
       element (first) is the key and the second element ( second)
       contains the data. find() returns an iterator which can be
       interpreted as a pointer to a pair. In order to obtain the data
       belonging to a key, the index operator can be called with the
       key as argument: */

    /////////////////////////////////////////////////////////////////////////
    // Function: operator[]
    //
    // Description:
    //
    // Inputs: k:         (in) key
    //
    // Returns: associated data
    /////////////////////////////////////////////////////////////////////////
    T& operator[](const Key& k)
    {
         return (*find(k)).second;
    }

    /* If the key does not exist, that is, if find() returns an end
       iterator, a run time error occurs while dereferencing! (See the
       dereferencing operator).
       As in the STL, insert() returns a pair whose first part
       consists of the iterator that points to the found position. The
       second part indicates whether the insertion has taken place or
       not. */

    /////////////////////////////////////////////////////////////////////////
    // Function: insert()
    //
    // Description:
    //
    // Inputs: P:         (in) element to be inserted
    //
    // Returns: iterator to found position
    /////////////////////////////////////////////////////////////////////////
    std::pair<iterator, bool> insert(const value_type& P)
    {
        iterator temp = find(P.first);
        bool inserted = false;

        if(!temp) { // not present
            size_type address = hf(P.first);
            if(!v[address])
               v[address] = OSAL_NEW list_type;
            v[address]->push_front(P);
            temp = find(P.first); // redefine temp
            inserted = true;
            ++count;
        }
        return std::make_pair(temp, inserted);
    }

    /* After the insertion, temp is redefined, because the iterator at
       first does not point to an existing element. The known
       auxiliary function makepair() generates a pair object to be
       returned. */

    /////////////////////////////////////////////////////////////////////////
    // Function: erase()
    //
    // Description:
    //
    // Inputs: q:         (in) iterator that specifies the obsolete element
    //
    // Returns:
    /////////////////////////////////////////////////////////////////////////
    void erase(iterator q)
    {
        /* If the iterator is defined at all, the element function
           erase() of the associated list is called. Subsequently, the
           list is deleted, provided it is now empty, and the vector
           element to which the list is attached, is set to 0. */

       if(q.pVec) {           // defined?
          v[q.Address]->erase(q.current);

          if(v[q.Address]->empty()) {
             OSAL_DELETE v[q.Address];
             v[q.Address] = 0;
          }
          --count;
       }
    }

    /* Sometimes, one would probably like to delete all elements of a
       map that have a given key. In an tHashMap, this can at most be one
       element, but in an HMultimap, several elements might be
       affected. */

    /////////////////////////////////////////////////////////////////////////
    // Function: erase()
    //
    // Description:
    //
    // Inputs: k:         (in) key that specifies the obsolete element
    //
    // Returns:
    /////////////////////////////////////////////////////////////////////////
    size_type erase(const Key& k)
    {
       size_type deleted_elements = 0; // count
       // calculate address
       size_type address = hf(k);
       if(!v[address])
          return 0;         // not present

       typename list_type::iterator temp =  v[address]->begin();

       /* In the following loop, the list is searched. An iterator
          called pos is used to remember the current position for the
          deletion itself. */

       while(temp != v[address]->end())
       {
          if((*temp).first == k)
          {
             typename list_type::iterator pos = temp++;

             v[address]->erase(pos);
             // pos is now undefined

             --count;
             ++deleted_elements;
          }
          else ++temp;
       }

       /* The temporary iterator temp is advanced in both branches of
          the if instruction. The operation ++ cannot be extracted in
          order to save the else, because temp would then be identical
          with pos which is undefined after the deletion, and a
          defined ++ operation would no longer be possible. */

       // delete hash table entry if needed
       if(v[address]->empty())
       {
          OSAL_DELETE v[address];
          v[address] = 0;
       }
       return deleted_elements;
    }

    /* We present here a couple of very simple methods. As opposed to
       other containers, max_size() does not indicate the maximum
       number of elements that can be stored in an tHashMap container,
       which is only limited by the capacity of the lists, but the
       number of available hash table entries. This information is
       more sensible, because the efficiency of an tHashMap depends on the
       occupation range alpha, assuming a good hash function. The
       occupation rate can easily be determined: alpha =
       size()/max_size(). */

    /////////////////////////////////////////////////////////////////////////
    // Function: size()
    //
    // Description:
    //
    // Inputs:
    //
    // Returns:
    /////////////////////////////////////////////////////////////////////////
    size_type size() const
    {
      return count;
    }

    /////////////////////////////////////////////////////////////////////////
    // Function: max_size()
    //
    // Description:
    //
    // Inputs:
    //
    // Returns:
    /////////////////////////////////////////////////////////////////////////
    size_type max_size() const
    {
      return v.size();
    }

    /////////////////////////////////////////////////////////////////////////
    // Function: empty()
    //
    // Description:
    //
    // Inputs:
    //
    // Returns:
    /////////////////////////////////////////////////////////////////////////
    bool empty() const
    {
      return count == 0;
    }

    /////////////////////////////////////////////////////////////////////////
    // Function: swap()
    //
    // Description:
    //
    // Inputs: s:         (in,out) hash map
    //
    // Returns:
    /////////////////////////////////////////////////////////////////////////
    void swap(tHashMap& s)
    {
      v.swap(s.v);
      std::swap(count, s.count);
      std::swap(hf, s.hf);        
    }

  private:
    // v is the vector whose elements are pointers to linked lists
    vector_type v;

    // hf is the function object used for calculation of the hash address
    tHashFunction hf;

    // count is the number of stored pairs of keys and data
    size_type count;
};



#endif   // T_HASHMAP_H_

