/******************************************************************************
FILE:           etg.cpp
SW-COMPONENT:   general library for ETG (extended trace generator)

DESCRIPTION:    see etg.h for more details

AUTHOR:         CM-DI/VW-PJ33 Peter Hassmann, Matthias Hessling
COPYRIGHT:      (c) 2009-2012 Robert Bosch Car Multimedia GmbH

HISTORY:
Date     | Rev. | Author                  | Modification
----------------------------------------------------------------------------
09-09-10 | 1.0  | CM-AI/PJ-VW36 Hassmann  | Initial revision
11-02-11 | 2.0  | CM-AI/PJ-VW36 Hessling  | Using function pointer to trace
11-04-11 | 2.1  | CM-AI/PJ-VW36 Hessling  | Adding errmem function pointer
24-09-12 | 2.2  | CM-AI/PJ-CB36 Hessling  | Adding ETG_I functions
19-12-13 | 2.3  | CM-AI/PJ-CB36 Hessling  | Make ETG independent from OSAL
02-01-18 | 2.4  | CM-CI1/ESW4 Hessling    | 64bit support
******************************************************************************/
#ifdef VARIANT_S_FTR_ENABLE_ETG_OSAL_THRCTRLBLOCK
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"
#endif

#define ETG_IMPLEMENTS
#include "etg.h"

#include <stdio.h>
#include <string.h>
#include <stdarg.h> //lint !e451 required for eSol tKernel compiler

#ifdef __cplusplus
extern "C"
{
#endif
/*
The component using this file has to set the fd via etg_vSetTraceDevice 
for tracing. 
*/
#ifdef ETG_USE_HGLOBALETGTRACE
static etg_tIODescriptor hGlobalETGTrace = 0;
#endif

   //#define ETG_DEBUG_PRINT(...) printf (__VA_ARGS__)
#define ETG_DEBUG_PRINT(...)
/*
In case of linux we releave from using OSAL functions.
Also in case of non OSAL defined we shall go with POSIX convention.
*/
#ifdef VARIANT_S_FTR_ENABLE_ETG_OSAL_THRCTRLBLOCK
#define etg_ThreadWhoAmI() ((etg_tThreadID)OSAL_ThreadWhoAmI())
#else
#if defined (_WIN32)
#include <windows.h>
#define etg_ThreadWhoAmI() ((etg_tThreadID)GetCurrentThreadId())
#elif defined(__APPLE__) || defined(ANDROID)
#include <pthread.h>
#define etg_ThreadWhoAmI() ((etg_tThreadID)pthread_self())
#else
#include <pthread.h>
#include <unistd.h>
#include <sys/syscall.h>
#define etg_ThreadWhoAmI() ((etg_tThreadID)syscall(SYS_gettid))
#endif
#endif

// legacy for navi group 2010-10-01
etg_tVoid etg_vSetTraceDevice(etg_tIODescriptor 
#ifdef ETG_USE_HGLOBALETGTRACE
ETGTrace
#endif
)
{
#ifdef ETG_USE_HGLOBALETGTRACE
   hGlobalETGTrace = ETGTrace;
#endif
}

etg_tBool etg_bIsTraceActive(etg_tU16 u16CompID, etg_tU16 u16LevelId)
{
   if(ETG_pfbIsTraceActive) { // make sure user has initialized trace output
      return ETG_pfbIsTraceActive(u16CompID, u16LevelId);
   } else {
      return FALSE;
   }
}

etg_tBool etg_bIsTraceActiveShort(etg_tU32 u32CompAndLevelId)
{
   if(ETG_pfbIsTraceActive) { // make sure user has initialized trace output
      return ETG_pfbIsTraceActive((etg_tU16)(u32CompAndLevelId >> 16), (etg_tU16)(u32CompAndLevelId & 0xFF));
   } else {
      return FALSE;
   }
}

// for DCL: check 2 trace-classes against level
etg_tBool etg_bIsTraceActiveDouble(etg_tU32 u32CompAndLevelId, etg_tU16 u16LevleId2)
{
   etg_tBool bActive = FALSE;
   if(ETG_pfbIsTraceActive) { // make sure user has initialized trace output
      if (ETG_pfbIsTraceActive((etg_tU16)(u32CompAndLevelId >> 16), (etg_tU16)(u32CompAndLevelId & 0xFF))) {
         bActive =  ETG_pfbIsTraceActive((etg_tU16)u16LevleId2, (etg_tU16)(u32CompAndLevelId & 0xFF));
      }
   }
   return bActive;
}

static etg_tU32 u32PushWord(etg_tU8* pu8Buf, etg_tU32 u32Index, etg_tU16 u16Word)
{
   pu8Buf[u32Index++] = (etg_tU8)(((u16Word)>>8) & 0xFF);
   pu8Buf[u32Index++] = (etg_tU8) ((u16Word)     & 0xFF);

   return u32Index;
}

static etg_tU32 u32PushLong(etg_tU8* pu8Buf, etg_tU32 u32Index, etg_tU32 u32Long)
{
   pu8Buf[u32Index++] = (etg_tU8)(((u32Long)>>24) & 0xFF);
   pu8Buf[u32Index++] = (etg_tU8)(((u32Long)>>16) & 0xFF);
   pu8Buf[u32Index++] = (etg_tU8)(((u32Long)>> 8) & 0xFF);
   pu8Buf[u32Index++] = (etg_tU8) ((u32Long)      & 0xFF);

   return u32Index;
}

static etg_tU32 u32PushLongLong(etg_tU8* pu8Buf, etg_tU32 u32Index, etg_tU64 u64LongLong)
{
   pu8Buf[u32Index++] = (etg_tU8)(((u64LongLong)>>56) & 0xFF);
   pu8Buf[u32Index++] = (etg_tU8)(((u64LongLong)>>48) & 0xFF);
   pu8Buf[u32Index++] = (etg_tU8)(((u64LongLong)>>40) & 0xFF);
   pu8Buf[u32Index++] = (etg_tU8)(((u64LongLong)>>32) & 0xFF);
   pu8Buf[u32Index++] = (etg_tU8)(((u64LongLong)>>24) & 0xFF);
   pu8Buf[u32Index++] = (etg_tU8)(((u64LongLong)>>16) & 0xFF);
   pu8Buf[u32Index++] = (etg_tU8)(((u64LongLong)>> 8) & 0xFF);
   pu8Buf[u32Index++] = (etg_tU8) ((u64LongLong)      & 0xFF);

   return u32Index;
}

// lint can't see it's already const due to macro
//lint -save -e818
// "template" to convert lists to strings in case of string-tracing
#define ETG_MAKE_LIST_STRING_IMP(entryLen, UorS, StringOrStringSigned) \
   etg_tChar *etg_MakeT##entryLen##List##StringOrStringSigned(etg_tChar *szOutBuf, etg_tCString coszFormat, etg_tU16 u16NumEntries, const etg_t##UorS##entryLen *p##UorS##entryLen##Buffer) { \
      etg_tS32 s32Res; \
      etg_tU32 u16StringPos = 0; \
      for (etg_tU16 u16Index = 0; u16Index < u16NumEntries;u16Index++) { \
         s32Res = ETG_FN_SNPRINTF(szOutBuf + u16StringPos, ETG_C_TRACE_MAX - u16StringPos, coszFormat, p##UorS##entryLen##Buffer[u16Index]); \
         if (s32Res != ETG_RES_ERROR) { \
            u16StringPos += (etg_tU32)s32Res; \
         } else { \
            break; \
         } \
      } \
      return szOutBuf; \
   }

ETG_MAKE_LIST_STRING_IMP(8,  U, String);
ETG_MAKE_LIST_STRING_IMP(16, U, String);
ETG_MAKE_LIST_STRING_IMP(32, U, String);
ETG_MAKE_LIST_STRING_IMP(64, U, String);

ETG_MAKE_LIST_STRING_IMP(8,  S, StringSigned);
ETG_MAKE_LIST_STRING_IMP(16, S, StringSigned);
ETG_MAKE_LIST_STRING_IMP(32, S, StringSigned);
ETG_MAKE_LIST_STRING_IMP(64, S, StringSigned);
//lint -restore

/* helper to print strings with fixed length:
   The string might be shorter than the given u16MaxNumChars
   The string will afterwards be printed as list and this requires, that all 
   bytes up to u16MaxNumChars are valid, even if the original string is shorter.
   So the string will be put into a buffer with the specified lentgth.
*/
etg_tU8 *etg_pu8MakeListStringBuf(etg_tU8 *pu8Dest, const etg_tChar *pcSrc, etg_tU8 u16MaxNumChars) {
   if (u16MaxNumChars >= ETG_C_TRACE_MAX) {
      u16MaxNumChars = ETG_C_TRACE_MAX - 1;
   }
   if (u16MaxNumChars > 0) {
      (etg_tVoid)ETG_FN_MEMSET(pu8Dest, 0, u16MaxNumChars);
      (etg_tVoid)ETG_FN_STRNCPY(pu8Dest, pcSrc, u16MaxNumChars);
   }
   return pu8Dest;    
}

// maximum len of the traced thread-name including \0
#define ETG_MAX_THR_NAME_LEN 9

/* helper to get id and nameString of the calling thread, has to be non-static for string-tracing */
etg_tVoid etg_vGetThreadData(etg_tThreadID *pId, etg_tChar *szName) {
   *pId = etg_ThreadWhoAmI();
#if defined VARIANT_S_FTR_ENABLE_ETG_OSAL_THRCTRLBLOCK
   OSAL_trThreadControlBlock rTcb; rTcb.szName = 0;
   (etg_tVoid)OSAL_s32ThreadControlBlock(*pId, &rTcb);
   (etg_tVoid)ETG_FN_STRNCPY(szName, (rTcb.szName?rTcb.szName:"unknown"), ETG_MAX_THR_NAME_LEN);
#else
#if defined (_WIN32) || defined(ANDROID)
   (etg_tVoid)ETG_FN_STRNCPY(szName, "    N/A", ETG_MAX_THR_NAME_LEN);
#else
   if(pthread_getname_np(pthread_self(),szName,ETG_MAX_THR_NAME_LEN) != 0 ) {
	   (etg_tVoid)ETG_FN_STRNCPY(szName, "    ???", ETG_MAX_THR_NAME_LEN);
   }
#endif // _WIN32
#endif // VARIANT_S_FTR_ENABLE_ETG_OSAL_THRCTRLBLOCK
}



// Helper to exit etg_vTraceBinary()
#define BREAK_AND_DONE_ON_NULL(ptr)                 \
   if (NULL == ptr) { bDone = TRUE; break;}

/*
Parameters:
   u32CompAndLevelId: u16TraceClass and u16TraceLevel in one u32 
   u32TraceId: (consisting unique u16FileID for this traceClass and uniqued u16TraceID inside this file)
   u16LineNr: the lineNr of the trace inside the file
   u32EtgFormats: format of the following up to 10 vargs: each format according to etg_enTraceType has 3 bits, starting with the lsb.
                  A list-arg used 6 bits (eg. ETG_EN_LIST followed by ETG_EN_T8)
                  Bit ETG_THR_IND_MASK indicates, if thread-info shall be added to the trace.
   ...            Variable number of up to 10 arguments.

*/
etg_tVoid etg_vTraceBinary(etg_tU32 u32CompAndLevelId, etg_tU32 u32TraceId, etg_tU16 u16LineNr, etg_tU32 u32EtgFormats, ...) //lint !e579 PQM_authorized_221 reason: automatically generated code, reviewed and not critical
{
   etg_tU8 au8Buffer[ETG_C_TRACE_MAX];
   au8Buffer[0]=0;

   etg_tU32 u32Length = 0;
   ETG_tVarArgList argList;
   etg_tBool bDone = FALSE;
   etg_tU16 u16CompId = (etg_tU16)(u32CompAndLevelId>> 16);
   etg_tU16 u16LevelId = (etg_tU16)u32CompAndLevelId;

   etg_tBool bErrMemTrace = FALSE;

   if (u16LevelId == (etg_tU16)ETG_LEVEL_ERRMEM) {
      bErrMemTrace = TRUE;
      u16LevelId = (etg_tU16)ETG_LEVEL_FATAL;
   }

   etg_enTraceType etgFormats[ETG_MAX_FORMATS + 1];
   etgFormats[0] = ETG_EN_DONE;
   // first byte of etg-trace is allways 0xFD
   // string-tracing is indicated by  u16LineNr==0
   if (u16LineNr) {
      ETG_DEBUG_PRINT("ETG:line=%u\n",u16LineNr);

      au8Buffer[u32Length++] = 0xFD;
      // if bit is set, thread-information shall be printed
      etg_tBool bWithThr = ((u32EtgFormats &  ETG_THR_IND_MASK) != 0);
      
      // add traceId 
      u32Length = u32PushLong(au8Buffer, u32Length, u32TraceId);
      u32Length = u32PushWord(au8Buffer, u32Length, u16LineNr);
      
      if (bWithThr) {
         // add thread-info (name + id)
         etg_tChar szThreadName[ETG_MAX_THR_NAME_LEN + 1];
         etg_tThreadID threadId;
         etg_vGetThreadData(&threadId, szThreadName);
         (etg_tVoid)ETG_FN_MEMCPY(&au8Buffer[u32Length], szThreadName, ETG_MAX_THR_NAME_LEN);
         u32Length+=8;
         u32Length = u32PushLong(au8Buffer, u32Length, (etg_tU32)threadId);
      }
   } 
   

   ETG_VarArgStart(argList, u32EtgFormats); //lint !e1773 !e1055 !e64 !e516 !e530 !e746 !e534 !e416 /*Review happened by Grosse-Pawig*/
   etg_enTraceType etgFormat=ETG_EN_DONE;
   // parse the format-entries
   etg_tBool hasMore=(etg_tBool)(u32EtgFormats & ETG_FORMAT_MORE_IND_MASK) ? 1 : 0;
   etg_tU8 numFormats = 0;
   ETG_DEBUG_PRINT("START: u32EtgFormats=%u\n", u32EtgFormats);
   while ((numFormats < ETG_MAX_FORMATS)) {
      ETG_DEBUG_PRINT("WHILE: u32EtgFormats numFormats=%u\n", numFormats);
      if (numFormats && !(numFormats%10)) {
         if (hasMore) {
            u32EtgFormats = (etg_tU32) ETG_VarArg(argList, etg_tInt); //lint !e64 !e78 !e516 !e530 !e662 !e746 !e826 !e1055 /*Review happened by Grosse-Pawig*/ 
            hasMore=(etg_tBool)(u32EtgFormats & ETG_FORMAT_MORE_IND_MASK) ? 1 : 0;
            ETG_DEBUG_PRINT("MORE: u32EtgFormats=%u hasMore=%u\n", u32EtgFormats, hasMore);
         } else {
            ETG_DEBUG_PRINT("FULL EtgFormats FULL and last\n");
            break;
         }
      }
      etgFormat = (etg_enTraceType)(u32EtgFormats & ETG_FORMAT_MASK);
      ETG_DEBUG_PRINT("ADD: etgFormat=%u\n", etgFormat);
      etgFormats[numFormats] = etgFormat;
      if (etgFormat==ETG_EN_DONE) {
         ETG_DEBUG_PRINT("DONE: u32EtgFormats=%u\n", u32EtgFormats);
         break;
      }
      numFormats++;
      u32EtgFormats >>=ETG_NUM_FORMAT_BITS;
   }
   etgFormats[numFormats] = ETG_EN_DONE;

   //  handle vargs according to etgFormats
   etg_tU8 formatIndex = 0;
   while (bDone == FALSE)
   {
      etgFormat = etgFormats[formatIndex++];
      ETG_DEBUG_PRINT("WHILE2: etgFormat=%u\n", etgFormat);
      switch (etgFormat)
      {
      case ETG_EN_T8:
         {
            etg_tU8 u8Arg = (etg_tU8) ETG_VarArg(argList, etg_tInt/*etg_tU8*/); //lint !e64 !e78 !e516 !e530 !e662 !e746 !e826 !e1055 /*Review happened by Grosse-Pawig*/ 
            
            if (u32Length + sizeof(etg_tU8) < ETG_C_TRACE_MAX) {
               au8Buffer[u32Length++] = u8Arg;
            } else {
               bDone = TRUE;
            }
            break;
         }

      case ETG_EN_T16:
         {
            etg_tU16 u16Arg = (etg_tU16) ETG_VarArg(argList, etg_tInt/*etg_tU16*/); //lint !e64 !e78 !e516 !e530 !e662 !e746 !e826 !e1055 /*Review happened by Grosse-Pawig*/             

            if (u32Length + sizeof(etg_tU16) < ETG_C_TRACE_MAX) {
               u32Length = u32PushWord(au8Buffer, u32Length, u16Arg);
            } else {
               bDone = TRUE;
            }
            break;
         }

      case ETG_EN_T32:
         {
            etg_tU32 u32Arg = (etg_tU32)ETG_VarArg(argList, etg_tU32); //lint !e64 !e78 !e516 !e530 !e662 !e746 !e826 !e1055 /*Review happened by Grosse-Pawig*/ 

            if (u32Length + sizeof(etg_tU32) < ETG_C_TRACE_MAX) {
               u32Length = u32PushLong(au8Buffer, u32Length, u32Arg);
            } else {
               bDone = TRUE;
            }
            break;
         }

      case ETG_EN_T64:
         {
            etg_tU64 u64Arg = (etg_tU64)ETG_VarArg(argList, etg_tU64); //lint !e64 !e78 !e516 !e530 !e662 !e746 !e826 !e1055 /*Review happened by Grosse-Pawig*/ 

            if (u32Length + sizeof(etg_tU64) < ETG_C_TRACE_MAX) {
               u32Length = u32PushLongLong(au8Buffer, u32Length, u64Arg);
            } else {
               bDone = TRUE;
            }
            break;
         }

      case ETG_EN_LIST:
         {
            // check for next format to get size of list-elements
            etg_ListParams listParams=(etg_ListParams)ETG_VarArg(argList, etg_ListParams);
            ETG_DEBUG_PRINT("LIST: LIST_LEN=%u\n", listParams.listLen);
            ETG_DEBUG_PRINT("LIST: LIST_TYPE=%u\n", listParams.listType);
            ETG_DEBUG_PRINT("LIST: LIST_PTR=%p\n", listParams.listPtr);

            switch (listParams.listType) {
            case ETG_T8:
               {
                  ETG_DEBUG_PRINT("LIST:U8:len=%u\n", listParams.listLen);
                  const etg_tU8* pu8Arg = (const etg_tU8*)listParams.listPtr;
                  BREAK_AND_DONE_ON_NULL(pu8Arg);
                  for (etg_tUInt nIdx = 0;
                  (nIdx < listParams.listLen) && (bDone == FALSE);
                  ++nIdx)
                  {
                     if ((u32Length + sizeof(etg_tU8)) < ETG_C_TRACE_MAX) {
                        au8Buffer[u32Length++] = pu8Arg[nIdx];
                     } else {
                        bDone = TRUE;
                     }
                  }
                  break;

               }


            case ETG_T16:
               {
                  const etg_tU16* pu16Arg = (const etg_tU16*)listParams.listPtr;
                  BREAK_AND_DONE_ON_NULL(pu16Arg);
                  
                  for (etg_tUInt nIdx = 0;
                  (nIdx < listParams.listLen) && (bDone == FALSE);
                  ++nIdx)
                  {
                     if ((u32Length + sizeof(etg_tU16)) < ETG_C_TRACE_MAX) {
                        u32Length = u32PushWord(au8Buffer, u32Length, pu16Arg[nIdx]);
                     } else {
                        bDone = TRUE;
                     }
                  }
                  break;
               }


            case ETG_T32:
               {
                  const etg_tU32* pu32Arg = (const etg_tU32*)listParams.listPtr;
                  BREAK_AND_DONE_ON_NULL(pu32Arg);
                  
                  for (etg_tUInt nIdx = 0;
                  (nIdx < listParams.listLen) && (bDone == FALSE);
                  ++nIdx)
                  {
                     if ((u32Length + sizeof(etg_tU32)) < ETG_C_TRACE_MAX) {
                        u32Length = u32PushLong(au8Buffer, u32Length, pu32Arg[nIdx]);
                     } else {
                        bDone = TRUE;
                     }
                  }
                  break;
               }
            case ETG_T64:
               {
                  const etg_tU64* pu64Arg = (const etg_tU64*)listParams.listPtr;
                  BREAK_AND_DONE_ON_NULL(pu64Arg);
                  
                  for (etg_tUInt nIdx = 0;
                  (nIdx < listParams.listLen) && (bDone == FALSE);
                  ++nIdx)
                  {
                     if ((u32Length + sizeof(etg_tU64)) < ETG_C_TRACE_MAX) {
                        u32Length = u32PushLongLong(au8Buffer, u32Length, pu64Arg[nIdx]);
                     } else {
                        bDone = TRUE;
                     }
                  }
                  break;
               }


            case ETG_TSTRING:
               {
                  const etg_tCString coszArg = (etg_tCString)listParams.listPtr;
                  BREAK_AND_DONE_ON_NULL(coszArg);
                  etg_tU32 strLen=(etg_tU32)ETG_FN_STRLEN(coszArg);
                  etg_tU32 listLen=listParams.listLen;
                  etg_tU32 maxLen=ETG_C_TRACE_MAX - u32Length;
                  if (listParams.listLen>maxLen) {
                     // requested list is longer than buffer
                     listLen=maxLen;
                     bDone=true;
                  }
                  if (strLen>listLen) {
                     // actual string is longer than limit listParams.listLen
                     strLen=listLen;
                  }

                  memcpy(au8Buffer+u32Length, coszArg, strLen);
                  u32Length+=strLen;


                  if (strLen<listLen && u32Length<ETG_C_TRACE_MAX) {
                     etg_tU32 fillLen=listLen-strLen;
                     memset(au8Buffer+u32Length, 0, fillLen);
                     u32Length+=fillLen;
                  }
                  break;
               }

            default:
               bDone = TRUE;
               break;
            }
            break;
         }
         
      case ETG_EN_STRING:
         {
            etg_tCString coszArg = ETG_VarArg(argList, etg_tChar*);  //lint !e64 !e78 !e1055 !e516 !e530 !e662 !e746 !e826 !e1055 /*Review happened by Grosse-Pawig*/
            BREAK_AND_DONE_ON_NULL(coszArg);
            etg_tU32 u32SLen = (etg_tU32)ETG_FN_STRLEN(coszArg)+1;
            
            if ((u32Length + u32SLen) > ETG_C_TRACE_MAX) {
               u32SLen = ETG_C_TRACE_MAX - u32Length;
            }
            (etg_tVoid)ETG_FN_STRNCPY((etg_tString)&au8Buffer[u32Length], coszArg, u32SLen);
            au8Buffer[ETG_C_TRACE_MAX-1] = '\0';
            u32Length += u32SLen;
            break;
         }
         
      case ETG_EN_DONE:
      default:
         {
            ETG_DEBUG_PRINT("WHILE2: case ETG_EN_DONE\n");

            bDone = TRUE;
            break;
         }
      }
      if (bDone) {
         ETG_DEBUG_PRINT("WHILE2: bDone was set\n");
      }

   }
   ETG_VarArgEnd(argList);

   // send trace data to backend
   if(bErrMemTrace) {
      if (ETG_pfvDoErrMemTrace && (u32Length>0)) {
         // Byte 0 in buffer is 0xff, signalling string tracing which should be there in errmem output also.
#ifdef VARIANT_S_FTR_ENABLE_ETG_ERRMEM_LF
         // new requirement for adit errmemd since beginning of 2015: a \n behind the string terminator 'text\0\n'
    	 // data sent to /dev/errmem is a stream and it does not matter if \0 is there (it's handled by errmem)
         if (u32Length < ETG_C_TRACE_MAX) {
            ++u32Length;
         }
         au8Buffer[u32Length-1] = '\n';
#endif
         ETG_pfvDoErrMemTrace(u16CompId, (etg_tInt)etg_eErrmemEntryFatal, (etg_tU16)u32Length, &au8Buffer[0] );
      }
   }
   if(ETG_pfvDoTrace) { // make sure user has initialized trace output
      ETG_pfvDoTrace(u16CompId, u16LevelId, (const etg_tS8*)au8Buffer, u32Length);
   }
} //lint !e438 varg used... Last value assigned to variable 'argList' (defined at line 222) not used

// registration of a channel
etg_tS32 etg_i_s32RegisterTraceChannel(etg_tU16 enTraceChan, etg_i_tpfChnCallback pfChnCallback) {
   if(ETG_pfs32RegTraceCallback) {
      return ETG_pfs32RegTraceCallback(enTraceChan,(ETG_tpfCallback)pfChnCallback);
   } else {
      return ETG_RES_ERROR;
   }
}

// de-registration of a channel
etg_tS32 etg_i_s32UnRegisterTraceChannel(etg_tU16 enTraceChan, etg_i_tpfChnCallback pfChnCallback) {    
   if(ETG_pfs32UnregTraceCallback) {
      return ETG_pfs32UnregTraceCallback(enTraceChan,(ETG_tpfCallback)pfChnCallback);
   } else {
      return ETG_RES_ERROR;
   }
}

// set application name, required for DLT support
etg_tS32 etg_s32SetEtgDltAppId(const etg_tChar* dltAppId, const etg_tChar* description)
{
	if(ETG_pfs32SetEtgDltAppId) {
		return ETG_pfs32SetEtgDltAppId(dltAppId, description);
	} else {
		return ETG_RES_ERROR;
	}
}

// register individual commands, required for DLT command injection support
etg_tS32 etg_i_s32RegisterEtgICmds(etg_i_tpfFileCallback callback, etg_tU32 length, etg_tU32* cmdIds)
{
	if(ETG_pfs32RegisterEtgICmds) {
		return ETG_pfs32RegisterEtgICmds(callback, length, cmdIds);
	} else {
		return ETG_RES_ERROR;
	}
}

// de-register individual commands, required for DLT command injection support
etg_tS32 etg_i_s32DeregisterEtgICmds(etg_tU32 length, etg_tU32* cmdIds)
{
	if(ETG_pfs32DeregisterEtgICmds) {
		return ETG_pfs32DeregisterEtgICmds(length, cmdIds);
	} else {
		return ETG_RES_ERROR;
	}
}

// some byte-utilities
etg_tU32 etg_i_getU32(const etg_tU8* const pu8Ar) {
   etg_tU32 u32Res;
   u32Res=((etg_tU32)(((etg_tU32)((pu8Ar)[0])<<24) + ((etg_tU32)((pu8Ar)[1])<<16) + ((etg_tU32)((pu8Ar)[2])<<8) + (pu8Ar)[3]));
   return u32Res;
}

etg_tU16 etg_i_getU16(const etg_tU8* const pu8Ar) {
   etg_tU16 u16Res;
   u16Res=((etg_tU16)(((etg_tU16)((pu8Ar)[0])<<8) + (pu8Ar)[1]));
   return u16Res;
}

etg_tU8 etg_i_getU8(const etg_tU8* const pu8Ar) {
   return pu8Ar[0];
}

etg_tS32 etg_i_getS32(const etg_tU8* const pu8Ar) {
   return (etg_tS32)etg_i_getU32(pu8Ar);
}

etg_tS16 etg_i_getS16(const etg_tU8* const pu8Ar) {
   return (etg_tS16)etg_i_getU16(pu8Ar);
}

etg_tS8 etg_i_getS8(const etg_tU8* const pu8Ar) {
   return (etg_tS8)etg_i_getU8(pu8Ar);
}

// sets the base pointer to the function to be called when non of the ETG_I functions are on turn
etg_tVoid etg_i_vDefineDefProc(etg_i_tCmdInfos *poCmdInfos, ETG_tpfCallback pfDefProc) {
   if(poCmdInfos) {
      poCmdInfos->pfFileCallback = (etg_i_tpfFileCallback)pfDefProc; // in case no other cmdid found we make a call to this func
   }
}

// function to add commands of a file to cmd-list of the channel
etg_tVoid etg_i_vAddCmdInfos(etg_i_tCmdInfos *prBase, etg_i_tCmdInfos *prNew) {
   if (NULL == prBase || NULL==prNew) {
      return;
   }
   etg_i_tCmdInfos *prCur=prBase;
   while (prCur->prNext) {
      prCur=prCur->prNext;
      if (prCur==prNew) {
         return;
      }
   }
   prNew->prNext=NULL;
   prCur->prNext=prNew;
}

// function to remove commands of a file from cmd-list of the channel
etg_tVoid etg_i_vRemoveCmdInfos(etg_i_tCmdInfos *prBase, const etg_i_tCmdInfos *prRemove) {
   if (NULL==prRemove) {
      return;
   }
   etg_i_tCmdInfos *prCur=prBase;
   while (prCur) {
      if (prCur->prNext==prRemove) {
         prCur->prNext=prRemove->prNext;
         break;
      } else {
         prCur=prCur->prNext;
      }
   }
}

// byte-positions in ttfis-buffer for ETG_I
//#define ETG_I_ETG_LEN_OFFSET 0
#define ETG_I_MIN_CMD_LEN 5
#define ETG_I_ETG_MARKER_OFFSET 1
#define ETG_I_ETG_MARKER 0xFD
#define ETG_I_FILE_ID_OFFSET 2
#define ETG_I_CMD_ID_OFFSET 4
#define ETG_I_CMD_DATA_OFFSET 6

// fn to process a ttfis-command of a trace-channel
etg_tBool etg_i_bChnProcess(etg_i_tCmdInfos *prBase, const etg_tUChar* const pu8TtfisCmdBuf) {
   if (prBase==0||pu8TtfisCmdBuf==0) return FALSE;
   etg_i_tCmdInfos *prCur=prBase;
   etg_tU8 u8Len=pu8TtfisCmdBuf[0];
   if (u8Len < ETG_I_MIN_CMD_LEN) {
      // not a ETG_I_ command check for legacy function and call default process handler
      if(prBase->pfFileCallback) {
         ((ETG_tpfCallback)prBase->pfFileCallback)((etg_tPVoid)pu8TtfisCmdBuf); //lint !e1773 : reviewed hem2hi calling convention from OSAL define
      }
      return FALSE;
   }
   if (pu8TtfisCmdBuf[ETG_I_ETG_MARKER_OFFSET]!=ETG_I_ETG_MARKER) {
      if(prBase->pfFileCallback) {
         ((ETG_tpfCallback)prBase->pfFileCallback)((etg_tPVoid)pu8TtfisCmdBuf); //lint !e1773 : reviewed hem2hi calling convention from OSAL define
      }
      return FALSE;
   }
   etg_tU16 u16FileId=etg_i_getU16(&pu8TtfisCmdBuf[ETG_I_FILE_ID_OFFSET]);
   etg_tU16 u16CmdId=etg_i_getU16(&pu8TtfisCmdBuf[ETG_I_CMD_ID_OFFSET]);
   while (prCur->prNext!=NULL) {
      prCur=prCur->prNext;
      if (prCur->u16FileId==u16FileId) {
         if (prCur->pfFileCallback != NULL) {
            return prCur->pfFileCallback(u16CmdId, (etg_tU8)(u8Len - ETG_I_MIN_CMD_LEN), &pu8TtfisCmdBuf[ETG_I_CMD_DATA_OFFSET]);
         }
      }
   }
   if(prBase->pfFileCallback) {
      ((ETG_tpfCallback)prBase->pfFileCallback)((etg_tPVoid)pu8TtfisCmdBuf); //lint !e1773 : reviewed hem2hi calling convention from OSAL define
   }
   return FALSE;
}

#ifdef __cplusplus
}
#endif
