/*
 * vd_clock_datetime.h
 *
 *  Created on: Apr 23, 2015
 *      Author: vee4kor
 */

#ifndef VD_CLOCK_DATETIME_H_
#define VD_CLOCK_DATETIME_H_

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"
#include "vd_clock_types.h"


// ****************************************************************************
// ***						G l o b a l   D e c l a r a t i o n s					 ***
// ****************************************************************************

// define bit field for date and time status
//    used as
//    - indicators for changed components			(get)
//    - indicators for valid date and/or time	(get)
//    as well as
//    - field mapping for change orders	(set/update)




// ****************************************************************************
// ***			C l a s s   v d _ c l o c k _ t c l D a t e T i m e			 ***
// ****************************************************************************
/*lint -e641 "Converting enum..." */

class vdclk_tclDateTime
{
	// Date and Time storage and conversion class
	// - storage divided in
	//    . century base (1900, 2000, 2100 etc) and
	//    . seconds elapsed since then
	// - includes conversion functions
	//
	// Note: Dates before 1582 will be treated as extrapolated gregorian calendar
	// - No transformation to Julian calendar or so.

	// =========================================================================
	// ===								I n t e r f a c e									 ===

	public:
	// --------------------------------------
	// ---   constructor(s), destructor   ---

		vdclk_tclDateTime	(tVoid);

		vdclk_tclDateTime (const vdclk_tclDateTime& roDT);

		~vdclk_tclDateTime(tVoid);

	// --------------------------------
	// ---   high level functions   ---

		// Function to extract components (year, month, day, hour, minute, second, weekday)
		// from internal state.
      // returns VDCLK_DateValid | VDCLK_TimeValid, if decomposition successful.
		tU8 u8GetComponents   ( tS16*	ps16Year,		// location for result
				                  tU8*	pu8Month,		//          "
				                  tU8*	pu8Day, 			//          "
				                  tU8*	pu8Hour,			//          "
				                  tU8*	pu8Minute,		//          "
				                  tU8*	pu8Second,		//          "
                              tU8*  pu8Weekday=NULL
			                   );


		// Member function to update internal date and time value
		// from given components.
		// A bit field mask is used to indicate which components to update.
      // returns: VDCLK_EN_DateValid | VDCLK_EN_TimeValid if decomposition is successful.
      tU8 u8Update	      (  tS16 s16Year,			// input, any valid tS16
				                  tU8 u8Month,			// input, 1 .. 12
				                  tU8 u8Day,				// input, 1 .. number of days
				                  tU8 u8Hour,				// input, 1 .. 24
				                  tU8 u8Minute,			// input, 0 .. 59
				                  tU8 u8Second,			// input, 0 .. 59
				                  tU8 enChangeMask		// input, any combination of VDCLK_EN_..Change values
											                  //			 from VDCLK_TEN_TimeDateMask
                           );


	// ----------------------------------
	// ---   medium level functions   ---

		// Member access function for internal value.
   	// returns: seconds elapsed since century base.
		tU32  u32GetValue ( tVoid ) const;


      // Member update function.
		// Implemented algorithm allows for smooth century transition.
		//
		// Note: Input value only spans little more than one century,
		// so century base remains uneffected,
		// but will be adjusted in case of over-/underflow.
      // Returns: combination of <Change> and <Valid> values
		//				from VDCLK_TEN_TimeDateMask

      // input, 0 .. century length (in seconds)
		// overflow condition: within 10 years above century length
		// underflow condition: within 10 years below 0xFFFFFFFF
		tU8	 u8SetValue ( tU32 u32Value );



      // Member access function for internal value.
      // returns: century base.
		tS16  s16GetCenturyBase ( tVoid ) const;


      // Member function to inform, whether date and time components
		// have been initalized by using u8Update().
      // Returns:
		// - VDCLK_TimeValid if hour, minute and second have been initialized
		// - VDCLK_DateValid if year, month and day have been initialized
		// - VDCLK_AllValid  if both.
		tU8	u8IsDateTimeValid ( tVoid ) const;


      // alternative to using the == operator
      tU8   u8GetChanges ( vdclk_tclDateTime* poPrevDT );



 // Member function to convert the u32value into yy,mm,dd,hh,mm.ss format
      tVoid vGetDateFormat(tBool bIsPositive, tU32 u32Value, tS16 s16UTCYear, tU8 u8UTCMonth, tS16* ps16Year, tU8* pu8Months, tU8* pu8Day, tU8* pu8Hour, tU8* pu8Minute, tU8* pu8Second);

      //Method to find the day of the week
      tU8 u8GetDayoftheWeek(tU16 u16Year, tU8 u8Month, tU8 u8Day);

      // Method to find the date for Nth day of the Nth Week of the Month
      // used to find the start of summer time and Winter time
      tU8 u8GetNthDate_DST(tU16 u16Year, tU8 u8Month, tU8 DayofWeek, tU8 u8NthWeek);

      //static tU32 u32NewValue
      //   // utility for calculating the term
      //   //   u32NewValue = u32Value + u32Summand - u32Subtrahend
      //   // if all participients may use the full tU32 range
      //   (
      //      const tU32 u32Value,
      //      const tU32 u32Summand,
      //      const tU32 u32Subtrahend
      //   );

	// ---------------------
	// ---   Operators   ---

		vdclk_tclDateTime&	operator	=
			//
			// Assignment operator =.
			//
			(
				const vdclk_tclDateTime& roDT
			);

      tU8	operator	==
			//
			// Comparison operator ==.
			//
			// Note: opposite logic: 0 means both operands are identical
			(
			   const vdclk_tclDateTime  oDT
			);
			// Returns: any combination of <VDCLK_TEN_TimeDateMask> values

		vdclk_tclDateTime&	operator	+=
			//
			// AddInPlace operator +=.
			//
			// used to add dynamic time offset (DTO) or duration values
			(
				tS64	s64Offset
			);
			// Returns: ---


	// -----------------------------
   // ---	Public Utilities	  ---

      // Static class member to calculate the number of days
		// within a given month of a given year.
		// Return value is completely independent of internal state.
		static tU8 u8DaysPerMonth ( tU8  u8Month,		// input, 1 .. 12; returns 0 days otherwise
                                  tS16 s16Year		// input, any tS16 Value
                        		   );

      // calculates the current weekday from a given date and returns 0=Monday, 7=Sunday
      static tU8 u8Weekday ( tS16 s16Year,		// input, any valid tS16
				                 tU8 u8Month,			// input, 1 .. 12
				                 tU8 u8Day				// input, 1 .. number of days
                           );

      // calculates the current day of the year
      // Returns: number of current day of the given year
      static tU16 u16_DayOfYear ( tS16 s16Year,    // input, any valid tS16
				                    tU8 u8Month,		// input, 1 .. 12
				                    tU8 u8Day			// input, 1 .. number of days
                              );


      // Static class member to distinguish leap years (having 29 days in february)
		// from non-leap years.
		// Leap years are years dividable by 4 with exception of 2100, 2200, 2300
		// and 2500, but regularly in 2000, 2400, 2800.
		// Return value is completely independent of internal state.
      // Returns: TRUE if the given year is a leap year
		static tBool bIsLeapYear ( tS16 s16Year );



      // Return value is completely independent of internal state.
      // Returns: TRUE if given parameters represent a valid date and time
		static tBool bIsDateTimeValid	( tS16 s16Year,			// input, any valid tS16
				                          tU8 u8Month,			// input, 1 .. 12
				                          tU8 u8Day,				// input, 1 .. number of days
				                          tU8 u8Hour,				// input, 1 .. 24
				                          tU8 u8Minute,			// input, 0 .. 59
				                          tU8 u8Second,			// input, 0 .. 59
                                      tU8 u8Weekday,        // input, 1 .. 7
				                          tU8 enChangeMask		// input, any combination of VDCLK_EN_..Change values
											                           //			 from VDCLK_TEN_TimeDateMask
			                            );


	// =========================================================================
	// ===						P r i v a t e   S e c t i o n							 ===

	private:
		tU32	_u32Value;			//  seconds elapsed from Jan 1st, CenturyBase, 0:0h
		tS16  _s16CenturyBase;	//  base of century (e.g. 1900, 2000, 2100)
		tU8	_u8FieldsValid;	//	 cumulatve indicator of initialized fields
										//  (year, month, day, hour, minut, second)

		tVoid	vMarkField
			// Function to set / reset the individual fields of _u8FieldsValid
			(
			   VDCLK_TEN_TimeDateStatus	enField,	// input, any valid VDCLK_EN_...Change
				tBool								bIsValid	// condition
			);
};

// ****************************************************************************


#endif /* VD_CLOCK_DATETIME_H_ */
