/************************************************************************
 * FILE:           vd_clock_Event.cpp
 * PROJECT:        VW-LowLine-NF
 * SW-COMPONENT:   vd_clock
 *----------------------------------------------------------------------
 *
 * DESCRIPTION:    Wrapper class for OSAL event handling
 *                 (Just to reduce writing of code)
 *
 *      Note:      Make sure to have included
 *                 -  system_types.h
 *                 -  vd_clock_Trace.h
 *
 *----------------------------------------------------------------------
 * COPYRIGHT:      (c) 2007 Blaupunkt Werke GmbH, Hildesheim
 *
 * HISTORY:
 * Date      | Author                | Modification
 *----------------------------------------------------------------------
 * 02.08.07  | Martin Koch (ESE)     | initial version
 ************************************************************************/

// System includes
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"       

// vd_clock includes
#include "vd_clock_types.h"
#include "../framework/app/vd_clock_AppMain_Trace.h"
#include "../framework/app/vd_clock_AppMain.h"
#include "vd_clock_Event.h"
//#include "vd_clock_updater.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_VD_CLOCK_EVENT
#include "trcGenProj/Header/vd_clock_Event.cpp.trc.h"
#endif

// ****************************************************************************
// ***         C l a s s   I n t e r f a c e   F u n c t i o n s            ***
// ****************************************************************************

// ---   Constructor   ---
vdclk_tclEvent::                          vdclk_tclEvent
(
		tCString const szEventName
)

{
	_szEventName = szEventName;
	_hEvent = NULL;			//Coverity Fix

	tS32  s32Err = OSAL_s32EventCreate
			(
					szEventName,
					&_hEvent
			);
	if (OSAL_OK != s32Err) {
		ETG_TRACE_ERR(("vdclk_tclEvent:EventCreate failed, code=%s",VDCLK_OsalErrorString()));
	}
	VDCLOCK_ASSERT_NORMAL_RETURN(_hEvent!=OSAL_C_INVALID_HANDLE);
};

// ****************************************************************************

vdclk_tclEvent::                          ~vdclk_tclEvent
(
)
{
	tS32 s32Err = OSAL_s32EventClose (_hEvent);
	if (OSAL_OK != s32Err) {
		ETG_TRACE_ERR(("~vdclk_tclEvent:EventClose failed, code=%s",VDCLK_OsalErrorString()));
	}
	s32Err = OSAL_s32EventDelete (_szEventName);
	if (OSAL_OK != s32Err) {
		ETG_TRACE_ERR(("~vdclk_tclEvent:EventDelete failed, code=%s",VDCLK_OsalErrorString()));
	}
	_szEventName = NULL;
	_hEvent = OSAL_C_INVALID_HANDLE;

	ETG_TRACE_COMP(("Flow-Event: %d", ETG_ENUM(VD_CLOCK_FlowEvents,TR_EN_FlowEventDTOR)));

};

// ****************************************************************************

tVoid   vdclk_tclEvent::                  vPost
(
		const OSAL_tEventMask   u32EventMask,
		const tU16              u16Class,         // Tracing information _CLASS
		const tU32              u32Line           // Tracing information _LINE
)

{
	(tVoid)u16Class;
	(tVoid)u32Line;
#if 0
	ETG_TRACE_COMP(("Event(%d): Mask=08%x caller:[%d(%d)]",
			ETG_ENUM(VD_CLOCK_FlowEvents,TR_EN_EventPosted),
			u32EventMask,
			ETG_ENUM(VD_CLOCK_FILES, u16Class),
			u32Line));
#endif
OSAL_s32EventPost
(
		_hEvent,
		u32EventMask ,
		OSAL_EN_EVENTMASK_OR
);
};

// ****************************************************************************

tVoid   vdclk_tclEvent::                  vClear
(
		const OSAL_tEventMask   u32EventMask,
		const tU16              u16Class,         // Tracing information _CLASS
		const tU32              u32Line           // Tracing information _LINE
)

{
	(tVoid)u16Class;
	(tVoid)u32Line;

	tS32  s32Err = OSAL_s32EventPost
			(
					_hEvent,
					~u32EventMask ,
					OSAL_EN_EVENTMASK_AND
			);
	if (OSAL_OK ==s32Err) {
		ETG_TRACE_COMP(("Event(%d): Mask=08%x caller:[%d(%d)]",
				ETG_ENUM(VD_CLOCK_FlowEvents,TR_EN_EventReset),
				u32EventMask,
				ETG_ENUM(VD_CLOCK_FILES, u16Class),
				u32Line));
	}
	else
	{
		ETG_TRACE_ERR(("Event(%d) Failed: Mask=08%x caller:[%d(%d)]",
				ETG_ENUM(VD_CLOCK_FlowEvents,TR_EN_EventReset),
				u32EventMask,
				ETG_ENUM(VD_CLOCK_FILES, u16Class),
				u32Line));
	}
};

// ****************************************************************************

tVoid   vdclk_tclEvent::                  vStatus
(
		const OSAL_tEventMask   u32EventMask,
		OSAL_tEventMask*  pu32EventStatus,  // location for actual event status
		const tU16              u16Class,         // Tracing information _CLASS
		const tU32              u32Line           // Tracing information _LINE
)

{
	(tVoid)u16Class;
	(tVoid)u32Line;

	tS32  s32Err = OSAL_s32EventStatus
			(
					_hEvent,
					u32EventMask,
					pu32EventStatus
			);
	if (OSAL_OK ==s32Err) {
		ETG_TRACE_COMP(("Event(%d): Mask=08%x caller:[%d(%d)]",
				ETG_ENUM(VD_CLOCK_FlowEvents,TR_EN_EventStatus),
				u32EventMask,
				ETG_ENUM(VD_CLOCK_FILES, u16Class),
				u32Line));
	}
	else
	{
		ETG_TRACE_ERR(("Event(%d) Failed: Mask=08%x caller:[%d(%d)]",
				ETG_ENUM(VD_CLOCK_FlowEvents,TR_EN_EventStatus),
				u32EventMask,
				ETG_ENUM(VD_CLOCK_FILES, u16Class),
				u32Line));
	}
};

// ****************************************************************************

tVoid   vdclk_tclEvent::                  vWait
(
		const OSAL_tEventMask   u32EventMask,
		OSAL_tMSecond     u32TimeOut,
		OSAL_tEventMask*  pu32EventStatus,  // location for actual event status
		const tU16  u16Class,         // Tracing information _CLASS
		const tU32  u32Line           // Tracing information _LINE
)

{
	(tVoid)u16Class;
	(tVoid)u32Line;

	ETG_TRACE_COMP(("Event(%d): Mask=08%x", ETG_ENUM(VDCLK_EventAction,TR_EN_EventMask), u32EventMask));
	tS32  s32Err = OSAL_s32EventWait
			(
					_hEvent,
					u32EventMask,
					OSAL_EN_EVENTMASK_OR,
					u32TimeOut,
					OSAL_NULL   // pu32EventStatus
			);
	if (OSAL_OK != s32Err) {
		ETG_TRACE_ERR(("OSAL_s32EventWait failed:code=%s",VDCLK_OsalErrorString()));
	}

	s32Err = OSAL_s32EventStatus
			(
					_hEvent,
					u32EventMask,
					pu32EventStatus
			);
	if (OSAL_OK ==s32Err) {
		ETG_TRACE_COMP(("Event(%d): Mask=08%x caller:[%d(%d)]",
				ETG_ENUM(VDCLK_EventAction,TR_EN_EventReceived),
				u32EventMask,
				ETG_ENUM(VD_CLOCK_FILES, u16Class),
				u32Line));
	}
	else
	{
		ETG_TRACE_ERR(("Event(%d) Failed: Mask=08%x caller:[%d(%d)]",
				ETG_ENUM(VDCLK_EventAction,TR_EN_EventReceived),
				u32EventMask,
				ETG_ENUM(VD_CLOCK_FILES, u16Class),
				u32Line));
	}
};

// ****************************************************************************

tBool   vdclk_tclEvent::                  bAwaitClearance
(
		const OSAL_tEventMask   u32EventMask,
		OSAL_tMSecond     u32TimeOut,
		const tU16              u16Class,         // Tracing information _CLASS
		const tU32              u32Line           // Tracing information _LINE
)

// waits for all bits defined by mask, set to zero

// as there is no corrsepondant OSAL function, we emulate with a
// periodic query of event status.

{
	(tVoid)u16Class;
	(tVoid)u32Line;

	OSAL_tEventMask      u32EventStatus = 0L;
	const OSAL_tMSecond  u32Delay       = 10L;   // set to 10 msec

	ETG_TRACE_COMP(("Event(%d): Mask=08%x caller:[%d(%d)]",
			ETG_ENUM(VD_CLOCK_FlowEvents,TR_EN_EventMask),
			u32EventMask,
			ETG_ENUM(VD_CLOCK_FILES, u16Class),
			u32Line));

	// query status first time
	tS32 s32Err = OSAL_s32EventStatus
			(
					_hEvent,
					u32EventMask,
					&u32EventStatus
			);
	if (OSAL_OK != s32Err) {
		ETG_TRACE_ERR(("OSAL_s32EventStatus failed:code=%s",VDCLK_OsalErrorString()));
	}

	// loop until event bits are cleared
	while  (u32EventStatus != 0L)
	{
		// just delay a little
		s32Err = OSAL_s32ThreadWait(u32Delay);
		if (OSAL_OK != s32Err) {
			ETG_TRACE_ERR(("OSAL_s32ThreadWait failed:code=%s",VDCLK_OsalErrorString()));
		}

		// if time runs out, fire OSAL_error and break loop
		if (u32TimeOut < u32Delay)
		{
			ETG_TRACE_ERR(("Emulate Timeout:code=%s",OSAL_coszErrorText(OSAL_E_TIMEOUT)));
			return FALSE;
		}


		// decrement time control variable
		u32TimeOut -= u32Delay;

		// query again
		s32Err = OSAL_s32EventStatus
				(
						_hEvent,
						u32EventMask,
						&u32EventStatus
				);
		if (OSAL_OK != s32Err) {
			ETG_TRACE_ERR(("OSAL_s32EventStatus failed:code=%s",VDCLK_OsalErrorString()));
		}
	};

	ETG_TRACE_COMP(("Event(%d): Mask=08%x", ETG_ENUM(VD_CLOCK_FlowEvents,TR_EN_EventStatus), u32EventStatus));
	return TRUE;
};

// ****************************************************************************
// ***            C l a s s   P r i v a t e   S e c t i o n                 ***
// ****************************************************************************

// standard constructor (suppressed); forces to give event name
vdclk_tclEvent::                  vdclk_tclEvent
(
)
{
	_hEvent = OSAL_C_INVALID_HANDLE;
	_szEventName = NULL;
};
