/************************************************************************
* FILE:         midw_common_trace_macros.h
* PROJECT:      several Paramount projects
* SW-COMPONENT: DemoServer
*----------------------------------------------------------------------
*
* DESCRIPTION: Handle trace outputs efficiently
* This class is used as template for all middleware server to provide
* trace output in an efficient way.
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author                   | Modification
*************************************************************************/

#ifndef MIDW_COMMON_TRACE_MACROS_H
#define MIDW_COMMON_TRACE_MACROS_H

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"
/** --- MS VC2005 Info
IntelliSense does not work in C++ projects under the following circumstances:
- The cursor is inside a code comment.
- You are writing a string literal.
- A syntax error appears above the cursor.
- IntelliSense is not _fully_ supported under the following circumstances:
  When you reference a header file multiple times using the #include directive,
  and the meaning of that header file changes due to various macro states (which
  are defined through the #define directive). In other words, when you include a
  header file several times whose usage changes under different macro states,
  IntelliSense will not always work.
*/

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ETRACE_S_IMPORT_INTERFACE_ERRMEM
#include "etrace_if.h"

/** --------------------- trace macros ---------------------
Trace outputs are written as one-line-macros for two reasons:
   the __LINE__ macro evaluates the correct linenumber automatically before
   the given output is parsed the evaluate function is called,
   which saves much of processing time, in case that trace for that
   class and level is switched of.
   ------------ Quick Intro to ETG (Extended Trace Generator)------------
This chapter shall give the reader a quick but complete intro in the functions of ETG.
1.	ETG is macro based because it encapsulates the trace function execution into an
   if-clause to avoid parameter calculation if the trace is not active for the given
   call. Please do not wrap the macros, so that the performance optimization is
   achieved.
2.	ETG macros have two parameters (either explicit or implicit)
   a.	The trace level is coded into the name of the macro:
      FATAL for unrecoverable system failures.
      ERR (=ERROR) for situations where the component cannot stay in the defined
         interface
      SYS_MIN (= SYSTEM_MIN) for situations where the function goes into defined
	      error handling covered by the interface.
      SYS (=SYSTEM) for situations where the function detects a recoverable error
         situation e.g. with extending the waiting time.
      COMP (=COMPONENT) for important function states being reached
      USR1-4 (USER_1-4) Debug information under developers decision (4 is most output).
   b.	The trace class is either set by a default parameter in the begin of the source
      file, or provided as first explicit parameter in a call to trace. The class is
      a 2-byte filter information enabling selection reduction to specific logging
      information. The trace class is defined for representing the given source code.
      1. byte (trace component) is company wide defined. 2. byte is up to the
      developers component structuring. The extension ..._CLS enables tracing with
      explicit class parameter.
3.	ETGs output is format string controlled like printf()!
   The supported specifiers are:
	%u for unsigned decimal,	%d and %i for signed decimal, 	%f for 32bit float
	%x for hexadecimal	%s for strings,	%c for 8bit ASCII	%p for pointer format
   All format specifiers may get a length indicator %nd where n is the count of chars
   of the argument.
   The format string can contain constant text. This text will be transported to the
   TRC-file by the generator. There is no need to restrict in words neither need for
   cryptic abbreviations.
4.	The ETG macro must be in double brackets to support the function of variable count
   of parameters inside of a macro ETG_TRACE_ERR(( a trace text ));
5.	All files (c and cpp) must be instrumented with including the corresponding ETG
   header at begin.
   This is also a very good place to define the default trace class for the file. The
   generated header will be placed into the build folder of the corresponding product
   and not in the source code directory.
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_MYCLASS
      #include "trcGenProj/Header/filename.cpp.trc.h"
      #endif
6.	ETG supports automatic translation of enums. Arguments like %d can be stuffed with
   a declaration of an enum definition to be used to translate automatically the
   transferred value. Developers can use an enum definition already available in the
   trc files (e.g. AIL power states or CCA service- or application ids):
   ETG_ENUM(ttfisEnumName,u8Val) defines that ttfisEnumName shall be used for
   translation in the trc file. Also the developer can use ETG_CENUM(cEnumName,u8Val)
   where cEnumName shall is being searched in all header files named in the components
   xml-file and a ARRAY statement will automatically created in addition to the trace
   output in the generated trace file.
7.	ETG also supports generic inheritance of trace output intended to be used for
   frameworks. The extension ..._DCL enables a second, dynamic (varaible) trace class
   parameter in the macros.
8.	ETG also supports tracing FATAL traces into the error memory: see
   ETG_TRACE_ERRMEM(("err"))

If you want further and detailed information to the previously mentioned items, you may read the following sections which provide a complete description and detailed usage information.
For FAQ please look at WIKI page http://hi0vm019.de.bosch.com/wiki/index.php?title=ETG
For the complete ETG manual read https://hi-dms.de.bosch.com/docushare/dsweb/Services/Document-151948
*/

/**
   This class encapsulates all trace comunication.
   TRACE defines the following trace levels.
   TR_LEVEL_FATAL       = 0,  [CUSTOMER]        used for FATAL output (always shown)
   TR_LEVEL_ERRORS      = 1,  [EEZ]             used for ERRORS including ASSERT
   TR_LEVEL_SYSTEM_MIN  = 2,  [FATAL_ERROR]
   TR_LEVEL_SYSTEM      = 3,  [ERROR]           used for WARNINGS allowing cont. of prg.
   TR_LEVEL_COMPONENT   = 4,  [WARNING]
   TR_LEVEL_USER_1      = 5,  [STATE_TRANSITION]used for normal trace infos for debugging
   TR_LEVEL_USER_2      = 6,  [EVENT]
   TR_LEVEL_USER_3      = 7,  [MESSAGE]         used for trace infos comming often
   TR_LEVEL_USER_4      = 8   [DATA]            used for function tracing if enabled

   General Message Structure for this trace messages
   Byte 0 = Length of message
   Byte 1 = TraceComponentNumber / also ClassNumber hi-byte
   Byte 2 = ClassNumber
   Byte 3 = TRACE LEVEL as define above
   Byte 4 = Statistic (trace specific value)
   Byte 5 = Reserved (trace specific value)
   Byte 6 = 0xfd == ETG KLARTEXT coding; 0xff == ETG string tracing; else manual tracing
   Byte 7 = FileNumber hi-Byte; hash coded ARRAY generated by ETG
   Byte 8 = FileNumber lo-Byte
   Byte 9 = Message hi-Byte; used to identify the final trace to be noted down.
   Byte 10= Message lo-Byte
   Byte 11= LineNumber hi-Byte; of the trace macro in the code
   Byte 12= LineNumber lo-Byte
   Byte 13... Subsequent message structure is defined buy MessageType

The beginning of the traces is the component number which is defined by
X:\di_trace\components\trace\include\tri_types.h
 ------------------------------------------------------------------------------- */

/**
ccademo_tenTraceClasses defines all trace classes available in this component
The names must be of TR_CLASS_<compName>_<className> (identical to the one defined in
the trc-entry-file of your compnent It will be used as default trace class each
in one file. And also when stating it explicitly in ..._CLS(()) macros.
*/

/**
ccad_tenTraceClasses defines all classes available in this component
*/
enum midw_common_tenTrcClassName               // TTFis use
{
   midw_common_Utf8StringBase       = 0x00,   // TR_CLASS_MIDW_COMMON_UTF8STRINGBASE
   midw_common_ipod_Utf8StringBase  = 0x01,   // TR_CLASS_MIDW_COMMON_IPOD_UTF8STRINGBASE
   midw_common_Utf8StringGroups     = 0x02,   // TR_CLASS_MIDW_COMMON_UTF8STRINGGROUPS
   midw_common_CodeConv             = 0x03,   // TR_CLASS_MIDW_COMMON_CODECONV
   midw_common_Trace                = 0x04,   // TR_CLASS_MIDW_COMMON_TRACE
   midw_common_AutoConfiguration    = 0x05,   // TR_CLASS_MIDW_COMMON_AUTO_CONFIGURATION
   midw_common_maxClassNames                   // count used for max trace channel inputs
};

/**
ccademo_tenTraceClasses defines all trace classes available in this component
The names must be of TR_CLASS_<compName>_<className> (identical to the one defined in
the trc-entry-file of your compnent It will be used as default trace class each
in one file. And also when stating it explicitly in ..._CLS(()) macros.
*/
enum midw_common_tenTraceClasses            // TTFis
{
   TR_CLASS_MIDW_COMMON_UTF8STRINGBASE       = TR_COMP_MIDDLEWARE + midw_common_Utf8StringBase,
   TR_CLASS_MIDW_COMMON_IPOD_UTF8STRINGBASE  = TR_COMP_MIDDLEWARE + midw_common_ipod_Utf8StringBase,
   TR_CLASS_MIDW_COMMON_UTF8STRINGGROUPS     = TR_COMP_MIDDLEWARE + midw_common_Utf8StringGroups,
   TR_CLASS_MIDW_COMMON_CODECONV             = TR_COMP_MIDDLEWARE + midw_common_CodeConv,
   TR_CLASS_MIDW_COMMON_TRACE                = TR_COMP_MIDDLEWARE + midw_common_Trace,
   TR_CLASS_MIDW_COMMON_AUTO_CONFIGURATION   = TR_COMP_MIDDLEWARE + midw_common_AutoConfiguration
};

// trace class which uses creation and destroying of it as trigger to trace
#define CCAD_SETFUNCTIONTRACING 123
// the flow control
class midw_common_tclTraceFlow
{
public:
   // enable function tracing flag allows generally switching function tracing off
   // function tracing does not consume additional processing time
   // when one has enabled trace LEVEL 8 and has set this variable TRUE
   static tBool mbEnableFuncTracing; // see ccademo_main.cpp for the implementation
};

// macro defines a conditional around the given command being executed only if
// the global flag for function tracing is enabled
#define TRACE_FLOW(call) if(midw_common_tclTraceFlow::mbEnableFuncTracing) { call };

#endif // MIDW_COMMON_TRACE_MACROS_H
/*EOF*/
