//----------------------------------------------------------------------
// FILE:         MCANInterface.h
// PROJECT:      NISSAN LCN2 (kai)
// SW-COMPONENT: middleware common
//----------------------------------------------------------------------
//
// DESCRIPTION:  header of general interface for MCAN communication
//
//----------------------------------------------------------------------
// COPYRIGHT:    (c) 2012 Robert Bosch GmbH, Hildesheim
// HISTORY:
// Date      | Author             | Modification
// 31.05.12  | TMS Kempen         | initial version
// 16.11.13  | Vinod Bhargav B S  | Fix for SMS issue seen for EUR variant
//           |                    |  Corrected the Audio config wrt SMS
// 11.06.14  | Ravi Deora         | Generic Handling of Status-Wait-Timer 
//           |                    | for all MCAN units
//----------------------------------------------------------------------

#ifndef _MCAN_IF_H_
#define _MCAN_IF_H_

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#include "TxSinkItf.h"

#define SYSTEM_S_IMPORT_INTERFACE_SET
#include "stl_pif.h"

#include "MCANItf.h"
#include "MCANSinkRxHandlerItf.h"
#include "AutoConfiguration/ConnectionConfigurator.h"
#include "AutoConfiguration/ConnectionConfiguratorCallbackInterface.h"
#include "MessageHandler/ConfigurationGetMessageHandler.h"
#include "MessageHandler/ConfigurationStatusMessageHandler.h"
#include "SendTriggerContainer.h"

#define MCAN_WORKER_THREAD_BASE_NAME       "MCAN_WK_NAME"
#define MCAN_WORKER_EVENT_BASE_NAME        "MCAN_EV_NAME"
#define MCAN_WORKER_ACCESS_SEM_BASE_NAME   "MCAN_AS_NAME"


#define MCAN_EV_COMMUNICATION_CON            0x00010000
#define MCAN_EV_COMMUNICATION_IND            0x00020000
#define MCAN_EV_CHANGED                      0x00040000
#define MCAN_EV_COMMUNICATION_TEST_TIMER     0x00080000
#define MCAN_EV_CONFIGURATION_GET_TIMER      0x00100000
#define MCAN_EV_HARDWARE_VERSION_GET_TIMER   0x00200000
#define MCAN_EV_SOFTWARE_VERSION_GET_TIMER   0x00400000
#define MCAN_EV_STATUS_WAITING_TIMER         0x00800000

#define MCAN_EV_BASE_MASK (   MCAN_EV_COMMUNICATION_CON           \
                            | MCAN_EV_COMMUNICATION_IND           \
                            | MCAN_EV_CHANGED                     \
                            | MCAN_EV_COMMUNICATION_TEST_TIMER    \
                            | MCAN_EV_CONFIGURATION_GET_TIMER     \
                            | MCAN_EV_HARDWARE_VERSION_GET_TIMER  \
                            | MCAN_EV_SOFTWARE_VERSION_GET_TIMER  \
                            | MCAN_EV_STATUS_WAITING_TIMER)


// the message-ID is coded in the highest 11 bits, the op-type in the lowest 5 bits

#define MCAN_ASSEMBLE_COMMAND( MsgId, OpType ) ( (tU16)( ( ( MsgId & 0x07ff ) << 5 ) | OpType ) )

#define MCAN_EXTRACT_MSG_ID( Command ) ( (tU16)( ( Command & 0xffe0 ) >> 5 ) & 0x07ff )
#define MCAN_EXTRACT_OP_TYPE( Command ) ( (tU8)( Command & 0x001f ) )


#define MCAN_C_BASE_MSG_CONFIGURATION       0x001
#define MCAN_C_BASE_MSG_HARDWARE_VERSION    0x010
#define MCAN_C_BASE_MSG_SOFTWARE_VERSION    0x011

#define MCAN_C_OP_TYPE_SET           0x00
#define MCAN_C_OP_TYPE_GET           0x01
#define MCAN_C_OP_TYPE_INCREMENT     0x02
#define MCAN_C_OP_TYPE_DECREMENT     0x03
#define MCAN_C_OP_TYPE_STATUS        0x0c
#define MCAN_C_OP_TYPE_INDICATION    0x0d

// We have to write an error 2 seconds after the error masking has changed to non-masked
#define MCAN_C_COMMUNICATION_TEST_TIMER_VALUE_FOR_CONNECTION_WAIT_AFTER_ERROR_IN_MS   2000

#define MCAN_C_CONFIGURATION_GET_TIMER_VALUE_IN_MS      1500

#define MCAN_C_HARDWARE_VERSION_GET_TIMER_VALUE_IN_MS   1500
#define MCAN_C_SOFTWARE_VERSION_GET_TIMER_VALUE_IN_MS   1500


#define MCAN_C_INIT_STATUS_CONNECTED                0x0001
#define MCAN_C_INIT_STATUS_CONFIGURATION_COMPLETE   0x0002


#define MCAN_C_HARDWARE_VERSION_GET_MAX_RETRIES   3
#define MCAN_C_SOFTWARE_VERSION_GET_MAX_RETRIES   3


#define MCAN_C_ERROR_STATE_OK              0x00
#define MCAN_C_ERROR_STATE_CURRENTLY_NOK   0x80
#define MCAN_C_ERROR_STATE_CONFIRMED_NOK   0xc0
#define MCAN_C_ERROR_STATE_MASKED          0xff



typedef enum 
{
   MCAN_EN_COMM_STATUS_OFF,
   MCAN_EN_COMM_STATUS_NOT_CONNECTED,
   MCAN_EN_COMM_STATUS_CONNECTION_WAIT,
   MCAN_EN_COMM_STATUS_CONNECTED,
   MCAN_EN_COMM_STATUS_CONNECTION_ERROR,
   MCAN_EN_COMM_STATUS_CONNECTION_WAIT_AFTER_ERROR
} MCAN_tenCommunicationStatus;


typedef enum 
{
   MCAN_EN_TEST_RESULT_UNKNOWN,
   MCAN_EN_TEST_RESULT_PASSED,
   MCAN_EN_TEST_RESULT_FAILED
} MCAN_tenTestResult;


typedef enum 
{
   MCAN_EN_ERROR_CODE_DELETE_COMMUNICATION_TEST_TIMER,
   MCAN_EN_ERROR_CODE_DELETE_CONFIGURATION_GET_TIMER,
   MCAN_EN_ERROR_CODE_DELETE_HARDWARE_VERSION_GET_TIMER,
   MCAN_EN_ERROR_CODE_DELETE_SOFTWARE_VERSION_GET_TIMER,
   MCAN_EN_ERROR_CODE_ACCESS_WORKER_EVENTS,
   MCAN_EN_ERROR_CODE_CREATE_WORKER_EVENTS,
   MCAN_EN_ERROR_CODE_CREATE_WORKER_ACCESS_SEMAPHORE,
   MCAN_EN_ERROR_CODE_SPAWN_WORKER_THREAD,
   MCAN_EN_ERROR_CODE_DELETE_WORKER_THREAD,
   MCAN_EN_ERROR_CODE_CLOSE_WORKER_ACCESS_SEMAPHORE,
   MCAN_EN_ERROR_CODE_DELETE_WORKER_ACCESS_SEMAPHORE,
   MCAN_EN_ERROR_CODE_CLOSE_WORKER_EVENTS,
   MCAN_EN_ERROR_CODE_DELETE_WORKER_EVENTS,
   MCAN_EN_ERROR_CODE_CREATE_COMMUNICATION_TEST_TIMER,
   MCAN_EN_ERROR_CODE_CREATE_CONFIGURATION_GET_TIMER,
   MCAN_EN_ERROR_CODE_CREATE_HARDWARE_VERSION_GET_TIMER,
   MCAN_EN_ERROR_CODE_CREATE_SOFTWARE_VERSION_GET_TIMER,
   MCAN_EN_ERROR_CODE_CREATE_STATUS_WAITING_TIMER,
   MCAN_EN_ERROR_CODE_SET_COMMUNICATION_TEST_TIMER,
   MCAN_EN_ERROR_CODE_SET_CONFIGURATION_GET_TIMER,
   MCAN_EN_ERROR_CODE_SET_HARDWARE_VERSION_GET_TIMER,
   MCAN_EN_ERROR_CODE_SET_SOFTWARE_VERSION_GET_TIMER,
   MCAN_EN_ERROR_CODE_SET_STATUS_WAITING_TIMER
}MCAN_tenErrorCode;

typedef enum 
{
   MCAN_EN_COMMAND_RECEIVED_IGNORED,
   MCAN_EN_COMMAND_ADDED_TO_WAITING_LIST,
   MCAN_EN_COMMAND_ADDED_TO_EXPIRED_LIST,
   MCAN_EN_COMMAND_ERASED_FROM_WAITING_LIST,
   MCAN_EN_COMMAND_ERASED_FROM_EXPIRED_LIST
} MCAN_tenCommandInfo;


typedef enum 
{
   MCAN_EN_BASIC_TIMER_COMMUNICATION_TEST,
   MCAN_EN_BASIC_TIMER_CONFIGURATION_GET,
   MCAN_EN_BASIC_TIMER_HARDWARE_VERSION_GET,
   MCAN_EN_BASIC_TIMER_SOFTWARE_VERSION_GET,
   MCAN_EN_BASIC_TIMER_STATUS_WAITING
} MCAN_tenBasicTimer;

#define MCAN_NO_REQUEST_ACTIVE      0xFFFFFFFF

// Command and the corresponding timeout time (current system timer value + timeout value)
// This type of list is formed to keep track of all the waiting commands.
class WaitingCommand
{
public:
   tU16 u16Command;
   tS32 s32Timeout;   

   WaitingCommand() { u16Command = 0; s32Timeout = 0; } // Default Constructor
   WaitingCommand( const tU16& Command, const tS32& Timeout ) { u16Command = Command; s32Timeout = Timeout; } // User-defined Constructor
};






// We need the Waiting list to be readily sorted in ascending order according to the timeout value.
// So that we have the next timeout value always at the beginning of the list 
// for the next timer to be started on add/remove of a command from the list.

// This struct could be used in a SET, so that while inserting a new element to the SET.
// It will implicitly take care of keeping the elements in the right order according to their corresponding timeout value.
struct WaitListCompare
{
   bool operator() (const WaitingCommand& cmd1, const WaitingCommand& cmd2) const
   {
      return ( cmd1.s32Timeout < cmd2.s32Timeout );
   }
};


class tclMCANInterface: public MCANItf, public MCANSinkRxHandlerItf, public ConnectionConfiguratorCallbackInterface
{
public:
   // constructors and destructor

   tclMCANInterface();          // Lint requested default constructor
   tVoid initVersionInfo();
   tVoid initMCANInterface();

   tclMCANInterface
      ( 
         tU16            u16EventMask,
         tU8             u8LocCompID,
         tU8             u8RemCompID,
         tBool           bUseAvailableFunctionForConfiguration,
         tU8             u8AvailableFunctionForConfiguration
      );

   tclMCANInterface
      (
         tU16            u16EventMask,
         tVoid*          pvCallbacks,
         tU8             u8LocCompID,
         tU8             u8RemCompID,
         tBool            bUseAvailableFunctionForConfiguration,
         tU8             u8AvailableFunctionForConfiguration
      );

   virtual ~tclMCANInterface();


   // start, setup, and shutdown

   tBool bOnStart( TxSinkItf* poTxSinkInterface, tS32 s32StackSize, tU32 u32ThreadPrio );

   tVoid ActivateChannel();
   tVoid DeactivateChannel(tU32 u32OldAppState);
   tVoid ForceDeactivateChannel();
   tVoid vOnNewAppState( tU32 u32OldAppState, tU32 u32AppState );

   tVoid vOnClose();

   // function called when the system changes to ON without change of connection
   // -> retrigger all messages that were blocked during "not ON"

   virtual tVoid vDoActionsAfterChangeToOn(){}

   // function called when the system leaves ON without change of connection
   // -> send special messages

   virtual tVoid vDoActionsAfterLeavingOn(){}


   // (de)activation the sending of application messages

   tVoid vDeactivateApplicationMessages();

   tVoid vActivateApplicationMessages();


   // information that error masking has changed

   tVoid vErrorMaskingChanged();

   // connection state

   tVoid vSetConnection( tBool bConnected, tBool bForceUpdate );

   inline virtual tBool bIsConnected() const { return _bConnected; }

   virtual tU8 u8GetAbsenceStateOfConnection();
   virtual tU8 u8GetMuteStateOfConnection();
   virtual tU8 u8GetCommunicationStatus();


   // function to get the current mute state of the data transmission

   virtual tU8 u8GetMuteStateOfDataTransmission();


   // configuration and version info

   tVoid vSetConfiguration( tU8 u8UnitID, tU8 u8UpperVersion, tU8 u8MiddleVersion, tU8 u8LowerVersion, tBool bForceUpdate );

   virtual tVoid vRequestHardwareVersion( tVoid );
   tVoid vSetHardwareVersion( tU8 u8UpperVersion, tU8 u8MiddleVersion, tU8 u8LowerVersion, tBool bForceUpdate );

   virtual tVoid vRequestSoftwareVersion( tVoid );
   tVoid vSetSoftwareVersion( tU8 u8UpperVersion, tU8 u8MiddleVersion, tU8 u8LowerVersion, tBool bForceUpdate );

   inline virtual tU8 u8GetUnitID() const { return _u8UnitID; }

   inline virtual tBool bIsMsgVersionValid() const { return _rMsgVersion.bValid; }
   inline virtual tU8  u8GetMsgVersionUpper() const { return _rMsgVersion.u8Upper; }
   inline virtual tU8  u8GetMsgVersionMid()   const { return _rMsgVersion.u8Middle; }
   inline virtual tU8  u8GetMsgVersionLower() const { return _rMsgVersion.u8Lower; }

   inline virtual tBool bIsHardwareVersionValid() const { return _rHardwareVersion.bValid; }
   inline virtual tU8  u8GetHardwareVersionUpper() const { return _rHardwareVersion.u8Upper; }
   inline virtual tU8  u8GetHardwareVersionMid()   const { return _rHardwareVersion.u8Middle; }
   inline virtual tU8  u8GetHardwareVersionLower() const { return _rHardwareVersion.u8Lower; }

   inline virtual tBool bIsSoftwareVersionValid() const { return _rSoftwareVersion.bValid; }
   inline virtual tU8  u8GetSoftwareVersionUpper() const { return _rSoftwareVersion.u8Upper; }
   inline virtual tU8  u8GetSoftwareVersionMid()   const { return _rSoftwareVersion.u8Middle; }
   inline virtual tU8  u8GetSoftwareVersionLower() const { return _rSoftwareVersion.u8Lower; }


   // function to ask for the version infos (hardware and software version)
   // can be used, if the version info is not requesetd during initialization

   tVoid vAskForVersionInfo();
   tVoid vStopAskingForVersionInfo();

   tVoid takeIncomingConfigurationStatusIntoAccount();
   virtual tVoid fakeIncomingMsgConfigurationStatus();
   //ToDo : ASP1BMH : Have to be a Private Function.
   virtual tVoid _vHandleMsgConfiguration( tU8 u8OpType, tU8 u8UnitID, tU8 u8UpperVersion, tU8 u8MiddleVersion, tU8 u8LowerVersion );

   // function to get the actual connection status

   inline MCAN_tenCommunicationStatus enGetCommStatus() const { return _enCommunicationStatus; }


   // function to get the current communication test result

   inline MCAN_tenTestResult enGetCommTestResult() const { return _enCommTestResult; }


   // functions to get the current data confirmation test result

   MCAN_tenTestResult enGetCDataConfTestResult() const;
   MCAN_tenTestResult enGetDDataConfTestResult() const;

   tU16 getCommandFromMessagePayload(tU8* pu8Data);

   // functions used for dealing with incoming messages from the tx sink interface

   virtual tVoid vHandleCommunicationCon( tU8 u8ConnectState );

   virtual tVoid vHandleCommunicationInd( tU8 u8ConnectState );

   virtual tVoid vHandleCDataCon( tU8 u8State );

   virtual tVoid vHandleDDataCon( tU8 u8State );
   virtual tVoid vHandleDataInd( tU8* pu8Data, tU16 u16Length );


   virtual tVoid vHandleUnitSpecificDataInd( tU16 u16MsgId, tU8 u8OpTypeCode, tU8* pu8Data, tU16 u16Length )
   {
      (tVoid)u16MsgId;
      (tVoid)u8OpTypeCode;
      (tVoid)pu8Data;
      (tVoid)u16Length;
   }
   virtual tVoid CallbackForStoringHistoryPersistant(tBool History);

protected:

   //
   // functions
   //

   // function to create the unit specific timers

   virtual tVoid vCreateUnitSpecificTimers(){}


   // functions for tx sink access
#if defined (__METER_UNIT_TESTING__)
public:
#endif
   tBool bDoCDataReq( const tU8* pcu8Data, tU16 u16DataLength, tU16 u16TimeoutInfo );
   /* Not using the default value function prototype to handle the commands with no timeout */
   tBool bDoCDataReq( const tU8* pcu8Data, tU16 u16DataLength ); 
#if defined (__METER_UNIT_TESTING__)
protected:
#endif

#if defined (__METER_UNIT_TESTING__)
public:
#endif
   tBool bDoDDataReq( const tU8* pcu8Data, tU16 u16DataLength, tU16 u16TimeoutInfo );
   /* Not using the default value function prototype to handle the commands with no timeout */
   tBool bDoDDataReq( const tU8* pcu8Data, tU16 u16DataLength );
#if defined (__METER_UNIT_TESTING__)
protected:
#endif

   // functions to set events evaluated by worker thread

   virtual tVoid vSetChangedEvent();

   tVoid vSetUnitSpecificEvent( tU16 u16Event );

   
   // functions to set available functions used for configuration message

   tVoid vSetAvailableFunctionForConfiguration( tU8 u8AvailableFunctionForConfiguration )
   { 
      _u8AvailableFunctionForConfiguration = u8AvailableFunctionForConfiguration; 
   }


   // function used to check whether the unit should be available

   virtual tBool bUnitShouldBeAvailable(){ return TRUE; }


   // function used to check whether the error masking is currently active

   virtual tBool bErrorMaskingActive(){ return FALSE; }


   // funcions used in "event changed" branch of working thread

   virtual tVoid vCheckForUnitSpecificDataRequests(){}
   
   virtual tVoid vCheckForSendingUnitSpecificInfo(){}

   
   // functions called to send a data update to possible CCA clients

   virtual tVoid vSendConnection(){}

   virtual tVoid vSendConfiguration(){}

   virtual tVoid vSendHardwareVersion(){}

   virtual tVoid vSendSoftwareVersion(){}

   virtual tVoid vSendDataTransmission(){}

   virtual tVoid vSendCurrentCommTestResult(){}

   virtual tVoid vSendCurrentCDataConfTestResult(){}

   virtual tVoid vSendCurrentDDataConfTestResult(){}


   // function used to evaluate unit specific events of the working thread

   virtual OSAL_tEventMask hHandleUnitSpecificEvents( OSAL_tEventMask hEvents ){ return hEvents; }


   // do the actions when initialization is set to incomplete, i.e. when configuration has to be exchanged

   virtual tVoid vDoActionsWhenInitializationIncomplete(){}

   // do the actions after initialization is complete, i.e. after configuration has been received

   virtual tVoid vDoActionsAfterInitializationComplete(){}


   // for sequencing of tx sink messages

   inline tBool bIsAllowedToSendCData() const 
   {
      return( ( FALSE == _bInSendingCData ) && ( FALSE == _bSendingOfApplicationMessagesDeactivated ) );
   }

   inline tBool bIsAllowedToSendDData() const 
   {
      return( ( FALSE == _bInSendingDData ) && ( FALSE == _bSendingOfApplicationMessagesDeactivated ) );
   }

   // timing information for requets
   inline tU32 u32GetTimeLastCDataReq() {return _u32TsLastCDataReq;}
   inline tU32 u32GetTimeLastDDataReq() {return _u32TsLastDDataReq;}
   inline tU32 u32GetTimeLastCommunicationReq() {return _u32TsLastCommunicationReq;}

   // for trace outputs

   virtual tVoid vTraceError( MCAN_tenErrorCode enErrorCode ){ (tVoid)enErrorCode; }
   virtual tVoid vTraceIncomingMsgConfiguration(tU8 OpType){}
   virtual tVoid vTraceTakeIncomingConfigurationStatusIntoAccount(tBool ChannelConnected, tBool HistorySet){}
   virtual tVoid vTraceIncomingMessage( tU16 u16MsgId, tU8 u8OpTypeCode, tU8* pu8Data, tU16 u16DataLength )
   {
      (tVoid)u16MsgId; 
      (tVoid)u8OpTypeCode; 
      (tVoid)pu8Data; 
      (tVoid)u16DataLength; 
   }

   virtual tVoid vTraceConnectionStateConnected(){}

   virtual tVoid vTraceCommunicationStatus(tU8 CommunicationStatus){(tVoid)CommunicationStatus;}

   virtual tVoid vTraceConnectionStateDisconnected( tBool bInfoFromIndication ){ (tVoid)bInfoFromIndication; }

   virtual tVoid vTraceCommunicationRequestConnect(){}

   virtual tVoid vTraceCommunicationRequestDisconnect(){}

   virtual tVoid vTraceSendingConfigurationGet(){}
   virtual tVoid vTraceConfigurationGet_NotSent(){}

   virtual tVoid vTraceSendingConfigurationStatus(){}

   virtual tVoid vTraceSendingHardwareVersionGet(){}

   virtual tVoid vTraceSendingSoftwareVersionGet(){}

   virtual tVoid vTraceBasicTimerEvent( MCAN_tenBasicTimer enTimer ){ (tVoid)enTimer; }

   virtual tVoid vTraceBasicTimerSet( MCAN_tenBasicTimer enTimer, OSAL_tMSecond msTimeout )
   { 
      (tVoid)enTimer; 
      (tVoid)msTimeout; 
   }

   virtual tVoid vTraceCommandInfo( MCAN_tenCommandInfo enCmdInfo, tU16 u16Command )
   {
      (tVoid)u16Command;
      (tVoid)enCmdInfo;
   }

   virtual tVoid vTracefakeIncomingMsgConfigurationStatus(
         tBool isConfigurationGetMessageBeenSentToSink,
         tBool isConfigurationStatusMessageFromSinkBeenReceived)
   {
      (tVoid)isConfigurationGetMessageBeenSentToSink;
      (tVoid)isConfigurationStatusMessageFromSinkBeenReceived;
   }

   // All the MCAN units could use this function to check the status of a particular command.
   // Returns TRUE if we are waiting for the Status response.
   tBool _bIsAlreadyWaitingCommand( const tU16& NewCommand ) const;

   // Checks whether the command has already expired so as to not bother MCAN units with those commands and simply ignore them in future
   tBool _bIsCommandAlreadyExpired( const tU16& NewCommand ) const;

   // All MCAN units would be informed via this virtual function that timeout of a particular command has expired
   virtual tVoid vStatusWaitingTimerTimedOut( tU16 u16Command ) { (tVoid) u16Command; }

public:
   tVoid setUnitID(tU8 UnitID){_u8UnitID = UnitID;}
   ConnectionConfigurator oConnectionConfigurator;
#if defined (__METER_UNIT_TESTING__)
public:
#else
private:
#endif

   //
   // types
   //
#if defined (__METER_UNIT_TESTING__)
public:
#endif
   typedef enum 
   {
      EN_DATA_CONF_STATUS_UNKNOWN,
      EN_DATA_CONF_STATUS_OK,
      EN_DATA_CONF_STATUS_FAULT
   } _tenDataConfStatus;

#if defined (__METER_UNIT_TESTING__)
private:
#endif

   typedef struct
   {
      tBool bValid;
      tU8  u8Upper;
      tU8  u8Middle;
      tU8  u8Lower;
   }_trVersion;


   //
   // worker thread
   //

   // thread body method

   static tVoid _vWorkerThreadFunction( tVoid *pvArg );


   // implementation of the worker thread function
public:
   tVoid ApplicationCallbackInit ( );
private:
   tVoid _vWorkerThreadFunctionImplementation();


   // setup and start of the worker thread

   tBool _bWorkerThreadSetup( tS32 s32StackSize, tU32 u32ThreadPrio );


   // termination and deletion of the worker thread and its resources

   tVoid _vWorkerThreadDelete();


   // functions to set a basic event evaluated by worker thread

   tVoid _vSetCommunicationConEvent();
   tVoid _vSetCommunicationIndEvent();
   tVoid _vSetCommunicationTestTimerEvent();
   tVoid _vSetConfigurationGetTimerEvent();
   tVoid _vSetHardwareVersionGetTimerEvent();
   tVoid _vSetSoftwareVersionGetTimerEvent();
   tVoid _vSetStatusWaitingTimerEvent();

   tVoid setParameterAndTriggerForFurtherActionsOnDisconnectedChannel();

   // function to evaluate the current connection state after reception of communication event
   tVoid handleConnectionTimerInCaseConnectionRequstIsPendingAndDisonnectConfirmationReceived(tBool CommunicationIndicationReceived);
   tVoid _vEvaluateConnectionState( tBool bInfoFromCommunicationIndication );

   tVoid traceCommunicationRequest(tU8 Request);
   tVoid sendCommunicationRequestToSink(tU8 Request);
   // funcions used in "event changed" branch of working thread

   tVoid _vCheckForCommunicationRequest();
public:
   tVoid sendConfigurationGetViaMCAN();
private:
   tVoid _vCheckForConfiguration();
   tBool _bInitializationComplete() const;
   tVoid _vCheckForVersionInfo();
   tVoid _vCheckForSendingBasicInfo();


   // function used to evaluate basic events of the working thread

   OSAL_tEventMask _hHandleBasicEvents( OSAL_tEventMask hEvents );


   //
   // communication and data confirmation tests
   //

   // function checking whether the communication test timer has to be started or stopped

   tVoid _vDoCommunicationTestTimerHandling();


   // function to trigger the saving of the communication test result

   tVoid _vTriggerSavingOfCommTestResult();


   // functions to trigger the saving of the data confirmation test result

   tVoid _vTriggerSavingOfCDataConfTestResult();
   tVoid _vTriggerSavingOfDDataConfTestResult();


   //
   // functions to trigger the update flag for sending data to registered clients
   //

   tVoid _vHandleConnectionUpdate();
   tVoid _vHandleConfigurationUpdate();
   tVoid _vHandleHardwareVersionUpdate();
   tVoid _vHandleSoftwareVersionUpdate();
   tVoid _vHandleDataTransmissionUpdate();


   // general handler functions for incoming standard MCAN messages

   //ToDo : ASP1BMH : Uncomment
  // virtual tVoid _vHandleMsgConfiguration( tU8 u8OpType, tU8 u8UnitID, tU8 u8UpperVersion, tU8 u8MiddleVersion, tU8 u8LowerVersion );
   tVoid _vHandleMsgHardwareVersion( tU8 u8OpType, tU8 u8UpperVersion, tU8 u8MiddleVersion, tU8 u8LowerVersion );
   tVoid _vHandleMsgSoftwareVersion( tU8 u8OpType, tU8 u8UpperVersion, tU8 u8MiddleVersion, tU8 u8LowerVersion );


   // 
   // timer functionality
   //

   // create the basic timers used for communication test and initialization phase

   tVoid _vCreateBasicTimers();


   // timer callbacks (note: we can use the same callback function for each unit)

   static tVoid _vCommunicationTestTimerCallback( tVoid *pvArg );
   static tVoid _vConfigurationGetTimerCallback( tVoid *pvArg );
   static tVoid _vHardwareVersionGetTimerCallback( tVoid *pvArg );
   static tVoid _vSoftwareVersionGetTimerCallback( tVoid *pvArg );
   static tVoid _vStatusWaitingTimerCallback( tVoid *pvArg );


   // functions to set (start/stop) a timer

   tVoid _vSetCommunicationTestTimer( OSAL_tMSecond msTimeout );
   tVoid _vSetConfigurationGetTimer( OSAL_tMSecond msTimeout );
   tVoid _vSetHardwareVersionGetTimer( OSAL_tMSecond msTimeout );
   tVoid _vSetSoftwareVersionGetTimer( OSAL_tMSecond msTimeout );
   tVoid _vSetStatusWaitingTimer( OSAL_tMSecond msTimeout );



   //
   // private variables
   //

   // for thread handling

   OSAL_tThreadID     _hWorkerThreadId;          // ID of the active thread
   OSAL_tEventHandle  _hWorkerEvents;            // events of the worker thread
   OSAL_tSemHandle    _hWorkerSem;               // access semaphore of the worker thread
   tBool              _bTerminateWorkerThread;   // flag for termination of worker thread, TRUE terminates worker thread


   // to store the tx sink interface
#if defined (__METER_UNIT_TESTING__)
public:
#endif
   TxSinkItf* _poTxSinkInterface;
#if defined (__METER_UNIT_TESTING__)
private:
#endif


   // to store the unit specific event mask

   tU16 _u16UnitSpecificEventMask;


   // to store the tx sink  callbacks and the content of the tx sink address field

   tVoid* _pvCallbacks;
#if defined (__METER_UNIT_TESTING__)
public:
#endif
   tU8    _u8LocalCompID;
   tU8    _u8RemoteCompID;
#if defined (__METER_UNIT_TESTING__)
private:
#endif

   // to store data for a possible parameter "available function" of the message "configuration"

   tBool _bUseAvailableFunctionForConfiguration;
   tU8   _u8AvailableFunctionForConfiguration;


   // to store the requested application state

   tU32 _u32RequestedAppState;


   // timer handles

   OSAL_tTimerHandle  _hCommunicationTestTimer;   // for communication test
   OSAL_tTimerHandle  _hConfigurationGetTimer;    // for retrying to send the configuration-get message
   OSAL_tTimerHandle  _hHardwareVersionGetTimer;  // for retrying to send the hardware_version-get message
   OSAL_tTimerHandle  _hSoftwareVersionGetTimer;  // for retrying to send the software_version-get message
   OSAL_tTimerHandle  _hStatusWaitingTimer;          // for retrying to send the MCAN unit specific messages      


   // for sequencing of tx sink messages

   tBool _bInSendingCData;
   tBool _bInSendingDData;

   tU32  _u32TsLastCDataReq;
   tU32  _u32TsLastDDataReq;
   tU32  _u32TsLastCommunicationReq;


   // for the result of data confirmations

   _tenDataConfStatus  _enCDataConfStatus;
   _tenDataConfStatus  _enDDataConfStatus;


   // for connection and communication handling

   tU8  _u8InitializationStatus;
   tU8  _u8ActualConnectionState;
   MCAN_tenCommunicationStatus  _enCommunicationStatus;

   MCAN_tenTestResult  _enCommTestResult;


   // trigger evaluated during event changed

   tBool _bTriggerCommunicationRequestConnect;
   tBool _bTriggerCommunicationRequestDisconnect;

   tBool _bCommunicationRequestBlocked;
   tU8   _u8CommunicationRequestBlockedState;

   tBool _bTriggerConfigurationGet;
   tBool _bTriggerConfigurationStatus;

   tBool _bTriggerHardwareVersionGet;
   tBool _bTriggerHardwareVersionGetOneTime;
   tBool _bTriggerSoftwareVersionGet;
   tBool _bTriggerSoftwareVersionGetOneTime;
   tBool _bAskForSoftwareVersionAfterHardwareVersion;

   tBool _bTriggerSendConnection;
   tBool _bTriggerSendConfiguration;
   tBool _bTriggerSendHardwareVersion;
   tBool _bTriggerSendSoftwareVersion;
   tBool _bTriggerSendDataTransmission;

   tBool _bTriggerSendCommTestResult;

   tBool _bTriggerSendCDataConfTestResult;
   tBool _bTriggerSendDDataConfTestResult;


   // to store the number of retries for version info

   tU8  _u8HardwareVersionGetRetries;
   tU8  _u8SoftwareVersionGetRetries;


   // to force sending the result after a version info request

   tBool _bForceUpdateOfHardwareVersion;
   tBool _bForceUpdateOfSoftwareVersion;


   // to deactivate the sending of application messages

   tBool _bSendingOfApplicationMessagesDeactivated;


   // to store the last known connection state

   tBool _bConnected;


   // for the data received via MCAN 

   tU8        _u8UnitID;
   _trVersion  _rMsgVersion;
   _trVersion  _rHardwareVersion;
   _trVersion  _rSoftwareVersion;
public:
   SendTriggerContainer oSendCommTestResult;
private:

   // to handle status-waiting-timer required for all MCAN based communication
   // add the command (=FunctionID + OpType) which we are waiting for with the timer value

   bpstl::multiset< WaitingCommand, WaitListCompare, bpstl::allocator<WaitingCommand> > _CommandWaitingList;

   // List of the commands expired
   bpstl::set< tU16 > _ExpiredCommandList;

   // Just to insert a new command to the wait list   
   tVoid _vAddCommandToWaitingList( const WaitingCommand& tempCommand );

   // Handler for the commands received for which we are waiting, remove the command and take care of a starting the new timer if required.
   tVoid _vHandleReceivedCommand( const tU16& u16Command );

   // Handler for the timeout, takes care of removing the corresponding command(s) and starting the next timer if required.
   tVoid _vHandleStatusWaitingTimerTimeout( tVoid );  

   tVoid _vRemoveCommandFromExpiredList( const tU16& u16Command );

   tVoid _vRemoveCommandFromWaitingList( const tU16& u16Command );  

   ConfigurationGetMessageHandler oConfigurationGetMessage;
   ConfigurationStatusMessageHandler oConfigurationStatusMessage;
};

#endif  // _MCAN_IF_H_
