//----------------------------------------------------------------------
// FILE:         MCANInterface.h
// PROJECT:      NISSAN LCN2 (kai)
// SW-COMPONENT: middleware common
//----------------------------------------------------------------------
//
// DESCRIPTION:  implementation of general interface for MCAN communication
//
//----------------------------------------------------------------------
// COPYRIGHT:    (c) 2012 Robert Bosch GmbH, Hildesheim
// HISTORY:
// Date      | Author             | Modification
// 31.05.12  | TMS Kempen         | initial version
//
//----------------------------------------------------------------------
#include "MCANInterface.h"
#define AMT_S_IMPORT_INTERFACE_GENERIC
#include <amt_if.h>


tVoid tclMCANInterface::initVersionInfo()
{
   _rMsgVersion.bValid   = FALSE;
   _rMsgVersion.u8Upper  = 0;
   _rMsgVersion.u8Middle = 0;
   _rMsgVersion.u8Lower  = 0;

   _rHardwareVersion.bValid   = FALSE;
   _rHardwareVersion.u8Upper  = 0;
   _rHardwareVersion.u8Middle = 0;
   _rHardwareVersion.u8Lower  = 0;

   _rSoftwareVersion.bValid   = FALSE;
   _rSoftwareVersion.u8Upper  = 0;
   _rSoftwareVersion.u8Middle = 0;
   _rSoftwareVersion.u8Lower  = 0;

}

tVoid tclMCANInterface::initMCANInterface()
{
   initVersionInfo();

   _CommandWaitingList.clear();
   _ExpiredCommandList.clear();
   oConfigurationGetMessage.forgetAboutMessageWasSendToSink();
   oConfigurationStatusMessage.forgetAboutMessageWasReceivedFromSink();
}

tclMCANInterface::tclMCANInterface()
   : _hWorkerThreadId(OSAL_ERROR)
   , _hWorkerEvents(OSAL_C_INVALID_HANDLE)
   , _hWorkerSem(OSAL_C_INVALID_HANDLE)
   , _bTerminateWorkerThread(FALSE)
   , _poTxSinkInterface(NULL)
   , _u16UnitSpecificEventMask(0)
   , _pvCallbacks(NULL)
   , _u8LocalCompID(0)
   , _u8RemoteCompID(0)
   , _bUseAvailableFunctionForConfiguration(FALSE)
   , _u8AvailableFunctionForConfiguration(0)
   , _u32RequestedAppState(AMT_C_U32_STATE_INVALID)
   , _hCommunicationTestTimer(OSAL_C_INVALID_HANDLE)
   , _hConfigurationGetTimer(OSAL_C_INVALID_HANDLE)
   , _hHardwareVersionGetTimer(OSAL_C_INVALID_HANDLE)
   , _hSoftwareVersionGetTimer(OSAL_C_INVALID_HANDLE)
   , _hStatusWaitingTimer(OSAL_C_INVALID_HANDLE)
   , _bInSendingCData(FALSE)
   , _bInSendingDData(FALSE)
   , _u32TsLastCDataReq(MCAN_NO_REQUEST_ACTIVE)
   , _u32TsLastDDataReq(MCAN_NO_REQUEST_ACTIVE)
   , _u32TsLastCommunicationReq(MCAN_NO_REQUEST_ACTIVE)
   , _enCDataConfStatus(EN_DATA_CONF_STATUS_UNKNOWN)
   , _enDDataConfStatus(EN_DATA_CONF_STATUS_UNKNOWN)
   , _u8InitializationStatus(0x00)
   , _u8ActualConnectionState(TX_SINK_DISCONNECTED)
   , _enCommunicationStatus(MCAN_EN_COMM_STATUS_OFF)
   , _enCommTestResult(MCAN_EN_TEST_RESULT_UNKNOWN)
   , _bTriggerCommunicationRequestConnect(FALSE)
   , _bTriggerCommunicationRequestDisconnect(FALSE)
   , _bCommunicationRequestBlocked(FALSE)
   , _u8CommunicationRequestBlockedState(TX_SINK_DISCONNECTED)
   , _bTriggerConfigurationGet(FALSE)
   , _bTriggerConfigurationStatus(FALSE)
   , _bTriggerHardwareVersionGet(FALSE)
   , _bTriggerHardwareVersionGetOneTime(FALSE)
   , _bTriggerSoftwareVersionGet(FALSE)
   , _bTriggerSoftwareVersionGetOneTime(FALSE)
   , _bAskForSoftwareVersionAfterHardwareVersion(FALSE)
   , _bTriggerSendConnection(FALSE)
   , _bTriggerSendConfiguration(FALSE)
   , _bTriggerSendHardwareVersion(FALSE)
   , _bTriggerSendSoftwareVersion(FALSE)
   , _bTriggerSendDataTransmission(FALSE)
   , _bTriggerSendCDataConfTestResult(FALSE)
   , _bTriggerSendDDataConfTestResult(FALSE)
   , _u8HardwareVersionGetRetries(0)
   , _u8SoftwareVersionGetRetries(0)
   , _bForceUpdateOfHardwareVersion(FALSE)
   , _bForceUpdateOfSoftwareVersion(FALSE)
   , _bSendingOfApplicationMessagesDeactivated(FALSE)
   , _bConnected(FALSE)
   , _u8UnitID(0)
{
initMCANInterface();
#if !defined(__METER_UNIT_TESTING__)
   //You are not supposed to use default constructor
   NORMAL_M_ASSERT_ALWAYS();
#endif
}

// ToDo: Components beside fc_meter must be adapted to the new interface. Callback function are not available any more.
// use CSm interface
tclMCANInterface::tclMCANInterface
   (
      tU16            u16EventMask,
      tVoid*          pvCallbacks,
      tU8             u8LocCompID,
      tU8             u8RemCompID,
      tBool            bUseAvailableFunctionForConfiguration,
      tU8             u8AvailableFunctionForConfiguration
   )
   : _hWorkerThreadId(OSAL_ERROR)
   , _hWorkerEvents(OSAL_C_INVALID_HANDLE)
   , _hWorkerSem(OSAL_C_INVALID_HANDLE)
   , _bTerminateWorkerThread(FALSE)
   , _poTxSinkInterface(NULL)
   , _u16UnitSpecificEventMask(u16EventMask)
   , _pvCallbacks(pvCallbacks)
   , _u8LocalCompID(u8LocCompID)
   , _u8RemoteCompID(u8RemCompID)
   , _bUseAvailableFunctionForConfiguration(bUseAvailableFunctionForConfiguration)
   , _u8AvailableFunctionForConfiguration(u8AvailableFunctionForConfiguration)
   , _u32RequestedAppState(AMT_C_U32_STATE_INVALID)
   , _hCommunicationTestTimer(OSAL_C_INVALID_HANDLE)
   , _hConfigurationGetTimer(OSAL_C_INVALID_HANDLE)
   , _hHardwareVersionGetTimer(OSAL_C_INVALID_HANDLE)
   , _hSoftwareVersionGetTimer(OSAL_C_INVALID_HANDLE)
   , _hStatusWaitingTimer(OSAL_C_INVALID_HANDLE)
   , _bInSendingCData(FALSE)
   , _bInSendingDData(FALSE)
   , _u32TsLastCDataReq(MCAN_NO_REQUEST_ACTIVE)
   , _u32TsLastDDataReq(MCAN_NO_REQUEST_ACTIVE)
   , _u32TsLastCommunicationReq(MCAN_NO_REQUEST_ACTIVE)
   , _enCDataConfStatus(EN_DATA_CONF_STATUS_UNKNOWN)
   , _enDDataConfStatus(EN_DATA_CONF_STATUS_UNKNOWN)
   , _u8InitializationStatus(0x00)
   , _u8ActualConnectionState(TX_SINK_DISCONNECTED)
   , _enCommunicationStatus(MCAN_EN_COMM_STATUS_OFF)
   , _enCommTestResult(MCAN_EN_TEST_RESULT_UNKNOWN)
   , _bTriggerCommunicationRequestConnect(FALSE)
   , _bTriggerCommunicationRequestDisconnect(FALSE)
   , _bCommunicationRequestBlocked(FALSE)
   , _u8CommunicationRequestBlockedState(TX_SINK_DISCONNECTED)
   , _bTriggerConfigurationGet(FALSE)
   , _bTriggerConfigurationStatus(FALSE)
   , _bTriggerHardwareVersionGet(FALSE)
   , _bTriggerHardwareVersionGetOneTime(FALSE)
   , _bTriggerSoftwareVersionGet(FALSE)
   , _bTriggerSoftwareVersionGetOneTime(FALSE)
   , _bAskForSoftwareVersionAfterHardwareVersion(FALSE)
   , _bTriggerSendConnection(FALSE)
   , _bTriggerSendConfiguration(FALSE)
   , _bTriggerSendHardwareVersion(FALSE)
   , _bTriggerSendSoftwareVersion(FALSE)
   , _bTriggerSendDataTransmission(FALSE)
   , _bTriggerSendCDataConfTestResult(FALSE)
   , _bTriggerSendDDataConfTestResult(FALSE)
   , _u8HardwareVersionGetRetries(0)
   , _u8SoftwareVersionGetRetries(0)
   , _bForceUpdateOfHardwareVersion(FALSE)
   , _bForceUpdateOfSoftwareVersion(FALSE)
   , _bSendingOfApplicationMessagesDeactivated(FALSE)
   , _bConnected(FALSE)
   , _u8UnitID(0)
   , oConnectionConfigurator((ConnectionConfiguratorCallbackInterface*)this)
{
   initMCANInterface();

}

// use INC as CAN interface
tclMCANInterface::tclMCANInterface
   (
      tU16            u16EventMask,
      tU8             u8LocCompID,
      tU8             u8RemCompID,
      tBool            bUseAvailableFunctionForConfiguration,
      tU8             u8AvailableFunctionForConfiguration
   )
   : _hWorkerThreadId(OSAL_ERROR)
   , _hWorkerEvents(OSAL_C_INVALID_HANDLE)
   , _hWorkerSem(OSAL_C_INVALID_HANDLE)
   , _bTerminateWorkerThread(FALSE)
   , _poTxSinkInterface(NULL)
   , _u16UnitSpecificEventMask(u16EventMask)
   , _pvCallbacks(NULL)
   , _u8LocalCompID(u8LocCompID)
   , _u8RemoteCompID(u8RemCompID)
   , _bUseAvailableFunctionForConfiguration(bUseAvailableFunctionForConfiguration)
   , _u8AvailableFunctionForConfiguration(u8AvailableFunctionForConfiguration)
   , _u32RequestedAppState(AMT_C_U32_STATE_INVALID)
   , _hCommunicationTestTimer(OSAL_C_INVALID_HANDLE)
   , _hConfigurationGetTimer(OSAL_C_INVALID_HANDLE)
   , _hHardwareVersionGetTimer(OSAL_C_INVALID_HANDLE)
   , _hSoftwareVersionGetTimer(OSAL_C_INVALID_HANDLE)
   , _hStatusWaitingTimer(OSAL_C_INVALID_HANDLE)
   , _bInSendingCData(FALSE)
   , _bInSendingDData(FALSE)
   , _enCDataConfStatus(EN_DATA_CONF_STATUS_UNKNOWN)
   , _enDDataConfStatus(EN_DATA_CONF_STATUS_UNKNOWN)
   , _u8InitializationStatus(0x00)
   , _u8ActualConnectionState(TX_SINK_DISCONNECTED)
   , _enCommunicationStatus(MCAN_EN_COMM_STATUS_OFF)
   , _enCommTestResult(MCAN_EN_TEST_RESULT_UNKNOWN)
   , _bTriggerCommunicationRequestConnect(FALSE)
   , _bTriggerCommunicationRequestDisconnect(FALSE)
   , _bTriggerConfigurationGet(FALSE)
   , _bTriggerConfigurationStatus(FALSE)
   , _bTriggerHardwareVersionGet(FALSE)
   , _bTriggerHardwareVersionGetOneTime(FALSE)
   , _bTriggerSoftwareVersionGet(FALSE)
   , _bTriggerSoftwareVersionGetOneTime(FALSE)
   , _bAskForSoftwareVersionAfterHardwareVersion(FALSE)
   , _bTriggerSendConnection(FALSE)
   , _bTriggerSendConfiguration(FALSE)
   , _bTriggerSendHardwareVersion(FALSE)
   , _bTriggerSendSoftwareVersion(FALSE)
   , _bTriggerSendDataTransmission(FALSE)
   , _bTriggerSendCDataConfTestResult(FALSE)
   , _bTriggerSendDDataConfTestResult(FALSE)
   , _u8HardwareVersionGetRetries(0)
   , _u8SoftwareVersionGetRetries(0)
   , _bForceUpdateOfHardwareVersion(FALSE)
   , _bForceUpdateOfSoftwareVersion(FALSE)
   , _bSendingOfApplicationMessagesDeactivated(FALSE)
   , _bConnected(FALSE)
   , _u8UnitID(0)
   , oConnectionConfigurator((ConnectionConfiguratorCallbackInterface*)this)
{
   initMCANInterface();
}

tclMCANInterface::~tclMCANInterface()
{
   _poTxSinkInterface = NULL;

   _CommandWaitingList.clear();
   _ExpiredCommandList.clear();
}

tVoid tclMCANInterface::CallbackForStoringHistoryPersistant(tBool History)
{

}

//----------------------------------------------------------------------
//
// FUNCTION: tBool tclMCANInterface::bOnStart( TxSinkItf* poTxSinkInterface, tS32 s32StackSize, tU32 u32ThreadPrio )
//
// DESCRIPTION: setups timer and worker thread
//
// PARAMETER:   pointer to the tx sink interface
//              size of the stack for the worker thread
//              priority of the worker thread
//
// RETURNVALUE: result of the thread setup
//
//----------------------------------------------------------------------
tBool tclMCANInterface::bOnStart( TxSinkItf* poTxSinkInterface, tS32 s32StackSize, tU32 u32ThreadPrio )
{
   tBool bRetVal = TRUE;

   _poTxSinkInterface = poTxSinkInterface;

   // creation of the basic timer 
   _vCreateBasicTimers();

   // creation of unit specific timers
   vCreateUnitSpecificTimers();

   // Unfolding the worker thread
   bRetVal = _bWorkerThreadSetup( s32StackSize, u32ThreadPrio );

   return bRetVal;
}
tVoid tclMCANInterface::ActivateChannel()
{
	vTraceCommunicationStatus(_enCommunicationStatus);
    if (_u32TsLastCommunicationReq != MCAN_NO_REQUEST_ACTIVE) {
        _bCommunicationRequestBlocked = TRUE;
        _u8CommunicationRequestBlockedState = TX_SINK_CONNECT;
    } else {
        if (TRUE == oConnectionConfigurator.isCommunicationGranted()) //ToDO: this belongs into bUnitShouldBeAvailable function
        {
            _u32TsLastCommunicationReq = OSAL_ClockGetElapsedTime();

            if ( MCAN_EN_COMM_STATUS_OFF == _enCommunicationStatus )
            {
                _enCommunicationStatus = MCAN_EN_COMM_STATUS_NOT_CONNECTED;
                vTraceCommunicationStatus(_enCommunicationStatus);
            }

            if ( bUnitShouldBeAvailable() )
            {
                _enCommTestResult = MCAN_EN_TEST_RESULT_UNKNOWN;
                _enCommunicationStatus = MCAN_EN_COMM_STATUS_CONNECTION_WAIT;
                vTraceCommunicationStatus(_enCommunicationStatus);
            }

            // communication status changed -> update the connection status
            _vHandleConnectionUpdate();

            _bTriggerCommunicationRequestConnect = TRUE;
            vSetChangedEvent();
        }
    }
}
tVoid tclMCANInterface::DeactivateChannel(tU32 u32OldAppState)
{
	vTraceCommunicationStatus(_enCommunicationStatus);
    if (_u32TsLastCommunicationReq != MCAN_NO_REQUEST_ACTIVE) {
        _bCommunicationRequestBlocked = TRUE;
        _u8CommunicationRequestBlockedState = TX_SINK_DISCONNECTED;
    } else {
        if (TRUE == oConnectionConfigurator.isCommunicationGranted())
        {
            _u32TsLastCommunicationReq = OSAL_ClockGetElapsedTime();

            _enCommTestResult = MCAN_EN_TEST_RESULT_UNKNOWN;

            // set the status to NOT_CONNECTED, because we want to disconnect
            _enCommunicationStatus = MCAN_EN_COMM_STATUS_NOT_CONNECTED;
            vTraceCommunicationStatus(_enCommunicationStatus);

            // stop a possibly running communication test timer
            _vDoCommunicationTestTimerHandling();

            // communication status changed -> update the connection status
            _vHandleConnectionUpdate();

            if ( u32OldAppState == AMT_C_U32_STATE_NORMAL )
            {
                _bTriggerCommunicationRequestDisconnect = TRUE;
                vSetChangedEvent();
            }
        }
    }
}

/**
 * For deactivating the channel, force method is been used, because the deactivate channel function is using the application state for avoiding redundant actions.
 * But for some components channel must be deactivated even though application state is still normal. Because on ACC off, application state won't be changed.
 *
 */
tVoid tclMCANInterface::ForceDeactivateChannel()
{
   DeactivateChannel(AMT_C_U32_STATE_NORMAL);
}

//----------------------------------------------------------------------
//
// FUNCTION: tVoid tclMCANInterface::vOnNewAppState( tU32 u32OldAppState, tU32 u32AppState )
//
// DESCRIPTION: is called after the initialisation of the framework is finished
//
// PARAMETER: tU32 u32OldAppState (actual state),
//            tU32 u32AppState (state to change to)
//
// RETURNVALUE: void
//
//----------------------------------------------------------------------
tVoid tclMCANInterface::vOnNewAppState( tU32 u32OldAppState, tU32 u32AppState )
{
   _u32RequestedAppState = u32AppState;

   if ( u32OldAppState != u32AppState )
   {
      switch ( u32AppState )
      {
         case AMT_C_U32_STATE_NORMAL:
         {
//            ActivateChannel(); // not required according to specification
         }
         break;
         case AMT_C_U32_STATE_DIAGNOSIS:
         case AMT_C_U32_STATE_PAUSE:
         case AMT_C_U32_STATE_OFF:
         {
//            DeactivateChannel(u32OldAppState); // not required according to specification
         }
         break;
         default:
         {
            /* +++
            this should never happen, you can call a trace or assert here
            +++ */
         }
         break;
      }
   }
}


//----------------------------------------------------------------------
//
// FUNCTION: tVoid tclMCANInterface::vOnClose()
//
// DESCRIPTION: releases all resources and sends close - message
//
// PARAMETER: void
//
// RETURNVALUE: void
//
//----------------------------------------------------------------------
tVoid tclMCANInterface::vOnClose()
{
   if ( OSAL_C_INVALID_HANDLE != _hCommunicationTestTimer )
   {
      _vSetCommunicationTestTimer( 0 );  // stop the timer before deleting it
      if ( OSAL_OK == OSAL_s32TimerDelete( _hCommunicationTestTimer ) )
      {
         _hCommunicationTestTimer = OSAL_C_INVALID_HANDLE;
      }
      else
      {
         vTraceError( MCAN_EN_ERROR_CODE_DELETE_COMMUNICATION_TEST_TIMER );
      }
   }

   if ( OSAL_C_INVALID_HANDLE != _hConfigurationGetTimer )
   {
      _vSetConfigurationGetTimer( 0 );  // stop the timer before deleting it
      if ( OSAL_OK == OSAL_s32TimerDelete( _hConfigurationGetTimer ) )
      {
         _hConfigurationGetTimer = OSAL_C_INVALID_HANDLE;
      }
      else
      {
         vTraceError( MCAN_EN_ERROR_CODE_DELETE_CONFIGURATION_GET_TIMER );
      }
   }

   if ( OSAL_C_INVALID_HANDLE != _hHardwareVersionGetTimer )
   {
      _vSetHardwareVersionGetTimer( 0 );  // stop the timer before deleting it
      if ( OSAL_OK == OSAL_s32TimerDelete( _hHardwareVersionGetTimer ) )
      {
         _hHardwareVersionGetTimer = OSAL_C_INVALID_HANDLE;
      }
      else
      {
         vTraceError( MCAN_EN_ERROR_CODE_DELETE_HARDWARE_VERSION_GET_TIMER );
      }
   }

   if ( OSAL_C_INVALID_HANDLE != _hSoftwareVersionGetTimer )
   {
      _vSetSoftwareVersionGetTimer( 0 );  // stop the timer before deleting it
      if ( OSAL_OK == OSAL_s32TimerDelete( _hSoftwareVersionGetTimer ) )
      {
         _hSoftwareVersionGetTimer = OSAL_C_INVALID_HANDLE;
      }
      else
      {
         vTraceError( MCAN_EN_ERROR_CODE_DELETE_SOFTWARE_VERSION_GET_TIMER );
      }
   }

   if ( OSAL_C_INVALID_HANDLE != _hStatusWaitingTimer )
   {
      _vSetStatusWaitingTimer( 0 );  // stop the timer before deleting it
      if ( OSAL_OK == OSAL_s32TimerDelete( _hStatusWaitingTimer ) )
      {
         _hStatusWaitingTimer = OSAL_C_INVALID_HANDLE;
      }
      else
      {
         vTraceError( MCAN_EN_ERROR_CODE_SET_STATUS_WAITING_TIMER );
      }
   }

   // Clear all the list data valid for the current ignition cycle
   _CommandWaitingList.clear();
   _ExpiredCommandList.clear();

   _vWorkerThreadDelete();
}


// (de)activation the sending of application messages

tVoid tclMCANInterface::vDeactivateApplicationMessages()
{
   _bSendingOfApplicationMessagesDeactivated = TRUE;
}

tVoid tclMCANInterface::vActivateApplicationMessages()
{
   _bSendingOfApplicationMessagesDeactivated = FALSE;
   vSetChangedEvent();
}


// information about error detection status (error detection masked or non-masked)

tVoid tclMCANInterface::vErrorMaskingChanged()
{

   if ( FALSE == bErrorMaskingActive() )
   {
      // Changing from masked to non-masked
      if ( MCAN_EN_COMM_STATUS_CONNECTION_ERROR == _enCommunicationStatus )
      {
         // During an active communication error the communication test 
         // has to be restarted, but with shorter timeout.
         _enCommTestResult = MCAN_EN_TEST_RESULT_UNKNOWN;
         _enCommunicationStatus = MCAN_EN_COMM_STATUS_CONNECTION_WAIT_AFTER_ERROR;
         vTraceCommunicationStatus(_enCommunicationStatus);
      }
      else
      {
         // Otherwise, sending of the test result can be triggered directly.
         _vTriggerSavingOfCommTestResult();
      }
// todo: check whether channel connection is allowed
      // Furthermore, sending of the data confirmation test results have to be triggered.
      _vTriggerSavingOfCDataConfTestResult();
      _vTriggerSavingOfDDataConfTestResult();

      // Note: If the current test result is UNKNOWN, we won't send any message.
   }
   // todo: check whether channel connection is allowed
   // eventually start the running communication test timer
   _vDoCommunicationTestTimerHandling();

   // error masking changed -> send the properties Connection and DataTransmission
   _vHandleConnectionUpdate();
   _vHandleDataTransmissionUpdate();

}


// connection

tVoid tclMCANInterface::vSetConnection( tBool bConnected, tBool bForceUpdate )
{

   tBool bChanged = FALSE;
   tBool bDataChanged = FALSE;

   if ( _bConnected != bConnected )
   {
      _bConnected = bConnected;
      bChanged = TRUE;

      if ( FALSE == _bConnected )
      {
         setUnitID(0);
         initVersionInfo();

         bDataChanged = TRUE;
      }
   }

   if ( bChanged || bForceUpdate )
   {
      _vHandleConnectionUpdate();
   }

   if ( bDataChanged )
   {
      _vHandleConfigurationUpdate();
      _vHandleHardwareVersionUpdate();
      _vHandleSoftwareVersionUpdate();
   }
}

tU8 tclMCANInterface::u8GetAbsenceStateOfConnection()
{
   tU8 u8RetVal = MCAN_C_ERROR_STATE_MASKED;

   if ( FALSE == bErrorMaskingActive() )
   {
      switch ( _enCommunicationStatus )
      {
         case MCAN_EN_COMM_STATUS_CONNECTION_WAIT:
         case MCAN_EN_COMM_STATUS_CONNECTED:
         {
            u8RetVal = MCAN_C_ERROR_STATE_OK;
         }
         break;
         case MCAN_EN_COMM_STATUS_CONNECTION_WAIT_AFTER_ERROR:
         {
            u8RetVal = MCAN_C_ERROR_STATE_CURRENTLY_NOK;
         }
         break;
         case MCAN_EN_COMM_STATUS_CONNECTION_ERROR:
         {
            u8RetVal = MCAN_C_ERROR_STATE_CONFIRMED_NOK;
         }
         break;
         case MCAN_EN_COMM_STATUS_OFF:
         case MCAN_EN_COMM_STATUS_NOT_CONNECTED:
         default:
         {
            // keep it as "masked"
         }
         break;
      }
   }

   return u8RetVal;
}

tU8 tclMCANInterface::u8GetMuteStateOfConnection()
{
   tU8 u8RetVal = MCAN_C_ERROR_STATE_MASKED;

   if ( FALSE == bErrorMaskingActive() )
   {
      if ( MCAN_EN_COMM_STATUS_CONNECTED == _enCommunicationStatus )
      {
         u8RetVal = MCAN_C_ERROR_STATE_OK;
      }
   }

   return u8RetVal;
}

tU8 tclMCANInterface::u8GetCommunicationStatus()
{
	return _enCommunicationStatus;
}

// function to get the current mute state of the data transmission

tU8 tclMCANInterface::u8GetMuteStateOfDataTransmission()
{
   tU8 u8RetVal = MCAN_C_ERROR_STATE_MASKED;

   if ( FALSE == bErrorMaskingActive() )
   {
      if (
            ( EN_DATA_CONF_STATUS_FAULT == _enCDataConfStatus )
            ||
            ( EN_DATA_CONF_STATUS_FAULT == _enDDataConfStatus )
         )
      {
         u8RetVal = MCAN_C_ERROR_STATE_CONFIRMED_NOK;
      }
      else
      {
         u8RetVal = MCAN_C_ERROR_STATE_OK;
      }
   }

   return u8RetVal;
}

// configuration

tVoid tclMCANInterface::vSetConfiguration
   (
      tU8 u8UnitID, tU8 u8UpperVersion, tU8 u8MiddleVersion, tU8 u8LowerVersion, tBool bForceUpdate
   )
{

   tBool bChanged = FALSE;

   if ( _u8UnitID != u8UnitID )
   {
      setUnitID(u8UnitID);
      bChanged = TRUE;
   }

   _rMsgVersion.bValid = TRUE;

   if ( _rMsgVersion.u8Upper != u8UpperVersion )
   {
      _rMsgVersion.u8Upper = u8UpperVersion;
      bChanged = TRUE;
   }

   if ( _rMsgVersion.u8Middle != u8MiddleVersion )
   {
      _rMsgVersion.u8Middle = u8MiddleVersion;
      bChanged = TRUE;
   }

   if ( _rMsgVersion.u8Lower != u8LowerVersion )
   {
      _rMsgVersion.u8Lower = u8LowerVersion;
      bChanged = TRUE;
   }

   if ( bChanged || bForceUpdate )
   {
      // perform update
      _vHandleConfigurationUpdate();
   }
}


// hardware version

tVoid tclMCANInterface::vRequestHardwareVersion( tVoid )
{
   _bTriggerHardwareVersionGetOneTime = TRUE;
   vSetChangedEvent();
}


tVoid tclMCANInterface::vSetHardwareVersion
   (
      tU8 u8UpperVersion, tU8 u8MiddleVersion, tU8 u8LowerVersion, tBool bForceUpdate
   )
{
   tBool bChanged = FALSE;

   if ( TRUE != _rHardwareVersion.bValid )
   {
      _rHardwareVersion.bValid = TRUE;
      bChanged = TRUE;
   }

   if ( _rHardwareVersion.u8Upper != u8UpperVersion )
   {
      _rHardwareVersion.u8Upper = u8UpperVersion;
      bChanged = TRUE;
   }

   if ( _rHardwareVersion.u8Middle != u8MiddleVersion )
   {
      _rHardwareVersion.u8Middle = u8MiddleVersion;
      bChanged = TRUE;
   }

   if ( _rHardwareVersion.u8Lower != u8LowerVersion )
   {
      _rHardwareVersion.u8Lower = u8LowerVersion;
      bChanged = TRUE;
   }

   if ( bChanged || bForceUpdate || _bForceUpdateOfHardwareVersion )
   {
      // perform update
      _vHandleHardwareVersionUpdate();
   }
}


// software version

tVoid tclMCANInterface::vRequestSoftwareVersion( tVoid )
{
   _bTriggerSoftwareVersionGetOneTime = TRUE;
   vSetChangedEvent();
}


tVoid tclMCANInterface::vSetSoftwareVersion
   (
      tU8 u8UpperVersion, tU8 u8MiddleVersion, tU8 u8LowerVersion, tBool bForceUpdate
   )
{

   tBool bChanged = FALSE;

   if ( TRUE != _rSoftwareVersion.bValid )
   {
      _rSoftwareVersion.bValid = TRUE;
      bChanged = TRUE;
   }

   if ( _rSoftwareVersion.u8Upper != u8UpperVersion )
   {
      _rSoftwareVersion.u8Upper = u8UpperVersion;
      bChanged = TRUE;
   }

   if ( _rSoftwareVersion.u8Middle != u8MiddleVersion )
   {
      _rSoftwareVersion.u8Middle = u8MiddleVersion;
      bChanged = TRUE;
   }

   if ( _rSoftwareVersion.u8Lower != u8LowerVersion )
   {
      _rSoftwareVersion.u8Lower = u8LowerVersion;
      bChanged = TRUE;
   }

   if ( bChanged || bForceUpdate || _bForceUpdateOfSoftwareVersion )
   {
      // perform update
      _vHandleSoftwareVersionUpdate();
   }
}


tVoid tclMCANInterface::vAskForVersionInfo()
{
   // At first we ask for hardware version.
   _bTriggerHardwareVersionGet = TRUE;

   // We have to ask for the software version after receiving the hardware version.
   _bAskForSoftwareVersionAfterHardwareVersion = TRUE;

   _u8HardwareVersionGetRetries = 0;
   _u8SoftwareVersionGetRetries = 0;

   vSetChangedEvent();
}


tVoid tclMCANInterface::vStopAskingForVersionInfo()
{
   _bTriggerHardwareVersionGet = FALSE;
   _vSetHardwareVersionGetTimer( 0 );

   _bTriggerSoftwareVersionGet = FALSE;
   _vSetSoftwareVersionGetTimer( 0 );

   _bAskForSoftwareVersionAfterHardwareVersion = FALSE;
}


// functions to get the current data confirmation test result

MCAN_tenTestResult tclMCANInterface::enGetCDataConfTestResult() const
{
   MCAN_tenTestResult enTestResult = MCAN_EN_TEST_RESULT_UNKNOWN;

   if ( EN_DATA_CONF_STATUS_OK == _enCDataConfStatus )
   {
      enTestResult = MCAN_EN_TEST_RESULT_PASSED;
   }
   else if ( EN_DATA_CONF_STATUS_FAULT == _enCDataConfStatus )
   {
      enTestResult = MCAN_EN_TEST_RESULT_FAILED;
   }

   return enTestResult;
}

MCAN_tenTestResult tclMCANInterface::enGetDDataConfTestResult() const
{
   MCAN_tenTestResult enTestResult = MCAN_EN_TEST_RESULT_UNKNOWN;

   if ( EN_DATA_CONF_STATUS_OK == _enDDataConfStatus )
   {
      enTestResult = MCAN_EN_TEST_RESULT_PASSED;
   }
   else if ( EN_DATA_CONF_STATUS_FAULT == _enDDataConfStatus )
   {
      enTestResult = MCAN_EN_TEST_RESULT_FAILED;
   }

   return enTestResult;
}


//
// protected functions
//

// functions for Sink access

tBool tclMCANInterface::bDoCDataReq( const tU8* pcu8Data, tU16 u16DataLength, tU16 u16TimeoutInfo )
{

   WaitingCommand tempCommand;

   tS32 s32OldTimeout;
   tS32 s32NewTimeout;

   tBool bRetVal = FALSE;
   tU8 au8AddressFieldMpdt[2] = { _u8LocalCompID, _u8RemoteCompID };

   if ( _poTxSinkInterface )
   {
      // Extract MsgId info from the byte sequence received
      tU16 Command = (tU16)( (tU16)(*pcu8Data) << 8 ) + ( *(pcu8Data + 1) );
      
      // For every Command sent, corresponding Status reply for the same MsgId is expected from Ext. MCAN Components.
      tU16 u16MsgId = MCAN_EXTRACT_MSG_ID( Command );
      tempCommand.u16Command = MCAN_ASSEMBLE_COMMAND( u16MsgId, MCAN_C_OP_TYPE_STATUS );



      // Are we already waiting for the reply ??
      if ( !_bIsAlreadyWaitingCommand( tempCommand.u16Command ) )
      {
         // If present in the expired list, remove it
         if ( _bIsCommandAlreadyExpired( tempCommand.u16Command ) )
         {
            _vRemoveCommandFromExpiredList( tempCommand.u16Command );
         }

         _u32TsLastCDataReq = OSAL_ClockGetElapsedTime();
         _bInSendingCData   = TRUE;

         // Calculate the new timeout value for this command
         s32NewTimeout = (tS32) u16TimeoutInfo + OSAL_ClockGetElapsedTime();

         tempCommand.s32Timeout = s32NewTimeout;

         // Only if this is not empty
         if ( !_CommandWaitingList.empty() )
         {
            // store the old timeout before adding the new command
            s32OldTimeout = _CommandWaitingList.begin()->s32Timeout;
         }
         else // Put some large value as OldTimeout
         {
            s32OldTimeout = 0x0FFFFFFF;
         }

         // Add the command to the List
         _vAddCommandToWaitingList(tempCommand);

         if ( TX_SINK_NO_ERROR > _poTxSinkInterface->DataReq( NULL,
                                                               TX_SINK_PTYPE_RN_MPDT_C,
                                                               au8AddressFieldMpdt,
                                                               pcu8Data,
                                                               u16DataLength ) )

         {
            // Sending was not successful
            _bInSendingCData = FALSE;
            _u32TsLastCDataReq = MCAN_NO_REQUEST_ACTIVE;


            // Remove the received command from the waiting list
            _CommandWaitingList.erase( tempCommand );
         }
         else
         {
            // Check if the new timer has to be started (only if the new timeout value is less than the already running one)
            if ( s32NewTimeout < s32OldTimeout )
            {
               // Stop the currently running timer and start the new timer because 
               // the new message's timer should expire before the already running status-waiting-timer
               _vSetStatusWaitingTimer( u16TimeoutInfo );
            }

            bRetVal = TRUE;
         }
      }
   }

   return bRetVal;
}

tBool tclMCANInterface::bDoCDataReq( const tU8* pcu8Data, tU16 u16DataLength )
{
   tBool bRetVal = FALSE;
   tU8 au8AddressFieldMpdt[2] = { _u8LocalCompID, _u8RemoteCompID };

   if ( _poTxSinkInterface )
   {
      _bInSendingCData = TRUE;
      _u32TsLastDDataReq = OSAL_ClockGetElapsedTime();

      //ToDo: replace function with the correct sink function
      if ( TX_SINK_NO_ERROR > _poTxSinkInterface->DataReq(
            NULL,
            TX_SINK_PTYPE,
            au8AddressFieldMpdt,
            pcu8Data,
            u16DataLength ) )

      {
         // Sending was not successful
         _bInSendingCData = FALSE;
         _u32TsLastDDataReq = MCAN_NO_REQUEST_ACTIVE;
      }
      else
      {
         bRetVal = TRUE;
      }
   }

   return bRetVal;
}

tBool tclMCANInterface::bDoDDataReq( const tU8* pcu8Data, tU16 u16DataLength )
{
   tBool bRetVal = FALSE;
   tU8 au8AddressFieldMpdt[2] = { _u8LocalCompID, _u8RemoteCompID };

   if ( _poTxSinkInterface )
   {
      _bInSendingDData = TRUE;
      _u32TsLastDDataReq = OSAL_ClockGetElapsedTime();

      //ToDo: replace function with the correct sink function
      if ( TX_SINK_NO_ERROR > _poTxSinkInterface->DataReq(
            NULL,
            0x00,//CSM_C_PTYPE_RN_MPDT_D, /** Jayashree:commented as it is aivi-relavant **/
            au8AddressFieldMpdt,
            pcu8Data,
            u16DataLength ) )

      {
         // Sending was not successful
         _bInSendingDData = FALSE;
         _u32TsLastDDataReq = MCAN_NO_REQUEST_ACTIVE;

      }
      else
      {
         bRetVal = TRUE;
      }
   }

   return bRetVal;
}


// functions used in the Tx Sink callbacks

tVoid tclMCANInterface::vHandleCommunicationCon( tU8 u8ConnectState )
{
   _u32TsLastCommunicationReq = MCAN_NO_REQUEST_ACTIVE;

   if (_bCommunicationRequestBlocked == TRUE) {
       _bCommunicationRequestBlocked = FALSE;

       if (_u8CommunicationRequestBlockedState == TX_SINK_CONNECT) {
           ActivateChannel();
       } else {
           ForceDeactivateChannel();
       }
   }

   if (oConnectionConfigurator.isCommunicationGranted())
   {
      _u8ActualConnectionState = u8ConnectState;

      _vSetCommunicationConEvent();

   }
}

tVoid tclMCANInterface::vHandleCommunicationInd( tU8 u8ConnectState )
{

   if (oConnectionConfigurator.isCommunicationGranted())
   {

      _u8ActualConnectionState = u8ConnectState;

      _vSetCommunicationIndEvent();
   }
}

tVoid tclMCANInterface::vHandleCDataCon( tU8 u8State )
{
   if ( TX_SINK_CONF_OK == u8State )
   {
      if ( EN_DATA_CONF_STATUS_OK != _enCDataConfStatus )
      {
         _enCDataConfStatus = EN_DATA_CONF_STATUS_OK;
         _vTriggerSavingOfCDataConfTestResult();
         // data confirmation status changed -> update the data transmission status
         _vHandleDataTransmissionUpdate();
      }
   }
   else
   {
      if ( EN_DATA_CONF_STATUS_FAULT != _enCDataConfStatus )
      {
         _enCDataConfStatus = EN_DATA_CONF_STATUS_FAULT;
         _vTriggerSavingOfCDataConfTestResult();
         // data confirmation status changed -> update the data transmission status
         _vHandleDataTransmissionUpdate();
      }
   }

   _u32TsLastCDataReq = MCAN_NO_REQUEST_ACTIVE;
   _bInSendingCData = FALSE;

   vSetChangedEvent();
}

tVoid tclMCANInterface::vHandleDDataCon( tU8 u8State )
{
   if ( TX_SINK_CONF_OK == u8State )
   {
      if ( EN_DATA_CONF_STATUS_OK != _enDDataConfStatus )
      {
         _enDDataConfStatus = EN_DATA_CONF_STATUS_OK;
         _vTriggerSavingOfDDataConfTestResult();
         // data confirmation status changed -> update the data transmission status
         _vHandleDataTransmissionUpdate();
      }
   }
   else
   {
      if ( EN_DATA_CONF_STATUS_FAULT != _enDDataConfStatus )
      {
         _enDDataConfStatus = EN_DATA_CONF_STATUS_FAULT;
         _vTriggerSavingOfDDataConfTestResult();
         // data confirmation status changed -> update the data transmission status
         _vHandleDataTransmissionUpdate();
      }
   }

   _u32TsLastDDataReq = MCAN_NO_REQUEST_ACTIVE;
   _bInSendingDData = FALSE;

   vSetChangedEvent();
}

tU16 tclMCANInterface::getCommandFromMessagePayload(tU8* pu8Data)
{
   tU16 Command = (tU16)( (tU16)(*pu8Data) << 8 ) + ( *(pu8Data + 1) );
#if defined (__METER_UNIT_TESTING__)
   printf("Here I am:Payload[0]: 0x%x, Payload[1]: 0x%x\n", (tU16)(*pu8Data), *(pu8Data+1));
   printf("Here I am:Payload[0]: <<8 0x%x\n", (tU16)( (tU16)(*pu8Data) << 8 ));
   printf("Here I am:Command: 0x%x\n", Command);
#endif
   return(Command);
}

tVoid tclMCANInterface::vHandleDataInd( tU8* pu8Data, tU16 u16Length )
{
   if ( pu8Data )
   {
      tU16 u16Command;
      tU16 u16MsgId;
      tU8  u8OpTypeCode;
      tU16 u16DataLength = 0;

      u16Command = getCommandFromMessagePayload(pu8Data);

      u16MsgId     = MCAN_EXTRACT_MSG_ID( u16Command );
      u8OpTypeCode = MCAN_EXTRACT_OP_TYPE( u16Command );

      if ( u16Length > 2 )
      {
         u16DataLength = u16Length - 2;
         vTraceIncomingMessage( u16MsgId, u8OpTypeCode, pu8Data + 2, u16DataLength );
      }
      else
      {
         vTraceIncomingMessage( u16MsgId, u8OpTypeCode, NULL, 0 );
      }

      // Are we waiting for this Status-message reply ??
      if ( _bIsAlreadyWaitingCommand( u16Command ) )
      {
         // Remove the command from the list and eventually start the next timer.
         _vHandleReceivedCommand( u16Command );
      }

      switch ( u16MsgId )
      {
         case MCAN_C_BASE_MSG_CONFIGURATION:
         {
            _vHandleMsgConfiguration( u8OpTypeCode, *(pu8Data + 2), *(pu8Data + 3), *(pu8Data + 4), *(pu8Data + 5) );
         }
         break;

         case MCAN_C_BASE_MSG_HARDWARE_VERSION:
         {
            _vHandleMsgHardwareVersion( u8OpTypeCode, *(pu8Data + 2), *(pu8Data + 3), *(pu8Data + 4) );
            vHandleUnitSpecificDataInd( u16MsgId, u8OpTypeCode, pu8Data + 2, u16DataLength );
         }
         break;

         case MCAN_C_BASE_MSG_SOFTWARE_VERSION:
         {
            _vHandleMsgSoftwareVersion( u8OpTypeCode, *(pu8Data + 2), *(pu8Data + 3), *(pu8Data + 4) );
            vHandleUnitSpecificDataInd( u16MsgId, u8OpTypeCode, pu8Data + 2, u16DataLength );
         }
         break;

         default:
         {
            // Only if the command is not expired already, send it to MCAN units to handle it further
            if ( !_bIsCommandAlreadyExpired( u16Command ) )
            {
               vHandleUnitSpecificDataInd( u16MsgId, u8OpTypeCode, pu8Data + 2, u16DataLength );
            }
            else // else ignore the expired command
            {
               // Trace the Info
               vTraceCommandInfo( MCAN_EN_COMMAND_RECEIVED_IGNORED, u16Command );
            }
         }
         break;
      }
   }
}


// functions to set events evaluated by worker thread

tVoid tclMCANInterface::vSetChangedEvent()
{

#if defined (__METER_UNIT_TESTING__)
   printf("vSetChangedEvent real function called\n");
#endif
   if ( OSAL_C_INVALID_HANDLE != _hWorkerEvents )
   {
      OSAL_s32EventPost( _hWorkerEvents, MCAN_EV_CHANGED, OSAL_EN_EVENTMASK_OR );
   }
}


tVoid tclMCANInterface::vSetUnitSpecificEvent( tU16 u16Event )
{
   if ( OSAL_C_INVALID_HANDLE != _hWorkerEvents )
   {
      OSAL_s32EventPost( _hWorkerEvents, (tU32)u16Event, OSAL_EN_EVENTMASK_OR );
   }
}



//
// private functions
//

// general worker thread function, calls the real implementation function

tVoid tclMCANInterface::_vWorkerThreadFunction( tVoid *pvArg )
{
   tclMCANInterface *poMyself = (tclMCANInterface *) pvArg;

   if ( poMyself )
   {
      poMyself->_vWorkerThreadFunctionImplementation();
   }
}
tVoid tclMCANInterface::ApplicationCallbackInit ( )
{

   tU8 au8AddressFieldMpdt[2] =
      { _u8LocalCompID, _u8RemoteCompID };

   if (_poTxSinkInterface && _pvCallbacks)
   {
      _poTxSinkInterface->ApplCallbackInit( NULL, 0x00/*CSM_C_PTYPE_RN_MPDT_C Jayashree:commented as it is aivi-relavant **/, au8AddressFieldMpdt, _pvCallbacks);
   }
}

// real implementation of gthe worker thread function

tVoid tclMCANInterface::_vWorkerThreadFunctionImplementation()
{
   OSAL_tEventMask hEvRequest = 0;
   tU32 u32mask = MCAN_EV_BASE_MASK | _u16UnitSpecificEventMask;

   ApplicationCallbackInit();

   // trigger the first changed event for possible communication request
   vSetChangedEvent();
#if defined (__METER_UNIT_TESTING__)
   printf("Test While:\n");
#endif

   // main input loop
   // wait for new Message in the InputQueue (no busy waiting)
   while ( FALSE == _bTerminateWorkerThread )
   {
#if defined (__METER_UNIT_TESTING__)
   printf("Test4 While in:\n");
#endif

      if ( OSAL_OK == OSAL_s32EventWait
                         (
                            _hWorkerEvents,
                            u32mask,
                            OSAL_EN_EVENTMASK_OR,
                            OSAL_C_TIMEOUT_FOREVER,
                            &hEvRequest
                         )
         )
      {
         // Take sem :
#if !defined (__METER_UNIT_TESTING__)
         //ahl_bEnterCritical( _hWorkerSem );
#endif
         if ( OSAL_OK != OSAL_s32EventPost( _hWorkerEvents, ~hEvRequest, OSAL_EN_EVENTMASK_AND ) )
         {
            vTraceError( MCAN_EN_ERROR_CODE_ACCESS_WORKER_EVENTS );
         }

         if ( MCAN_EV_COMMUNICATION_CON & hEvRequest )
         {
            hEvRequest = hEvRequest & ~(MCAN_EV_COMMUNICATION_CON);

            _vEvaluateConnectionState( FALSE );
         }

         if ( MCAN_EV_COMMUNICATION_IND & hEvRequest )
         {
            hEvRequest = hEvRequest & ~(MCAN_EV_COMMUNICATION_IND);

            _vEvaluateConnectionState( TRUE );
         }

         if ( MCAN_EV_CHANGED & hEvRequest )
         {
            hEvRequest = hEvRequest & ~(MCAN_EV_CHANGED);

            _vCheckForCommunicationRequest();

            _vCheckForConfiguration();

            if ( _bInitializationComplete() )
            {
               _vCheckForVersionInfo();

               vCheckForUnitSpecificDataRequests();
            }

            _vCheckForSendingBasicInfo();

            vCheckForSendingUnitSpecificInfo();
         }

         hEvRequest = hHandleUnitSpecificEvents( hEvRequest );

         hEvRequest = _hHandleBasicEvents( hEvRequest );

         // Release Sem :
#if !defined (__METER_UNIT_TESTING__)
         //ahl_bReleaseCritical( _hWorkerSem );
#endif

      } // Event wait

   } // while ( FALSE == _bTerminateWorkerThread )
}


// setup and start of the worker thread

tBool tclMCANInterface::_bWorkerThreadSetup( tS32 s32StackSize, tU32 u32ThreadPrio )
{
   tBool bRetVal = TRUE;

   tChar szResourceName[OSAL_C_U32_MAX_NAMELENGTH];
   OSAL_trThreadAttribute  rAttr;

   // create my events
   OSALUTIL_s32SaveNPrintFormat( szResourceName, OSAL_C_U32_MAX_NAMELENGTH, "%s_%d_%d", MCAN_WORKER_EVENT_BASE_NAME, _u8LocalCompID, _u8RemoteCompID );

   if ( OSAL_ERROR == OSAL_s32EventCreate( szResourceName, &_hWorkerEvents ) )
   {
      vTraceError( MCAN_EN_ERROR_CODE_CREATE_WORKER_EVENTS );
      bRetVal = FALSE;
   }

   // create my thread access semaphore
   OSALUTIL_s32SaveNPrintFormat( szResourceName, OSAL_C_U32_MAX_NAMELENGTH, "%s_%d_%d", MCAN_WORKER_ACCESS_SEM_BASE_NAME, _u8LocalCompID, _u8RemoteCompID );

   if ( OSAL_OK != OSAL_s32SemaphoreCreate( szResourceName, &_hWorkerSem, 1 ) )
   {
      vTraceError( MCAN_EN_ERROR_CODE_CREATE_WORKER_ACCESS_SEMAPHORE );
      bRetVal = FALSE;
   }

   // try spawning the thread only if all preparation was successful
   if ( TRUE == bRetVal )
   {
      // configure the thread with the read values
      OSALUTIL_s32SaveNPrintFormat( szResourceName, OSAL_C_U32_MAX_NAMELENGTH, "%s_%d_%d", MCAN_WORKER_THREAD_BASE_NAME, _u8LocalCompID, _u8RemoteCompID );

      rAttr.szName       = szResourceName;
      rAttr.s32StackSize = s32StackSize;
      rAttr.u32Priority  = u32ThreadPrio;
      rAttr.pfEntry      = (OSAL_tpfThreadEntry)_vWorkerThreadFunction;
      rAttr.pvArg        = (tPVoid)this;

      _hWorkerThreadId = OSAL_ThreadSpawn( &rAttr );

      // if our handle cannot be used
      if ( OSAL_ERROR == _hWorkerThreadId )
      {
         vTraceError( MCAN_EN_ERROR_CODE_SPAWN_WORKER_THREAD );

         NORMAL_M_ASSERT_ALWAYS();

         // for release give return value false
         bRetVal = FALSE;
      }
   }

   // keep in mind : TRUE means the task runs properly, false means some POS.
   return bRetVal;
}


// termination and deletion of the worker thread and its resources

tVoid tclMCANInterface::_vWorkerThreadDelete()
{
   tChar szResourceName[OSAL_C_U32_MAX_NAMELENGTH];

   _bTerminateWorkerThread = TRUE;

   /* +++
   free all allocated ressources
   +++ */

   if ( OSAL_OK != OSAL_s32ThreadDelete( _hWorkerThreadId ) )
   {
      vTraceError( MCAN_EN_ERROR_CODE_DELETE_WORKER_THREAD );
   }

   if ( OSAL_OK != OSAL_s32SemaphoreClose( _hWorkerSem ) )
   {
      vTraceError( MCAN_EN_ERROR_CODE_CLOSE_WORKER_ACCESS_SEMAPHORE );
   }

   OSALUTIL_s32SaveNPrintFormat( szResourceName, OSAL_C_U32_MAX_NAMELENGTH, "%s_%d_%d", MCAN_WORKER_ACCESS_SEM_BASE_NAME, _u8LocalCompID, _u8RemoteCompID );

   if ( OSAL_OK != OSAL_s32SemaphoreDelete( szResourceName ) )
   {
      vTraceError( MCAN_EN_ERROR_CODE_DELETE_WORKER_ACCESS_SEMAPHORE );
   }

   if ( OSAL_OK != OSAL_s32EventClose( _hWorkerEvents ) )
   {
      vTraceError( MCAN_EN_ERROR_CODE_CLOSE_WORKER_EVENTS );
   }

   OSALUTIL_s32SaveNPrintFormat( szResourceName, OSAL_C_U32_MAX_NAMELENGTH, "%s_%d_%d", MCAN_WORKER_EVENT_BASE_NAME, _u8LocalCompID, _u8RemoteCompID );

   if ( OSAL_OK == OSAL_s32EventDelete( szResourceName ) )
   {
      _hWorkerEvents = OSAL_C_INVALID_HANDLE;
   }
   else
   {
      vTraceError( MCAN_EN_ERROR_CODE_DELETE_WORKER_EVENTS );
   }
}



// functions to set a basic event evaluated by worker thread

tVoid tclMCANInterface::_vSetCommunicationConEvent()
{
   if ( OSAL_C_INVALID_HANDLE != _hWorkerEvents )
   {
      OSAL_s32EventPost( _hWorkerEvents, MCAN_EV_COMMUNICATION_CON, OSAL_EN_EVENTMASK_OR );
   }
}

tVoid tclMCANInterface::_vSetCommunicationIndEvent()
{
   if ( OSAL_C_INVALID_HANDLE != _hWorkerEvents )
   {
     OSAL_s32EventPost( _hWorkerEvents, MCAN_EV_COMMUNICATION_IND, OSAL_EN_EVENTMASK_OR );
   }
}

tVoid tclMCANInterface::_vSetCommunicationTestTimerEvent()
{
   if ( OSAL_C_INVALID_HANDLE != _hWorkerEvents )
   {
     OSAL_s32EventPost( _hWorkerEvents, MCAN_EV_COMMUNICATION_TEST_TIMER, OSAL_EN_EVENTMASK_OR );
   }
}

tVoid tclMCANInterface::_vSetConfigurationGetTimerEvent()
{

   if ( OSAL_C_INVALID_HANDLE != _hWorkerEvents )
   {
      OSAL_s32EventPost( _hWorkerEvents, MCAN_EV_CONFIGURATION_GET_TIMER, OSAL_EN_EVENTMASK_OR );
   }
}

tVoid tclMCANInterface::_vSetHardwareVersionGetTimerEvent()
{
   if ( OSAL_C_INVALID_HANDLE != _hWorkerEvents )
   {
      OSAL_s32EventPost( _hWorkerEvents, MCAN_EV_HARDWARE_VERSION_GET_TIMER, OSAL_EN_EVENTMASK_OR );
   }
}

tVoid tclMCANInterface::_vSetSoftwareVersionGetTimerEvent()
{
   if ( OSAL_C_INVALID_HANDLE != _hWorkerEvents )
   {
     OSAL_s32EventPost( _hWorkerEvents, MCAN_EV_SOFTWARE_VERSION_GET_TIMER, OSAL_EN_EVENTMASK_OR );
   }
}

tVoid tclMCANInterface::_vSetStatusWaitingTimerEvent()
{
   if ( OSAL_C_INVALID_HANDLE != _hWorkerEvents )
   {
     OSAL_s32EventPost( _hWorkerEvents, MCAN_EV_STATUS_WAITING_TIMER, OSAL_EN_EVENTMASK_OR );
   }
}

tVoid tclMCANInterface::setParameterAndTriggerForFurtherActionsOnDisconnectedChannel()
{
    // communication disconnected
    vSetConnection( FALSE, FALSE );

    _u8InitializationStatus = 0x00;

    oConfigurationGetMessage.forgetAboutMessageWasSendToSink();
    oConfigurationStatusMessage.forgetAboutMessageWasReceivedFromSink();

    vDoActionsWhenInitializationIncomplete();

}

///
/// connection confirmation will be sent also in case of trying to connect, but channel is disconnected. The confirmation gives the actual state back, but requested state.
/// we want to wait for timeout, before reporting an error
tVoid tclMCANInterface::handleConnectionTimerInCaseConnectionRequstIsPendingAndDisonnectConfirmationReceived(tBool CommunicationIndicationReceived)
{
    if ((MCAN_EN_COMM_STATUS_CONNECTION_WAIT == _enCommunicationStatus)
            &&
            ( FALSE == CommunicationIndicationReceived ) // confirmation received
            )
    {
        // start timer to wait for connection. It might happen, that channel is getting connected afterwards. The confirmation mean only, that channel was not connected as we tried to connect it.
        _vSetCommunicationTestTimer(MCAN_C_COMMUNICATION_TEST_TIMER_VALUE_FOR_CONNECTION_WAIT_AFTER_ERROR_IN_MS);
    }
    else
    {
        // stop a possibly running communication test timer, in case we don't wait for connection.
        _vDoCommunicationTestTimerHandling();
    }
}

// function to evaluate the current connection state
tVoid tclMCANInterface::_vEvaluateConnectionState( tBool bInfoFromCommunicationIndication )
{
   if ( _u8ActualConnectionState == TX_SINK_CONNECTED )
   {
      vTraceConnectionStateConnected();

      _enCommTestResult = MCAN_EN_TEST_RESULT_PASSED;

      if ( MCAN_EN_COMM_STATUS_CONNECTED != _enCommunicationStatus )
      {
         // This is the first connection or a re-connection after connection error 
         // -> we have to send a test result message
         _enCommunicationStatus = MCAN_EN_COMM_STATUS_CONNECTED;
         vTraceCommunicationStatus(_enCommunicationStatus);
         _vTriggerSavingOfCommTestResult();
         // communication status changed -> update the connection status
         _vHandleConnectionUpdate();
      }

      // stop a possibly running communication test timer
      _vDoCommunicationTestTimerHandling();

      _u8InitializationStatus = MCAN_C_INIT_STATUS_CONNECTED;
      vDoActionsWhenInitializationIncomplete();// ToDo: evtl nur aufrufen, wenn tats?chlich configuration.set aufgerufen wurden. Oder das raus, wenn gefakt wird.

      vSetConnection( TRUE, FALSE );

      if ( AMT_C_U32_STATE_NORMAL == _u32RequestedAppState )
      {
         if (FALSE == oConfigurationGetMessage.isMessageBeenSentToSink())
         {
            _bTriggerConfigurationGet = TRUE;   // send Configuration get
            vSetChangedEvent();
         }
         else
         {
            // the configuration.get was sent in a cyle before. So we must fake the answer.
            vTraceConfigurationGet_NotSent();
            fakeIncomingMsgConfigurationStatus();
         }
      }
   }
   else
   {
      vTraceConnectionStateDisconnected( bInfoFromCommunicationIndication );

      if (
            ( MCAN_EN_COMM_STATUS_CONNECTED == _enCommunicationStatus )
            ||
            (
               ( TRUE == bInfoFromCommunicationIndication )
               &&
               ( MCAN_EN_COMM_STATUS_CONNECTION_WAIT == _enCommunicationStatus )
            )
         )
      {
         // an active communication is interrupted -> this is a connection error
         if ( AMT_C_U32_STATE_NORMAL == _u32RequestedAppState )
         {
            _enCommunicationStatus = MCAN_EN_COMM_STATUS_CONNECTION_ERROR;
            vTraceCommunicationStatus(_enCommunicationStatus);
            _enCommTestResult = MCAN_EN_TEST_RESULT_FAILED;
            _vTriggerSavingOfCommTestResult();
         }
         else
         {
            // we can expect to be disconnected
            _enCommunicationStatus = MCAN_EN_COMM_STATUS_NOT_CONNECTED;
            vTraceCommunicationStatus(_enCommunicationStatus);
            _enCommTestResult = MCAN_EN_TEST_RESULT_UNKNOWN;
         }
         // communication status changed -> update the connection status
         _vHandleConnectionUpdate();
      }

      handleConnectionTimerInCaseConnectionRequstIsPendingAndDisonnectConfirmationReceived(bInfoFromCommunicationIndication);
      setParameterAndTriggerForFurtherActionsOnDisconnectedChannel();
   }
}

tVoid tclMCANInterface::traceCommunicationRequest(tU8 Request)
{
   switch (Request)
   {
      case TX_SINK_CONNECT:
      {
         vTraceCommunicationRequestConnect();
         break;
      }
      case TX_SINK_DISCONNECTED:
      {
         vTraceCommunicationRequestDisconnect();
         break;

      }
      default:
      {
         break;
      }
   }
}

/**
 * The corresponding MCAN device will be called telling it, that the head unit want to start or stop the communication.
 * @param Request
 */
tVoid tclMCANInterface::sendCommunicationRequestToSink(tU8 Request)
{
   tU8 au8AddressFieldMpdt[2] = { _u8LocalCompID, _u8RemoteCompID };

   if (_poTxSinkInterface) {

       traceCommunicationRequest(Request);
       _u32TsLastCommunicationReq = OSAL_ClockGetElapsedTime();


       _poTxSinkInterface->CommunicationReq( NULL,
               TX_SINK_BUS_MCAN,
               TX_SINK_PTYPE_RN_MPDT_C, au8AddressFieldMpdt, Request,
               TX_SINK_APPL_ID_DEFAULT);
   } else {
       _u32TsLastCommunicationReq == MCAN_NO_REQUEST_ACTIVE;
   }
}
// funcions used in "event changed" branch of working thread

tVoid tclMCANInterface::_vCheckForCommunicationRequest()
{

   if (_bTriggerCommunicationRequestConnect)
   {
      _bTriggerCommunicationRequestConnect = FALSE;
      sendCommunicationRequestToSink(TX_SINK_CONNECT);
   }

   if (_bTriggerCommunicationRequestDisconnect)
   {
      _bTriggerCommunicationRequestDisconnect = FALSE;
      sendCommunicationRequestToSink(TX_SINK_DISCONNECTED);
   }
}

tVoid tclMCANInterface::sendConfigurationGetViaMCAN()
{
   tU16 u16Command = MCAN_ASSEMBLE_COMMAND( MCAN_C_BASE_MSG_CONFIGURATION, MCAN_C_OP_TYPE_GET );

   tU8  au8Data[3];
   tU16 u16DataLength;
   au8Data[0] = TX_SINK_GET_HIBYTE( u16Command );
   au8Data[1] = TX_SINK_GET_LOBYTE( u16Command );

   if ( _bUseAvailableFunctionForConfiguration )
   {
      au8Data[2] = _u8AvailableFunctionForConfiguration;
      u16DataLength = 3;
   }
   else
   {
      u16DataLength = 2;
   }

   vTraceSendingConfigurationGet();

   bDoCDataReq( au8Data, u16DataLength );
   oConfigurationGetMessage.rememberMessageWasSendToSink();
}
tVoid tclMCANInterface::_vCheckForConfiguration()
{
   if (
         ( _u8ActualConnectionState == TX_SINK_CONNECTED )
         &&
         ( ( _u8InitializationStatus & MCAN_C_INIT_STATUS_CONNECTED ) == MCAN_C_INIT_STATUS_CONNECTED )
      )
   {
      if ( ( bIsAllowedToSendCData() ) && ( _bTriggerConfigurationGet ) )
      {

         _bTriggerConfigurationGet = FALSE;

         // start the configuration get timer
         _vSetConfigurationGetTimer( MCAN_C_CONFIGURATION_GET_TIMER_VALUE_IN_MS );

         sendConfigurationGetViaMCAN();
      }

      // ToDo : Always the bIsAllowedToSendCData is failing. Check why it is failing...!!
      if ( ( bIsAllowedToSendCData() ) && ( _bTriggerConfigurationStatus ) )
      //if ( ( _bTriggerConfigurationStatus ) )
      {
         tU16 u16Command = MCAN_ASSEMBLE_COMMAND( MCAN_C_BASE_MSG_CONFIGURATION, MCAN_C_OP_TYPE_STATUS );

         tU8 au8Data[3];
         tU16 u16DataLength;

         _bTriggerConfigurationStatus = FALSE;

         au8Data[0] = TX_SINK_GET_HIBYTE( u16Command );
         au8Data[1] = TX_SINK_GET_LOBYTE( u16Command );

         if ( _bUseAvailableFunctionForConfiguration )
         {
            au8Data[2] = _u8AvailableFunctionForConfiguration;
            u16DataLength = 3;
         }
         else
         {
            u16DataLength = 2;
         }

         vTraceSendingConfigurationStatus();

         if ( bDoCDataReq( au8Data, u16DataLength ) )
         {
            // stop the configuration get timer
            _vSetConfigurationGetTimer( 0 );

            // set flag "Configuration Complete" if not already set
            if ( !_bInitializationComplete() )
            {
               _u8InitializationStatus |= MCAN_C_INIT_STATUS_CONFIGURATION_COMPLETE;

               vDoActionsAfterInitializationComplete();
			   oConnectionConfigurator.connectionEstablished();

               // Clear all the list data valid for the current ignition cycle
               _CommandWaitingList.clear();
               _ExpiredCommandList.clear();
            }
         }
      }
   } // if connected
}

tBool tclMCANInterface::_bInitializationComplete() const
{
   if (
         ( _u8ActualConnectionState == TX_SINK_CONNECTED )
         &&
         (
            ( _u8InitializationStatus & ( MCAN_C_INIT_STATUS_CONNECTED | MCAN_C_INIT_STATUS_CONFIGURATION_COMPLETE ) )
            ==
            ( MCAN_C_INIT_STATUS_CONNECTED | MCAN_C_INIT_STATUS_CONFIGURATION_COMPLETE )
         )
      )
   {
      return TRUE;
   }
   else
   {
      return FALSE;
   }
}

tVoid tclMCANInterface::_vCheckForVersionInfo()
{
	if ( ( bIsAllowedToSendCData() ) && ( _bTriggerHardwareVersionGet || _bTriggerHardwareVersionGetOneTime ) )
	//if ( ( _bTriggerHardwareVersionGet || _bTriggerHardwareVersionGetOneTime ) )
   {
      tU16 u16Command = MCAN_ASSEMBLE_COMMAND( MCAN_C_BASE_MSG_HARDWARE_VERSION, MCAN_C_OP_TYPE_GET );

      tU8 au8Data[2];

      if ( _bTriggerHardwareVersionGetOneTime )
      {
         _bForceUpdateOfHardwareVersion = TRUE;
      }
      else
      {
         // Otherwise we have to increase the retry counter.
         _u8HardwareVersionGetRetries++;
      }

      _bTriggerHardwareVersionGet = FALSE;
      _bTriggerHardwareVersionGetOneTime = FALSE;

      au8Data[0] = TX_SINK_GET_HIBYTE( u16Command );
      au8Data[1] = TX_SINK_GET_LOBYTE( u16Command );

      // start the hardware version get timer
      _vSetHardwareVersionGetTimer( MCAN_C_HARDWARE_VERSION_GET_TIMER_VALUE_IN_MS );

      vTraceSendingHardwareVersionGet();

      bDoCDataReq( au8Data, 2 );
   }

	//ToDo : bIsAllowedToSendCData is commented now...!! Dont know who will make it true. Check Why the condition is failing.
   if ( ( bIsAllowedToSendCData() ) && ( _bTriggerSoftwareVersionGet || _bTriggerSoftwareVersionGetOneTime ) )
	//if ( ( _bTriggerSoftwareVersionGet || _bTriggerSoftwareVersionGetOneTime ) )
   {
      tU16 u16Command = MCAN_ASSEMBLE_COMMAND( MCAN_C_BASE_MSG_SOFTWARE_VERSION, MCAN_C_OP_TYPE_GET );

      tU8 au8Data[2];

      if ( _bTriggerSoftwareVersionGetOneTime )
      {
         // This is triggered by an explicit one-time request
         // Therefore, we have to force an update at the latest after the timeout.
         _bForceUpdateOfSoftwareVersion = TRUE;
      }
      else
      {
         // Otherwise we have to increase the retry counter.
         _u8SoftwareVersionGetRetries++;
      }

      _bTriggerSoftwareVersionGet = FALSE;
      _bTriggerSoftwareVersionGetOneTime = FALSE;

      au8Data[0] = TX_SINK_GET_HIBYTE( u16Command );
      au8Data[1] = TX_SINK_GET_LOBYTE( u16Command );

      // start the software version get timer
      _vSetSoftwareVersionGetTimer( MCAN_C_SOFTWARE_VERSION_GET_TIMER_VALUE_IN_MS );

      vTraceSendingSoftwareVersionGet();

      bDoCDataReq( au8Data, 2 );
   }
}


tVoid tclMCANInterface::_vCheckForSendingBasicInfo()
{
   if ( _bTriggerSendConnection )
   {
      _bTriggerSendConnection = FALSE;
      vSendConnection();
   }

   if ( _bTriggerSendConfiguration )
   {
      _bTriggerSendConfiguration = FALSE;
      vSendConfiguration();
   }

   if ( _bTriggerSendHardwareVersion )
   {
      _bTriggerSendHardwareVersion = FALSE;
      vSendHardwareVersion();
   }

   if ( _bTriggerSendSoftwareVersion )
   {
      _bTriggerSendSoftwareVersion = FALSE;
      vSendSoftwareVersion();
   }

   if ( _bTriggerSendDataTransmission )
   {
      _bTriggerSendDataTransmission = FALSE;
      vSendDataTransmission();
   }


   if ( oSendCommTestResult.isSendingOfMessageAllowed())
   {
      oSendCommTestResult.consumeSendingTrigger();
      vSendCurrentCommTestResult();
   }


   if ((oSendCommTestResult.isSendingOfMessageAllowed())
         && (_bTriggerSendCDataConfTestResult))
   {
      _bTriggerSendCDataConfTestResult = FALSE;
      vSendCurrentCDataConfTestResult();
   }

   if ((oSendCommTestResult.isSendingOfMessageAllowed())
         && (_bTriggerSendDDataConfTestResult))
   {
      _bTriggerSendDDataConfTestResult = FALSE;
      vSendCurrentDDataConfTestResult();
   }
}


// function used to evaluate several basic events of the working thread

OSAL_tEventMask tclMCANInterface::_hHandleBasicEvents( OSAL_tEventMask hEvents )
{
   OSAL_tEventMask hRetVal = hEvents;

   if ( MCAN_EV_COMMUNICATION_TEST_TIMER & hRetVal )
   {
      hRetVal = hRetVal & ~(MCAN_EV_COMMUNICATION_TEST_TIMER);

      vTraceBasicTimerEvent( MCAN_EN_BASIC_TIMER_COMMUNICATION_TEST );

      if ( AMT_C_U32_STATE_NORMAL == _u32RequestedAppState )
      {
         vTraceCommunicationStatus(_enCommunicationStatus);
         if (( MCAN_EN_COMM_STATUS_CONNECTION_WAIT_AFTER_ERROR == _enCommunicationStatus )
             ||
             (MCAN_EN_COMM_STATUS_CONNECTION_WAIT == _enCommunicationStatus)
             )

         {
            _enCommunicationStatus = MCAN_EN_COMM_STATUS_CONNECTION_ERROR;
            vTraceCommunicationStatus(_enCommunicationStatus);
            _enCommTestResult = MCAN_EN_TEST_RESULT_FAILED;
            _vTriggerSavingOfCommTestResult();
            // communication status changed -> update the connection status
            _vHandleConnectionUpdate();
         }
      }
      else
      {
         // we can expect to be disconnected
         _enCommTestResult = MCAN_EN_TEST_RESULT_UNKNOWN;
         _enCommunicationStatus = MCAN_EN_COMM_STATUS_NOT_CONNECTED;
         vTraceCommunicationStatus(_enCommunicationStatus);
         // communication status changed -> update the connection status
         _vHandleConnectionUpdate();
      }
   } // if ( MCAN_EV_COMMUNICATION_TEST_TIMER & hRetVal )

   if ( MCAN_EV_CONFIGURATION_GET_TIMER & hRetVal )
   {
      hRetVal = hRetVal & ~(MCAN_EV_CONFIGURATION_GET_TIMER);

      vTraceBasicTimerEvent( MCAN_EN_BASIC_TIMER_CONFIGURATION_GET );

      _bTriggerConfigurationGet = TRUE;
      vSetChangedEvent();
   } // if ( MCAN_EV_CONFIGURATION_GET_TIMER & hRetVal )

   if ( MCAN_EV_HARDWARE_VERSION_GET_TIMER & hRetVal )
   {
     hRetVal = hRetVal & ~(MCAN_EV_HARDWARE_VERSION_GET_TIMER);

      vTraceBasicTimerEvent( MCAN_EN_BASIC_TIMER_HARDWARE_VERSION_GET );

      if ( _bForceUpdateOfHardwareVersion )
      {
         _vHandleHardwareVersionUpdate();
      }
      else if ( _u8HardwareVersionGetRetries < MCAN_C_HARDWARE_VERSION_GET_MAX_RETRIES )
      {
         _bTriggerHardwareVersionGet = TRUE;
         vSetChangedEvent();
      }
      else if ( _bAskForSoftwareVersionAfterHardwareVersion )
      {
         _bAskForSoftwareVersionAfterHardwareVersion = FALSE;
         _bTriggerSoftwareVersionGet = TRUE;
         vSetChangedEvent();
      }
   } // if ( MCAN_EV_HARDWARE_VERSION_GET_TIMER & hRetVal )

   if ( MCAN_EV_SOFTWARE_VERSION_GET_TIMER & hRetVal )
   {
      hRetVal = hRetVal & ~(MCAN_EV_SOFTWARE_VERSION_GET_TIMER);

      vTraceBasicTimerEvent( MCAN_EN_BASIC_TIMER_SOFTWARE_VERSION_GET );

      if ( _bForceUpdateOfSoftwareVersion )
      {
         _vHandleSoftwareVersionUpdate();
      }
      else if ( _u8SoftwareVersionGetRetries < MCAN_C_SOFTWARE_VERSION_GET_MAX_RETRIES )
      {
         _bTriggerSoftwareVersionGet = TRUE;
         vSetChangedEvent();
      }
   } // if ( MCAN_EV_SOFTWARE_VERSION_GET_TIMER & hRetVal )

   if ( MCAN_EV_STATUS_WAITING_TIMER & hRetVal )
   {
      hRetVal = hRetVal & ~(MCAN_EV_STATUS_WAITING_TIMER);

      vTraceBasicTimerEvent( MCAN_EN_BASIC_TIMER_STATUS_WAITING );

      _vHandleStatusWaitingTimerTimeout();
   } // if ( MCAN_EV_STATUS_WAITING_TIMER & hRetVal )

   return hRetVal;
}


// function checking whether the communication test timer has to be started or stopped

tVoid tclMCANInterface::_vDoCommunicationTestTimerHandling()
{
  OSAL_tMSecond msTimeout = 0;

   ///
   /// Check whether error shall be reported.
   if ( FALSE == bErrorMaskingActive() )
   {
       /// diagnosis still wants errors to be reported. So decide here whether something is wrong once timer expires
       switch (_enCommunicationStatus)
       {
       case MCAN_EN_COMM_STATUS_CONNECTION_ERROR:
       {
           _enCommunicationStatus = MCAN_EN_COMM_STATUS_CONNECTION_WAIT_AFTER_ERROR;
           vTraceCommunicationStatus(_enCommunicationStatus);
           _vHandleConnectionUpdate();
           _vSetCommunicationTestTimer(MCAN_C_COMMUNICATION_TEST_TIMER_VALUE_FOR_CONNECTION_WAIT_AFTER_ERROR_IN_MS);
           break;
       }
       default:
       {
           // stop timer
           _vSetCommunicationTestTimer( msTimeout );
       }
       }
   }
}

// function to trigger the saving of the communication test result

tVoid tclMCANInterface::_vTriggerSavingOfCommTestResult()
{
   oSendCommTestResult.setSendingTrigger();  // set trigger flag
   vSetChangedEvent();                  // trigger worker thread
}


// function to trigger the saving of the data confirmation test result

tVoid tclMCANInterface::_vTriggerSavingOfCDataConfTestResult()
{
   _bTriggerSendCDataConfTestResult = TRUE;  // set trigger flag
   vSetChangedEvent();                       // trigger worker thread
}


// function to trigger the saving of the data confirmation test result

tVoid tclMCANInterface::_vTriggerSavingOfDDataConfTestResult()
{
   _bTriggerSendDDataConfTestResult = TRUE;  // set trigger flag
   vSetChangedEvent();                       // trigger worker thread
}


// update functions called from data pool to send data to registered clients

tVoid tclMCANInterface::_vHandleConnectionUpdate()
{
   _bTriggerSendConnection = TRUE;       // set trigger flag
   vSetChangedEvent();                   // trigger worker thread
}

tVoid tclMCANInterface::_vHandleConfigurationUpdate()
{
   _bTriggerSendConfiguration = TRUE;    // set trigger flag
   vSetChangedEvent();                   // trigger worker thread
}

tVoid tclMCANInterface::_vHandleHardwareVersionUpdate()
{
   _bForceUpdateOfHardwareVersion = FALSE;

   _bTriggerSendHardwareVersion = TRUE;  // set trigger flag
   vSetChangedEvent();                   // trigger worker thread
}

tVoid tclMCANInterface::_vHandleSoftwareVersionUpdate()
{
   _bForceUpdateOfSoftwareVersion = FALSE;

   _bTriggerSendSoftwareVersion = TRUE;  // set trigger flag
   vSetChangedEvent();                   // trigger worker thread
}

tVoid tclMCANInterface::_vHandleDataTransmissionUpdate()
{
   _bTriggerSendDataTransmission = TRUE; // set trigger flag
   vSetChangedEvent();                   // trigger worker thread
}


tVoid tclMCANInterface::takeIncomingConfigurationStatusIntoAccount()
{
   // stop the configuration get timer
   _vSetConfigurationGetTimer(0);

   // set flag "Configuration Complete" if not already set
   if (!_bInitializationComplete())
   {
      _u8InitializationStatus |= MCAN_C_INIT_STATUS_CONFIGURATION_COMPLETE;
      vDoActionsAfterInitializationComplete();

      // Clear all the list data valid for the current ignition cycle
      _CommandWaitingList.clear();
      _ExpiredCommandList.clear();
   }
   oConnectionConfigurator.connectionEstablished();
   // trace status
   vTraceTakeIncomingConfigurationStatusIntoAccount(oConnectionConfigurator.isChannelConnected(), oConnectionConfigurator.oHistoryContainer.isHistorySet());
}



tVoid tclMCANInterface::fakeIncomingMsgConfigurationStatus()
{
   vTracefakeIncomingMsgConfigurationStatus(
         oConfigurationGetMessage.isMessageBeenSentToSink(),
         oConfigurationStatusMessage.isMessageFromSinkBeenReceived());
   // a real configuration status is been expected, if configuration get is been sent to sink.
   // if configuration status is been received in the meantime, it means that we must fake the configuration status. Otherwise we wait for the real configuration status.
   if (
         (TRUE == oConfigurationGetMessage.isMessageBeenSentToSink())
         &&
         (TRUE == oConfigurationStatusMessage.isMessageFromSinkBeenReceived())
         )
   {
      takeIncomingConfigurationStatusIntoAccount();
//      _bMessageMeterSettingInfoReceived = TRUE;  // Meter won't send the Setting Info. This variable must be set, but couldn't from here and it's not yet tested. This line is just to reminde us that it need to be done.
   }
}

// general handler functions for incoming standard MCAN messages
tVoid tclMCANInterface::_vHandleMsgConfiguration( tU8 u8OpType, tU8 u8UnitID, tU8 u8UpperVersion, tU8 u8MiddleVersion, tU8 u8LowerVersion )
{
   vTraceIncomingMsgConfiguration(u8OpType);
   switch( u8OpType )
   {
      case MCAN_C_OP_TYPE_SET:
      {
         vSetConfiguration( u8UnitID, u8UpperVersion, u8MiddleVersion, u8LowerVersion, FALSE );

         // communication partner did a re-initialization -> clear the "complete" flag to re-initialize the data
         _u8InitializationStatus &= ~MCAN_C_INIT_STATUS_CONFIGURATION_COMPLETE;
         vDoActionsWhenInitializationIncomplete();

         _bTriggerConfigurationStatus = TRUE;
         vSetChangedEvent();
      }
      break;
      case MCAN_C_OP_TYPE_STATUS:
      {
         oConfigurationStatusMessage.MessageReceivedFromSink();
         vSetConfiguration( u8UnitID, u8UpperVersion, u8MiddleVersion, u8LowerVersion, FALSE );

         takeIncomingConfigurationStatusIntoAccount();
         // trace connection status. Required for diagnosis. We need this at this place once history is been cleared and system start up again.
         vSendConnection();
      }
      break;
      default:
      {
      }
      break;
   }
}


tVoid tclMCANInterface::_vHandleMsgHardwareVersion( tU8 u8OpType, tU8 u8UpperVersion, tU8 u8MiddleVersion, tU8 u8LowerVersion )
{
   if ( MCAN_C_OP_TYPE_STATUS == u8OpType )
   {
      // stop the hardware version get timer
      _vSetHardwareVersionGetTimer( 0 );

      vSetHardwareVersion( u8UpperVersion, u8MiddleVersion, u8LowerVersion, FALSE );
      if ( _bAskForSoftwareVersionAfterHardwareVersion )
      {
         _bAskForSoftwareVersionAfterHardwareVersion = FALSE;
         _bTriggerSoftwareVersionGet = TRUE;
         vSetChangedEvent();
      }
   }
}


tVoid tclMCANInterface::_vHandleMsgSoftwareVersion( tU8 u8OpType, tU8 u8UpperVersion, tU8 u8MiddleVersion, tU8 u8LowerVersion )
{
   if ( MCAN_C_OP_TYPE_STATUS == u8OpType )
   {
      // stop the software version get timer
      _vSetSoftwareVersionGetTimer( 0 );

      vSetSoftwareVersion( u8UpperVersion, u8MiddleVersion, u8LowerVersion, FALSE );
   }
}

// timer functionality

//----------------------------------------------------------------------
//
// FUNCTION: tVoid tclMCANInterface::_vCreateBasicTimers()
//
// DESCRIPTION: create the basic timers used for communication test and 
//              initialization phase
//
// PARAMETER:   none
//
// RETURNVALUE: void
//
//----------------------------------------------------------------------
tVoid tclMCANInterface::_vCreateBasicTimers()
{
   // create the timer
   if ( OSAL_OK != OSAL_s32TimerCreate( (OSAL_tpfCallback)_vCommunicationTestTimerCallback, (tPVoid)this, &_hCommunicationTestTimer ) )
   {
      vTraceError( MCAN_EN_ERROR_CODE_CREATE_COMMUNICATION_TEST_TIMER );
      _hCommunicationTestTimer = OSAL_C_INVALID_HANDLE;
   }

   if ( OSAL_OK != OSAL_s32TimerCreate( (OSAL_tpfCallback)_vConfigurationGetTimerCallback, (tPVoid)this, &_hConfigurationGetTimer ) )
   {
      vTraceError( MCAN_EN_ERROR_CODE_CREATE_CONFIGURATION_GET_TIMER );
      _hConfigurationGetTimer = OSAL_C_INVALID_HANDLE;
   }

   if ( OSAL_OK != OSAL_s32TimerCreate( (OSAL_tpfCallback)_vHardwareVersionGetTimerCallback, (tPVoid)this, &_hHardwareVersionGetTimer ) )
   {
      vTraceError( MCAN_EN_ERROR_CODE_CREATE_HARDWARE_VERSION_GET_TIMER );
      _hHardwareVersionGetTimer = OSAL_C_INVALID_HANDLE;
   }

   if ( OSAL_OK != OSAL_s32TimerCreate( (OSAL_tpfCallback)_vSoftwareVersionGetTimerCallback, (tPVoid)this, &_hSoftwareVersionGetTimer ) )
   {
      vTraceError( MCAN_EN_ERROR_CODE_CREATE_SOFTWARE_VERSION_GET_TIMER );
      _hSoftwareVersionGetTimer = OSAL_C_INVALID_HANDLE;
   }

   if ( OSAL_OK != OSAL_s32TimerCreate( (OSAL_tpfCallback)_vStatusWaitingTimerCallback, (tPVoid)this, &_hStatusWaitingTimer ) )
   {
      vTraceError( MCAN_EN_ERROR_CODE_CREATE_STATUS_WAITING_TIMER );
      _hStatusWaitingTimer = OSAL_C_INVALID_HANDLE;
   }
}

// timer callbacks

tVoid tclMCANInterface::_vCommunicationTestTimerCallback( tVoid *pvArg )
{
   tclMCANInterface *poMyself = (tclMCANInterface *) pvArg;

   if ( poMyself )
   {
      poMyself->_vSetCommunicationTestTimerEvent();
   }
}

tVoid tclMCANInterface::_vConfigurationGetTimerCallback( tVoid *pvArg )
{
   tclMCANInterface *poMyself = (tclMCANInterface *) pvArg;

   if ( poMyself )
   {
      poMyself->_vSetConfigurationGetTimerEvent();
   }
}

tVoid tclMCANInterface::_vHardwareVersionGetTimerCallback( tVoid *pvArg )
{
   tclMCANInterface *poMyself = (tclMCANInterface *) pvArg;

   if ( poMyself )
   {
      poMyself->_vSetHardwareVersionGetTimerEvent();
   }
}

tVoid tclMCANInterface::_vSoftwareVersionGetTimerCallback( tVoid *pvArg )
{
   tclMCANInterface *poMyself = (tclMCANInterface *) pvArg;

   if ( poMyself )
   {
      poMyself->_vSetSoftwareVersionGetTimerEvent();
   }
}

tVoid tclMCANInterface::_vStatusWaitingTimerCallback( tVoid *pvArg )
{
   tclMCANInterface *poMyself = (tclMCANInterface *) pvArg;

   if ( poMyself )
   {
      poMyself->_vSetStatusWaitingTimerEvent();
   }
}

// functions to set (start/stop) a timer

tVoid  tclMCANInterface::_vSetCommunicationTestTimer( OSAL_tMSecond msTimeout )
{
   if ( OSAL_C_INVALID_HANDLE != _hCommunicationTestTimer )
   {
      if ( OSAL_s32TimerSetTime( _hCommunicationTestTimer, msTimeout, 0 ) != OSAL_OK )
      {
         vTraceError( MCAN_EN_ERROR_CODE_SET_COMMUNICATION_TEST_TIMER );
      }
      else
      {
         vTraceBasicTimerSet( MCAN_EN_BASIC_TIMER_COMMUNICATION_TEST , msTimeout );
      }
   }
}

tVoid  tclMCANInterface::_vSetConfigurationGetTimer( OSAL_tMSecond msTimeout )
{
   if ( OSAL_C_INVALID_HANDLE != _hConfigurationGetTimer )
   {
      if ( OSAL_s32TimerSetTime( _hConfigurationGetTimer, msTimeout, 0 ) != OSAL_OK )
      {
         vTraceError( MCAN_EN_ERROR_CODE_SET_CONFIGURATION_GET_TIMER );
      }
      else
      {
         vTraceBasicTimerSet( MCAN_EN_BASIC_TIMER_CONFIGURATION_GET, msTimeout );
      }
   }
}

tVoid  tclMCANInterface::_vSetHardwareVersionGetTimer( OSAL_tMSecond msTimeout )
{
   if ( OSAL_C_INVALID_HANDLE != _hHardwareVersionGetTimer )
   {
      if ( OSAL_s32TimerSetTime( _hHardwareVersionGetTimer, msTimeout, 0 ) != OSAL_OK )
      {
         vTraceError( MCAN_EN_ERROR_CODE_SET_HARDWARE_VERSION_GET_TIMER );
      }
      else
      {
         vTraceBasicTimerSet( MCAN_EN_BASIC_TIMER_HARDWARE_VERSION_GET, msTimeout );
      }
   }
}

tVoid  tclMCANInterface::_vSetSoftwareVersionGetTimer( OSAL_tMSecond msTimeout )
{
   if ( OSAL_C_INVALID_HANDLE != _hSoftwareVersionGetTimer )
   {
      if ( OSAL_s32TimerSetTime( _hSoftwareVersionGetTimer, msTimeout, 0 ) != OSAL_OK )
      {
         vTraceError( MCAN_EN_ERROR_CODE_SET_SOFTWARE_VERSION_GET_TIMER );
      }
      else
      {
         vTraceBasicTimerSet( MCAN_EN_BASIC_TIMER_SOFTWARE_VERSION_GET, msTimeout );
      }
   }
}

tVoid  tclMCANInterface::_vSetStatusWaitingTimer( OSAL_tMSecond msTimeout )
{
   if ( OSAL_C_INVALID_HANDLE != _hStatusWaitingTimer )
   {
      if ( OSAL_s32TimerSetTime( _hStatusWaitingTimer, msTimeout, 0 ) != OSAL_OK )
      {
         vTraceError( MCAN_EN_ERROR_CODE_SET_STATUS_WAITING_TIMER );
      }
      else
      {
         vTraceBasicTimerSet( MCAN_EN_BASIC_TIMER_STATUS_WAITING, msTimeout );
      }
   }
}

// All the MCAN units could use this function to check the status of a particular command.
// Returns TRUE if we are waiting for the Status response.
tBool tclMCANInterface::_bIsAlreadyWaitingCommand( const tU16& NewCommand ) const
{   
   // List should not be empty
   if ( !_CommandWaitingList.empty() )
   {
      bpstl::multiset< WaitingCommand, WaitListCompare, bpstl::allocator<WaitingCommand> >::iterator it = _CommandWaitingList.begin();

      // Cannot use "multiset::find" because the Key_compare/Value_compare is using s32timeout Info to compare.
      for ( ; it != _CommandWaitingList.end() ; it++ )
      {
         // Is command already in the list
         if ( it->u16Command == NewCommand )
         {
            return TRUE;
         }
      }
   }

   return FALSE;
}

// Just to insert a new command to the wait list   
tVoid tclMCANInterface::_vAddCommandToWaitingList(const WaitingCommand& tempCommand)
{
   // Add to the list of waiting commands
   _CommandWaitingList.insert( tempCommand );

   vTraceCommandInfo( MCAN_EN_COMMAND_ADDED_TO_WAITING_LIST, tempCommand.u16Command );
}

// Handler for the commands received for which we are waiting, remove the command and take care of a starting the new timer if required.
tVoid tclMCANInterface::_vHandleReceivedCommand( const tU16& u16Command )
{
    // List should not be empty
    if ( !_CommandWaitingList.empty() )
    {
        // Remove the received command from the waiting list

        // Is the command at the top of the list, if true, this is a special case where we would also need to re-set the timer.
        if ( _CommandWaitingList.begin()->u16Command == u16Command )
        {
        	_CommandWaitingList.erase( _CommandWaitingList.begin() );

            vTraceCommandInfo( MCAN_EN_COMMAND_ERASED_FROM_WAITING_LIST, u16Command );        

            if ( !_CommandWaitingList.empty() )
            {
                // If the status reply for the currently running timer was just received timer should be re started with the next timeout
                _vSetStatusWaitingTimer( _CommandWaitingList.begin()->s32Timeout - OSAL_ClockGetElapsedTime() );          
            }
            else
            {
                // If the list is empty stop the timer
                _vSetStatusWaitingTimer( 0 );
            }
        }
        else
        {
            _vRemoveCommandFromWaitingList( u16Command );
        }
    }
}

// Handler for the timeout, takes care of removing the corresponding command(s) and starting the next timer if required.
tVoid tclMCANInterface::_vHandleStatusWaitingTimerTimeout( tVoid )
{
   tS32 s32CurrentTime = OSAL_ClockGetElapsedTime();
   tU16 u16Command = 0;
   OSAL_tMSecond Timeout;

   if ( !_CommandWaitingList.empty() )
   {
      bpstl::multiset< WaitingCommand, WaitListCompare, bpstl::allocator<WaitingCommand> >::iterator it, it_newBegin;      
      WaitingCommand tempCommand(u16Command, s32CurrentTime);

      // This will return the first iterator which does not follow --> s32Timeout<= s32CurrentTime
      it_newBegin = _CommandWaitingList.upper_bound(tempCommand);

      it = _CommandWaitingList.begin();
      
      // Atleast the first element from the Wait List should be removed.
      do {
         // Inform the MCAN units about the timeout of this Command            
         vStatusWaitingTimerTimedOut ( it->u16Command );

         // Add to the Expired Command list (to be ignored list)
         _ExpiredCommandList.insert( it->u16Command );

         vTraceCommandInfo( MCAN_EN_COMMAND_ADDED_TO_EXPIRED_LIST, it->u16Command );

         it++;

      } while ( (it != it_newBegin) && (it_newBegin != _CommandWaitingList.end()) );      
         
      // Remove all the elements which are expired s32Timeout<= s32CurrentTime
      _CommandWaitingList.erase( _CommandWaitingList.begin(), it_newBegin );      

   }

   // List should not be empty
   if ( !_CommandWaitingList.empty() )
   {
      // Eventually restart timer with the next timeout value.
      Timeout = _CommandWaitingList.begin()->s32Timeout - s32CurrentTime;

      _vSetStatusWaitingTimer( Timeout );
   }
}

// Checks whether the command has already expired so as to 
// not bother MCAN units with those commands and simply ignore them in future
tBool tclMCANInterface::_bIsCommandAlreadyExpired( const tU16& NewCommand ) const
{
   if ( !_ExpiredCommandList.empty() )
   {
      return ( _ExpiredCommandList.end() != _ExpiredCommandList.find( NewCommand ) );
   }
   else
   {
      return FALSE;
   }
}

tVoid tclMCANInterface::_vRemoveCommandFromExpiredList( const tU16& u16Command )
{
   bpstl::set< tU16 >::iterator it = _ExpiredCommandList.begin();
   
   if ( !_ExpiredCommandList.empty() )
   {
      it = _ExpiredCommandList.find( u16Command );

      if ( _ExpiredCommandList.end() != it )
      {
         vTraceCommandInfo( MCAN_EN_COMMAND_ERASED_FROM_EXPIRED_LIST, *it );
         
         // Remove the command from the expired list
         _ExpiredCommandList.erase( it );         
      }
   }   
}

tVoid tclMCANInterface::_vRemoveCommandFromWaitingList( const tU16& u16Command )
{
    // List should not be empty
    if ( !_CommandWaitingList.empty() )
    {
        bpstl::multiset< WaitingCommand, WaitListCompare, bpstl::allocator<WaitingCommand> >::iterator it = _CommandWaitingList.begin();

        // Cannot use "multiset::find" because the Key_compare/Value_compare is using s32timeout Info to compare.
        for ( ; it != _CommandWaitingList.end() ; it++ )
        {
            if ( it->u16Command == u16Command )
            {
                _CommandWaitingList.erase( it );

                vTraceCommandInfo( MCAN_EN_COMMAND_ERASED_FROM_WAITING_LIST, u16Command );

                // Only one entry per Command 
                // Once an element is erased via the iterator, 
                // the iterator is no longer valid which cannot be incremented (it++) as reqd. by "for loop"
                return;
            }
        }
    }
}

