/**************************************************************************//**
* \file     DatagramSocket.cpp
*
* \remark   Copyright: 2010 Robert Bosch GmbH, Hildesheim
******************************************************************************/
#include "DatagramSocket.h"

#ifdef __cplusplus
extern "C"
{
#include "inc.h"
}
#endif

#include <unistd.h>
#include <sys/poll.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <netdb.h>
#include <errno.h>
#include <stdlib.h>

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <sys/time.h>
#include <time.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <signal.h>
#include <fcntl.h>
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_MIDW_COMMON_TRACE
#include "trcGenProj/Header/DatagramSocket.cpp.trc.h"
#endif
#include "../midw_common_trace.h"

/**************************************************************************//**
* Destructor
******************************************************************************/
DatagramSocket::~DatagramSocket()
{
    if (m_socketThread != NULL)
    {
       m_socketThread->bStop();
    }
    dgram_exit(m_pdgram);
    m_pdgram = NULL;
    if (m_socketThread != NULL)
    {
    	delete m_socketThread;
    	m_socketThread = NULL;
    }
    
    m_DatagramSocketCallback = NULL;

    close(m_SocketDescriptor);
    close(m_SocketDescriptorFD);
}


/**************************************************************************//**
* Constructor
******************************************************************************/
DatagramSocket::DatagramSocket(DatagramSocketCallbackIf* cb, uint32_t threadStackSize, uint32_t threadPriority, int32_t port, int32_t addressFamily, tBool bConnectV850,tBool bFakeDevice, tBool bConnectToServerFD){
    m_SocketDescriptor = socket(addressFamily, SOCK_STREAM, 0);
            
    /** SUZUKI-22835 : Setting time out for send calls to 1.3 second
    As suggested by ADR team. ADR may stay in XOFF state for max 3 seconds*/
    struct timeval SendTimeOut;
    SendTimeOut.tv_sec = 1;
    SendTimeOut.tv_usec = 333000;
    setsockopt(m_SocketDescriptor, SOL_SOCKET, SO_SNDTIMEO,(char *) &SendTimeOut, sizeof SendTimeOut);
    m_pdgram = dgram_init(m_SocketDescriptor, DGRAM_MAX, NULL);
    m_DatagramSocketCallback = cb;
    m_bConnectV850 = bConnectV850;
    m_bFakeDevice = bFakeDevice;    
    m_bConnectToServerFD = bConnectToServerFD;
    if (bSetUpSocketConnection(addressFamily, SOCK_STREAM, port))
    {      
		vStartSocketThread(threadStackSize,threadPriority,port);
    }
    else
    {
        dgram_exit(m_pdgram);
        close(m_SocketDescriptor);
        cb->onSocketConnectionNotEstablished();
    }
}
/**************************************************************************//**
*
******************************************************************************/
bool DatagramSocket::bSetUpSocketConnection(int32_t addressFamily, int32_t type, int32_t port)
{
    (void) addressFamily;
    (void) type;
    struct sockaddr_in serv_addr;
    struct sockaddr_in *local_addr, *remote_addr;
    struct hostent *server, *localHost;
    ETG_TRACE_USR4( ( "DatagramSocket:bSetUpSocketConnection addressFamily = %d, type = %d, port = %d",addressFamily,type,port) );

    /**if m_bFakeDevice is true then connection is established with fake1-local else m_bConnectV850 is checked*/
    if(!m_bFakeDevice){
        if (m_bConnectV850) {
           localHost = gethostbyname("scc-local");
           ETG_TRACE_USR4( ( "DatagramSocket:bSetUpSocketConnection scc-local") );
        } else {
           localHost = gethostbyname("adr3-local");
           ETG_TRACE_USR4( ( "DatagramSocket:bSetUpSocketConnection adr3-local") );
        }

        if(NULL == localHost)
        {
            return false;
        }
    
        /** Bind to a local port to receive response*/
        serv_addr.sin_family = AF_INET;
        memcpy(&serv_addr.sin_addr.s_addr, localHost->h_addr, localHost->h_length);
        serv_addr.sin_port = htons(port);
        if (bind(m_SocketDescriptor, (struct sockaddr*) &serv_addr, sizeof serv_addr) < 0)
        {
            return false;
        }
    
        
        /**Get the ADR3 hosts IP Address and try to connect to the ADR3 socket*/
        if (m_bConnectV850) {
           server = gethostbyname("scc");
           ETG_TRACE_USR4( ( "DatagramSocket:bSetUpSocketConnection scc") );
        } else {
           server = gethostbyname("adr3");
           ETG_TRACE_USR4( ( "DatagramSocket:bSetUpSocketConnection adr3") );
        }

        if (server == NULL)
        {
            return false;
        }
    
        /* Connect to port of the ADR3/v850/fake1 */
        serv_addr.sin_family = AF_INET;
        memcpy(&serv_addr.sin_addr.s_addr, server->h_addr, server->h_length);
        serv_addr.sin_port = htons(port);
        if (connect(m_SocketDescriptor,	(struct sockaddr*) &serv_addr, sizeof serv_addr) < 0)
        {
            return false;
        }
    
        int32_t on =1;
        int32_t rc = ioctl(m_SocketDescriptor, FIONBIO, (char *)&on);
        if (rc < 0)
        {
            return false;
        }
    } else {
        if (m_bConnectToServerFD){
            struct addrinfo hints, *pHostAddrServer = NULL;
            
            char cPortNum[MAX_PORT_FD_DIGIT] = {0};
            sprintf(cPortNum, "%d", port);
            
            errno = 0;
            
            m_SocketDescriptor = socket(AF_INET, SOCK_STREAM, 0 );
            if(m_SocketDescriptor < 0) { 
            	return false;
            }
            
            memset( &hints, 0, sizeof( hints ) );
            hints.ai_family = AF_INET;
            
            /**Get the Local Host and Bind to that Host**/
            
            if( getaddrinfo( "localhost", cPortNum, &hints, &pHostAddrServer ) < 0 ) {
            	return false;
            }

            /** Bind to a local port to recieve response*/
            if ( bind(m_SocketDescriptor, pHostAddrServer->ai_addr, pHostAddrServer->ai_addrlen ) < 0 ) {
                ETG_TRACE_USR4( ( "DatagramSocket:bSetUpSocketConnection failed to bind socket. ERROR: %d %s", errno, strerror( errno ) ) );
            	return false;
            }
            
            freeaddrinfo( pHostAddrServer );           /* addr info no longer needed */      
            /*---- Listen on the socket, with 1 max connection requests queued ----*/
            if( listen(m_SocketDescriptor, 1) == 0 ){
               ETG_TRACE_USR4( ( "DatagramSocket:bSetUpSocketConnection listening on port %d", port ) );
            } else {
               ETG_TRACE_USR4( ( "DatagramSocket:bSetUpSocketConnection fail to listen. ERROR: %d %s", errno, strerror( errno ) ) );
            }
            /*---- Accept call creates a new socket for the incoming connection ----*/
            
            m_SocketDescriptorFD = accept(m_SocketDescriptor, (struct sockaddr*)NULL, NULL);
            if (m_SocketDescriptorFD < 0)
            {
                    ETG_TRACE_USR4( ( "DatagramSocket:bSetUpSocketConnection failed to accept client. ERROR: %d %s", errno, strerror( errno ) ) );
                    return false;
            }
            ETG_TRACE_USR4( ( "DatagramSocket:bSetUpSocketConnection accepted client. ERROR: %d %s", errno, strerror( errno ) ) );
         } else {
            struct addrinfo hints, *pHostAddrClient = NULL;
            char cPortNum[MAX_PORT_FD_DIGIT] = {0};
            sprintf(cPortNum, "%d", port);
            
            m_SocketDescriptor = socket(AF_INET, SOCK_STREAM, 0 );
            if(m_SocketDescriptor < 0) { 
               return false;
            }
            
            memset( &hints, 0, sizeof( hints ) );
            hints.ai_family = AF_INET;
            
            /**Get the Remote Host and Connect to that Host,
            which is dynamically created at runtime by kernel in the file /etc/hosts*/
            
            if ( getaddrinfo("localhost", cPortNum, &hints, &pHostAddrClient ) < 0) {
               return false;
            }
            
            /* Connect to remote port*/
            
            errno = 0;	
            ETG_TRACE_USR4( ( "DatagramSocket:bSetUpSocketConnection connect to server fake device" ) );
            if ( connect( m_SocketDescriptor, pHostAddrClient->ai_addr, pHostAddrClient->ai_addrlen ) < 0 ) {
                ETG_TRACE_USR4( ( "DatagramSocket:bSetUpSocketConnection failed to connect to server. ERROR: %d %s", errno, strerror( errno ) ) );
            	return false;
            }
            ETG_TRACE_USR4( ( "DatagramSocket:bSetUpSocketConnection connected" ) );
            
            freeaddrinfo( pHostAddrClient );           /* addr info no longer needed */
        }
    
    }
    return true;
}


/**************************************************************************//**
*
******************************************************************************/
int32_t DatagramSocket::sendMessage(void* ubuf, size_t ulen)
{
    if (m_pdgram != NULL)
    {
        unsigned char ucCount = 3;
        while(ucCount--)
        {
            ETG_TRACE_USR4( ( "DatagramSocket:sendMessage") );
            int32_t iNumberOfBytes = dgram_send(m_pdgram, ubuf, ulen);
            if(iNumberOfBytes != ulen)
            {
                /** store the errno before it gets overwritten */
                s32ErrorNumber = errno;
            }
            else
            {
                return iNumberOfBytes;
            }
        }
        
        
    }
    return 0;
}

/**************************************************************************//**
*
******************************************************************************/
void DatagramSocket::vStartSocketThread(uint32_t threadStackSize, uint32_t threadPriority, int32_t port){
	sprintf(name, "%s%X1%d", "SOCKET_", port, m_bConnectToServerFD);
	ThreadFunctionArguments* args = new ThreadFunctionArguments;
	if (args != NULL)
	{
		if(m_bFakeDevice && m_bConnectToServerFD){
		   args->socketDescriptor = m_SocketDescriptorFD;
                   m_pdgram = dgram_init(m_SocketDescriptorFD, DGRAM_MAX, NULL);
                } else {
		   args->socketDescriptor = m_SocketDescriptor;
                   m_pdgram = dgram_init(m_SocketDescriptor, DGRAM_MAX, NULL);
                }
		args->pdgram = m_pdgram;
		args->datagramSocketCallback = m_DatagramSocketCallback;
	}
	m_socketThread = new SocketThread(name, threadStackSize, threadPriority, args);
	if (m_socketThread != NULL)
	{
		m_socketThread->bStart(0);
	}
}
