
/*****************************************************************************
* FILE 			: IntercomNanoMessageClientHandler.h
*
* BRIEF			: Hand-crafted code
*
* SW-COMPONENT 	: VDVoice
*
* DESCRIPTION 	: This class provides the initialization
*                 and external interface of VDVoice subcomponent.
*
* AUTHOR 		: Monojit Chatterjee (CM/ECG5)
*
* COPYRIGHT 	: (c) Robert Bosch Car Multimedia GmbH
*
* HISTORY 		: 02/06/2020     Initial version
*
****************************************************************************/

#include "IntercomNanoMessageClientHandler.h"
#include "AudioSourceHandler/IntercomAudioClientHandler.h"
#include "IntercomComponent.h"
#include <dlt/dlt.h>

DLT_IMPORT_CONTEXT (INTERCOM_AUDIO_HANDLER);





namespace IntercomApp
{


using namespace asf::core;
using namespace ::bosch::cm::di::NanoMsgIntercomInterface;




std::unordered_map <int32_t , eTPhoneNumber > mapPhoneNumber =
{
    { 1,  eTPhoneNumber__eTPhoneNumber_RestRoom},
    { 2,  eTPhoneNumber__eTPhoneNumber_Kitchen},
    { 3, eTPhoneNumber__eTPhoneNumber_UpperDeck}
};

IntercomNanoMessageClientHandler* IntercomNanoMessageClientHandler::mIntercomNanoMessageClientHandler = NULL;

IntercomNanoMessageClientHandler::IntercomNanoMessageClientHandler()
    : nanoMsgIntercomInterfaceProxy_( ::bosch::cm::di::NanoMsgIntercomInterface::NanoMsgIntercomInterfaceProxy::createProxy("NanoMsgIntercomPort", *this) )
{
    mIntercomNanoMessageClientHandler = this;

    mIntercomService = IntercomService::createInstance();

    m_pEntityInfo = tclAvRoutingParser::pGetInstance();

    if ( setVersionInfo ( VERSION_STRING ) )
    {
        DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING( getVersionInfo ().c_str()));
    }




}
IntercomNanoMessageClientHandler::~IntercomNanoMessageClientHandler()
{
    delete m_pEntityInfo;
    m_pEntityInfo = nullptr;
}

void IntercomNanoMessageClientHandler::onAvailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ServiceStateChange& stateChange)
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler::onAvailable"));

    if ( nanoMsgIntercomInterfaceProxy_ == proxy )
    {
        if ( ! initializeXMLParser ( ) )
        {
            DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_FATAL, DLT_CSTRING("IntercomNanoMessageClientHandler::onAvailable()  xml parser parsing failed "));
            if ( writeToErrorMemory ( "IntercomNanoMessageClientHandler::onAvailable() CMA IP not found in avroutingadapter.xml " ) )
            {
                DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler::onAvailable() CMA IP not found in avroutingadapter.xml- written to error memory "));
            }
            mIntercomService->sendConfigError ( INTERCOM_CONFIG_AVROUTING_ADAPTER_CMA_IP_NOT_FOUND_ERROR ) ;
			
			

        }
        else
        {
            this->registerInterested ();
            IntercomService::createInstance()->sendIntercomServiceAvailabilitySignal();
            mIntercomService->sendConfigError ( No_ERROR_CONFIG );
            mIntercomService->sendNanoMessageError ( No_ERROR_NANOMESSAGE ) ;
			DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler::onAvailable gstreamer initialized "));
			InterComAudioProcessing::createInstance()->init();
            DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler::onAvailable sending sendIntercomServiceAvailabilitySignal"));
        }









        // ReqPhoneStatus
    }
}


bool IntercomNanoMessageClientHandler::initializeXMLParser ()
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler::initializeXMLParser  ") );



    m_bParseState = m_pEntityInfo->bParseXml();
    bool status = false ;

    if ( m_bParseState  )
    {

        vector<std::string> vecEntityIpCMA = m_pEntityInfo->vectorGetEntityIP( DEV_NAME_CMA );


        if ( vecEntityIpCMA.size() > 0  )
        {
            AmpIpAddress = vecEntityIpCMA [ PRIMARY_CMA ] ;
            status = true ;
            DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("initializeXMLParser ip address parsing success "), DLT_CSTRING(getAmpIpAddress().c_str()));
        }

    }

    else
    {

        if ( writeToErrorMemory ( "IntercomNanoMessageClientHandler::initializeXMLParser() avroutingadapter.xml parsing failed " ) )
        {
            DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler::initializeXMLParser() avroutingadapter.xml parsing failed - written to error memory "));
        }

        mIntercomService->sendConfigError ( INTERCOM_CONFIG_AVROUTING_ADAPTER_PARSE_ERROR ) ;

    }



    return status ;

}
void IntercomNanoMessageClientHandler::onUnavailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ServiceStateChange& stateChange)
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler::onUnavailable"));
    mIntercomService->sendNanoMessageError ( NANOMESSAGE_UNAVAILABLE_ERROR ) ;
}

void IntercomNanoMessageClientHandler::onEventAcceptHUCallError(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< EventAcceptHUCallError >& error)
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onEventAcceptHUCallError"));
    mIntercomService->sendNanoMessageError ( NANOMESSAGE_SENDMESSAGE_ERROR ) ;
}

void IntercomNanoMessageClientHandler::onEventAcceptHUCallResponse(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< EventAcceptHUCallResponse >& response)
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onEventAcceptHUCallResponse"));

}




void IntercomNanoMessageClientHandler::onEventCancelHUCallError(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< EventCancelHUCallError >& error)
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onEventCancelHUCallError"));
    mIntercomService->sendNanoMessageError ( NANOMESSAGE_RECEIVEMESSAGE_ERROR ) ;
}

void IntercomNanoMessageClientHandler::onEventCancelHUCallResponse(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< EventCancelHUCallResponse >& response)
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onEventCancelHUCallResponse"));

}




void IntercomNanoMessageClientHandler::onEventHUCallError(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< EventHUCallError >& error)
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onEventHUCallError"));
    mIntercomService->sendNanoMessageError ( NANOMESSAGE_SENDMESSAGE_ERROR ) ;
}

void IntercomNanoMessageClientHandler::onEventHUCallResponse(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< EventHUCallResponse >& response)
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onEventHUCallResponse"));

}



void IntercomNanoMessageClientHandler::onReqGetVersionError(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< ReqGetVersionError >& error)
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onReqGetVersionError"));
    mIntercomService->sendNanoMessageError ( NANOMESSAGE_SENDMESSAGE_ERROR ) ;
}

void IntercomNanoMessageClientHandler::onReqGetVersionResponse(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< ReqGetVersionResponse >& response)
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onReqGetVersionResponse"));
}




void IntercomNanoMessageClientHandler::onReqGetVolumeError(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< ReqGetVolumeError >& error)
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onReqGetVolumeError"));
    mIntercomService->sendNanoMessageError ( NANOMESSAGE_SENDMESSAGE_ERROR ) ;
}

void IntercomNanoMessageClientHandler::onReqGetVolumeResponse(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< ReqGetVolumeResponse >& response)
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onReqGetVolumeResponse"));
}







void IntercomNanoMessageClientHandler::onReqSetVolumeError(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< ReqSetVolumeError >& error)
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onReqSetVolumeError"));
    mIntercomService->sendNanoMessageError ( NANOMESSAGE_SENDMESSAGE_ERROR ) ;
}

void IntercomNanoMessageClientHandler::onReqSetVolumeResponse(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< ReqSetVolumeResponse >& response)
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onReqSetVolumeResponse"));
}




void IntercomNanoMessageClientHandler::onReqSetVolumeDownError(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< ReqSetVolumeDownError >& error)
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onReqSetVolumeDownError"));
    mIntercomService->sendNanoMessageError ( NANOMESSAGE_SENDMESSAGE_ERROR ) ;
}

void IntercomNanoMessageClientHandler::onReqSetVolumeDownResponse(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< ReqSetVolumeDownResponse >& response)
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onReqSetVolumeDownResponse"));
}


bool IntercomNanoMessageClientHandler::makeOutGoingCall  ( int32_t PhoneNo )
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler makeOutGoingCall with dest ip  and phone No :"), DLT_UINT ((uint32_t)PhoneNo), DLT_CSTRING("ip address"), DLT_CSTRING(getAmpIpAddress ().c_str()));

    this->setOutGoingPhoneNo ( PhoneNo );
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("setOutGoingCall PhoneNo "), DLT_UINT( PhoneNo ));
    nanoMsgIntercomInterfaceProxy_->sendEventHUCallRequest ( *this, mapPhoneNumber [ PhoneNo ], enDevID_Media_Apm, DEV_NUM,   EVENT_HU_CALL_MSG_ID , getAmpIpAddress () );
	
	return true;

}

void IntercomNanoMessageClientHandler::onReqSetVolumeUpError(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< ReqSetVolumeUpError >& error)
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onReqSetVolumeUpError"));
    mIntercomService->sendNanoMessageError ( NANOMESSAGE_SENDMESSAGE_ERROR ) ;
}

void IntercomNanoMessageClientHandler::onReqSetVolumeUpResponse(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< ReqSetVolumeUpResponse >& response)
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onReqSetVolumeUpResponse"));
}

void IntercomNanoMessageClientHandler::onReqPhoneStatusError(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< ReqPhoneStatusError >& error)
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onReqPhoneStatusError"));
    mIntercomService->sendNanoMessageError ( NANOMESSAGE_SENDMESSAGE_ERROR ) ;
}

void IntercomNanoMessageClientHandler::onReqPhoneStatusResponse(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< ReqPhoneStatusResponse >& response)
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onReqPhoneStatusResponse"));
}



void IntercomNanoMessageClientHandler::onSig_EventAcceptIntercomCallError(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< Sig_EventAcceptIntercomCallError >& error)
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onSig_EventAcceptIntercomCallError"));
    mIntercomService->sendNanoMessageError ( NANOMESSAGE_SENDMESSAGE_ERROR ) ;
}

void IntercomNanoMessageClientHandler::onSig_EventAcceptIntercomCallSignal(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< Sig_EventAcceptIntercomCallSignal >& signal)
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onSig_EventAcceptIntercomCallSignal phone Number "), DLT_UINT ( signal->getPhoneNumber () ));
}




void IntercomNanoMessageClientHandler::onSig_EventCancelIntercomCallError(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< Sig_EventCancelIntercomCallError >& error)
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onSig_EventCancelIntercomCallError"));
    mIntercomService->sendNanoMessageError ( NANOMESSAGE_SENDMESSAGE_ERROR ) ;
}

void IntercomNanoMessageClientHandler::onSig_EventCancelIntercomCallSignal(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< Sig_EventCancelIntercomCallSignal >& signal)
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onSig_EventCancelIntercomCallSignal"));

    if ( IntercomAudioClientHandler::poGetIntercomAudioClientInstance()->getAudioChannelStatus () == true )
    {
        IntercomService::createInstance()->sendEventCanCellCall();
        IntercomAudioClientHandler::poGetIntercomAudioClientInstance()->rejectIntercomAudioSource();
        DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onSig_EventCancelIntercomCallSignal and reject audiosource"));

    }
    else
    {

        DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onSig_EventCancelIntercomCallSignal allocation failed so not sending"));

    }




}



void IntercomNanoMessageClientHandler::onSig_EventIntercomCallError(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< Sig_EventIntercomCallError >& error)
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onSig_EventIntercomCallError"));
    mIntercomService->sendNanoMessageError ( NANOMESSAGE_SENDMESSAGE_ERROR ) ;
}

void IntercomNanoMessageClientHandler::onSig_RspGetIntercomAvailableError(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< Sig_RspGetIntercomAvailableError >& error)
{
}

void IntercomNanoMessageClientHandler::onSig_RspGetIntercomAvailableSignal(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< Sig_RspGetIntercomAvailableSignal >& signal)
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onSig_RspGetIntercomAvailableSignal"));
    eTIntercomAvailable pAvailability =  signal->getAvailability ();
    IntercomService::createInstance()->SendResponseAvailability ( (uint8_t ) pAvailability );
}

void IntercomNanoMessageClientHandler::onReqGetIntercomAvailableError(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< ReqGetIntercomAvailableError >& error)
{
	DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler::onReqGetIntercomAvailableError"));
}

void IntercomNanoMessageClientHandler::onReqGetIntercomAvailableResponse(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< ReqGetIntercomAvailableResponse >& response)
{
	DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler::onReqGetIntercomAvailableResponse"));
}


void IntercomNanoMessageClientHandler::onSig_EventIntercomAvailableError(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< Sig_EventIntercomAvailableError >& error)
{
		DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler::onSig_EventIntercomAvailableError"));
}

void IntercomNanoMessageClientHandler::onSig_EventIntercomAvailableSignal(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< Sig_EventIntercomAvailableSignal >& signal)
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onSig_EventIntercomAvailableSignal"));
    eTIntercomAvailable pAvailability =  signal->getAvailability ();
    IntercomService::createInstance()->SendEventAvailability ( (uint8_t ) pAvailability );
}

void IntercomNanoMessageClientHandler::onSig_EventIntercomCallSignal(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< Sig_EventIntercomCallSignal >& signal)
{
	
	if ( IntercomComponent::getInstance ()->getLowVoltageErrorStatus () == true )
	{
	DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onSig_EventIntercomCallSignal : intercom call blocked due to USER_OFF_LOW_VOLTAGE"));
	this->CallAction ( REJECT );
	DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onSig_EventIntercomCallSignal : intercom call blocked due to USER_OFF_LOW_VOLTAGE Reject request sent to CMA"));
		return;
	}
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onSig_EventIntercomCallSignal"));
    eTPhoneNumber ePhoneNo = signal->getPhoneNumber ();

    auto it = mapPhoneNumber.find ( ePhoneNo  );


    //

    this->setPhoneNumber ( (int32_t )it->first );


    if (IntercomAudioClientHandler::poGetIntercomAudioClientInstance()->acceptIntercomSource ( ) == false )
    {
        DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler sending signal since audio already allocated"));

    }

    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler sending incoming call signal to HMI"));

    IntercomService::createInstance()->sendIncomingCall  ( (int32_t )it->first );



    //	nanoMsgIntercomInterfaceProxy_->sendReqACKEventHandshakeRequest ( *this );

}




void IntercomNanoMessageClientHandler::onSig_EventPhoneStatusChangedError(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< Sig_EventPhoneStatusChangedError >& error)
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onSig_EventPhoneStatusChangedError"));
    mIntercomService->sendNanoMessageError ( NANOMESSAGE_RECEIVEMESSAGE_ERROR ) ;
}

void IntercomNanoMessageClientHandler::onSig_EventPhoneStatusChangedSignal(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< Sig_EventPhoneStatusChangedSignal >& signal)
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onSig_EventPhoneStatusChangedSignal phoneNum "));

    eTPhoneNumber ePhoneNumber = (eTPhoneNumber) signal->getPhoneNumber ();
    uint8_t       PhoneStatus  =                 signal->getPhonestatus ();
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("ePhoneNumber"), DLT_UINT(ePhoneNumber),DLT_CSTRING("PhoneStatus"), DLT_UINT(PhoneStatus) );

    IntercomService::createInstance()->sendEventPhoneStatusChangedSignal ( PhoneStatus , ( uint8_t ) ePhoneNumber );

    // when the phonestatus is on hook and not ringing then sending cancel call request
    if ( ! checkStBit ( PhoneStatus , DISABLED_ENABLED  ) && ! checkStBit ( PhoneStatus , CONNECTED_DISCONNECTED  ) && checkStBit ( PhoneStatus , ONHOOK_PICKUP  ) && !checkStBit ( PhoneStatus , RINGING_NOTRINGING  ) && ( uint8_t ) ePhoneNumber == this->getOutGoingPhoneNo ()  )
    {
        this->CallAction ( REJECT );
        bool status = IntercomAudioClientHandler::poGetIntercomAudioClientInstance()->rejectIntercomAudioSource();
        DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler Sending Reject request and deallocation audio due to on hook  not ringing status"));
        this->setOutGoingPhoneNo ( 0 );
    }

    else if ( !checkStBit ( PhoneStatus , DISABLED_ENABLED  ) && !checkStBit ( PhoneStatus , CONNECTED_DISCONNECTED  ) && !checkStBit ( PhoneStatus , ONHOOK_PICKUP  ) && ( uint8_t ) ePhoneNumber == this->getOutGoingPhoneNo () )
    {
        this->setOutGoingPhoneNo ( 0 );
        DLT_LOG(INTERCOM_AUDIO_HANDLER,DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler clearing outgoing call phone No since PhoneStatus is PickUP"));

    }

}






void IntercomNanoMessageClientHandler::onSig_EventSetVolumeError(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< Sig_EventSetVolumeError >& error)
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onSig_EventSetVolumeError"));
    mIntercomService->sendNanoMessageError ( NANOMESSAGE_RECEIVEMESSAGE_ERROR ) ;
}

void IntercomNanoMessageClientHandler::onSig_EventSetVolumeSignal(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< Sig_EventSetVolumeSignal >& signal)
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onSig_EventSetVolumeSignal"));

}




void IntercomNanoMessageClientHandler::onSig_RspGetVersionError(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< Sig_RspGetVersionError >& error)
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onSig_RspGetVersionError"));
    mIntercomService->sendNanoMessageError ( NANOMESSAGE_RECEIVEMESSAGE_ERROR ) ;
}

void IntercomNanoMessageClientHandler::onSig_RspGetVersionSignal(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< Sig_RspGetVersionSignal >& signal)
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onSig_RspGetVersionSignal"));
}




void IntercomNanoMessageClientHandler::onSig_RspGetVolumeError(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< Sig_RspGetVolumeError >& error)
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onSig_RspGetVolumeError"));
    mIntercomService->sendNanoMessageError ( NANOMESSAGE_RECEIVEMESSAGE_ERROR ) ;
}

void IntercomNanoMessageClientHandler::onSig_RspGetVolumeSignal(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< Sig_RspGetVolumeSignal >& signal)
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onSig_RspGetVolumeSignal"));
}




void IntercomNanoMessageClientHandler::onSig_RspPhoneStatusError(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< Sig_RspPhoneStatusError >& error)
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onSig_RspPhoneStatusError"));
    mIntercomService->sendNanoMessageError ( NANOMESSAGE_RECEIVEMESSAGE_ERROR ) ;
}



/* void IntercomNanoMessageClientHandler::onReqACKEventHandshakeError(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< ReqACKEventHandshakeError >& error)
{
   DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onSig_RspPhoneStatusSignal"));
}

void IntercomNanoMessageClientHandler::onReqACKEventHandshakeResponse(const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< ReqACKEventHandshakeResponse >& response)
{
   DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onSig_RspPhoneStatusSignal"));

} */


void IntercomNanoMessageClientHandler::onSig_RspPhoneStatusSignal  (const ::boost::shared_ptr< NanoMsgIntercomInterfaceProxy >& proxy, const ::boost::shared_ptr< Sig_RspPhoneStatusSignal >& signal)
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler onSig_RspPhoneStatusSignal"));
    uint8_t phoneStatus1 = signal->getPhonestatus1 ();
    uint8_t phoneStatus2 = signal->getPhonestatus2 ();
    uint8_t phoneStatus3 = signal->getPhonestatus3 ();
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("onSig_RspPhoneStatusSignal ::RestRoomStatus "), DLT_UINT( phoneStatus1 ),DLT_CSTRING("KitchenStatus "), DLT_UINT( phoneStatus2 ) ,DLT_CSTRING("upperDeck "), DLT_UINT( phoneStatus3 ) );
    IntercomService::createInstance()->sendRspPhoneStatusSignal ( phoneStatus1 , phoneStatus2, phoneStatus3 );

}

void IntercomNanoMessageClientHandler::registerInterested ()
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler registerInterested"));
    if ( nanoMsgIntercomInterfaceProxy_ != nullptr )
    {
        DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler nanoMsgIntercomInterfaceProxy"));
        nanoMsgIntercomInterfaceProxy_->sendSig_EventIntercomCallRegister ( *this );
        nanoMsgIntercomInterfaceProxy_->sendSig_EventCancelIntercomCallRegister ( *this );
        nanoMsgIntercomInterfaceProxy_->sendSig_EventAcceptIntercomCallRegister ( *this );
        nanoMsgIntercomInterfaceProxy_->sendSig_RspGetVolumeRegister ( *this );
        nanoMsgIntercomInterfaceProxy_->sendSig_EventSetVolumeRegister ( *this );
        nanoMsgIntercomInterfaceProxy_->sendSig_RspGetVersionRegister ( *this );
        nanoMsgIntercomInterfaceProxy_->sendSig_EventPhoneStatusChangedRegister ( *this );
        nanoMsgIntercomInterfaceProxy_->sendSig_RspPhoneStatusRegister ( *this );
        nanoMsgIntercomInterfaceProxy_->sendSig_EventIntercomAvailableRegister ( *this );
        nanoMsgIntercomInterfaceProxy_->sendSig_RspGetIntercomAvailableRegister ( *this );
    }
    else
    {
        mIntercomService->sendNanoMessageError ( NANOMESSAGE_COMMUNICATION_ERROR ) ;
    }

}

IntercomNanoMessageClientHandler* IntercomNanoMessageClientHandler::createInstance()
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler : createInstance"));


    return mIntercomNanoMessageClientHandler;
}


bool IntercomNanoMessageClientHandler::CallAction  ( eCallAction Action )
{
    if ( Action == ACCEPT )
    {
        DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler : CallAction ACCEPT with dest ip"));

        if ( nanoMsgIntercomInterfaceProxy_!= nullptr )
            nanoMsgIntercomInterfaceProxy_->sendEventAcceptHUCallRequest ( *this, enDevID_Media_Apm, DEV_NUM,  ACCEPT_INTERCOM_EVENT_MSG_ID, getAmpIpAddress () );
    }
    else if ( Action == REJECT )
    {
        DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler : CallAction REJECT with dest ip"));
        if ( nanoMsgIntercomInterfaceProxy_!= nullptr )
            nanoMsgIntercomInterfaceProxy_->sendEventCancelHUCallRequest ( *this, enDevID_Media_Apm, DEV_NUM,  CANCEL_INTERCOM_EVENT_MSG_ID, getAmpIpAddress () );

    }
    return true;

}

bool IntercomNanoMessageClientHandler::getPhoneStatus ()
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler : CallAction getPhoneStatus "), DLT_CSTRING( getVersionInfo ().c_str()));
    if ( nanoMsgIntercomInterfaceProxy_ != nullptr  )
        nanoMsgIntercomInterfaceProxy_->sendReqPhoneStatusRequest (*this, enDevID_Media_Apm, DEV_NUM,  PHONE_STATUS_REQUEST_MSG_ID, getAmpIpAddress ());


    return true;
}

bool IntercomNanoMessageClientHandler::getAvailabilityStatus ()
{
    DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomNanoMessageClientHandler : getAvailabilityStatus "), DLT_CSTRING( getVersionInfo ().c_str()));
    if ( nanoMsgIntercomInterfaceProxy_ != nullptr  )
        nanoMsgIntercomInterfaceProxy_->sendReqGetIntercomAvailableRequest (*this, enDevID_Media_Apm, DEV_NUM,  INTERCOM_GET_AVAILABILITY_MSG_ID, getAmpIpAddress ());


    return true;
}

bool IntercomNanoMessageClientHandler::writeToErrorMemory ( const std::string& error )
{
    bool status = true;
    int FileHandle = open( "/dev/errmem", O_WRONLY );
    if ( FileHandle != - 1 )
    {
        if ( write( FileHandle, error.c_str( ), error.length( ) ) == - 1 )
        {
            DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomApp::failed to write to errormem "), DLT_CSTRING( strerror( errno ) ));
            status = false ;
        }

        if ( close( FileHandle ) == - 1 )
        {
            DLT_LOG(INTERCOM_AUDIO_HANDLER, DLT_LOG_INFO, DLT_CSTRING("IntercomApp::failed to close /dev/errmem "), DLT_CSTRING( strerror( errno ) ));
            status = false;
        }

    }
    else
    {

        status = false;

    }

    return status;
}

}
