/*!
*******************************************************************************
* \file              ihl_tclThreader.cpp
*******************************************************************************
*  - PROJECT:        GM Gen2
*  - SW-COMPONENT:   Infotainment Helper Library
*  - DESCRIPTION:    Thread Utility Class - Threads as objects
*                    Command Processor Model Thread class
*  - COPYRIGHT:      &copy; 2010 Robert Bosch Car Multimedia Gmbh
*******************************************************************************
* \date 30.06.2011 \version 1.1 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* - Updated event handling to All Application events.
* - Removed static cast in vStartThread() so that polymorphism can be supported 
* in principle.
*
* \date 06.12.2010 \version 1.0 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* \bug No known bugs
******************************************************************************/

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/

#include "ihl_tclThreader.h"

/******************************************************************************
| defines and macros (scope: module-local)
|----------------------------------------------------------------------------*/

#define IHL_POSTEVENT_NAME            "Event"
#define IHL_POSTSEMP_NAME             "Sem"

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| function prototype (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| function implementation (scope: external-interfaces)
|----------------------------------------------------------------------------*/

namespace ihl {
   namespace thread {

/******************************************************************************
** FUNCTION:  ihl_tclThreader::ihl_tclThreader(const OSAL_trThre..
******************************************************************************/

ihl_tclThreader::ihl_tclThreader
(
   const ihl_tThreadAttribute &rfcoThreadAttrib
   , const ihl_rEventFieldAttribute &rfcoEvtFieldAttrib
   , ihl_tclThreadable* const cpoThreadable
   , tBool bExecThread
):m_coEvtFieldAttrib(rfcoEvtFieldAttrib), m_bThreadAlive(FALSE), m_oThreadId(OSAL_ERROR)
, m_hEvntHandle(OSAL_C_INVALID_HANDLE), m_oEventName()
, m_hSemHandle(OSAL_C_INVALID_HANDLE), m_oSemName()
, m_cpoThreadable(cpoThreadable), m_oThreadAttrib(rfcoThreadAttrib)
{   
   if(TRUE == bAllocResrc(m_oThreadAttrib.m_oOsalThreadAtt.szName))
   {
      // Set the Thread entry function
      m_oThreadAttrib.m_oOsalThreadAtt.pfEntry  =  ((OSAL_tpfThreadEntry)vStartThread);
      // Arguement to the entry function.
      m_oThreadAttrib.m_oOsalThreadAtt.pvArg    =  ((tPVoid)this);

      // Check if it is required to spawn the thread now?
      if(TRUE == bExecThread)
      {
         // Spawn the thread.
         tBool bRetVal = bRunThread();

         // Assert for Thread is really spawn or not?
         NORMAL_M_ASSERT(TRUE == bRetVal);
      }  // if(TRUE == bExecThread)
   }  // if(TRUE == bAllocResrc(m_oThreadAttrib.m_oOsalThreadAtt.szName, TRUE))
   else
   {
      // Creation of Event failed. Cannot continue... [:(]
      NORMAL_M_ASSERT_ALWAYS();
   }  // End of if-else; if(TRUE == bAllocResrc())
   
}  // ihl_tclThreader::ihl_tclThreader(ihl_tclThread &rfco....

/******************************************************************************
** FUNCTION:  virtual ihl_tclThreader::~ihl_tclThreader()
******************************************************************************/

/*virtual*/
ihl_tclThreader::~ihl_tclThreader()
{
   try   // try
   {
      //--------------------------------------------------------------------------
      // :NOTE: Since the Thread delete cannot be used, there exists 
      // synchronization problems during shutdown of the system, between creator
      // thread and the worker thread. This synchronization is provided using
      // semaphores.
      //--------------------------------------------------------------------------

      (tVoid)bEventPost(IHL_THREAD_TERMINATE_EVENT);

      // Wait for the Worker Thread termination event
      if (OSAL_ERROR == OSAL_s32SemaphoreWait(m_hSemHandle, m_oThreadAttrib.m_u32SyncTime))
      {
         // Get the OSAL_ERROR Code and do an assert.
         tU32 u32ErrorCode   =  OSAL_u32ErrorCode();
         // Assert the Errorcodes
         // Error - Time out error?
         NORMAL_M_ASSERT(OSAL_E_TIMEOUT != u32ErrorCode); 
         // Error - Interrupt?
         NORMAL_M_ASSERT(OSAL_E_INTERRUPT != u32ErrorCode);
         // Error - Invalid Value?
         NORMAL_M_ASSERT(OSAL_E_INVALIDVALUE != u32ErrorCode);
         // Assert for debugging purpose.
         NORMAL_M_ASSERT_ALWAYS();
      }  // if (OSAL_ERROR == OSAL_s32SemaphoreWait(m_hSemHandle, m_oThreadAt..

      // Semaphore Clean up.
      if(OSAL_C_INVALID_HANDLE != m_hSemHandle)
      {
         // First close the semaphore and then delete it.
         (tVoid)OSAL_s32SemaphoreClose(m_hSemHandle);   
         (tVoid)OSAL_s32SemaphoreDelete(m_oSemName.szValue);
         m_hSemHandle = OSAL_C_INVALID_HANDLE;
      }  // if(OSAL_C_INVALID_HANDLE != m_hSemHandle)

      // Close the event field.
      vEventClose();
   }  // try
   catch(...)  // catch (...)
   {
   }  // catch (...)
   // 
}  // ihl_tclThreader::~ihl_tclThreader()

/******************************************************************************
** FUNCTION:  tVoid ihl_tclThreader::vEventPost(OSAL_tEventMask ev..
******************************************************************************/

tBool ihl_tclThreader::bEventPost
(
   OSAL_tEventMask evtEvent
   , OSAL_tenEventMaskFlag enFlags
)
{
   tBool bRetVal  =  FALSE;

   // Post the event, if the event handle is valid.
   if(OSAL_C_INVALID_HANDLE != m_hEvntHandle)
   {
      bRetVal = (OSAL_ERROR != OSAL_s32EventPost(m_hEvntHandle, evtEvent, enFlags));
   }  // if(OSAL_ERROR != m_hEvntHandle)

   return bRetVal;
}  // tVoid ihl_tclThreader::vEventPost(OSAL_tEventMask evtEvent, OSAL_tenEven..

/******************************************************************************
** FUNCTION:  tBool ihl_tclThreader::bRunThread()
******************************************************************************/

tBool ihl_tclThreader::bRunThread()
{
   // Check if thread is already spawned?
   if(OSAL_ERROR == m_oThreadId)
   {
      m_oThreadId  =  OSAL_ThreadSpawn(&(m_oThreadAttrib.m_oOsalThreadAtt));
   }  // if(OSAL_ERROR == m_oThreadId)

   return (OSAL_ERROR != m_oThreadId);
}  // tBool ihl_tclThreader::bRunThread()

/******************************************************************************
** FUNCTION:  virtual tVoid ihl_tclThreader::vExecute()
******************************************************************************/

/*virtual*/
tVoid ihl_tclThreader::vExecute()
{
   m_bThreadAlive = TRUE;

   // Loop through when Thread is alive.
   while(TRUE == m_bThreadAlive)
   {
      OSAL_tEventMask evntMask   =  0;

      if (OSAL_OK == OSAL_s32EventWait(m_hEvntHandle
            , m_coEvtFieldAttrib.m_evntMask, m_coEvtFieldAttrib.m_enFlags
            , m_coEvtFieldAttrib.m_evntTimeOut, &evntMask))
      {
         if(IHL_THREAD_TERMINATE_EVENT & evntMask)
         {
            // Terminate event has arrived, make the Thread alive flag FALSE and 
            // exit the looping
            m_bThreadAlive =  FALSE;
            break;   
         }  // if(ihl_THREAD_TERMINATE_EVENT & evntMask)

         tS32 s32Error = OSAL_s32EventPost(m_hEvntHandle, ~evntMask, OSAL_EN_EVENTMASK_AND);
         NORMAL_M_ASSERT(OSAL_ERROR != s32Error);
         
         if(IHL_THREAD_APP_EVENTS & evntMask)
         {
            if(OSAL_NULL != m_cpoThreadable)
            {
               // Execute the customized threadable function
               m_cpoThreadable->vExecute(evntMask);

            }  // if(OSAL_NULL != m_cpoThreadable)

         }  // if(IHL_THREAD_APP_EVENTS & evntMask)

      }  // if (OSAL_OK == OSAL_s32EventWait(m_hEvntHa...
      else
      {
         tU32 u32ErrorCode =  OSAL_u32ErrorCode();

         switch(u32ErrorCode)
         {
            case OSAL_E_INVALIDVALUE:
            {
               // Event Field has been closed and deleted.
               // Just exit the looping
               m_bThreadAlive = FALSE;

            }  // case OSAL_E_INVALIDVALUE:
            break;

            case OSAL_E_EVENTINUSE:
            {
               // Assert for debugging purpose.
               NORMAL_M_ASSERT(OSAL_E_EVENTINUSE != u32ErrorCode); 
            }  // case OSAL_E_EVENTINUSE:, case OSAL_E_TIMEOUT:
            break;

            default:
            {
               // Assert for debugging purpose.
               // FATAL ERROR - ASSERT
               NORMAL_M_ASSERT_ALWAYS();
            }  // default:
            break;               
         }  // switch(u32ErrorCode)
      }  // End of if-else; if (OSAL_OK == OSAL_s32EventWait(m_hEvntHa...
   }  // while(TRUE == m_bThreadAlive)

   // Close the Event field
   vEventClose();

   if(OSAL_C_INVALID_HANDLE != m_hSemHandle)
   {
      // Synchronization mechanism - Semaphore post.
      (tVoid)OSAL_s32SemaphorePost(m_hSemHandle);
   }  // if(OSAL_C_INVALID_HANDLE != m_hSemHandle)
   
   // Exit the thread context
   OSAL_vThreadExit();
}  // tVoid ihl_tclThreader::vExecute()

/******************************************************************************
** FUNCTION:  tVoid ihl_tclThreader::vEventClose()
******************************************************************************/

tVoid ihl_tclThreader::vEventClose()
{
   //-----------------------------------------------------------------------
   // :NOTE: Defensive Programming Technique.
   // Event field can be closed and deleted from Creator thread context as 
   // well as the Worker Thread context. Usually Event field is deleted from
   // Worker Thread context. The synchronization between Creator thread & 
   // Worker thread is carried out using semaphore. Creator thread waits for  
   // 5 seconds for clearing the resources. If the Worker thread did not get
   // CPU time, then creator thread uses brute force and closes / deletes 
   // the Event field. This way the Worker thread comes out of the while 
   // looping. So for safety reasons the event handle is checked for before 
   // deletion.
   //-----------------------------------------------------------------------

   if (OSAL_C_INVALID_HANDLE != m_hEvntHandle)
   {
      // :NOTE: Close the Event field before deletion. This avoids thread 
      // waits for this event to prevent blocking. If the event field is 
      // still open, the deletion is pending and accomplished with the final
      // close. Hence an event close is essential
      (tVoid)OSAL_s32EventClose(m_hEvntHandle);

      // Delete the event field.
      (tVoid)OSAL_s32EventDelete(m_oEventName.szValue);

      // Explicitly make the Event handle Invalid.
      m_hEvntHandle  =  OSAL_C_INVALID_HANDLE;
   }  // if(OSAL_C_INVALID_HANDLE != m_hEvntHandle)
}  // tVoid ihl_tclThreader::vEventClose()

/******************************************************************************
** FUNCTION:  tBool ihl_tclThreader::bAllocResrc(tCString cszName)
******************************************************************************/

tBool ihl_tclThreader::bAllocResrc(tCString cszName)
{
   tBool bRetVal  =  FALSE;

   ihl_tclFiString oLocPrefixName(cszName);
   // Create the Event name
   m_oEventName =  oLocPrefixName;
   m_oEventName += IHL_POSTEVENT_NAME;

   if(OSAL_ERROR == OSAL_s32EventCreate(m_oEventName.szValue, &m_hEvntHandle))
   {
      // Event creation has failed - FATAL ERROR
      // Query the OSAL_ERROR type. 
      tU32 u32ErrorCode =  OSAL_u32ErrorCode();

      // Check for error type
      // Event name already exists
      NORMAL_M_ASSERT(OSAL_E_ALREADYEXISTS != u32ErrorCode);
      // No space to create a new event
      NORMAL_M_ASSERT(OSAL_E_NOSPACE != u32ErrorCode);
      // No permissions to create a new event
      NORMAL_M_ASSERT(OSAL_E_NOPERMISSION != u32ErrorCode);
      // Invalid parameter passed while creating an event field.
      NORMAL_M_ASSERT(OSAL_E_INVALIDVALUE != u32ErrorCode);
      // Assert for debugging purpose.
      NORMAL_M_ASSERT_ALWAYS();

      m_hEvntHandle  = OSAL_C_INVALID_HANDLE;
   }  // if(OSAL_ERROR == OSAL_s32EventCreate(m_oEventName.szValue, &m_hEvntHa..
   else
   {
      bRetVal  =  TRUE;
   }  // End of if-else; if(OSAL_ERROR == OSAL_s32EventCreate(cszName, ..

   if(TRUE == bRetVal)
   {
      // Generate the sempahore name
      m_oSemName  =  oLocPrefixName;
      m_oSemName  += IHL_POSTSEMP_NAME;

      // Create Semaphore for signalling mechanism
      if(OSAL_ERROR == OSAL_s32SemaphoreCreate(m_oSemName.szValue, &m_hSemHandle, 0))
      {
         // Semaphore creation Failed - FATAL ERROR
         // Query the OSAL_ERROR type. 
         tU32 u32ErrorCode    =  OSAL_u32ErrorCode();

         // Check for the error
         // Assert reason - Platform Issue
         NORMAL_M_ASSERT(OSAL_E_NOPERMISSION != u32ErrorCode); 
         // Assert reason - Platform Issue
         NORMAL_M_ASSERT(OSAL_E_NOSPACE != u32ErrorCode);
         // Invalid parameter passed while creating an event field.
         NORMAL_M_ASSERT(OSAL_E_INVALIDVALUE != u32ErrorCode);
         // Sempahore already exists with this name.
         NORMAL_M_ASSERT(OSAL_E_ALREADYEXISTS != u32ErrorCode);
         // Assert for debugging purpose.
         NORMAL_M_ASSERT_ALWAYS();

         m_hSemHandle = OSAL_C_INVALID_HANDLE;
      }  // if (OSAL_ERROR == OSAL_s32SemaphoreCreate(m_s8...
      else
      {
         bRetVal  =  TRUE;
      }
   }  // if(TRUE == bRetVal)

   return bRetVal;
}  // tBool ihl_tclThreader::bAllocResrc(tCString cszName)

/******************************************************************************
** FUNCTION:  static tVoid ihl_tclThreader::vStartThread(tVoid *pvArg)
******************************************************************************/

/*static*/
tVoid ihl_tclThreader::vStartThread(tVoid *pvArg)
{
   // Null pointer check
   if(OSAL_NULL != pvArg)
   {
      ihl_tclThreader* poThreader = reinterpret_cast<ihl_tclThreader*>(pvArg);

      // Execute the Thread Function
      poThreader->vExecute();
   }  // if(OSAL_NULL != pvArg)
   else
   {
      // Assert on NULL Pointer access.
      NORMAL_M_ASSERT_ALWAYS();
   }  // End of if-else; if(OSAL_NULL != pvArg)
}  // tVoid ihl_tclThreader::vStartThread(tVoid *pvArg)

   }  // namespace thread
}  // namespace ihl

///////////////////////////////////////////////////////////////////////////////

// <EOF>
