/*!
*******************************************************************************
* \file              ihl_tclMsgQThreader.cpp
*******************************************************************************
*  - PROJECT:        GM Gen2
*  - SW-COMPONENT:   Infotainment Helper Library
*  - DESCRIPTION:    Thread Utility Class - Threads as objects
*                    Command Processor Model Thread class
*                    Support for Message Queues & Event
*  - COPYRIGHT:      &copy; 2010 Robert Bosch Car Multimedia Gmbh
*******************************************************************************
* \date 23.06.2011 \version 1.0 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* \bug No known bugs
******************************************************************************/

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/

#include "ihl_tclMsgQThreader.h"
#include "ihl_tclMsgQThreadable.h"
#include "ihl_Trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_IHL_TASK
#include "trcGenProj/Header/ihl_tclMsgQThreader.cpp.trc.h"
#endif

/******************************************************************************
| defines and macros (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| function prototype (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| function implementation (scope: external-interfaces)
|----------------------------------------------------------------------------*/

namespace ihl {
   namespace thread {

/******************************************************************************
** FUNCTION:  ihl_tclMsgQThreader::ihl_tclMsgQThreader(const OSAL_trThre..
******************************************************************************/

ihl_tclMsgQThreader::ihl_tclMsgQThreader
(
   const ihl_tThreadAttribute &rfcoThreadAttrib
   , const ihl_tMsgQAttribute &rfcoMsgQAttrib
   , const ihl_rEventFieldAttribute &rfcoEvtFieldAttrib
   , ihl_tclMsgQThreadable* const cpoThreadable
   , tBool bExecThread
):ihl_tclThreader(rfcoThreadAttrib, rfcoEvtFieldAttrib, cpoThreadable, FALSE)
, m_coMsgQAttrib(rfcoMsgQAttrib), m_cpoMsgQThreadable(cpoThreadable)
, m_oMsgQHandle(OSAL_C_INVALID_HANDLE)
{
   // Create the Message Queue
   if (OSAL_ERROR != OSAL_s32MessageQueueCreate(m_coMsgQAttrib.m_oMsgQName.szValue
                        , m_coMsgQAttrib.m_u32MaxMessages
                        , m_coMsgQAttrib.m_u32MaxMsgLen, OSAL_EN_READWRITE
                        , &m_oMsgQHandle))
   {
      // Plant the message notification.
      if (OSAL_ERROR == OSAL_s32MessageQueueNotify(m_oMsgQHandle
                           , (OSAL_tpfCallback)vOnNewMessage, this))
      {
         NORMAL_M_ASSERT_ALWAYS();
      }
   }
   else
   {
      NORMAL_M_ASSERT_ALWAYS();
   }

   // Delayed spawn
   if (TRUE == bExecThread)
   {
      // Spawn the thread.
      tBool bRetVal = bRunThread();

      // Assert for Thread is really spawn or not?
      NORMAL_M_ASSERT(TRUE == bRetVal);
   }
}  // ihl_tclMsgQThreader::ihl_tclMsgQThreader(const ihl_tThreadAttribute &rfc..

/******************************************************************************
** FUNCTION:  virtual ihl_tclMsgQThreader::~ihl_tclMsgQThreader()
******************************************************************************/

/*virtual*/
ihl_tclMsgQThreader::~ihl_tclMsgQThreader()
{
   // Post the terminate event.
   (tVoid)bEventPost(IHL_THREAD_TERMINATE_EVENT);

   if (OSAL_C_INVALID_HANDLE != m_oMsgQHandle)
   {
      // Remove the callback mechanism
      tS32 s32RetVal =  OSAL_s32MessageQueueNotify(m_oMsgQHandle, OSAL_NULL, OSAL_NULL);
      NORMAL_M_ASSERT(OSAL_ERROR != s32RetVal);
      s32RetVal      =  OSAL_s32MessageQueueClose(m_oMsgQHandle);
      NORMAL_M_ASSERT(OSAL_ERROR != s32RetVal);
      s32RetVal      =  OSAL_s32MessageQueueDelete(m_coMsgQAttrib.m_oMsgQName.szValue);
      NORMAL_M_ASSERT(OSAL_ERROR != s32RetVal);
      m_oMsgQHandle  =  OSAL_C_INVALID_HANDLE;

   }  // if (OSAL_C_INVALID_HANDLE != m_oMsgQHandle)

}  // ihl_tclMsgQThreader::~ihl_tclMsgQThreader()

/******************************************************************************
** FUNCTION:  virtual tVoid ihl_tclMsgQThreader::vExecute()
******************************************************************************/

/*virtual*/
tVoid ihl_tclMsgQThreader::vExecute()
{
   m_bThreadAlive = TRUE;

   // Loop through when Thread is alive.
   while (TRUE == m_bThreadAlive)
   {
      OSAL_tEventMask evntMask   =  0;

      if (OSAL_OK == OSAL_s32EventWait(m_hEvntHandle
            , m_coEvtFieldAttrib.m_evntMask, m_coEvtFieldAttrib.m_enFlags
            , m_coEvtFieldAttrib.m_evntTimeOut, &evntMask))
      {
         if (IHL_THREAD_TERMINATE_EVENT & evntMask)
         {
            // Terminate event has arrived, make the Thread alive flag FALSE and 
            // exit the looping
            m_bThreadAlive =  FALSE;
            break;   
         }  // if(ihl_THREAD_TERMINATE_EVENT & evntMask)

         tS32 s32Error = OSAL_s32EventPost(m_hEvntHandle, ~evntMask, OSAL_EN_EVENTMASK_AND);
         NORMAL_M_ASSERT(OSAL_ERROR != s32Error);
         
         // Check for app events.
         if (IHL_THREAD_APP_EVENTS & evntMask)
         {
            if(OSAL_NULL != m_cpoThreadable)
            {
               // Execute the customized threadable function
               m_cpoThreadable->vExecute(evntMask);

            }  // if(OSAL_NULL != m_cpoThreadable)

         }  // if(IHL_THREAD_APP_EVENTS & evntMask)

         // Check for new message event.
         if (IHL_THREAD_NEW_MSG_EVENT & evntMask)
         {
            // Dispatch messages.
            vOnMessage();
         }  // if(IHL_THREAD_NEW_MSG_EVENT & evntMask)

      }  // if (OSAL_OK == OSAL_s32EventWait(m_hEvntHa...
      else
      {
         tU32 u32ErrorCode =  OSAL_u32ErrorCode();

         switch (u32ErrorCode)
         {
            case OSAL_E_INVALIDVALUE:
            {
               // Event Field has been closed and deleted.
               // Just exit the looping
               m_bThreadAlive = FALSE;

            }  // case OSAL_E_INVALIDVALUE:
            break;

            case OSAL_E_EVENTINUSE:
            {
               // Assert for debugging purpose.
               NORMAL_M_ASSERT(OSAL_E_EVENTINUSE != u32ErrorCode); 
            }  // case OSAL_E_EVENTINUSE:, case OSAL_E_TIMEOUT:
            break;

            default:
            {
               // Assert for debugging purpose.
               // FATAL ERROR - ASSERT
               NORMAL_M_ASSERT_ALWAYS();
            }  // default:
            break;               
         }  // switch (u32ErrorCode)
      }  // End of if-else; if (OSAL_OK == OSAL_s32EventWait(m_hEvntHa...
   }  // while (TRUE == m_bThreadAlive)

   // Close the Event field
   vEventClose();

   // Close and delete the Message Queue.
   if (OSAL_C_INVALID_HANDLE != m_oMsgQHandle)
   {
      // Remove the callback mechanism
      tS32 s32RetVal =  OSAL_s32MessageQueueNotify(m_oMsgQHandle, OSAL_NULL, OSAL_NULL);
      NORMAL_M_ASSERT(OSAL_ERROR != s32RetVal);
      s32RetVal      =  OSAL_s32MessageQueueClose(m_oMsgQHandle);
      NORMAL_M_ASSERT(OSAL_ERROR != s32RetVal);
      s32RetVal      =  OSAL_s32MessageQueueDelete(m_coMsgQAttrib.m_oMsgQName.szValue);
      NORMAL_M_ASSERT(OSAL_ERROR != s32RetVal);
      m_oMsgQHandle  =  OSAL_C_INVALID_HANDLE;
   }

   if (OSAL_C_INVALID_HANDLE != m_hSemHandle)
   {
      // Synchronization mechanism - Semaphore post.
      (tVoid)OSAL_s32SemaphorePost(m_hSemHandle);
   }  // if(OSAL_C_INVALID_HANDLE != m_hSemHandle)
   
   // Exit the thread context
   OSAL_vThreadExit();
}  // tVoid ihl_tclMsgQThreader::vExecute()

/******************************************************************************
** FUNCTION:  virtual tVoid ihl_tclMsgQThreader::vOnMessage()
******************************************************************************/

/*virtual*/
tVoid ihl_tclMsgQThreader::vOnMessage()
{
	tU32 u32NumMsg =  0;

	if (OSAL_ERROR != OSAL_s32MessageQueueStatus(m_oMsgQHandle, OSAL_NULL
		, OSAL_NULL, &u32NumMsg))
	{
		ETG_TRACE_USR4(("ihl_tclMsgQThreader::vOnMessage NumMsg: %d", u32NumMsg));

		while ((0 < u32NumMsg) && (OSAL_NULL != m_cpoMsgQThreadable))
		{
			// Get the buffer to read the message
			ihl_tMessage* poMessage =  m_cpoMsgQThreadable->poGetMsgBuffer();
			NORMAL_M_ASSERT(OSAL_NULL != poMessage);

		if(OSAL_NULL != poMessage)
		 {
			 if((OSAL_NULL != poMessage->m_pu8Buffer)
				 && (0 < OSAL_s32MessageQueueWait(m_oMsgQHandle, poMessage->m_pu8Buffer
				 , poMessage->m_u32Length, OSAL_NULL, OSAL_C_TIMEOUT_NOBLOCKING)))
			 {
				 // Dispatch the message for processing.
				 m_cpoMsgQThreadable->vExecute(poMessage);

				 // Decrement the number of messages
				 --u32NumMsg;
			 }  // if((OSAL_NULL != poMessage) && (0 < OSAL_s32MessageQueueWait(m..
			 else
			 {
				 NORMAL_M_ASSERT(OSAL_NULL != poMessage->m_pu8Buffer);
				 NORMAL_M_ASSERT(0 < poMessage->m_u32Length);
			 }  // End if-else; if((OSAL_NULL != poMessage) && (0 < OSAL_s32Messa...
		}//if(OSAL_NULL != poMessage)

			// Query the Queue status again to continue dispatching.
			if ((0 == u32NumMsg) 
				&& (OSAL_ERROR == OSAL_s32MessageQueueStatus(m_oMsgQHandle, OSAL_NULL
				, OSAL_NULL, &u32NumMsg)))
			{
				// Reset the number of messages to 0; This would break the looping.
				u32NumMsg   =  0;

				// Since there was an error in querying the status, it is possible
				// we have new messages. Post an event back to thread so that
				// it will come back to the check the Queue status later after event
				// processing.
				(tVoid)bEventPost(IHL_THREAD_NEW_MSG_EVENT);
				break;
			}  // if ((0 == u32NumMsg) && (OSAL_ERROR == OSAL_s32MessageQueueSta..

		}  // while ((0 < u32NumMsg) && (OSAL_NULL != m_cpoMsgQThreadable))
	}  // if (OSAL_ERROR != OSAL_s32MessageQueueStatus(m_oMsgQHandle, OSAL_NULL..
	else
	{
		// Since there was an error in querying the status, it is possible
		// we have new messages. Post an event back to thread so that
		// it will come back to the check the Queue status later after event
		// processing.
		(tVoid)bEventPost(IHL_THREAD_NEW_MSG_EVENT);

	}  // End if-else; if (OSAL_ERROR != OSAL_s32MessageQueueStatus(m_oMsgQHan..

}  // tVoid ihl_tclMsgQThreader::vOnMessage()

/******************************************************************************
** FUNCTION:  OSAL_tMQueueHandle ihl_tclMsgQThreader::u32GetMsgQHandle()
******************************************************************************/

OSAL_tMQueueHandle ihl_tclMsgQThreader::u32GetMsgQHandle() const
{
   return m_oMsgQHandle;
}  // OSAL_tMQueueHandle ihl_tclMsgQThreader::u32GetMsgQHandle() const

/******************************************************************************
** FUNCTION:  static tVoid ihl_tclMsgQThreader::vOnNewMessage(tPVoid pvArg)
******************************************************************************/

/*static*/
tVoid ihl_tclMsgQThreader::vOnNewMessage(tPVoid pvArg)
{  
   if (OSAL_NULL != pvArg)
   {
      ihl_tclMsgQThreader* poMsgQThreader = reinterpret_cast<ihl_tclMsgQThreader*>(pvArg);
      tBool bRetVal  =  poMsgQThreader->bEventPost(IHL_THREAD_NEW_MSG_EVENT);
      NORMAL_M_ASSERT(FALSE != bRetVal);

   }  // if (OSAL_NULL != pvArg)

}  // tVoid ihl_tclMsgQThreader::vOnNewMessage(tPVoid pvArg)

   }  // namespace thread
}  // namespace ihl

///////////////////////////////////////////////////////////////////////////////
// <EOF>
