/*!
*******************************************************************************
* \file              ihl_tclMsgQThreader.h
*******************************************************************************
*  - PROJECT:        GM Gen2
*  - SW-COMPONENT:   Infotainment Helper Library
*  - DESCRIPTION:    Thread Utility Class - Threads as objects
*                    Command Processor Model Thread class
*                    Support for Message Queues & Event
*  - COPYRIGHT:      &copy; 2010 Robert Bosch Car Multimedia Gmbh
*******************************************************************************
* \date 23.06.2011 \version 1.0 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* \bug No known bugs
******************************************************************************/

#ifndef IHL_TCLMSGQTHREADER_H_
#define IHL_TCLMSGQTHREADER_H_

/******************************************************************************
| includes of component-internal interfaces, if necessary
| (scope: component-local)
|----------------------------------------------------------------------------*/

#include "ihl_tclThreader.h"

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/

#define IHL_MSG_Q_MAX_MSGS          25

/******************************************************************************
| typedefs (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| forward declarations
|----------------------------------------------------------------------------*/

namespace ihl {
   namespace thread {

      using namespace ihl::fi_helpers::midw_fi;

class ihl_tclMsgQThreadable;

/*! 
 * \struct ihl_rMsgQAttribute
 * \brief This structure contains the attributes required to setup the message 
 * queue
 */

typedef struct ihl_rMsgQAttribute
{
   /// Max number of messages
   tU32 m_u32MaxMessages;
   /// Maximum message size/length
   tU32 m_u32MaxMsgLen;
   /// Message Queue Name
   ihl_tclFiString m_oMsgQName;
   
   ihl_rMsgQAttribute(tU32 u32MaxMsgLen, tU32 u32MaxMessages = IHL_MSG_Q_MAX_MSGS)
      :m_u32MaxMessages(u32MaxMessages), m_u32MaxMsgLen(u32MaxMsgLen)
      , m_oMsgQName()
   {
      tU8 u8Buffer[OSAL_C_U32_MAX_NAMELENGTH]  =  {0};
      // Auto generate the message queue names.
      OSAL_vRandomSeed(OSAL_ClockGetElapsedTime());  // Randomize
      OSAL_s32PrintFormat((tChar*)&u8Buffer[0], "%x_%x_", OSAL_ThreadWhoAmI(), OSAL_s32Random());

      snprintf((tChar*)&u8Buffer[0], OSAL_C_U32_MAX_NAMELENGTH, "%x_%x_", OSAL_ThreadWhoAmI(), OSAL_s32Random());

      //OSAL_tVarArgList argList ={0};
      //OSAL_VarArgStart(argList, "%x_%x_");
      //OSAL_s32VarNPrintFormat((tChar*)&u8Buffer[0], OSAL_C_U32_MAX_NAMELENGTH, "%x_%x_", argList);
      //OSAL_VarArgEnd(argList);

      u8Buffer[OSAL_C_U32_MAX_NAMELENGTH - 1]   =  '\0';    // Null Termination protection
      ihl_tclFiString oTempString(((tCString)&u8Buffer[0]));
      oTempString += "MsgQ";
      m_oMsgQName = oTempString;
   };

   ihl_rMsgQAttribute(const ihl_rMsgQAttribute& rfcoMsgQAttrib)
      :m_u32MaxMessages(rfcoMsgQAttrib.m_u32MaxMessages)
      , m_u32MaxMsgLen(rfcoMsgQAttrib.m_u32MaxMsgLen)
      , m_oMsgQName(rfcoMsgQAttrib.m_oMsgQName)
   {};

   ihl_rMsgQAttribute& operator=(const ihl_rMsgQAttribute& rfcoMsgQAttrib)
   {
      // Identity Test - Self assignment test.
      if (this != &rfcoMsgQAttrib)
      {
         m_u32MaxMessages  =  rfcoMsgQAttrib.m_u32MaxMessages;
         m_u32MaxMsgLen    =  rfcoMsgQAttrib.m_u32MaxMsgLen;
         m_oMsgQName       =  rfcoMsgQAttrib.m_oMsgQName;
      }
      return *this;
   };
} ihl_tMsgQAttribute;

/*! 
 * \class ihl_tclMsgQThreader
 * \brief Customized threader class to support message queue along with events.
 */

class ihl_tclMsgQThreader : public ihl_tclThreader
{
public:
   
   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  ihl_tclMsgQThreader::ihl_tclMsgQThreader(const OSAL_trThre..
   ***************************************************************************/
   /*!
    * \brief   Parameterized Constructor, based on Dependency Injection 
    *          Principle (DIP)
    * \param   [rfcoThreadAttrib]:     (I)   Thread attributes
    * \param   [rfcoMsgQAttrib]:       (I)   Message Queue attributes
    * \param   [rfcoEvtFieldAttrib]:   (I)   Event Field attributes
    * \param   [cpoThreadable]:        (->I) Command Object
    * \param   [bExecThread]:          (I)   Flag to signal whether to spawn the 
    *          thread immediately after creation or not?
    * \retval  NONE
    * \date 23.06.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   ihl_tclMsgQThreader
   (
      const ihl_tThreadAttribute &rfcoThreadAttrib
      , const ihl_tMsgQAttribute &rfcoMsgQAttrib
      , const ihl_rEventFieldAttribute &rfcoEvtFieldAttrib      
      , ihl_tclMsgQThreadable* const cpoThreadable
      , tBool bExecThread  =  TRUE
   );

   /***************************************************************************
   ** FUNCTION:  virtual ihl_tclMsgQThreader::~ihl_tclMsgQThreader()
   ***************************************************************************/
   /*!
    * \brief   Destructor
    * \param   NONE
    * \retval  NONE
    * \date 23.06.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual ~ihl_tclMsgQThreader();

  /***************************************************************************
   ** FUNCTION:  OSAL_tMQueueHandle ihl_tclMsgQThreader::u32GetMsgQHandle()
   ***************************************************************************/
   /*!
    * \brief   Get the message queue handle
    * \param   NONE
    * \retval  [OSAL_tMQueueHandle]: OSAL Message Queue Handle
    * \date 23.06.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   OSAL_tMQueueHandle u32GetMsgQHandle() const;

   /***************************************************************************
   ****************************END OF PUBLIC***********************************
   ***************************************************************************/

protected:

   /***************************************************************************
   *******************************PROTECTED************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  virtual tVoid ihl_tclMsgQThreader::vExecute()
   ***************************************************************************/
   /*!
    * \brief   Command interface to execute the commands using command objects.
    * \param   NONE
    * \retval  NONE
    * \date 23.06.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tVoid vExecute();

   /***************************************************************************
   ** FUNCTION:  virtual tVoid ihl_tclMsgQThreader::vOnMessage()
   ***************************************************************************/
   /*!
    * \brief   Dispatcher function on new message.
    * \param   NONE
    * \retval  NONE
    * \date 23.06.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tVoid vOnMessage();

    /*!
    * \addtogroup tclMem
    */
   /*! @{*/

   /// Message Queue attributes
   const ihl_tMsgQAttribute m_coMsgQAttrib;
   /// Reference to the Threadable.
   ihl_tclMsgQThreadable* const m_cpoMsgQThreadable; 
   /// Message Queue Handle
   OSAL_tMQueueHandle m_oMsgQHandle;

   /*! @}*/

   /***************************************************************************
   ****************************END OF PROTECTED********************************
   ***************************************************************************/

private:

   /***************************************************************************
   *********************************PRIVATE************************************
   ***************************************************************************/  
   
   /***************************************************************************
   ** FUNCTION:  static tVoid ihl_tclMsgQThreader::vOnNewMessage(tVoid *pvArg)
   ***************************************************************************/
   /*!
    * \brief   Callback function for new message notification.
    * \param   [pvArg]:    (->I) "this" pointer as void pointer
    * \retval  NONE
    * \date 23.06.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   static tVoid vOnNewMessage(tPVoid pvArg);

   /***************************************************************************
   ****************************END OF PRIVATE**********************************
   ***************************************************************************/

}; // class ihl_tclMsgQThreader : class ihl_tclThreadable

      }  // namespace thread
}  // namespace ihl

#endif   // #ifndef IHL_TCLMSGQTHREADER_H_

////////////////////////////////////////////////////////////////////////////////
// <EOF>
