/*!
********************************************************************************
* \file              ihl_tclTraceStreamable.cpp
********************************************************************************
*  - PROJECT:        GM Gen2
*  - SW-COMPONENT:   Infotainment Helper Library 
*  - DESCRIPTION:    Generalized Trace Streamer which can handle trace commands
*  - COPYRIGHT:      &copy; 2010 Robert Bosch GmbH, Hildesheim
********************************************************************************
* \date 17.11.2010 \version 1.0 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* \bug No known bugs
*******************************************************************************/

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/

#include "ihl_tclTraceStreamable.h"
#include "ihl_Trace.h"

#define AHL_S_IMPORT_INTERFACE_GENERIC
#define AHL_S_IMPORT_INTERFACE_CCA_EXTENSION
#include <ahl_if.h>

#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include <generic_msgs_if.h>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_IHL_LOOPBACK
#include "trcGenProj/Header/ihl_tclTraceStreamable.cpp.trc.h"
#endif

/******************************************************************************
| defines and macros (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| function prototype (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| function implementation (scope: external-interfaces)
|----------------------------------------------------------------------------*/

namespace ihl {
   namespace loopback {

/******************************************************************************
** FUNCTION:  ihl_tclTraceStreamable::ihl_tclTraceStreamable(ahl_tc..
******************************************************************************/

/*explicit*/
ihl_tclTraceStreamable::ihl_tclTraceStreamable
(
   ahl_tclBaseOneThreadApp *const cpoApp
   , tU16 u16MajorVer
):m_coMainApp(cpoApp), m_oCmdMapper(), m_u16MajorVer(u16MajorVer)
{}

/******************************************************************************
** FUNCTION:  virtual ihl_tclTraceStreamable::~ihl_tclTraceStreamab..
******************************************************************************/

/*virtual*/
ihl_tclTraceStreamable::~ihl_tclTraceStreamable()
{
   try //try  // Currently disabled - try-catch construct giving Lint Prio1
   {
      for (ihl_tCmdMapIter iterMap = m_oCmdMapper.begin(); 
         iterMap != m_oCmdMapper.end(); ++iterMap)
      {
         OSAL_DELETE (iterMap->second);
         iterMap->second = OSAL_NULL;
      }

      m_oCmdMapper.clear();
   }    // try
   catch(...)//catch (...)
   {
   }  // catch (...)
   //

}  // ihl_tclTraceStreamable::~ihl_tclTraceStreamable()

/******************************************************************************
** FUNCTION:  tBool ihl_tclTraceStreamable::bStream(tU8 const* const cpu..
******************************************************************************/

tBool ihl_tclTraceStreamable::bStream(tU8 const* const cpu8Buffer)
{
   tBool bRetVal  =  FALSE;
   
   tU16 u16Cmd    =  ((tU16)cpu8Buffer[2]|((tU16)cpu8Buffer[1]<<8));  

   ihl_tCmdMapIter iterMap = m_oCmdMapper.find(u16Cmd);

   if (m_oCmdMapper.end() != iterMap)
   {
      bRetVal  =  (*(iterMap->second))(cpu8Buffer);
   }  // if(m_oCmdMapper.end() != iterMap)
   else
   {
      // Trace streamer did not find any handler function
      // so using default trace streamer loopback.
      ETG_TRACE_USR4(("Using default trace stream loopback.."));

      bRetVal  =  bTraceStream(cpu8Buffer);

   }  // End of if-else; if(m_oCmdMapper.end() != iterMap)
   
   ETG_TRACE_USR4(("TraceStreamer MapSize: %d, searched command: %x, Ret: %d"
      , m_oCmdMapper.size(), u16Cmd, ETG_CENUM(ihl_tenSuccess, bRetVal)));

   return bRetVal;

}  // tBool ihl_tclTraceStreamable::bStream(tU8 const* const cpu8Buffer)

/******************************************************************************
** FUNCTION:  tVoid ihl_tclTraceStreamable::vAddCmd(tU16 u16Cmd, ipodau..
******************************************************************************/

tVoid ihl_tclTraceStreamable::vAddCmd
(
   tU16 u16Cmd
   , ihl_tclCmdFunctor* poCmdFunctor
)
{
   ihl_tCmdMapIter iterMap = m_oCmdMapper.find(u16Cmd);

   // Command not found, hence add.
   if (m_oCmdMapper.end() != iterMap)
   {
      // Command already exists in the map.
      ETG_TRACE_ERR(("Command already exists in the trace loopback streamer."));
   }
   else
   {
      (tVoid)m_oCmdMapper.insert(std::make_pair(u16Cmd, poCmdFunctor));
   }
}  // tVoid ihl_tclTraceStreamable::vAddCmd(tU16 u16Cmd, ihl_tclCmdF..

/******************************************************************************
** FUNCTION:  tBool ihl_tclTraceStreamable::bSendMsg(const fi_tclMessageBase..
******************************************************************************/

tBool ihl_tclTraceStreamable::bSendMsg(const fi_tclMessageBase& rfcoMsgBase) const
{
   tBool bRetVal  =  TRUE;

   fi_tclVisitorMessage oMsg(rfcoMsgBase.corfoGetTypeBase(), m_u16MajorVer);

   oMsg.vInitServiceData
   ( 
      m_coMainApp->u16GetAppId()                // Source app-ID
      , m_coMainApp->u16GetAppId()              // Dest. app-ID
      , AMT_C_U8_CCAMSG_STREAMTYPE_NODATA       // Stream type
      , 0                                       // Stream counter
      , 0                                       // Reg ID
      , 0                                       // Command counter
      , rfcoMsgBase.u16GetServiceID()           // Service-ID
      , rfcoMsgBase.u16GetFunctionID()          // Function-ID
      , rfcoMsgBase.u8GetOpCode()               // OpCode
   );

   if (TRUE == oMsg.bIsValid())
   {
      ETG_TRACE_USR4(("Posting self message."));

      ail_tenCommunicationError enCommError = 
         m_coMainApp->enPostMessage(&oMsg, TRUE);

      if (AIL_EN_N_NO_ERROR != enCommError)
      {
         ETG_TRACE_ERR(("Posting the self message returned an error: %d"
            , ETG_ENUM(AIL_ERROR, enCommError)));
         bRetVal  =  FALSE;
      }
   }
   else
   {
      ETG_TRACE_ERR(("Self message is not valid"));
      bRetVal  =  FALSE;
   }

   return bRetVal;
}  // tBool ihl_tclTraceStreamable::bSendMsg(const fi_tclTypeBase &rfcoTyp..

/*******************************************************************************
** FUNCTION:  tBool ihl_tclTraceStreamable::bTraceStream(tU8 const* const ..
*******************************************************************************/

tBool ihl_tclTraceStreamable::bTraceStream(tU8 const* const cpu8Buffer) const
{
   tBool bRetVal  =  TRUE;

   // Create a Stream message.
   gm_tclStreamMessage oMsg
      (
         m_coMainApp->u16GetAppId()                // Source app-ID
         , m_coMainApp->u16GetAppId()              // Dest. app-ID
         , 0                                       // Reg ID
         , 0                                       // Command counter
         , IHL_C_U16_SRV_TRACE_LOOPBACK            // ServiceId
         , IHL_C_U16_TRACE_STREAM_FID              // FID
         , IHL_C_U8_OPCODE_LOOPBACK                // Opcode
         , cpu8Buffer[0]                           // Buffer Size
      );
   
   // Set the data from trace into the stream.
   oMsg.vSetData((const tChar*)(&cpu8Buffer[1]));
   
   if (TRUE == oMsg.bIsValid())
   {
      ETG_TRACE_USR4(("Posting Trace stream self message."));

      ail_tenCommunicationError enCommError = 
         m_coMainApp->enPostMessage(&oMsg, TRUE);

      if (AIL_EN_N_NO_ERROR != enCommError)
      {
         ETG_TRACE_ERR(("Posting Trace stream returned an error: %d"
            , ETG_ENUM(AIL_ERROR, enCommError)));
         bRetVal  =  FALSE;
      }
   }
   else
   {
      ETG_TRACE_ERR(("Self message is not valid"));
      bRetVal  =  FALSE;
   }

   return bRetVal;

}  // tBool ihl_tclTraceStreamable::bTraceStream(tU8 const* const cpu8Buffer) ..


   }  // namespace loopback
}  // namespace ihl

////////////////////////////////////////////////////////////////////////////////

// <EOF>
