/*!
********************************************************************************
* \file              ihl_tclTraceStreamable.h
********************************************************************************
*  - PROJECT:        GM Gen2
*  - SW-COMPONENT:   Infotainment Helper Library 
*  - DESCRIPTION:    Generalized Trace Streamer which can handle trace commands
*  - COPYRIGHT:      &copy; 2010 Robert Bosch GmbH, Hildesheim
********************************************************************************
* \date 17.11.2010 \version 1.0 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* \bug No known bugs
*******************************************************************************/

#ifndef IHL_TCLTRACESTREAMABLE_H_
#define IHL_TCLTRACESTREAMABLE_H_

/******************************************************************************
| includes of component-internal interfaces, if necessary
| (scope: component-local)
|----------------------------------------------------------------------------*/

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include <osal_if.h>

// Includes to utilize the BP STL exceptional handler
#include <iterator>
#include <map>


//using namespace std;

#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
#include <common_fi_if.h>

#include "ihl_tclTraceCmdFunctor.h"

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/

#define IHL_C_U16_TRACE_STREAM_FID                 0xFFFE
#define IHL_C_U16_SRV_TRACE_LOOPBACK               0xFFFE
#define IHL_C_U8_OPCODE_LOOPBACK                   CCA_C_U8_OPCODE_METHODSTART

/******************************************************************************
| typedefs (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| forward declarations (scope: global)
|----------------------------------------------------------------------------*/

class ahl_tclBaseOneThreadApp;

namespace ihl {
   namespace loopback {

/*! 
 * \class ihl_tclTraceStreamable
 * \brief Trace Command Streamable class - Abstract base class.
 * This class provides a trace command dispatcher interface. 
 * The dispatcher mechanism is generic as this depends on the command mapping, 
 * which has to be performed by the concrete class by deriving from this base
 * class.
 * The derived class uses the vSetupCmds() interface to setup the commands which
 * need to be supported.
 */

class ihl_tclTraceStreamable
{
public:
   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  ihl_tclTraceStreamable::ihl_tclTraceStreamable(ahl_tclB..
   ***************************************************************************/
   /*!
    * \brief   Parameterized Constructor, based on Dependency Injection 
    *          Principle (DIP)
    * \param   [cpoApp]:        (->I) Pointer to the main application
    * \retval  NONE
    * \date 17.11.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   explicit ihl_tclTraceStreamable(ahl_tclBaseOneThreadApp* const cpoApp
      , tU16 u16MajorVer FI_DEFAULT_VERSION);

   /***************************************************************************
   ** FUNCTION:  virtual ihl_tclTraceStreamable::~ihl_tclTraceStreamable()
   ***************************************************************************/
   /*!
    * \brief   Destructor
    * \param   NONE
    * \retval  NONE
    * \date 17.11.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual ~ihl_tclTraceStreamable();

   /***************************************************************************
   ** FUNCTION:  tBool ihl_tclTraceStreamable::bStream(tU8 const* const cpu..
   ***************************************************************************/
   /*!
    * \brief   Streamer interface which performs the mapping based on the input
    *          trace stream provided.
    * \param   [cpu8Buffer]:   (->I) Input trace channel buffer.
    * \retval  [tBool]: TRUE if streaming was successful, FALSE otherwise.
    * \date 17.11.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tBool bStream(tU8 const* const cpu8Buffer);
   
   /***************************************************************************
   ** FUNCTION:  virtual tVoid ihl_tclTraceStreamable::vSetupCmds() = 0
   ***************************************************************************/
   /*!
    * \brief   The function sets up the commands supported via trace.
    *          This is a virtual function which has to be overloaded by
    *          deriving, to provide a component specific implementation.
    *          This is used for mapping trace commands provided via the Trace 
    *          input channel interface.
    * \param   NONE
    * \retval  NONE
    * \date 17.11.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tVoid vSetupCmds() {};
   
   /***************************************************************************
   ****************************END OF PUBLIC***********************************
   ***************************************************************************/

protected:

   /***************************************************************************
   *******************************PROTECTED************************************
   ***************************************************************************/
   
   /***************************************************************************
   ** FUNCTION:  tVoid ihl_tclTraceStreamable::vAddCmd(tU16 u16Cmd, ihl_..
   ***************************************************************************/
   /*!
    * \brief   Adds the Command & Command functor to the stream map.
    * \param   [u16Cmd]:         (I) Trace Command
    * \param   [poCmdFunctor]:   (->I) Command Functor
    * \retval  NONE
    * \date 17.11.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tVoid vAddCmd(tU16 u16Cmd, ihl_tclCmdFunctor* poCmdFunctor);   

   /***************************************************************************
   ** FUNCTION:  tBool ihl_tclTraceStreamable::bSendMsg(const fi_tclType..
   ***************************************************************************/
   /*!
    * \brief   Posts a Self message 
    * \param   [rfcoMsgBase]:   (I) Fi Message Base type object
    * \retval  [tBool]: TRUE if message post is successful, FALSE otherwise.
    * \date 17.11.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tBool bSendMsg(const fi_tclMessageBase& rfcoMsgBase) const;

   /***************************************************************************
   ** FUNCTION:  tBool ihl_tclTraceStreamable::bTraceStream(tU8 const* const ..
   ***************************************************************************/
   /*!
    * \brief   Function to directly stream the bytes from Trace to component 
    *          through
    * \param   [rfcoTypeBase]:   (I) Fi Base type object
    * \retval  [tBool]: TRUE if message post is successful, FALSE otherwise.
    * \date 17.11.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tBool bTraceStream(tU8 const* const cpu8Buffer) const;

   /***************************************************************************
   ** FUNCTION:  ihl_tclTraceStreamable::ihl_tclTraceStreamable()
   ***************************************************************************/
   /*!
    * \brief   Default Constructor
    * \param   NONE
    * \retval  NONE
    * \note    Default constructor is declared protected to disable it. So 
    *          that any attempt to create without parameter will be caught by 
    *          the compiler.
    * \date 17.11.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   ihl_tclTraceStreamable();   // No definition exists.

   /***************************************************************************
   ** FUNCTION:  ihl_tclTraceStreamable::ihl_tclTraceStreamable(const audc..
   ***************************************************************************/
   /*!
    * \brief   Copy Constructor
    * \param   [rfcoTraceStrm]: (I) Const reference to object to be copied
    * \retval  NONE
    * \note    Default copy constructor is declared protected to disable it. So 
    *          that any attempt to copy will be caught by the compiler.
    * \date 17.11.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   ihl_tclTraceStreamable(const ihl_tclTraceStreamable& rfcoTraceStrm);

   /***************************************************************************
   ** FUNCTION:  ihl_tclTraceStreamable& ihl_tclTraceStreamable::operator=()
   ***************************************************************************/
   /*!
    * \brief   Assignment Operator
    * \param   [rfcoTraceStrm]: (I) Const reference to object to be copied
    * \retval  [ihl_tclTraceStreamable&]: Reference to this pointer.
    * \note    Assignment operator is declared protected to disable it. So 
    *          that any attempt for assignment will be caught by the compiler.
    * \date 17.11.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   ihl_tclTraceStreamable& operator=(const ihl_tclTraceStreamable& rfcoTraceStrm);

   /***************************************************************************
   ****************************END OF PROTECTED********************************
   ***************************************************************************/

private:

   /***************************************************************************
   *********************************PRIVATE************************************
   ***************************************************************************/
   
   /*! 
    * \typedef map<tU16, ihl_tclCmdFunctor*> ihl_tCmdMapper
    * \brief Trace command to Functor Mapper data structure.
    */

   typedef std::map<tU16, ihl_tclCmdFunctor*> ihl_tCmdMapper;

   /*! 
    * \typedef map<tU16, ihl_tclCmdFunctor*>::iterator ihl_tCmdMapIter
    * \brief Trace command to Functor Mapper data structure iterator.
    */
   typedef std::map<tU16, ihl_tclCmdFunctor*>::iterator ihl_tCmdMapIter;
   
   /*!
    * \addtogroup tclMem
    */
   /*! @{*/

   /// Main application pointer.
   ahl_tclBaseOneThreadApp* const m_coMainApp;

   /// Trace Streamer Mapper object.
   ihl_tCmdMapper m_oCmdMapper;

   /// Major version of FI
   tU16 m_u16MajorVer;

   /*! @}*/

   /***************************************************************************
   ****************************END OF PRIVATE**********************************
   ***************************************************************************/
}; // class ihl_tclTraceStreamable

   }  // namespace loopback
}  // namespace ihl

#endif   // #ifndef IHL_TCLTRACESTREAMER_H_

////////////////////////////////////////////////////////////////////////////////
// <EOF>
