/*!
********************************************************************************
* \file              ihl_tclTraceCmdFunctor.h
********************************************************************************
*  - PROJECT:        GM Gen2
*  - SW-COMPONENT:   Infotainment Helper Library 
*  - DESCRIPTION:    Generalized Trace command functor - Functors are functions 
*                    with a state.
*  - COPYRIGHT:      &copy; 2010 Robert Bosch GmbH, Hildesheim
********************************************************************************
* \date 17.11.2010 \version 1.0 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* \bug No known bugs
*******************************************************************************/

#ifndef IHL_TCLTRACECMDFUNCTOR_H_
#define IHL_TCLTRACECMDFUNCTOR_H_

/******************************************************************************
| includes of component-internal interfaces, if necessary
| (scope: component-local)
|----------------------------------------------------------------------------*/

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include <osal_if.h>

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| typedefs (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| forward declarations (scope: global)
|----------------------------------------------------------------------------*/

namespace ihl {
   namespace loopback {

/*! 
 * \class ihl_tclCmdFunctor
 * \brief Command Functor - Abstract base class.
 *
 * Functors are functions with a state.Functors can encapsulate C and C++ 
 * function pointers employing the concepts templates and polymorphism.
 * Functors can be used to build up a list of pointers to member functions of 
 * arbitrary classes and call them all through the same interface without 
 * bothering about the need of a pointer to an instance or their class. 
 * All the functions should have the same return-type and calling parameters. 
 * Functors are also known as closures or Functionoids. Functors can be used to
 * implement callbacks.
 */

class ihl_tclCmdFunctor
{
public:
   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  ihl_tclCmdFunctor::ihl_tclCmdFunctor()
   ***************************************************************************/
   /*!
    * \brief   Default Constructor
    * \param   NONE
    * \retval  NONE
    * \date 17.11.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   ihl_tclCmdFunctor(){};

   /***************************************************************************
   ** FUNCTION:  virtual ihl_tclCmdFunctor::~ihl_tclCmdFunctor()
   ***************************************************************************/
   /*!
    * \brief   Destructor
    * \param   NONE
    * \retval  NONE
    * \date 17.11.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual ~ihl_tclCmdFunctor(){};

   /***************************************************************************
   ** FUNCTION:  virtual tBool ihl_tclCmdFunctor::operator()(tU8 const* ..
   ***************************************************************************/
   /*!
    * \brief   The function executes the callback function.
    *          This is a pure virtual function which has to be overloaded by
    *          deriving, to provide an implementation.
    *          This is used by the Trace input channel interface.
    * \param   [cpu8Buffer]: (->I)  Pointer to the input buffer coming from 
    *          Trace input channel.
    * \retval  [tBool]: Status of callback mechanism.
    * \date 17.11.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tBool operator()(tU8 const* const cpu8Buffer) const = 0;

   /***************************************************************************
   ****************************END OF PUBLIC***********************************
   ***************************************************************************/
}; // class ihl_tclCmdFunctor

/*! 
 * \class ihl_tclTraceCmdFunctor
 * \brief Generic template class implementing the callback mechanism.
 *
 */

template <class ihl_tclGenericClass>
class ihl_tclTraceCmdFunctor : public ihl_tclCmdFunctor
{
public:

   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/

   /*! 
    * \typedef  tBool (ihl_tclGenericClass::*ihl_pfFuncCallbk)(tU8 const* const cpu8Buffer) const
    * \brief Callback function signature definition.
    */
   typedef  tBool (ihl_tclGenericClass::*ihl_pfFuncCallbk)(tU8 const* const cpu8Buffer) const;

   /***************************************************************************
   ** FUNCTION:  ihl_tclTraceCmdFunctor(ihl_tclGenericClass* poCmd, ihl_..
   ***************************************************************************/
   /*!
    * \brief   Parameterized Constructor - Dependency Injection Principle (DIP)
    * \param   [poCmd]:       (->I) Command Object pointer.
    * \param   [pfCallbk]:    (->I) Function pointer for callback.
    * \retval  NONE
    * \date 17.11.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   ihl_tclTraceCmdFunctor(ihl_tclGenericClass* poCmd, ihl_pfFuncCallbk pfCallbk)
      :m_poCmd(poCmd), m_pfCallbk(pfCallbk){};
   
   /***************************************************************************
   ** FUNCTION:  virtual tBool ihl_tclCmdFunctor::operator()(tU8 const* cons..
   ***************************************************************************/
   /*!
    * \brief   The function executes the callback function.
    *          Callback function is called from here using the command object.
    * \param   [cpu8Buffer]: (->I)  Pointer to the input buffer coming from 
    *          Trace input channel.
    * \retval  [tBool]: Status of callback mechanism.
    * \date 17.11.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tBool operator()(tU8 const* const cpu8Buffer) const
   {
      tBool bRetVal  = FALSE;
      if((OSAL_NULL != m_poCmd) && (OSAL_NULL != m_pfCallbk))
      {
         bRetVal  =  (*m_poCmd.*m_pfCallbk)(cpu8Buffer);   // Function callback.
      }
      return bRetVal;
   };

   /***************************************************************************
   ****************************END OF PUBLIC***********************************
   ***************************************************************************/

protected:

   /***************************************************************************
   *******************************PROTECTED************************************
   ***************************************************************************/
   
   /***************************************************************************
   ** FUNCTION:  ihl_tclTraceCmdFunctor::ihl_tclTraceCmdFunctor()
   ***************************************************************************/
   /*!
    * \brief   Default Constructor
    * \param   NONE
    * \retval  NONE
    * \note    Default constructor is declared protected to disable it. So 
    *          that any attempt to instantiate w/o parameter will be caught by 
    *          the compiler.
    * \date 17.11.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   ihl_tclTraceCmdFunctor();      // No definition exists.

   /***************************************************************************
   ****************************END OF PROTECTED********************************
   ***************************************************************************/

private:

   /***************************************************************************
   *********************************PRIVATE************************************
   ***************************************************************************/

   /*!
    * \addtogroup tclMem
    */
   /*! @{*/

   /// Command Object pointer
   ihl_tclGenericClass* m_poCmd;

   /// Function pointer for callback
   ihl_pfFuncCallbk m_pfCallbk;

   /*! @}*/

   /***************************************************************************
   ****************************END OF PRIVATE**********************************
   ***************************************************************************/

}; // class ihl_tclTraceCmdFunctor : public ihl_tclCmdFunctor

   }  // namespace loopback
}  // namespace ihl

#endif   // #ifndef IHL_TCLTRACECMDFUNCTOR_H_

////////////////////////////////////////////////////////////////////////////////
// <EOF>
