/*!
*******************************************************************************
* \file              ihl_tclDirHandler.cpp
*******************************************************************************
*  - PROJECT:        GM Gen2
*  - SW-COMPONENT:   Infotainment Helper Library
*  - DESCRIPTION:    Directory Handler - Exception safe Directory handler
*  - COPYRIGHT:      &copy; 2010 Robert Bosch Car Multimedia Gmbh
*******************************************************************************
* \date 06.12.2010 \version 1.0 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* \bug No known bugs
******************************************************************************/

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/

#include "ihl_tclDirHandler.h"

/******************************************************************************
| defines and macros (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| function prototype (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| function implementation (scope: external-interfaces)
|----------------------------------------------------------------------------*/

namespace ihl {
   namespace io {

/******************************************************************************
** FUNCTION:  ihl_tclDirHandler::ihl_tclDirHandler(tCString szDirectory)
******************************************************************************/

/*explicit*/
ihl_tclDirHandler::ihl_tclDirHandler(tCString szDirectory)
:m_pioCtrlDir(OSALUTIL_prOpenDir(szDirectory)), m_u32ErrorCode(OSAL_E_NOERROR)
{}  // ihl_tclDirHandler::ihl_tclDirHandler(tCString szDirectory)

/******************************************************************************
** FUNCTION:  virtual ihl_tclDirHandler::~ihl_tclDirHandler()
******************************************************************************/

/*virtual*/
ihl_tclDirHandler::~ihl_tclDirHandler()
{
   if(OSAL_NULL != m_pioCtrlDir)
   {
      tS32 s32StatCode = OSALUTIL_s32CloseDir(m_pioCtrlDir);
      NORMAL_M_ASSERT(OSAL_ERROR != s32StatCode);

      // Invalidate the IO Control directory structure
      m_pioCtrlDir   =  OSAL_NULL;
   }  // if(OSAL_NULL != m_pioCtrlDir)
}  // ihl_tclDirHandler::~ihl_tclDirHandler()

/***************************************************************************
** FUNCTION:  virtual tBool ihl_tclDirHandler::bIsValid() const
***************************************************************************/

/*virtual*/
tBool ihl_tclDirHandler::bIsValid() const
{
   return(OSAL_NULL != m_pioCtrlDir);
}  // tBool ihl_tclDirHandler::bIsValid() const

/******************************************************************************
** FUNCTION:  virtual tBool ihl_tclDirHandler::bMkDir(tCString szDirectory)
******************************************************************************/

/*virtual*/
tBool ihl_tclDirHandler::bMkDir(tCString szDirectory)
{
   tBool bRetVal  =  FALSE;

   if(OSAL_NULL != m_pioCtrlDir)
   {
      if(OSAL_ERROR != OSALUTIL_s32CreateDir(m_pioCtrlDir->fd, szDirectory))
      {
         bRetVal  =  TRUE;
         // Reset the Error code.
         m_u32ErrorCode =  OSAL_E_NOERROR;
      }  // if(OSAL_ERROR != OSALUTIL_s32CreateDir(m_pioCtrlDir->fd, szDirecto..
      else
      {
         // Query & Set the error code.
         m_u32ErrorCode =  OSAL_u32ErrorCode();
      }  // End of if-else; if(OSAL_ERROR != OSALUTIL_s32CreateDir(m_pioCtrl..
   }  // if(OSAL_NULL != m_pioCtrlDir)

   return (bRetVal);
}  // tBool ihl_tclDirHandler::bMkDir(tCString szDirectory)

/******************************************************************************
** FUNCTION:  virtual tBool ihl_tclDirHandler::bRmDir(tCString szDirectory)
******************************************************************************/

/*virtual*/
tBool ihl_tclDirHandler::bRmDir(tCString szDirectory)
{
   tBool bRetVal  =  FALSE;

   if(OSAL_NULL != m_pioCtrlDir)
   {     
      /// \brief A directory can be removed only if the directory is empty. 
      /// Reference -  
      /// <A HREF="http://www.opengroup.org/onlinepubs/009695399/functions/rmdir.html">
      /// The Open Group Base Spec Issue 6, IEEE Std 1003.1, 2004 Edition</A>.
      ///
      /// If the directory has to be removed the files in the directory have to 
      /// be removed first by iterating through the directory files list.
      /// Using the OSAL IO Control functionality to remove files recursively

      if(OSAL_ERROR != OSAL_s32IOControl(m_pioCtrlDir->fd, OSAL_C_S32_IOCTRL_FIORMRECURSIVE, (intptr_t)szDirectory))
      {
         bRetVal  =  TRUE;
         // Reset the Error code.
         m_u32ErrorCode =  OSAL_E_NOERROR;
      }  // if(OSAL_ERROR != OSAL_s32IOControl(m_pioCtrlDir->fd..
      else
      {
         // Query & Set the error code.
         m_u32ErrorCode =  OSAL_u32ErrorCode();
      }  // End of if-else; if(OSAL_ERROR != OSAL_s32IOControl(m_pioCtrlDir->fd.
   }  // if(OSAL_NULL != m_pioCtrlDir)

   return (bRetVal);
}  // tBool ihl_tclDirHandler::bRmDir(tCString szDirectory)

/******************************************************************************
** FUNCTION:  virtual OSAL_trIOCtrlDirent* ihl_tclDirHandler::prReadDir()
******************************************************************************/

/*virtual*/
OSAL_trIOCtrlDirent* ihl_tclDirHandler::prReadDir()
{
   OSAL_trIOCtrlDirent* pDirent  =  OSALUTIL_prReadDir(m_pioCtrlDir);
   
   if(OSAL_NULL != pDirent)
   {
      m_u32ErrorCode =  OSAL_E_NOERROR;
   }  // if(OSAL_NULL != pDirent)
   else
   {
      // :NOTE: From the description of the POSIX - readdir(), in case of reaching
      // end of directory, a NULL pointer is returned without changing the error
      // code value. This means that the value could be old error value if queried
      // Hence, here we explicitly set to resources does not exists as error code.
      m_u32ErrorCode =  OSAL_E_DOESNOTEXIST;
   }  // End of if-else; if(OSAL_NULL != pDirent)

   return pDirent;
}  // OSAL_trIOCtrlDirent* ihl_tclDirHandler::prReadDir()

/******************************************************************************
** FUNCTION:  tU32 ihl_tclDirHandler::u32ErrorCode() const
******************************************************************************/

tU32 ihl_tclDirHandler::u32ErrorCode() const
{
   return (m_u32ErrorCode);
}  // tU32 ihl_tclDirHandler::u32ErrorCode() const

   }  // namespace io
}  // namespace ihl

///////////////////////////////////////////////////////////////////////////////

// <EOF>
