/*!
*******************************************************************************
* \file              ihl_tclDirHandler.h
*******************************************************************************
*  - PROJECT:        GM Gen2
*  - SW-COMPONENT:   Infotainment Helper Library
*  - DESCRIPTION:    Directory Handler - Exception safe Directory handler
*  - COPYRIGHT:      &copy; 2010 Robert Bosch Car Multimedia Gmbh
*******************************************************************************
* \date 06.12.2010 \version 1.0 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* \bug No known bugs
******************************************************************************/

#ifndef IHL_TCLDIRHANDLER_H_
#define IHL_TCLDIRHANDLER_H_

/******************************************************************************
| includes of component-internal interfaces, if necessary
| (scope: component-local)
|----------------------------------------------------------------------------*/

// Includes to utilize the OSAL Interface
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include <osal_if.h>

// Includes to utilize the BP STL exceptional handler



/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/

namespace ihl {
   namespace io {

/*! 
 * \class ihl_tclDirHandler 
 * \brief Exception safe directory handler for performing directory operations.
 *
 * This is based on design pattern \ref RAII "RAII"
 */

class ihl_tclDirHandler
{
public:

   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  ihl_tclDirHandler::ihl_tclDirHandler(tCString szDirectory)
   ***************************************************************************/
   /*!
    * \brief   Parameterized Constructor
    * \param   [szDirectory]:  (->I) Directory name (inclusive of path)
    * \retval  NONE
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   explicit ihl_tclDirHandler(tCString szDirectory);

   /***************************************************************************
   ** FUNCTION:  virtual ihl_tclDirHandler::~ihl_tclDirHandler()
   ***************************************************************************/
   /*!
    * \brief   Destructor
    * \param   NONE
    * \retval  NONE
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual ~ihl_tclDirHandler();

   /***************************************************************************
   ** FUNCTION:  virtual tBool ihl_tclDirHandler::bIsValid() const
   ***************************************************************************/
   /*!
    * \brief   This function checks the validity of the directory control 
    *          structure.
    * \param   NONE
    * \retval  [tBool]: TRUE if directory control structure is valid, 
    *          FALSE otherwise.
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tBool bIsValid() const;

   /***************************************************************************
   ** FUNCTION:  virtual tBool ihl_tclDirHandler::bMkDir(tCString szDirectory)
   ***************************************************************************/
   /*!
    * \brief   This function to create a sub directory within the current 
    *          directory. This function corresponds to OSAL's 
    *          \c OSALUTIL_s32CreateDir().
    * \param   [szDirectory]: Sub directory name.
    * \retval  [tBool]: TRUE or FALSE in case of error.
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tBool bMkDir(tCString szDirectory);

   /***************************************************************************
   ** FUNCTION:  virtual tBool ihl_tclDirHandler::bRmDir(tCString szDirectory)
   ***************************************************************************/
   /*!
    * \brief   This function to remove a sub directory within the current 
    *          directory. This is a recursive call, hence all the contents of 
    *          this sub directory specified will be removed.
    *          This function corresponds to OSAL's \c OSAL_s32IOControl() with 
    *          parameter OSAL_C_S32_IOCTRL_FIORMRECURSIVE.
    * \param   [szDirectory]: Sub directory name.
    * \retval  [tBool]: TRUE or FALSE in case of error.
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tBool bRmDir(tCString szDirectory);

   /***************************************************************************
   ** FUNCTION:  virtual OSAL_trIOCtrlDirent* ihl_tclDirHandler::prReadDir()
   ***************************************************************************/
   /*!
    * \brief   This function reads the content of a directory.  This function 
    *          corresponds to OSAL's \c OSALUTIL_prReadDir()
    * \param   NONE
    * \retval  [OSAL_trIOCtrlDirent*]: : Pointer to directory entry or OSAL_NULL
    *          in case of error 
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual OSAL_trIOCtrlDirent* prReadDir();

   /***************************************************************************
   ** FUNCTION:  tU32 ihl_tclDirHandler::u32ErrorCode() const
   ***************************************************************************/
   /*!
    * \brief   This function returns the last error code during the file 
    *          handling
    * \param   NONE
    * \retval  [tU32]: Error code
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tU32 u32ErrorCode() const;

   /***************************************************************************
   ****************************END OF PUBLIC***********************************
   ***************************************************************************/

protected:
   
   /***************************************************************************
   *******************************PROTECTED************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  ihl_tclDirHandler::ihl_tclDirHandler()
   ***************************************************************************/
   /*!
    * \brief   Default Constructor
    * \param   NONE
    * \retval  NONE
    * \note    Default constructor is declared protected to disable default 
    *          construction.
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   ihl_tclDirHandler();

   /***************************************************************************
   ** FUNCTION:  ihl_tclDirHandler::ihl_tclDirHandler(const ihl_tclDirHandler..)
   ***************************************************************************/
   /*!
    * \brief   Copy Constructor
    * \param   [rfcoDirHandler]: (I) Const reference to object to be copied
    * \retval  NONE
    * \note    Default copy constructor is declared protected to disable it. So 
    *          that any attempt to copy will be caught by the compiler.
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   ihl_tclDirHandler(const ihl_tclDirHandler &rfcoDirHandler);

   /***************************************************************************
   ** FUNCTION:  ihl_tclDirHandler& ihl_tclDirHandler::operator=(const ..)
   ***************************************************************************/
   /*!
    * \brief   Assignment Operator
    * \param   [rfcoDirHandler]: (I) Const reference to object to be copied
    * \retval  [ihl_tclDirHandler&]: Reference to this pointer.
    * \note    Assignment operator is declared protected to disable it. So 
    *          that any attempt for assignment will be caught by the compiler.
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   ihl_tclDirHandler& operator=(const ihl_tclDirHandler &rfcoDirHandler);

   /*!
    * \addtogroup tclMem
    */
   /*! @{*/

   /// Resource : IO Control Structure for Directory to be acquired
   OSAL_trIOCtrlDir* m_pioCtrlDir;
   /// Last Error code
   tU32 m_u32ErrorCode;                      

   /*! @}*/

   /***************************************************************************
   ****************************END OF PROTECTED********************************
   ***************************************************************************/

private:

   /***************************************************************************
   *********************************PRIVATE************************************
   ***************************************************************************/

   /***************************************************************************
   ****************************END OF PRIVATE**********************************
   ***************************************************************************/
}; // class ihl_tclDirHandler

/******************************************************************************
| typedefs (scope: global)
|----------------------------------------------------------------------------*/

/*! 
 * \typedef ihl_tclDirHandler ihl_tDirHandler
 * \brief Exception safe Directory handler
 *
 * The objects of this type will be registered for stack unwinding.
 * 
 * \note If you leave a function by exception, only destructors of registered 
 * stack objects will be called. That is contrary to C++ exception handling 
 * where the destructors will be called automatically by a mechanism called 
 * stack unwinding.
 *
 * Ref: Howto Using the Blaupunkt STL, v1.0, by Wiedemann Frank(CM-AI/PJ-CF15)
 * 
 */

typedef ihl_tclDirHandler ihl_tDirHandler;

   }  // namespace io
}  // namespace ihl

#endif   // #ifndef IHL_TCLDIRHANDLER_H_

// <EOF>
