/*!
*******************************************************************************
* \file              ihl_tclRawData.h
*******************************************************************************
*  - PROJECT:        GM Gen2
*  - SW-COMPONENT:   Infotainment Helper Library 
*  - DESCRIPTION:    Container class for Raw data handling
*  - COPYRIGHT:      &copy; 2010 Robert Bosch Car Multimedia Gmbh
*******************************************************************************
* \date 16.11.2010 \version 1.0 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* \bug No known bugs
******************************************************************************/

#ifndef IHL_TCLRAWDATA_H_
#define IHL_TCLRAWDATA_H_

/******************************************************************************
| includes of component-internal interfaces, if necessary
| (scope: component-local)
|----------------------------------------------------------------------------*/

// Includes to utilize the OSAL Interface
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include <osal_if.h>

// Includes to utilize the BP STL exceptional handler



/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/

namespace ihl {
   namespace buffer {

/*! 
 * \class ihl_tclRawData
 * \brief Raw data Container similar to a STL vector.
 * 
 * This is based on design pattern \ref RAII "RAII"
 * 
 */

class ihl_tclRawData
{
public:

   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  ihl_tclRawData::ihl_tclRawData()
   ***************************************************************************/
   /*!
    * \brief   Default Constructor
    * \param   NONE
    * \retval  NONE
    * \date 16.11.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   ihl_tclRawData();

   /***************************************************************************
   ** FUNCTION:  virtual ihl_tclRawData::~ihl_tclRawData()
   ***************************************************************************/
   /*!
    * \brief   Destructor
    * \param   NONE
    * \retval  NONE
    * \date 16.11.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual ~ihl_tclRawData();

   /***************************************************************************
   ** FUNCTION:  ihl_tclRawData::ihl_tclRawData(tCU32 cu32Size, tU8..
   ***************************************************************************/
   /*!
    * \brief   Parameterized Constructor
    * \param   [cu32Size]:   (I)   Raw data buffer size.
    * \param   [u8Value]:    (I)   Buffer Initializer value
    * \retval  NONE
    * \date 16.11.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   explicit ihl_tclRawData(tCU32 cu32Size, tU8 u8Value = 0);

   /***************************************************************************
   ** FUNCTION:  ihl_tclRawData::ihl_tclRawData(tCU8 cu8Buffer[],..)
   ***************************************************************************/
   /*!
    * \brief   Parameterized Constructor
    * \param   [cu8Buffer]:  (->I) Reference to array of data to be copied
    * \param   [cu32Size]:   (I)   Size of the array
    * \retval  NONE
    * \date 16.11.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   ihl_tclRawData(tCU8 cu8Buffer[], tCU32 cu32Size);

   /***************************************************************************
   ** FUNCTION:  ihl_tclRawData::ihl_tclRawData(const ihl_tclR..
   ***************************************************************************/
   /*!
    * \brief   Copy Constructor
    * \param   [rfcoRawData]: (I) Const reference to object to be copied
    * \retval  NONE
    * \date 16.11.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   ihl_tclRawData(const ihl_tclRawData &rfcoRawData);

   /***************************************************************************
   **********************************OPERATORS*********************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  ihl_tclRawData& ihl_tclRawData::operator=(const ..)
   ***************************************************************************/
   /*!
    * \brief   Assignment Operator
    * \param   [rfcoRawData]: (I) Const reference to object to be copied
    * \retval  [ihl_tclRawData&]: Reference to this pointer.
    * \date 16.11.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   ihl_tclRawData& operator=(const ihl_tclRawData &rfcoRawData);

   /***************************************************************************
   ** FUNCTION:  tU8& ihl_tclRawData::operator[](tCU32 cu32Index)
   ***************************************************************************/
   /*!
    * \brief   Array subscripting Operator - Mutable(Read-Write) interface
    *          Random access
    * \param   [cu32Index]:   (I) Index to be accessed.
    * \retval  [tU8&]:        Returns the u32Index'th element.
    * \date 16.11.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tU8& operator[](tCU32 cu32Index);

   /***************************************************************************
   ** FUNCTION:  tCU8& ihl_tclRawData::operator[](tCU32 cu32Index) const
   ***************************************************************************/
   /*!
    * \brief   Array subscripting Operator - Immutable(Read only) interface 
    *          Random access
    * \param   [cu32Index]:   (I) Index to be accessed.
    * \retval  [tCU8&]:       Returns the u32Index'th element. 
    * \date 16.11.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tCU8& operator[](tCU32 cu32Index) const;
   
   /***************************************************************************
   ** FUNCTION:  tBool ihl_tclRawData::operator==(const ihl ..) const
   ***************************************************************************/
   /*!
    * \brief   Comparison Operator - "Equal to"
    * \param   [rfcoRawDataRhs]: (I) Const Reference to object to be compared
    * \retval  [tBool]: TRUE if both are equal, FALSE otherwise.
    * \date 16.11.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tBool operator==(const ihl_tclRawData &rfcoRawDataRhs) const;

   /***************************************************************************
   ** FUNCTION:  tBool ihl_tclRawData::operator!=(const ihl ..) const
   ***************************************************************************/
   /*!
    * \brief   Comparison Operator - "Not equal to"
    * \param   [rfcoRawDataRhs]: (I) Const Reference to object to be compared
    * \retval  [tBool]: TRUE if both are unequal, FALSE otherwise.
    * \date 16.11.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tBool operator!=(const ihl_tclRawData &rfcoRawDataRhs) const;

   /***************************************************************************
   ** FUNCTION:  ihl_tclRawData ihl_tclRawData::operator+(co..) const
   ***************************************************************************/
   /*!
    * \brief   Addition Operator
    * \param   [rfcoRawData]: (I) Const Reference to object to be added
    * \retval  [ihl_tclRawData]: Result of addition.
    * \date 16.11.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   ihl_tclRawData operator+(const ihl_tclRawData &rfcoRawData) const;

   /***************************************************************************
   ** FUNCTION:  ihl_tclRawData& ihl_tclRawData::operator+=(...) const
   ***************************************************************************/
   /*!
    * \brief   Assignment by addition Operator
    * \param   [rfcoRawData]: (I) Const Reference to object to be added
    * \retval  [ihl_tclRawData&]: Reference to this pointer.
    * \date 16.11.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   ihl_tclRawData& operator+=(const ihl_tclRawData &rfcoRawData);

   /***************************************************************************
   ** FUNCTION:  tVoid ihl_tclRawData::vClear()
   ***************************************************************************/
   /*!
    * \brief   Erases all of the elements
    * \param   NONE
    * \retval  NONE
    * \date 16.11.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tVoid vClear();

   /***************************************************************************
   ** FUNCTION:  virtual tBool ihl_tclRawData::bIsEmpty() const
   ***************************************************************************/
   /*!
    * \brief   Interface to find if the container is empty
    * \param   NONE
    * \retval  [tBool]: TRUE if the container is empty, FALSE otherwise.
    * \date 16.11.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tBool bIsEmpty() const;

   /***************************************************************************
   ** FUNCTION:  virtual tU32 ihl_tclRawData::u32GetDataSize() const
   ***************************************************************************/
   /*!
    * \brief   Returns the size of the container.
    * \param   NONE
    * \retval  [tU32]: Size of container.
    * \date 16.11.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tU32 u32GetDataSize() const;

   /***************************************************************************
   ** FUNCTION:  virtual tVoid ihl_tclRawData::vResize(tU32 u32Size)
   ***************************************************************************/
   /*!
    * \brief   Inserts or erases elements at the end such that the size becomes
    *          \c u32Size.
    * \param   [cu32Size]:     (I) Size of the final container.
    * \retval  NONE
    * \date 16.11.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tVoid vResize(tCU32 cu32Size);

   /***************************************************************************
   ** FUNCTION:  virtual tBool ihl_tclRawData::bInsert(tCU32 cu32Size, ..
   ***************************************************************************/
   /*!
    * \brief   Insert \c rfcoRawData before position \c u32Index
    * \param   [cu32Index]:   (I) Point of insertion
    * \param   [rfcoRawData]: (I) Raw data buffer
    * \retval  NONE
    * \date 16.11.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tBool bInsert(tCU32 cu32Index, const ihl_tclRawData &rfcoRawData);

   /***************************************************************************
   ** FUNCTION:  virtual tVoid ihl_tclRawData::vInsert(tCU32 cu32Index,..
   ***************************************************************************/
   /*!
    * \brief   Insert \c u8Buffer[] before position \c u32Index
    * \param   [cu32Index]:   (I) Point of insertion
    * \param   [cu8Buffer]:   (->I) Buffer to be inserted
    * \param   [cu32Size]:    (I) Size of the buffer
    * \retval  NONE
    * \date 16.11.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tBool bInsert(tCU32 cu32Index, tCU8 cu8Buffer[], tCU32 cu32Size);

   /***************************************************************************
   ** FUNCTION:  virtual tBool ihl_tclRawData::bGetRawData(tU8 u8Buffe..
   ***************************************************************************/
   /*!
    * \brief   Copy raw data into \c u8Buffer[] 
    * \param   [u8Buffer]:    (->O) Buffer into which the raw data has to be 
    *          copied
    * \param   [cu32MaxSize]: (I) Max size of the buffer
    * \retval  [tBool]: Status of copying.
    * \date 16.11.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tBool bGetRawData(tU8 u8Buffer[], tCU32 cu32MaxSize) const;

   /***************************************************************************
   ** FUNCTION:  tVoid ihl_tclRawData::vInit(tU8 u8Value)
   ***************************************************************************/
   /*!
    * \brief   Initializes all elements in raw data to the value specified.
    * \param   [u8Value]:    (I)   Buffer Initializer value
    * \retval  NONE
    * \date 16.11.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tVoid vInit(tU8 u8Value = 0);

   /***************************************************************************
   ****************************END OF PUBLIC***********************************
   ***************************************************************************/

protected:

   /***************************************************************************
   *******************************PROTECTED************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  tVoid ihl_tclRawData::vClone(const ihl_tclRawData..
   ***************************************************************************/
   /*!
    * \brief   Clone the raw data
    * \param   [rfcoRawData]:  (I) Const Reference to object to be cloned
    * \retval  NONE
    * \date 16.11.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tVoid vClone(const ihl_tclRawData &rfcoRawData);

   /***************************************************************************
   ** FUNCTION:  tPU8 ihl_tclRawData::pu8CreatBuffer(tU32 u32Size) const
   ***************************************************************************/
   /*!
    * \brief   Create a buffer of requested size
    * \param   [u32Size]:  (I) Size to allocate
    * \retval  [tPU8]   Pointer to allocated buffer.
    * \date 16.11.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tPU8 pu8CreatBuffer(tU32 u32Size) const;

   /*!
    * \addtogroup tclMem
    */
   /*! @{*/

   /// Size of the raw data
   tU32 m_u32RawDataSize;
   /// Pointer to the buffer holding the data.
   tPU8 m_pu8RawData;

   /*! @}*/

   /***************************************************************************
   ****************************END OF PROTECTED********************************
   ***************************************************************************/

private:

   /***************************************************************************
   *********************************PRIVATE************************************
   ***************************************************************************/

   /***************************************************************************
   ****************************END OF PRIVATE**********************************
   ***************************************************************************/

}; // class ihl_tclRawData

/******************************************************************************
| typedefs (scope: global)
|----------------------------------------------------------------------------*/

/*! 
 * \typedef ihl_tclRawData ihl_tRawData
 * \brief Exception Safe Raw data Container
 *
 * The objects of this type will be registered for stack unwinding.
 * 
 * \note If you leave a function by exception, only destructors of registered 
 * stack objects will be called. That is contrary to C++ exception handling 
 * where the destructors will be called automatically by a mechanism called 
 * stack unwinding.
 *
 * Ref: Howto Using the Blaupunkt STL, v1.0, by Wiedemann Frank(CM-AI/PJ-CF15)
 * 
 */

typedef ihl_tclRawData ihl_tRawData;

   }  // namespace buffer
}  // namespace ihl

#endif // #ifndef IHL_TCLRAWDATA_H_

////////////////////////////////////////////////////////////////////////////////
// <EOF>
