/************************************************************************
* FILE:        hc_tclThermalSensor_EF.cpp
* PROJECT:
* SW-COMPONENT:fc_heatctrl
*----------------------------------------------------------------------
*
* DESCRIPTION: class of thermal sensor specific emergency functions
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author             | Modification
* 11.02.2015| CM-AI/EPB2 Bernard | init
*
*************************************************************************/

/*******************************************************************************
| includes: system- and project- includes
|-----------------------------------------------------------------------------*/
// Basic OSAL includes
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"         // use Application Help Library

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

/*******************************************************************************
| set own module id
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| includes: needed interfaces from external components
|-----------------------------------------------------------------------------*/


/*******************************************************************************
| includes: internal and external interfaces from this component
|-----------------------------------------------------------------------------*/

#include "hc_tclThermalSensor_EF.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_HEATCTRL_APPLICATION
#include "trcGenProj/Header/hc_tclThermalSensor_EF.cpp.trc.h"
#endif

/*******************************************************************************
| defines and macros (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| typedefs (scope: module-local)
|-----------------------------------------------------------------------------*/


/*******************************************************************************
| variable definition (scope: global)
|-----------------------------------------------------------------------------*/


/*******************************************************************************
| variable definition (scope: module-local)
|-----------------------------------------------------------------------------*/


/*******************************************************************************
| function prototype (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function implementation (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
*
* FUNCTION:    hc_tclThermalSensor()
*
* DESCRIPTION: constructor
*
* PARAMETER:   none
*
* RETURNVALUE: none
*
*******************************************************************************/
hc_tclThermalSensor_EF::hc_tclThermalSensor_EF()
{
   ETG_TRACE_USR4(("hc_tclThermalSensor_EF() entered."));
   (tVoid)memset((tVoid*)m_atEFParameter, OSAL_NULL, sizeof(m_atEFParameter));
}

/*******************************************************************************
*
* FUNCTION:    ~hc_tclThermalSensor()
*
* DESCRIPTION: constructor
*
* PARAMETER:   none
*
* RETURNVALUE: none
*
*******************************************************************************/
hc_tclThermalSensor_EF::~hc_tclThermalSensor_EF()
{
   ETG_TRACE_USR4(("~hc_tclThermalSensor_EF() entered."));
   (tVoid)memset((tVoid*)m_atEFParameter, OSAL_NULL, sizeof(m_atEFParameter));
}

/*******************************************************************************
*
* FUNCTION:    bEFNotify()
*
* DESCRIPTION: Notification for emergency function
*
* PARAMETER:   tenEmergencyFunctionID enFunctionID: emergency function ID
*              tenThermalState enUpperState: upper state to activate emergency
*                                            function
*              tenThermalState enLowerState: lower state to activate emergency
*                                            function
*
* RETURNVALUE: tBool {successful or failed}
*
*******************************************************************************/
tBool hc_tclThermalSensor_EF::bEFNotify(tenEmergencyFunctionID enFunctionID,
		tenThermalState enUpperState, tenThermalState enLowerState)
{
   ETG_TRACE_USR4(("bEFNotify(function: %d, upper state: %d, lower state: %d) entered.",
		   enFunctionID, enUpperState, enLowerState ));

   if( enEmergencyFunctionID_MAX > enFunctionID)
   {
	   m_atEFParameter[enFunctionID].enUpperState = enThermalState_UNAVAILABLE;
	   if (enThermalState_NORMAL < enUpperState && enThermalState_MAX > enUpperState)
		   m_atEFParameter[enFunctionID].enUpperState = enUpperState;
	   m_atEFParameter[enFunctionID].enLowerState = enThermalState_UNAVAILABLE;
	   if (enThermalState_NORMAL > enLowerState && enThermalState_CRITICALLOW <= enLowerState)
		   m_atEFParameter[enFunctionID].enLowerState = enLowerState;
	   return TRUE;
   }
   return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    vUpdate_EmergencyFunctionState()
*
* DESCRIPTION: update all emergency function states depending of last valid state
*
* PARAMETER:   tBool &bChanged: one of emergency function state has been changed
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclThermalSensor_EF::vUpdate_EmergencyFunctionState(tBool &bChanged)
{
   ETG_TRACE_USR4(("vUpdate_EmergencyFunctionState() entered."));

   tenThermalState _enCurState = enGetSensorState();
   tenThermalState _enLastValidState = enGetSensorStateLastValid();
   for (tU8 _u8Index = 0; (tU8)enEmergencyFunctionID_MAX > _u8Index; ++_u8Index)
   {
	   tenEmergencyFunctionState _enState_New = enEmergencyFunctionState_DISABLED;
	   if (enThermalState_UNAVAILABLE != _enCurState)
	   {
		   if ((enThermalState_UNAVAILABLE != m_atEFParameter[_u8Index].enUpperState)
			   || (enThermalState_UNAVAILABLE != m_atEFParameter[_u8Index].enLowerState))
		   {
			   _enState_New = enEmergencyFunctionState_NoResult;
			   if (enThermalState_TEMPUNAVAIL < _enLastValidState && enThermalState_MAX > _enLastValidState)
			   {
				   _enState_New = enEmergencyFunctionState_Inactive;
				   // check upper state
				   if (enThermalState_UNAVAILABLE != m_atEFParameter[_u8Index].enUpperState)
					   if (m_atEFParameter[_u8Index].enUpperState <= _enLastValidState)
						   _enState_New = enEmergencyFunctionState_ActiveHigh;
				   // check lower state
				   if (enThermalState_UNAVAILABLE != m_atEFParameter[_u8Index].enLowerState)
					   if (m_atEFParameter[_u8Index].enLowerState >= _enLastValidState)
						   _enState_New = enEmergencyFunctionState_ActiveLow;
			   }
		   }
	   }
	   if (_enState_New != m_atEFParameter[_u8Index].enFunctionState)
		   bChanged = TRUE;
	   m_atEFParameter[_u8Index].enFunctionState = _enState_New;
   }
   ETG_TRACE_USR4(("vUpdate_EmergencyFunctionState(changed: %d) exit.", bChanged ));
}

/*******************************************************************************
*
* FUNCTION:    bGetNewEmergencyFunctionState()
*
* DESCRIPTION: getting the new emergency function state information
*
* PARAMETER:   tEmergencyFunctionInfo_Internal &FunctionStates
*
* RETURNVALUE: tBool		{successful or failed}
*
*******************************************************************************/
tBool hc_tclThermalSensor_EF::bGetNewEmergencyFunctionState(tEmergencyFunctionInfo_Internal &FunctionStates) const
{
   ETG_TRACE_USR4(("bGetNewEmergencyFunctionState() entered."));

   for (tU8 _u8Index = 0; (tU8)enEmergencyFunctionID_MAX > _u8Index; ++ _u8Index)
   {
	   FunctionStates.aenFunctionState[_u8Index] = m_atEFParameter[_u8Index].enFunctionState;
   }
   return TRUE;
}

/*******************************************************************************
*
* FUNCTION:    bIsEmergencyFunctionActive()
*
* DESCRIPTION: returns TRUE if the state of specified emergency function active
*
* PARAMETER:   tenEmergencyFunctionID enFunction
*
* RETURNVALUE: tBool		{active or not}
*
*******************************************************************************/
tBool hc_tclThermalSensor_EF::bIsEmergencyFunctionActive(tenEmergencyFunctionID enFunction) const
{
   ETG_TRACE_USR4(("bIsEmergencyFunctionActive() entered."));

   if ((tenEmergencyFunctionID)(sizeof(m_atEFParameter)/sizeof(m_atEFParameter[0])) > enFunction)
   {
	   return ((enEmergencyFunctionState_Active <= m_atEFParameter[enFunction].enFunctionState)
			   && (enEmergencyFunctionState_ActiveLow >= m_atEFParameter[enFunction].enFunctionState));
   }
   return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    vTraceInfo_EF()
*
* DESCRIPTION: trace status info
*
* PARAMETER:   	const char* strIdentifier		trace identifier
*                                               like "HC_APPL.TSEN.GYRO_.ATTR_"
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclThermalSensor_EF::vTraceInfo_EF(const char* strIdentifier)
{
   ETG_TRACE_USR4(("vTraceInfo_EF() entered."));

   for (tU8 _u8Index = 0; (tU8)enEmergencyFunctionID_MAX > _u8Index; ++ _u8Index)
   {
	   if (enThermalState_UNAVAILABLE != m_atEFParameter[_u8Index].enUpperState
		   || enThermalState_UNAVAILABLE != m_atEFParameter[_u8Index].enLowerState
		   || enEmergencyFunctionState_DISABLED != m_atEFParameter[_u8Index].enFunctionState )
	   {
		   tChar _strAttrName[20] = "";
		   (tVoid)sprintf(_strAttrName, "      EF_%5s_", strGetEmergencyFunctionAbbreviation((tenEmergencyFunctionID)_u8Index));

		   ETG_TRACE_USR2(("%26s: %20s%10s=%25s ", strIdentifier, _strAttrName, "_NAME",
				   strGetEmergencyFunctionID((tenEmergencyFunctionID)_u8Index) ));
		   ETG_TRACE_USR2(("%26s: %20s%10s=%25s ", strIdentifier, _strAttrName, "UPPER",
				   strGetThermalState(m_atEFParameter[_u8Index].enUpperState) ));
		   ETG_TRACE_USR2(("%26s: %20s%10s=%25s ", strIdentifier, _strAttrName, "LOWER",
				   strGetThermalState(m_atEFParameter[_u8Index].enLowerState) ));
		   ETG_TRACE_USR2(("%26s: %20s%10s=%25s ", strIdentifier, _strAttrName, "STATE",
				   strGetEmergencyFunctionState(m_atEFParameter[_u8Index].enFunctionState) ));
	   }
   }
}
